package edu.jhu.ece.iacl.algorithms.graphics.map;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import no.uib.cipr.matrix.Matrix;
import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.map.HemisphericalMap.WeightVectorFunc;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * Weighting function for uniform or Tuette weights. These weights are good for
 * initializing spherical parameterizations since they move the parameterization
 * away from degenerate solutions. However, they are a function of the mesh's
 * parameterization and not indicative of underlying mesh geometry.
 * 
 * @author Blake Lucas
 * 
 */
public class TuetteWeightFunc implements WeightVectorFunc {
	EmbeddedSurface surf;
	int[][] neighborVertexVertexTable;
	int[] labels;

	public TuetteWeightFunc(ProgressiveSurface ps, int[] labels) {
		this.surf = ps.surf;
		this.labels = labels;
		this.neighborVertexVertexTable = ps.neighborVertexVertexTable;
	}

	public double populate(Matrix Ax, Matrix Ay, int[] lookupTable,
			int[] revLookupTable, int id) {
		int origIndex = lookupTable[id];
		int len = neighborVertexVertexTable[origIndex].length;
		double curv = 1;// weights[id];
		if (len == 0)
			return 0;
		if (labels[origIndex] != 0) {
			int nbr;
			double w = curv * 1.0 / len;
			for (int i = 0; i < len; i++) {
				nbr = revLookupTable[neighborVertexVertexTable[origIndex][i]];
				Ax.set(id, nbr, -w);
				Ay.set(id, nbr, -w);
			}
		}
		Ax.set(id, id, 1);
		Ay.set(id, id, 1);

		return curv;
	}

	public double populate(Matrix Ar, int[] lookupTable, int[] revLookupTable,
			int id) {
		int origIndex = lookupTable[id];
		int len = neighborVertexVertexTable[origIndex].length;
		Point3f pivot = surf.getVertex(origIndex);
		int origNbr;
		if (len == 0)
			return 0;
		if (labels[origIndex] != 0) {
			int nbr;
			double w = 1.0 / len;
			for (int i = 0; i < len; i++) {
				nbr = revLookupTable[origNbr = neighborVertexVertexTable[origIndex][i]];
				Point3f pt = surf.getVertex(origNbr);
				Ar.set(id, nbr, -w * GeometricUtilities.dot(pivot, pt));
			}
		}
		Ar.set(id, id, 1);

		return 1;
	}

	public double populate(Matrix Ax, Matrix Ay, Matrix Az, int[] lookupTable,
			int[] revLookupTable, int id) {
		int origIndex = lookupTable[id];
		int len = neighborVertexVertexTable[origIndex].length;
		double curv = 1;// weights[id];
		if (len == 0)
			return 0;
		int nbr;
		if (labels[origIndex] != 0) {
			double w = 1 / (double) len;
			for (int i = 0; i < len; i++) {
				nbr = revLookupTable[neighborVertexVertexTable[origIndex][i]];
				Ax.add(id, nbr, -w);
				Ay.add(id, nbr, -w);
				Az.add(id, nbr, -w);
			}
		}
		Ax.set(id, id, 1);
		Ay.set(id, id, 1);
		Az.set(id, id, 1);
		return 1;
	}

	public Vector3f getResidualVector(int id) {
		int len = neighborVertexVertexTable[id].length;
		Point3f pivot = surf.getVertex(id);
		Vector3f umb = new Vector3f();
		if (len == 0)
			return umb;
		for (int i = 0; i < len; i++) {
			umb.add(surf.getVertex(neighborVertexVertexTable[id][i]));
		}
		umb.scale(1.0f / len);
		umb.sub(pivot);
		return umb;
	}
}
