package edu.jhu.ece.iacl.algorithms.graphics.smooth;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.XuGraphicsWrapper;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface.Edge;

/**
 * Smooth vertex data that is embedded in surface using different kernel types.
 * 
 * @author Blake Lucas
 * 
 */
public class SmoothSurfaceQuantity extends XuGraphicsWrapper {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.5 $");
	}
	

	public SmoothSurfaceQuantity(AbstractCalculation parent, EmbeddedSurface surf) {
		super(parent, surf, true);
		setLabel("Surface Metrics");
	}

	public SmoothSurfaceQuantity(EmbeddedSurface surf) {
		super(surf, true);
		setLabel("Surface Metrics");
	}	

	protected int maxIters = 1000;
	protected double convergence = 1E-6;
	protected Method method = Method.MEAN;
	protected int maxLayers = 1;
	protected double sigma = 1.0;
	protected boolean logSpace = false;
	protected int index = 0;

	public enum Method {
		MEAN, GAUSSIAN, MIN, MAX
	};

	public void setUseLogSpace(boolean logSpace) {
		this.logSpace = logSpace;
	}

	public void setMaxIters(int maxIters) {
		this.maxIters = maxIters;
	}

	public void setConvergence(double convergence) {
		this.convergence = convergence;
	}

	public void setMethod(Method method) {
		this.method = method;
	}

	public void setSigma(double sigma) {
		this.sigma = sigma;
	}

	public void setIndex(int index) {
		this.index = index;
	}
	
	public void setMaxLayers(int maxLayers) {
		this.maxLayers = maxLayers;
	}
	
	public SmoothSurfaceQuantity() {
		setLabel("Smooth Surface Quantity");
	}
	
	int NBR_MAX_NUM = 5000;
	/**
	 * Smooth vertex data embedded in surface
	 * @param origSurf surface
	 * @param index offset into vertex data
	 * @return surface with smoothed vertex data
	 */
	public EmbeddedSurface solve() {
		EmbeddedSurface smthSurf = surf.clone();
		double[][] vertData = surf.getVertexData();
		double[][] newVertData = new double[vertData.length][1];

		if (logSpace) {
			for (int i = 0; i < vertData.length; i++) {
				newVertData[i][index] = Math.log(1E-10 + Math
						.abs(vertData[i][index]));
			}
		} else {
			for (int i = 0; i < vertData.length; i++) {
				newVertData[i][index] = vertData[i][index];
			}
		}
		
		double value, w, wsum = 0, min, max;
		int vertCount = vertData.length;
	
		// modified by clara, Mar 16 2012
		// average neighbors (with specified number of layers), one iteration
		int[] nbrVtxNum = new int[1];
		int[] nbrPolyNum = new int[1];
		int[] nbrVtxID = new int[NBR_MAX_NUM];
		int[] nbrPolyID = new int[NBR_MAX_NUM];
        int nbr;

		min = 1E30;
		max = -1E30;
		for (int i = 0; i < vertCount; i++) {
			Point3f pt = surf.getVertex(i);
			GetNbr(i, nbrVtxID, nbrPolyID, nbrVtxNum, nbrPolyNum, maxLayers);

			
			value = 0;
			min = Math.min(min, value);
			max = Math.max(max, value);
			switch (method) {
			case MEAN:
				wsum = nbrVtxNum[0];
				break;
			case GAUSSIAN:
				wsum = 0;
				break;
			case MIN:
				break;
			case MAX:
				break;
			}
			
			for (int inbr = 0; inbr < nbrVtxNum[0]; inbr++) {
				nbr = nbrVtxID[inbr];
				switch (method) {
				case MEAN:
					value += vertData[nbr][index];
					break;
				case GAUSSIAN:
					w = Math.exp(-0.5
							* Math.pow(pt.distance(surf.getVertex(nbr)), 2)
							/ (sigma * sigma));
					value += w * vertData[nbr][index];
					wsum += w;
					break;
				case MIN:
					value = Math.min(value, vertData[nbr][index]);
					break;
				case MAX:
					value = Math.max(value, vertData[nbr][index]);
					break;
				}
			}
			switch (method) {
			case MEAN:
				if (wsum > 0)
					value /= wsum;
				break;
			case GAUSSIAN:
				if (wsum > 0)
					value /= wsum;
				break;
			case MIN:
				break;
			case MAX:
				break;
			}
			newVertData[i][index] = value;
		}

		if (logSpace) {
			for (int i = 0; i < vertData.length; i++) {
				newVertData[i][index] = Math.exp(newVertData[i][index]);
			}
		}
		smthSurf.setVertexData(newVertData);
		markCompleted();
		return smthSurf;
	}

	/*
	 * This routine is used to find the nearest Layer layers of neighbor of
	 * vertex CurrentVtxID. The neighboring vertex IDs are stored in VtxID and
	 * the neighboring polygon IDs are in PolyID. The VtxID is not ordered
	 * except the immidiate neighbors. Neither is PolyID.
	 */
	public void GetNbr(int CurrentVtxID, int[] VtxID, int[] PolyID, int[] VNum,
			int[] PNum, int Layer) {

		int i, j, k;
		int Vtx0, Poly0;
		int VtxNum;
		int VtxAdded;
		int v0, v1, p0, p1;
		Edge[] nb;

		Vtx0 = 0;
		Poly0 = 0;
		VtxAdded = 0;
		VtxID[Vtx0] = CurrentVtxID;
		VtxNum = 1;
		for (i = 0; i < Layer; i++) {
			VtxAdded = 0;
			for (j = Vtx0; j < VtxNum; j++) {
				nb = XUGetNeighbor(VtxID[j]);
				for (k = 0; k < nb.length; k++) {
					v0 = edges[nb[k].id].v1;
					v1 = edges[nb[k].id].v2;
					p0 = edges[nb[k].id].f1;
					p1 = edges[nb[k].id].f2;

					if (!IsInSet(VtxID, VtxNum + VtxAdded, v0)) {
						VtxID[VtxNum + VtxAdded] = v0;
						VtxAdded++;
					}
					if (!IsInSet(VtxID, VtxNum + VtxAdded, v1)) {
						VtxID[VtxNum + VtxAdded] = v1;
						VtxAdded++;
					}

					if (!IsInSet(PolyID, Poly0, p0))
						PolyID[Poly0++] = p0;

					if (!IsInSet(PolyID, Poly0, p1))
						PolyID[Poly0++] = p1;

				}
			}
			Vtx0 = VtxNum;
			VtxNum += VtxAdded;
		}
		VNum[0] = VtxNum;
		PNum[0] = Poly0;
		return;
	}

	protected boolean IsInSet(int[] set, int N, int ID) {
		int k;

		if (N == 0)
			return false;

		for (k = 0; k < N; k++) {
			if (ID == set[k])
				return true;
		}
		return false;
	}


}



