package edu.jhu.ece.iacl.algorithms.graphics.utilities;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.utilities.rbf.RadialBasisFuncFloat4D;
import edu.jhu.ece.iacl.algorithms.graphics.utilities.rbf.RadialFunctionType;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * Compute the gradient of a quantity defined on a surface using radial basis
 * functions. This is a terribly inefficient implementation.
 * 
 * @author Blake Lucas
 * 
 */
public class SurfaceQuantityGradient extends AbstractCalculation {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.4 $");
	}

	RadialFunctionType rbfType;

	public enum GradientComponent {
		GRADIENT, NORMAL, TANGET
	};

	public SurfaceQuantityGradient(RadialFunctionType rbfType) {
		this.rbfType = rbfType;
		setLabel("Surface Quantity Graident");
	}

	public double[][] gradient(EmbeddedSurface surf, int index,
			GradientComponent comp, boolean normalize) {
		double[][] vertData = surf.getVertexData();
		int vertCount = vertData.length;
		int[][] nbrTable = EmbeddedSurface.buildNeighborVertexVertexTable(surf,
				EmbeddedSurface.Direction.COUNTER_CLOCKWISE);
		Point3f[] nbrPoints;
		double[] data;
		double[][] gradient = new double[vertCount][3];
		setTotalUnits(vertCount);
		for (int i = 0; i < vertCount; i++) {
			int[] nbrs = nbrTable[i];
			nbrPoints = new Point3f[nbrs.length + 1];
			data = new double[nbrPoints.length + 1];
			for (int j = 0; j < nbrs.length; j++) {
				int nbr = nbrs[j];
				nbrPoints[j] = surf.getVertex(nbr);
				data[j] = vertData[nbr][index];
			}
			data[nbrs.length] = vertData[i][index];
			nbrPoints[nbrs.length] = surf.getVertex(i);
			RadialBasisFuncFloat4D rbf = new RadialBasisFuncFloat4D(nbrPoints,
					data, rbfType);
			Vector3f grad = rbf.interpolateGradient(nbrPoints[nbrs.length]);

			if (comp == GradientComponent.NORMAL) {
				Vector3f norm = surf.getNormal(i);
				norm.scale(grad.dot(norm));
			}
			if (comp == GradientComponent.TANGET) {
				Vector3f norm = surf.getNormal(i);
				norm.scale(grad.dot(norm));
				grad.sub(norm);
			}
			if (normalize) {
				GeometricUtilities.normalize(grad);
			}
			gradient[i][0] = grad.x;
			gradient[i][1] = grad.y;
			gradient[i][2] = grad.z;
			incrementCompletedUnits();
		}
		markCompleted();
		return gradient;
	}
}
