package edu.jhu.ece.iacl.algorithms.graphics.utilities.rbf;

import Jama.*;
import javax.vecmath.Point2f;
import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector3f;
/**
 * Use radial basis functions to interpolate w position for a function w=f(x,y,z)
 * where f(.) is known for several positions.
 * 
 * @author Blake Lucas
 * 
 */
public class RadialBasisFuncDouble4D extends RadialBasisFunctionDouble{
	protected Matrix Ainv;
	public RadialBasisFuncDouble4D(Point3d[] points,RadialFunctionType func){
		this.radialFunc=func;
	}
	public RadialBasisFuncDouble4D(RadialFunctionType func){
		this.radialFunc=func;
	}
	public void init(Point3d[] points){
		this.points=points;
		computeCoefficients();		
	}
	public double interpolate(Point3d p,double[] vals){
		double ret=0;
		Matrix b=new Matrix(points.length+4,1);
		for(int i=0;i<points.length;i++){
			b.set(i,0,vals[i]);
		}
		for(int i=0;i<4;i++){
			b.set(points.length+i,0,0);
		}
		Matrix coeff=Ainv.times(b);
		chi=new double[4];
		lambda=new double[points.length];
		chi[0]=coeff.get(points.length,0);
		chi[1]=coeff.get(points.length+1,0);
		chi[2]=coeff.get(points.length+2,0);
		chi[3]=coeff.get(points.length+3,0);
		ret+=chi[0]+chi[1]*p.x+chi[2]*p.y+chi[3]*p.z;
		for(int i=0;i<points.length;i++){
			lambda[i]=coeff.get(i,0);
			ret+=lambda[i]*evalFunc(p,points[i]);
		}
		return ret;
	}
	protected void computeCoefficients(){
		Matrix A=new Matrix(points.length+4,points.length+4);
		double d;
		//Populate s(x) entries
		for(int i=0;i<points.length;i++){
			Point3d p=points[i];
			A.set(i,points.length,1);
			A.set(i,points.length+1,p.x);
			A.set(i,points.length+2,p.y);
			A.set(i,points.length+3,p.z);
			A.set(points.length, i, 1);
			A.set(points.length+1, i, p.x);
			A.set(points.length+2, i, p.y);
			A.set(points.length+3, i, p.z);
			for(int j=0;j<points.length;j++){
				d=evalFunc(p,points[j]);
				A.set(i, j, d);
			}
		}
		for(int i=0;i<4;i++){
			for(int j=0;j<4;j++){
				A.set(points.length+i,points.length+j,0);
			}
		}
		SingularValueDecomposition svd=new SingularValueDecomposition(A);
		Matrix S=svd.getS();
		Matrix V=svd.getV();
		Matrix U=svd.getU();
		int zeros=0;
		for(int i=0;i<S.getColumnDimension();i++){
			if(Math.abs(S.get(i, i))>1E-12){
				S.set(i, i, 1/S.get(i, i));
			} else {
				zeros++;
				S.set(i,i,0);
			}
		}
		Ainv=V.times(S.times(U.transpose()));
		
	}
	public double evalFunc(Point3d p1,Point3d p2){
		double r=(double)Math.sqrt((p1.x-p2.x)*(p1.x-p2.x)+(p1.y-p2.y)*(p1.y-p2.y)+(p1.z-p2.z)*(p1.z-p2.z));
		switch(radialFunc){
			case LINEAR:return r;
			case THIN_PLATE:return (double)((r!=0)?r*r*Math.log(r):0);
			case GAUSSIAN:return (double)Math.exp(-alpha*r*r);
			case MULTIQUADRIC:return (double)Math.sqrt((r*r+C*C));
			default: return 0;
		}
	}
	public Vector3d evalFuncGradient(Point3d p1,Point3d p2){
		double r=(double)Math.sqrt((p1.x-p2.x)*(p1.x-p2.x)+(p1.y-p2.y)*(p1.y-p2.y)+(p1.z-p2.z)*(p1.z-p2.z));
		double rx=(r!=0)?(p1.x-p2.x)/r:0;
		double ry=(r!=0)?(p1.y-p2.y)/r:0;
		double rz=(r!=0)?(p1.z-p2.z)/r:0;

		double d=0;
		switch(radialFunc){
			case LINEAR:
				d=1;break;
			case THIN_PLATE:
				d=(double)((r>0)?2*r*Math.log(Math.abs(r))+r:0);break;
			case GAUSSIAN:
				d=(double)(-2*alpha*r*Math.exp(-alpha*r*r));break;
			case MULTIQUADRIC:
				d=(double)(r/Math.sqrt((r*r+C*C)));break;
			default:
		}
		return new Vector3d(d*rx,d*ry,d*rz);
	}
}
