package edu.jhu.ece.iacl.algorithms.thickness;

import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.thickness.grid.SimpleThicknessNode;
import edu.jhu.ece.iacl.jist.structures.data.BinaryMinHeap;
/**
 * Compute thickness using Yezzi, Prince Eulerian fast-marching method
 * @author Blake Lucas (bclucas@jhu.edu)
 *
 */
public class EulerThickness extends ThicknessSolver{
	public EulerThickness(double lambda,double isoVal,double lagrangeDeltaT){
		super(lambda,isoVal,lagrangeDeltaT);
		setLabel("Eulerian Thickness Solver");
	}
	protected void computeThickness() {
			byte[] nbhdX=neighborMask.getNeighborsX();
			byte[] nbhdY=neighborMask.getNeighborsY();
			byte[] nbhdZ=neighborMask.getNeighborsZ();
			int i,j,k,l,ni,nj,nk;
			SimpleThicknessNode cd;
			
			setTotalUnits(2*corticalData.length);
			setLabel("Computing Inner Length");
			int count=0;
			double len;
			initializeBoundary(INSIDE_TAG);
			while(heap.size()>0){
				SimpleThicknessNode data=(SimpleThicknessNode)heap.remove();
				//System.out.println("OUT "+data.getLength()+" "+data.getLocation());

				data.label=SOLVED;
				incrementCompletedUnits();
				i=data.i;
				j=data.j;
				k=data.k;
				for(l=0;l<nbhdX.length;l++){
					ni=i+nbhdX[l];
					nj=j+nbhdY[l];
					nk=k+nbhdZ[l];
					cd=getData(ni,nj,nk);
					//System.out.println(cd+" "+ni+","+nj+","+nk);
					if(cd!=null){
						if(cd.label==VISITED){
							len=solveInnerLength(ni,nj,nk,data);
							heap.change(cd,len);
						} else if(cd.label==UNVISITED){
							cd.setLength(solveInnerLength(ni,nj,nk,data));
							cd.label=VISITED;
							heap.add(cd);
						}
					}
				}
			}
			markCompleted();
			initializeBoundary(OUTSIDE_TAG);
			setLabel("Computing Outer Length");
			while(heap.size()>0){
				SimpleThicknessNode data=(SimpleThicknessNode)heap.remove();
				//System.out.println("OUT "+data.getLength()+" "+data.getLocation());
				data.label=SOLVED;
				incrementCompletedUnits();
				i=data.i;
				j=data.j;
				k=data.k;
				for(l=0;l<nbhdX.length;l++){
					ni=i+nbhdX[l];
					nj=j+nbhdY[l];
					nk=k+nbhdZ[l];
					cd=getData(ni,nj,nk);
					if(cd!=null){
						if(cd.label==VISITED){
							len=solveOuterLength(ni,nj,nk,data);
							heap.change(cd,len);
						} else if(cd.label==UNVISITED){
							cd.setLength(solveOuterLength(ni,nj,nk,data));
							cd.label=VISITED;
							heap.add(cd);
						}
					}
				}
			}
			markCompleted();
			/*
			for(CorticalThicknessData c:corticalData){
				System.out.println("OUTER "+c.getLength());
			}
			*/
	}

	/**
	 * Compute distance from outer boundary
	 * @param i i 
	 * @param j j
	 * @param k k
	 * @param parent parent node
	 * @return distance from outer boundary
	 */
	protected double solveOuterLength(int i,int j,int k,SimpleThicknessNode parent){
		int ai,aj,ak;
		double tx,ty, tz;
		float[] tan=tangetFieldMat[i][j][k];
		double numer=1;
		double denom=0;
		if(tan[0]<0){
			tx=-tan[0];ai=i-1;
		} else {
			tx=tan[0];ai=i+1;
		}

		if(tan[1]<0){
			ty=-tan[1];aj=j-1;
		} else {
			ty=tan[1];aj=j+1;
		}
		if(tan[2]<0){
			tz=-tan[2];ak=k-1;
		} else {
			tz=tan[2];ak=k+1;
		}
		if(getLabel(ai,j,k)!=UNVISITED){
			numer+=tx*getLength(ai,j,k);
			denom+=tx;
		}
		if(getLabel(i,aj,k)!=UNVISITED){
			numer+=ty*getLength(i,aj,k);
			denom+=ty;
		}
		if(getLabel(i,j,ak)!=UNVISITED){
			numer+=tz*getLength(i,j,ak);
			denom+=tz;
		}
		double len=numer/denom;
		if(Double.isNaN(len)||Double.isNaN(len)||Math.abs(len-parent.getLength())>MAX_LENGTH||len<parent.getLength()){
			Vector3f t =new Vector3f(tan);
			t.add(new Vector3f(tangetFieldMat[parent.i][parent.j][parent.k]));
			t.scale(0.5f);
			Vector3f v=new Vector3f(parent.i-i,parent.j-j,parent.k-k);
			float d=t.dot(v);
			if(d>0){
				return d+parent.getLength();
			} else {
				return parent.getLength()+v.length();
			}
		} else {
			return len;
		}
	}
	/**
	 * Compute distance from inner boundary
	 * @param i i 
	 * @param j j
	 * @param k k
	 * @param parent parent node
	 * @return distance from inner boundary
	 */
	protected double solveInnerLength(int i,int j,int k,SimpleThicknessNode parent){
		int ai,aj,ak;
		double tx,ty, tz;
		float[] tan=tangetFieldMat[i][j][k];
		double numer=1;
		double denom=0;
		if(tan[0]<0){
			tx=-tan[0];ai=i+1;
		} else {
			tx=tan[0];ai=i-1;
		}
		if(tan[1]<0){
			ty=-tan[1];aj=j+1;
		} else {
			ty=tan[1];aj=j-1;
		}
		if(tan[2]<0){
			tz=-tan[2];ak=k+1;
		} else {
			tz=tan[2];ak=k-1;
		}
		if(getLabel(ai,j,k)!=UNVISITED){
			numer+=tx*getLength(ai,j,k);
			denom+=tx;
		}
		if(getLabel(i,aj,k)!=UNVISITED){
			numer+=ty*getLength(i,aj,k);
			denom+=ty;
		}
		if(getLabel(i,j,ak)!=UNVISITED){
			numer+=tz*getLength(i,j,ak);
			denom+=tz;
		}
		double len=numer/denom;
		if(Double.isNaN(len)||Double.isNaN(len)||Math.abs(len-parent.getLength())>MAX_LENGTH||len<parent.getLength()){
			Vector3f t =new Vector3f(tan);
			t.add(new Vector3f(tangetFieldMat[parent.i][parent.j][parent.k]));
			t.scale(0.5f);
			Vector3f v=new Vector3f(i-parent.i,j-parent.j,k-parent.k);
			float d=t.dot(v);
			if(d>0){
				return d+parent.getLength();
			} else {
				return parent.getLength()+v.length();
			}
		} else {
			return len;
		}		
	}
	/**
	 * Initialize boundary 
	 * @param boundary {INSIDE_TAG,OUTSIDE_TAG}
	 */
	protected void initializeBoundary(int boundary){
		int n;
		int ni,nj,nk;
		int i,j,k;
		byte[] nbhdX=neighborMask.getNeighborsX();
		byte[] nbhdY=neighborMask.getNeighborsY();
		byte[] nbhdZ=neighborMask.getNeighborsZ();
		heap=new BinaryMinHeap(corticalData.length,rows,cols,slices);
		SimpleThicknessNode.setCompareInnerLength((boundary==INSIDE_TAG));
		for(SimpleThicknessNode cd:corticalData){
			cd.label=UNVISITED;
		}
		for(SimpleThicknessNode cd:corticalData){
			if(cd.label==UNVISITED){
				i=cd.i;j=cd.j;k=cd.k;
				for(int l=0;l<nbhdX.length;l++){
					ni=i+nbhdX[l];
					nj=j+nbhdY[l];
					nk=k+nbhdZ[l];
					if(ni<0||nj<0||nk<0||ni>=rows||nj>=cols||nk>=slices)continue;
					if(tags[ni][nj][nk]==boundary){
						cd.setLength(0);
						cd.label=VISITED;
						heap.add(cd);
						break;
					}
				}
			}
		}
	}

}
