package edu.jhu.ece.iacl.algorithms.thickness;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.isosurf.IsoSurfaceOnGrid;
import edu.jhu.ece.iacl.algorithms.graphics.map.CoupleSphericallyMappedSurfaces;
import edu.jhu.ece.iacl.algorithms.graphics.map.SphericalMapCorrection;
import edu.jhu.ece.iacl.algorithms.graphics.map.SurfaceToComplex;
import edu.jhu.ece.iacl.algorithms.tgdm.GenericTGDM;
import edu.jhu.ece.iacl.algorithms.thickness.grid.BoundaryNode;
import edu.jhu.ece.iacl.algorithms.thickness.grid.EmbeddedGrid;
import edu.jhu.ece.iacl.algorithms.thickness.grid.EmbeddedNode;
import edu.jhu.ece.iacl.algorithms.thickness.laplace.LaplaceSolveOnGraph;
import edu.jhu.ece.iacl.algorithms.thickness.laplace.LaplaceSolveOnGraphDirichlet;
import edu.jhu.ece.iacl.algorithms.topology.ConnectivityRule;
import edu.jhu.ece.iacl.jist.structures.data.BinaryMinHeap;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataDouble;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;

import Jama.*;

/**
 * Compute locations of correspondence points in an inter-surface conformal map and then measure
 * thickness as the euclidian distance between corresponding points
 * @author Blake Lucas (bclucas@jhu.edu)
 *
 */
public class RobustInterSurfaceConformalMapEuler extends SimpleLagrangianEmbedding implements RobustInterSurfaceMap{
	protected int intSurfs;
	protected boolean mapCoordinates=true;
	protected float smoothFactor;
	protected ImageData centralVol;
	protected double phongAlpha=5;

	protected float levelSetSpacing=1;
	protected float alpha=0.5f;
	protected final static float angle_thresh_tan=(float)(89*Math.PI/180);
	protected final static float angle_thresh_grad=(float)(89*Math.PI/180);
	protected EmbeddedSurface initInnerMap,initOuterMap;
	public RobustInterSurfaceConformalMapEuler(double lambda, double isoVal,double lagrangeDeltaT) {
		super(lambda, isoVal,lagrangeDeltaT);

		this.setLabel("Extended Lagrangian Embedding");
		// TODO Auto-generated constructor stub
	}
	public RobustInterSurfaceConformalMapEuler() {
		super();
		this.setLabel("Extended Lagrangian Embedding");
		// TODO Auto-generated constructor stub
	}
	protected void computeThickness() {
		int total=0;
		int mult=0;
		if(mapOuterToInner)mult++;
		if(mapInnerToOuter)mult++;
		setTotalUnits(mult * (grid.getValidNodeCount()));
		double len;
		if(!EmbeddedNode.isCompareToInner()){
			heap = new BinaryMinHeap(grid.getGridNodes().size()
					+ grid.getBoundaryNodes().length, rows, cols, slices);

			initializeBoundary(OUTSIDE_OUTER);
			while (heap.size() >0) {
				EmbeddedNode node = (EmbeddedNode) heap.remove();
				total++;
				node.setMarchingLabel(SOLVED);
				for (EmbeddedNode neighbor:node.getNeighbors()) {
					if (neighbor.getMarchingLabel() == VISITED) {
						updateNode(neighbor,node);
					} else if (neighbor.getMarchingLabel() == UNVISITED) {
						len=updateNode(neighbor,node);
						if(neighbor.isInnerBoundary()){
							neighbor.setMarchingLabel(SOLVED);	
							incrementCompletedUnits();	
						} else {
							heap.add(neighbor);
							neighbor.setMarchingLabel(VISITED);	
						} 
						        
					
					} 
				}
			}
			markCompleted();
			System.gc();
		}
		if(EmbeddedNode.isCompareToInner()){
			heap = new BinaryMinHeap(grid.getGridNodes().size()
					+ grid.getBoundaryNodes().length, rows, cols, slices);
			initializeBoundary(INSIDE_INNER);
			while (heap.size() >0) {
				EmbeddedNode node = (EmbeddedNode) heap.remove();
				total++;
				node.setMarchingLabel(SOLVED);
				incrementCompletedUnits();	
				for (EmbeddedNode neighbor:node.getNeighbors()) {
					if (neighbor.getMarchingLabel() == VISITED) {
						updateNode(neighbor,node);
					} else	if(neighbor.getMarchingLabel() == UNVISITED) {
						len=updateNode(neighbor,node);
						if(neighbor.isOuterBoundary()){
							neighbor.setMarchingLabel(SOLVED);	
							incrementCompletedUnits();
						} else {
							heap.add(neighbor);
							neighbor.setMarchingLabel(VISITED);	
						} 
					} 
				}
			}
			markCompleted();
		}
		surfaceIntersector=null;
		heap=null;
		System.gc();

	}
	ArrayList<EmbeddedSurface> surfs;
	ArrayList<ImageDataFloat> levelsets;
	public ArrayList<EmbeddedSurface> getSurfaces(){
		return surfs;
	}
	public ArrayList<ImageDataFloat> getLevelSets(){
		return levelsets;
	}
	protected int maxNonHarmonic=0;
	protected int decimationAmount=0;
	protected int maxVertCount=0;
	public int getMaxNonHarmonic(){
		return maxNonHarmonic;
	}
	public int getDecimationAmount(){
		return decimationAmount;
	}
	public int getMaxVertexCount(){
		return maxVertCount;
	}
	protected Vector3f computeTanget(EmbeddedNode node){
		
		List<EmbeddedNode> upwindNodes=new LinkedList<EmbeddedNode>();
		double up=node.getLength();
		for(EmbeddedNode nd:node.getNeighbors()){
			if(!nd.isOutside()){
				upwindNodes.add(nd);
			}
		}
		Matrix D=new Matrix(upwindNodes.size(),3);
		Matrix W=new Matrix(upwindNodes.size(),upwindNodes.size());
		Matrix GU=new Matrix(upwindNodes.size(),1);
		Point3f pivot=node.getLocation();
		Vector3f diff=new Vector3f();
		double w;
		
		for(int i=0;i<upwindNodes.size();i++){
			EmbeddedNode n=upwindNodes.get(i);
			diff.sub(pivot,n.getLocation());
			D.set(i, 0, diff.x);
			D.set(i, 1, diff.y);
			D.set(i, 2, diff.z);
			w=diff.length();
			w=(w>1E-6)?1/w:1E6;
			W.set(i, i, w);
			GU.set(i,0, up-n.getLength());
		}	
		Matrix Dsqr=D.transpose().times(W).times(D);
		SingularValueDecomposition svd=new SingularValueDecomposition(Dsqr);
		Matrix S=svd.getS();
		Matrix V=svd.getV();
		Matrix U=svd.getU();
		int zeros=0;
		for(int i=0;i<S.getColumnDimension();i++){
			if(Math.abs(S.get(i, i))>1E-12){
				S.set(i, i, 1/S.get(i, i));
			} else {
				zeros++;
				S.set(i,i,0);
			}
		}
		Matrix Dinv=V.times(S.times(U.transpose())).times(D.transpose()).times(W);
		Matrix T=Dinv.times(GU);
		Vector3f tan=new Vector3f((float)T.get(0,0),(float)T.get(1,0),(float)T.get(2,0));
		GeometricUtilities.normalize(tan);
		return tan;
	}
	protected void computeLaplace(boolean computeInner,float level) {		
		LaplaceSolveOnGraph laplaceSolver=new LaplaceSolveOnGraphDirichlet(this,rows,cols,slices);
		tangetField=new ImageDataFloat(rows,cols,slices,3);
		tangetFieldMat=tangetField.toArray4d();
		laplaceSolver.solve(grid);
		ImageDataDouble laplaceVol=laplaceSolver.getLaplaceVolume();
		double[][][] laplaceMat=laplaceVol.toArray3d();
		float[][][] levelSetMat=levelSetVol.toArray3d();
		Vector3f tan;

		for(EmbeddedNode node:grid.getBoundaryNodes()){
			if(node.isInnerBoundary()){
				node.setLength((EmbeddedNode.isCompareToInner())?0:1);
			}
			if(node.isOuterBoundary()){
				node.setLength((EmbeddedNode.isCompareToInner())?1:0);
			}
		}
		EmbeddedNode node;

		for(int i=0;i<rows;i++){
			for(int j=0;j<cols;j++){
				for(int k=0;k<slices;k++){
					node=grid.getNode(i, j, k);
					if(EmbeddedNode.isCompareToInner()){
						levelSetMat[i][j][k]=(float)laplaceMat[i][j][k];	
					} else{
						levelSetMat[i][j][k]=1-(float)laplaceMat[i][j][k];
					}
					if(node!=null){
						node.setLength(levelSetMat[i][j][k]);
					}
				}
			}
		}
		for(int i=0;i<rows;i++){
			for(int j=0;j<cols;j++){
				for(int k=0;k<slices;k++){
					node=grid.getNode(i, j, k);
					if(node!=null&&node.isInterior()){
						tan=computeTanget(node);
						if(!EmbeddedNode.isCompareToInner()){
							tan.negate();
						}
						node.setTanget(tan);
						tangetFieldMat[i][j][k][0]=tan.x;
						tangetFieldMat[i][j][k][1]=tan.y;
						tangetFieldMat[i][j][k][2]=tan.z;
					}
				}
			}
		}
		tangetField.setName(innerVol.getName()+"_tanget");
	}


	public void solve(ImageData inner, ImageData outer) {
		mapInnerToOuter=true;
		mapOuterToInner=false;
		IsoSurfaceOnGrid surfGen=new IsoSurfaceOnGrid();
		EmbeddedSurface surf=(ADJUST_BOUNDARY)?surfGen.solveNoMove(inner, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(inner, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);
		SurfaceToComplex s2c=new SurfaceToComplex(this,surf);
		double[][] dat=s2c.mapToCartesianCoordinates(surf.getCenterOfMass());
		surf.setVertexData(dat);
		EmbeddedSurface sphere=surf.getEmbeddedSphere(0, false);
		SphericalMapCorrection hmc=new SphericalMapCorrection(this);
		sphere=hmc.solve(sphere);
		solve(inner,outer,sphere,null,0);
		CoupleSphericallyMappedSurfaces cs=new CoupleSphericallyMappedSurfaces(this);
		cs.solve(surfs, 1, -0.05f, true);
		surfs=cs.getRemeshedSurfaces();
		innerMesh=surfs.get(0);
		innerMesh.setName(inner.getName());
		outerMesh=surfs.get(1);
		outerMesh.setName(outer.getName());
		updateSurfStats(innerMesh);
		updateSurfStats(outerMesh);
	}
	public void solve(ImageData inner, ImageData outer,EmbeddedSurface tmpInnerMap,EmbeddedSurface tmpOuterMap,int offset) {
		this.innerVol = inner;
		this.outerVol = outer;
		rows = innerVol.getRows();
		cols = innerVol.getCols();
		slices = innerVol.getSlices();
		mapInnerToOuter=(tmpInnerMap!=null);
		mapOuterToInner=(tmpOuterMap!=null);
		if(ADJUST_BOUNDARY)adjustBoundary(innerVol, outerVol);
		IsoSurfaceOnGrid surfGen=new IsoSurfaceOnGrid();
		innerMesh=(ADJUST_BOUNDARY)?surfGen.solveNoMove(innerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(innerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);
		this.innerVol=surfGen.getAlteredLevelSet();
		outerMesh=(ADJUST_BOUNDARY)?surfGen.solveNoMove(outerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(outerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);
		this.outerVol=surfGen.getAlteredLevelSet();
		EmbeddedSurface innerMap=null,outerMap=null;
		if(tmpInnerMap!=null){
			innerMap=initInnerMap=innerMesh.clone();
			initInnerMap.setVertexData(tmpInnerMap.getVertexData());
			initInnerMap.setName(innerMesh.getName()+"_orig");
		}
		if(tmpOuterMap!=null){
			outerMap=initOuterMap=outerMesh.clone();
			initOuterMap.setVertexData(tmpOuterMap.getVertexData());
			initOuterMap.setName(outerMesh.getName()+"_orig");
		}
		//System.out.println("NUMBER OF HOLES IN OUTER SURFACE "+EmbeddedSurface.getGenus(outerMesh)+" "+outerMesh.getVertexCount()+" "+outerMesh.getFaceCount());
		innerMesh.setName(innerMesh.getName()+"_map");
		outerMesh.setName(outerMesh.getName()+"_map");
		surfs=new ArrayList<EmbeddedSurface>();
		levelsets=new ArrayList<ImageDataFloat>();
		ImageDataFloat innerFloat=new ImageDataFloat(innerVol);
		ImageDataFloat outerFloat=new ImageDataFloat(outerVol);

		
		//Find the level set that is no farther than 1.5 voxels from the original
		ImageDataFloat initFloat;
		EmbeddedSurface tmp;
		if(mapInnerToOuter){
			initFloat=innerFloat;
			EmbeddedNode.setCompareInnerLength(true);
			surfs.add(innerMesh);
			float levelSet=0;
			System.out.println("MAXIMUM EUCLIDIAN DISTNACE BETWEEN SURFS "+levelSetSpacing);
			EmbeddedSurface initMesh=innerMesh;

			for(int lev=1;lev<=interSurfs+1;lev++){
				System.gc();
				//Instantiate graph of nodes
				if(mapCoordinates){
					grid = new EmbeddedGrid(this,rows,cols,slices,true,false,innerMap,null);
				}
				gridMat=grid.getGrid();
				thicknessVol = new ImageDataFloat(rows, cols, slices);
				thicknessMat = thicknessVol.toArray3d();
				levelSetVol=new ImageDataFloat(rows,cols,slices);
				thicknessVol.setName(innerFloat.getName()+"_thick");
				//Build graph
				grid.build(initFloat, outerFloat, initMesh,outerMesh,isoVal,EmbeddedGrid.Topology.EXTENDED);
				markCompleted();
				//Solve laplace's equation
				computeLaplace(true,levelSet);
				markCompleted();
				//Transfer Spherical Map
				computeThickness();
				//Copy graph values to volumes

				
				//Find the level set that is no farther than a step size in voxels from the original
				levelSet=1;
				for(int i=0;i<initFloat.getRows();i++){
					for(int j=0;j<initFloat.getCols();j++){
						for(int k=0;k<initFloat.getSlices();k++){
							if(initFloat.getFloat(i, j, k)>=levelSetSpacing){
								levelSet=Math.min(levelSet, levelSetVol.getFloat(i,j,k));
							}
						}
					}
				}
				//Find the outer surface which is already known
				levelsets.add(levelSetVol);
				levelSetVol.setName(innerFloat.getName()+"_level"+lev);
				System.out.println("TARGET LEVEL SET "+levelSet);
				if(lev==interSurfs+1||levelSet==1){
					//Do not find intermediate surface if this is the last iteration
					grid.updateOuterMeshThicknessAndCorrespondence(outerMesh);
					SphericalMapCorrection flatten=new SphericalMapCorrection(this);

					EmbeddedSurface sphere=flatten.solve(initMesh,outerMesh,0);
					maxNonHarmonic=flatten.getMaxNonHarmonic();
					decimationAmount=flatten.getDecimationAmount();
					maxVertCount=sphere.getVertexCount();
					surfs.add(outerMesh);
					break;
				}
				markCompleted();

				//Find intermediate surface at specified level set
				initFloat=computeIntermediateSurface(initFloat,initFloat,outerFloat,levelSet);
				tmp=(ADJUST_BOUNDARY)?surfGen.solveNoMove(initFloat, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(initFloat, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);
				tmp.setName(innerFloat.getName()+"_surf"+lev);
				//Correct Spherical map
				innerMap=correctSphericalMap(initMesh,tmp,levelSet);
				initMesh=tmp;
				surfs.add(initMesh);
			}
		}
		
		if(mapOuterToInner){
			EmbeddedNode.setCompareInnerLength(false);
			surfs.add(outerMesh);
			float levelSet=1;
			double stepsz=this.lagrangeDeltaT;
			EmbeddedSurface initMesh=outerMesh;
			initFloat=outerFloat;
			for(int lev=1;lev<=interSurfs+1;lev++){
				System.gc();
				//Instantiate graph of nodes
				if(mapCoordinates){
					grid = new EmbeddedGrid(this,rows,cols,slices,false,true,null,outerMap);
				}
				gridMat=grid.getGrid();
				thicknessVol = new ImageDataFloat(rows, cols, slices);
				thicknessMat = thicknessVol.toArray3d();
				levelSetVol=new ImageDataFloat(rows,cols,slices);
				thicknessVol.setName(innerFloat.getName()+"_thick");
				//Build graph
				grid.build(innerFloat, initFloat, innerMesh,initMesh,isoVal,EmbeddedGrid.Topology.EXTENDED);
				markCompleted();
				//Solve laplace's equation
				computeLaplace(false,levelSet);
				markCompleted();
				//Transfer Spherical Map
				computeThickness();
				//Copy graph values to volumes
				//Find the level set that is no farther than a step size in voxels from the original
				levelSet=1;
				for(int i=0;i<initFloat.getRows();i++){
					for(int j=0;j<initFloat.getCols();j++){
						for(int k=0;k<initFloat.getSlices();k++){
							if(initFloat.getFloat(i, j, k)<=-stepsz){
								levelSet=Math.min(levelSet, levelSetVol.getFloat(i,j,k));
							}
						}
					}
				}
				markCompleted();
				//Find the outer surface which is already known
				levelsets.add(levelSetVol);
				levelSetVol.setName(outerFloat.getName()+"_level"+lev);
				System.out.println("TARGET LEVEL SET "+levelSet);
				if(lev==interSurfs+1||levelSet==1){
					//Do not find intermediate surface if this is the last iteration
					grid.updateInnerMeshThicknessAndCorrespondence(innerMesh);
					SphericalMapCorrection flatten=new SphericalMapCorrection(this);
					EmbeddedSurface sphere=flatten.solve(initMesh,innerMesh,0);
					maxNonHarmonic=flatten.getMaxNonHarmonic();
					decimationAmount=flatten.getDecimationAmount();
					maxVertCount=sphere.getVertexCount();
					surfs.add(innerMesh);
					break;
				}

				//Find intermediate surface at specified level set
				initFloat=computeIntermediateSurface(initFloat,innerFloat,initFloat,levelSet);
				tmp=(ADJUST_BOUNDARY)?surfGen.solveNoMove(initFloat, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(initFloat, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);
				tmp.setName(outerFloat.getName()+"_surf"+lev);
				//Correct Spherical map
				outerMap=correctSphericalMap(initMesh,tmp,levelSet);
				initMesh=tmp;
				surfs.add(initMesh);
			}
		}
		
	}
	public EmbeddedSurface getInnerOriginalSurface(){
		return initInnerMap;
	}
	public EmbeddedSurface getOuterOriginalSurface(){
		return initOuterMap;
	}
	public void solve(ImageData inner, ImageData central, ImageData outer,EmbeddedSurface innerMap,EmbeddedSurface outerMap) {
		this.innerVol = inner;
		this.outerVol = outer;
		this.centralVol = central;
		rows = innerVol.getRows();
		cols = innerVol.getCols();
		slices = innerVol.getSlices();

		if(ADJUST_BOUNDARY)adjustBoundary(innerVol, outerVol);
		IsoSurfaceOnGrid surfGen=new IsoSurfaceOnGrid();
		innerMesh=(ADJUST_BOUNDARY)?surfGen.solveNoMove(innerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(innerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);
		this.innerVol=surfGen.getAlteredLevelSet();
		outerMesh=(ADJUST_BOUNDARY)?surfGen.solveNoMove(outerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(outerVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);
		this.outerVol=surfGen.getAlteredLevelSet();
		centralMesh=(ADJUST_BOUNDARY)?surfGen.solveNoMove(centralVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false):surfGen.solveOriginal(centralVol, ConnectivityRule.CONNECT_18_6, (float)isoVal, false);
		this.centralVol=surfGen.getAlteredLevelSet();
		
		//System.out.println("NUMBER OF HOLES IN OUTER SURFACE "+EmbeddedSurface.getGenus(outerMesh)+" "+outerMesh.getVertexCount()+" "+outerMesh.getFaceCount());

		surfs=new ArrayList<EmbeddedSurface>();
		levelsets=new ArrayList<ImageDataFloat>();
		ImageDataFloat innerFloat=new ImageDataFloat(innerVol);
		ImageDataFloat outerFloat=new ImageDataFloat(outerVol);

		
		//Find the level set that is no farther than 1.5 voxels from the original
		ImageDataFloat initFloat;
		EmbeddedSurface tmp;
		if(mapInnerToOuter){
			initFloat=innerFloat;
			EmbeddedNode.setCompareInnerLength(true);
			float levelSet=0;
			System.out.println("MAXIMUM EUCLIDIAN DISTNACE BETWEEN SURFS "+levelSetSpacing);
			EmbeddedSurface initMesh=innerMesh;
			thicknessVol = new ImageDataFloat(rows, cols, slices);
			thicknessMat = thicknessVol.toArray3d();
			levelSetVol=new ImageDataFloat(rows,cols,slices);
			thicknessVol.setName(innerFloat.getName()+"_thick");
			grid = new EmbeddedGrid(this,rows,cols,slices,true,false,innerMap,null);
			gridMat=grid.getGrid();
			//Build graph
			grid.build(initFloat, outerFloat, initMesh,outerMesh,isoVal,EmbeddedGrid.Topology.EXTENDED);
			markCompleted();
			//Solve laplace's equation
			computeLaplace(true,levelSet);
			markCompleted();
			//Transfer Spherical Map
			computeThickness();
			//Do not find intermediate surface if this is the last iteration
			grid.updateOuterMeshThicknessAndCorrespondence(outerMesh);
			SphericalMapCorrection flatten=new SphericalMapCorrection(this);
			EmbeddedSurface sphere=flatten.solve(initMesh,outerMesh,0);
			maxNonHarmonic=flatten.getMaxNonHarmonic();
			decimationAmount=flatten.getDecimationAmount();
			maxVertCount=sphere.getVertexCount();
			correctSphericalMap(initMesh,centralMesh,levelSet);

		}
		
		if(mapOuterToInner){
			EmbeddedNode.setCompareInnerLength(false);
			
			float levelSet=1;
			EmbeddedSurface initMesh=outerMesh;
			initFloat=outerFloat;
				System.gc();
				//Instantiate graph of nodes
				grid = new EmbeddedGrid(this,rows,cols,slices,false,true,null,outerMap);
				gridMat=grid.getGrid();
				thicknessVol = new ImageDataFloat(rows, cols, slices);
				thicknessMat = thicknessVol.toArray3d();
				levelSetVol=new ImageDataFloat(rows,cols,slices);
				thicknessVol.setName(innerFloat.getName()+"_thick");
				//Build graph
				grid.build(innerFloat, initFloat, innerMesh,initMesh,isoVal,EmbeddedGrid.Topology.EXTENDED);
				markCompleted();
				//Solve laplace's equation
				computeLaplace(false,levelSet);
				markCompleted();
				//Transfer Spherical Map
				computeThickness();
				//Copy graph values to volumes
				//Find the level set that is no farther than a step size in voxels from the original
				markCompleted();
				//Find the outer surface which is already known
				levelsets.add(levelSetVol);
				//Do not find intermediate surface if this is the last iteration
				grid.updateInnerMeshThicknessAndCorrespondence(innerMesh);
				SphericalMapCorrection flatten=new SphericalMapCorrection(this);
				EmbeddedSurface sphere=flatten.solve(initMesh,innerMesh,0);
				maxNonHarmonic=flatten.getMaxNonHarmonic();
				decimationAmount=flatten.getDecimationAmount();
				maxVertCount=sphere.getVertexCount();
				
				correctSphericalMap(initMesh,centralMesh,levelSet);
				
		}

		surfs.add(innerMesh);
		surfs.add(centralMesh);
		surfs.add(outerMesh);
	}
	protected EmbeddedSurface correctSphericalMap(EmbeddedSurface sourceSurf,EmbeddedSurface targetSurf,double levelSet){
		double[][] dat=new double[targetSurf.getVertexCount()][4];
		double err=0;
		double mean=0;
		for(int i=0;i<dat.length;i++){
			Point3f p=grid.interpolateCorrespondence(targetSurf.getVertex(i));
			double len=grid.interpolateLengthOnEdge(targetSurf.getVertex(i));
			err+=(levelSet-len)*(levelSet-len);
			mean+=len;
			
			dat[i][0]=p.x;
			dat[i][1]=p.y;
			dat[i][2]=p.z;
			dat[i][3]=len;
		}
		err=Math.sqrt(err/(dat.length-1));
		mean/=dat.length;
		
		System.out.println("Target Level Set "+levelSet+" Mean Level Set "+mean+" Std. Dev. "+err);
		targetSurf.setVertexData(dat);
		SphericalMapCorrection flatten=new SphericalMapCorrection(this);
		EmbeddedSurface sphere=flatten.solve(sourceSurf,targetSurf, 0);
		return sphere;
	}
	protected void sanityCheck(){
		Vector3f dir;
		int badNodes=0;
		int nodeCount=0;
		boolean isBadNode;
		System.out.println("SANITY CHECK");
		for(EmbeddedNode n:grid.getGridNodes()){
			if(n.getNeighbors().contains(n)){
				System.out.println("CONTAINS LOOP "+n);
			}
			if(n.isInside()){
				if(n.getNeighbors().size()!=18){
					System.out.println("NOT CORRECT NUMBER OF NEIGHBORS "+n);
				}
			}
			
		}
		for(BoundaryNode n:grid.getBoundaryNodes()){
			if(n.getNeighbors().contains(n)){
				System.out.println("CONTAINS LOOP "+n);
			}			
		}
		
		for(EmbeddedNode n:grid.getGridNodes()){
			if(n.isJustInside()){
				isBadNode=false;
				
				for(EmbeddedNode nbr:n.nbhd){
					if(nbr.isInnerBoundary()){
						nodeCount++;
						dir=n.getDirection(nbr);
						if(dir.dot(nbr.getTanget())<0){
							System.out.println("INNER ANGLE TOO BIG "+nbr.getTanget()+" "+nbr.getClass().getCanonicalName()+" "+nbr.getLocation()+" "+(180/Math.PI)*GeometricUtilities.angle(dir, nbr.getTanget()));
							isBadNode=true;
							for(EmbeddedNode snbr:nbr.getNeighbors()){
								if(nbr.regionLabel==snbr.regionLabel){
									//System.out.println("CONTAINS SELF? "+n.getRegionLabelString()+n.getNeighbors().contains(n));
									//System.out.println("SURF NEIGHBORS "+snbr.getClass().getCanonicalName()+" "+snbr.getLocation()+" "+nbr.distance(snbr)+" "+(snbr==nbr)+" "+(180/Math.PI)*GeomUtil.angle(snbr.getTanget(), nbr.getTanget()));
								}
							}
						}
					} else if(nbr.isOuterBoundary()){
						dir=nbr.getDirection(n);
						if(dir.dot(nbr.getTanget())<0){
							System.out.println("OUTER ANGLE TOO BIG "+nbr.getLocation()+" "+(180/Math.PI)*GeometricUtilities.angle(dir, nbr.getTanget()));
							isBadNode=true;
						}						
					}
				}
				if(isBadNode)badNodes++;
			}
		}
		
		System.out.println("BAD NODES "+badNodes+"/"+ nodeCount);
	}
	public ImageDataFloat computeIntermediateSurface(ImageDataFloat initLevelSet,ImageDataFloat inner,ImageDataFloat outer,float levelSet){
		GenericTGDM tgdm=new GenericTGDM(this,ConnectivityRule.CONNECT_18_6);	

		ImageDataFloat tmpVol=tgdm.solveIntermediateSurface(!EmbeddedNode.isCompareToInner(),initLevelSet,levelSetVol,null,null,levelSet,smoothFactor,0f,1,4);
		float[][][] tmpMat=tmpVol.toArray3d();
		float[][][] innerMat=inner.toArray3d();
		float[][][] outerMat=outer.toArray3d();
		for(int i=0;i<tmpVol.getRows();i++){
			for(int j=0;j<tmpVol.getCols();j++){
				for(int k=0;k<tmpVol.getSlices();k++){
					tmpMat[i][j][k]=Math.max(outerMat[i][j][k]+0.0001f,Math.min(tmpMat[i][j][k],innerMat[i][j][k]-0.0001f));
				}
			}
		}
		return tmpVol;
	}


	protected double updateNode(EmbeddedNode node,EmbeddedNode parent){
		Vector3f tan=node.getTanget();
		Point3f pivot=node.getLocation();
		Vector3f diff=new Vector3f();
		boolean sgn=EmbeddedNode.isCompareToInner();
		double w,l;
			List<EmbeddedNode> upwindNodes=new LinkedList<EmbeddedNode>();
			//double minAngle=Math.PI,ang;
			//EmbeddedNode closestNeighborNode=null;
			for(EmbeddedNode nd:node.getNeighbors()){
				if(nd.getMarchingLabel()!=UNVISITED&&(nd.isInterior()||(nd.isInnerBoundary()&&EmbeddedNode.isCompareToInner())||(nd.isOuterBoundary()&&!EmbeddedNode.isCompareToInner()))){
					if(sgn){
						diff.sub(pivot,nd.getLocation());
					} else {
						diff.sub(nd.getLocation(),pivot);
					}
					if(tan.dot(diff)>=0){
						upwindNodes.add(nd);
					}		
				}
			}
			if(upwindNodes.size()<3){
				
				Point3f q=null;
				q=new Point3f();
				double wsum=0;
				for(EmbeddedNode nd:upwindNodes){
					//Weight by distance to iso-level
					w=node.getLocation().distance(nd.getLocation());
					w=(w>1E-6)?1/w:1E6;
					wsum+=w;
					Point3f p=new Point3f(nd.getCorrespodence());
					p.scale((float)w);
					q.add(p);
				}
				/*
				if(wsum>0){
					q.scale((float)(1.0/wsum));
				}
				*/
				l=Math.sqrt(q.x*q.x+q.y*q.y+q.z*q.z);
				//System.out.println("INSUFFICIENT NEIGHBORS "+node.getRegionLabelString()+" "+node.getLocation()+" "+upwindNodes.size());
				if(l>1E-8){
					q.scale(1.0f/(float)l);
				} else {
					System.out.println("A COULD NOT MAP "+l+" "+q+" "+parent.getCorrespodence()+" "+upwindNodes.size()+" "+node.getNeighbors().size()+" "+node.getRegionLabelString());
					q=parent.getCorrespodence();
				}
				node.setCorrespodence(q);
				return 0;
			} else {
				Matrix D=new Matrix(upwindNodes.size(),3);
				Matrix W=new Matrix(upwindNodes.size(),upwindNodes.size());
				Matrix One=new Matrix(upwindNodes.size(),1);
				Matrix Qx,Qy,Qz;
				Matrix GU;
				Qx=new Matrix(upwindNodes.size(),1);
				Qy=new Matrix(upwindNodes.size(),1);
				Qz=new Matrix(upwindNodes.size(),1);
				GU=new Matrix(upwindNodes.size(),1);
				double up=node.getLength();
				for(int i=0;i<upwindNodes.size();i++){
					EmbeddedNode n=upwindNodes.get(i);
					diff.sub(pivot,n.getLocation());
					D.set(i, 0, diff.x);
					D.set(i, 1, diff.y);
					D.set(i, 2, diff.z);
					w=diff.length();
					w=(w>1E-6)?1/w:1E6;
					W.set(i, i, w);
					Point3f q=n.getCorrespodence();
					Qx.set(i,0, q.x);
					Qy.set(i,0, q.y);
					Qz.set(i,0, q.z);
					GU.set(i,0, up-n.getLength());
					One.set(i,0,1);
				}	
				Matrix Dsqr=D.transpose().times(W).times(D);
				SingularValueDecomposition svd=new SingularValueDecomposition(Dsqr);
				Matrix S=svd.getS();
				Matrix V=svd.getV();
				Matrix U=svd.getU();
				int zeros=0;
				for(int i=0;i<S.getColumnDimension();i++){
					if(Math.abs(S.get(i, i))>1E-12){
						S.set(i, i, 1/S.get(i, i));
					} else {
						zeros++;
						S.set(i,i,0);
					}
				}
				Matrix Dinv=V.times(S.times(U.transpose())).times(D.transpose()).times(W);
				Matrix T=new Matrix(1,3);
				if(sgn){
					T.set(0,0,tan.x);
					T.set(0,1,tan.y);
					T.set(0,2,tan.z);
				} else {
					T.set(0,0,-tan.x);
					T.set(0,1,-tan.y);
					T.set(0,2,-tan.z);
				}
				
				Matrix ot=T.times(Dinv);
				double denom=ot.times(One).get(0,0);
				if(Math.abs(denom)<1E-10){
					denom=1;
				}
				Point3f q=new Point3f();	
				q.x=(float)((ot.times(Qx)).get(0,0)/denom);
				q.y=(float)((ot.times(Qy)).get(0,0)/denom);
				q.z=(float)((ot.times(Qz)).get(0,0)/denom);
				l=Math.sqrt(q.x*q.x+q.y*q.y+q.z*q.z);
					if(l>1E-5){
						q.scale(1.0f/(float)l);	
					} else {
						q=new Point3f();
						double wsum=0;
						for(EmbeddedNode nd:upwindNodes){
							//Weight by distance to iso-level
							w=node.getLocation().distance(nd.getLocation());
							w=(w>1E-6)?1/w:1E6;
							wsum+=w;
							Point3f p=new Point3f(nd.getCorrespodence());
							p.scale((float)w);
							q.add(p);
						}
						System.out.println("TAN "+tan);
						System.out.println("OT ");
						for(int i=0;i<ot.getRowDimension();i++){
							for(int j=0;j<ot.getColumnDimension();j++){
								System.out.print(ot.get(i, j)+" ");
							}
						}
						System.out.print("\n");

						
						l=(float)Math.sqrt(q.x*q.x+q.y*q.y+q.z*q.z);
						if(l>1E-6){
							q.scale(1.0f/(float)l);
						} else {
							q=(Point3f)parent.getCorrespodence().clone();
						}

						System.out.flush();
					}
					
					node.setCorrespodence(q);
				
				return 0;
			}
	}
	public double maxAngle(List<EmbeddedNode> nbrs,Point3f q){
		double ang=0;
		double maxAngle=0;
		for(EmbeddedNode nd:nbrs){
			ang=GeometricUtilities.angle(q, nd.getCorrespodence());
			maxAngle=Math.max(ang,maxAngle);
		}
		return maxAngle;		
	}
	public EmbeddedGrid getGrid(){
		grid.setName(innerVol.getName()+"_grid");
		return grid;
	}
	protected int interSurfs=0;
	public void setIntermediateSurfaces(int interSurfs) {
		this.interSurfs=interSurfs;
	}
	public float getLevelSetSpacing() {
		return levelSetSpacing;
	}
	public void setLevelSetSpacing(float levelSetSpacing) {
		this.levelSetSpacing = levelSetSpacing;
	}

}
