package edu.jhu.ece.iacl.algorithms.thickness.grid;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.Vector;

import javax.vecmath.Point2d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.map.SurfaceToComplex;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.geom.NormalGenerator;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataUByte;

import Jama.*;
/**
 * Eulerian grid with the iso-surface boundary immersed in the grid as boundary nodes
 * @author Blake Lucas (bclucas@jhu.edu)
 *
 */
public class EmbeddedGrid extends AbstractCalculation {
	private int rows, cols, slices;
	private GridNode[][][] grid;
	private ImageDataUByte labelVol;
	private BoundaryNode[] boundaryNodes;
	private LinkedList<GridNode> gridNodes;
	public static final byte NARROW_BAND = 5;
	public static final byte INSIDE_NARROW_BAND = EmbeddedNode.INSIDE_NARROW_BAND;
	public static final byte OUTSIDE_NARROW_BAND = EmbeddedNode.OUTSIDE_NARROW_BAND;
	public static final byte OUTSIDE_OUTER = EmbeddedNode.OUTSIDE_OUTER;
	public static final byte JUST_OUTSIDE_OUTER_BOUNDARY = EmbeddedNode.JUST_OUTSIDE_OUTER_BOUNDARY;
	public static final byte OUTER_BOUNDARY = EmbeddedNode.OUTER_BOUNDARY;
	public static final byte JUST_INSIDE = EmbeddedNode.JUST_INSIDE;
	public static final byte INSIDE = EmbeddedNode.INSIDE;
	public static final byte JUST_INSIDE_INNER_BOUNDARY = EmbeddedNode.JUST_INSIDE_INNER_BOUNDARY;
	public static final byte INNER_BOUNDARY = EmbeddedNode.INNER_BOUNDARY;
	public static final byte INSIDE_INNER = EmbeddedNode.INSIDE_INNER;

	public enum Topology {
		SIMPLE, EXTENDED
	};

	private int validNodeCount = 0;
	protected String name;

	public void setName(String name) {
		this.name = name;
	}

	public String getName() {
		return name;
	}

	private boolean mapInnerToOuterSurface, mapOuterToInnerSurface;
	private EmbeddedSurface innerMap, outerMap;

	// private GraphNode quickNodes[];
	public String toString() {
		return "grid";
	}

	public EmbeddedGrid(int rows, int cols, int slices,
			boolean mapInnerSurface, boolean mapOuterSurface) {
		super();
		this.mapInnerToOuterSurface = mapInnerSurface;
		this.mapOuterToInnerSurface = mapOuterSurface;
		this.setLabel("Non-Uniform Grid");
		this.rows = rows;
		this.cols = cols;
		this.slices = slices;
		grid = new GridNode[rows][cols][slices];
		gridNodes = new LinkedList<GridNode>();
	}

	public EmbeddedGrid(int rows, int cols, int slices,
			boolean mapInnerSurface, boolean mapOuterSurface,
			EmbeddedSurface innerMap, EmbeddedSurface outerMap) {
		super();
		this.mapInnerToOuterSurface = mapInnerSurface;
		this.mapOuterToInnerSurface = mapOuterSurface;
		this.setLabel("Embedded Non-Uniform Grid");
		this.rows = rows;
		this.cols = cols;
		this.slices = slices;
		this.innerMap = innerMap;
		this.outerMap = outerMap;
		grid = new GridNode[rows][cols][slices];
		gridNodes = new LinkedList<GridNode>();
	}

	public EmbeddedGrid(AbstractCalculation parent, int rows, int cols,
			int slices, boolean mapInnerSurface, boolean mapOuterSurface,
			EmbeddedSurface innerMap, EmbeddedSurface outerMap) {
		super(parent);
		this.mapInnerToOuterSurface = mapInnerSurface;
		this.mapOuterToInnerSurface = mapOuterSurface;
		this.setLabel("Embedded Non-Uniform Grid");
		this.rows = rows;
		this.cols = cols;
		this.slices = slices;
		this.innerMap = innerMap;
		this.outerMap = outerMap;
		grid = new GridNode[rows][cols][slices];
		gridNodes = new LinkedList<GridNode>();
	}

	public EmbeddedGrid(AbstractCalculation parent, int rows, int cols,
			int slices, boolean mapInnerSurface, boolean mapOuterSurface) {
		super(parent);
		this.mapInnerToOuterSurface = mapInnerSurface;
		this.mapOuterToInnerSurface = mapOuterSurface;
		this.setLabel("Embedded Non-Uniform Grid");
		this.rows = rows;
		this.cols = cols;
		this.slices = slices;
		grid = new GridNode[rows][cols][slices];
		gridNodes = new LinkedList<GridNode>();
	}

	public int getRows() {
		return rows;
	}

	public int getCols() {
		return cols;
	}

	public int getSlices() {
		return slices;
	}

	public BoundaryNode[] getBoundaryNodes() {
		return boundaryNodes;
	}

	public LinkedList<GridNode> getGridNodes() {
		return gridNodes;
	}

	public EmbeddedNode[] getNodes() {
		EmbeddedNode[] list = new EmbeddedNode[gridNodes.size()
				+ boundaryNodes.length];
		int i = 0;
		for (EmbeddedNode node : boundaryNodes) {
			list[i++] = node;
		}
		for (EmbeddedNode node : gridNodes) {
			list[i++] = node;
		}
		return list;
	}

	public GridNode[][][] getGrid() {
		return grid;
	}

	public byte[][][] getLabels() {
		return labelVol.toArray3d();
	}

	public GridNode getNode(int i, int j, int k) {
		return grid[i][j][k];
	}

	public double getNormalizedLength(int i, int j, int k) {
		if (grid[i][j][k] != null) {
			return grid[i][j][k].getNormalizedLength();
		} else if (labelVol.toArray3d()[i][j][k] > 0) {
			return 2;
		} else {
			return -1;
		}
	}
	/*
	public EmbeddedNode getNodeNeighbor(int i, int j, int k, int ai, int aj,
			int ak) {
		EmbeddedNode node = grid[i][j][k];
		if (i == 1) {
			node = node.nbhd[EmbeddedNode.EAST];
			if (j == 1) {
				node = node.nbhd[EmbeddedNode.NORTH];
				if (k == 1) {
					node = node.nbhd[EmbeddedNode.UP];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.NORTH].nbhd[EmbeddedNode.EAST].nbhd[EmbeddedNode.UP];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.NORTH].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.EAST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.NORTH].nbhd[EmbeddedNode.EAST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.EAST].nbhd[EmbeddedNode.NORTH];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.EAST].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.NORTH];
				} else {
					node = node.nbhd[EmbeddedNode.DOWN];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.NORTH].nbhd[EmbeddedNode.EAST].nbhd[EmbeddedNode.DOWN];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.NORTH].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.EAST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.NORTH].nbhd[EmbeddedNode.EAST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.EAST].nbhd[EmbeddedNode.NORTH];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.EAST].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.NORTH];
				}
			} else {
				node = node.nbhd[EmbeddedNode.SOUTH];
				if (k == 1) {
					node = node.nbhd[EmbeddedNode.UP];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.SOUTH].nbhd[EmbeddedNode.EAST].nbhd[EmbeddedNode.UP];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.SOUTH].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.EAST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.SOUTH].nbhd[EmbeddedNode.EAST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.EAST].nbhd[EmbeddedNode.SOUTH];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.EAST].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.SOUTH];
				} else {
					node = node.nbhd[EmbeddedNode.DOWN];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.SOUTH].nbhd[EmbeddedNode.EAST].nbhd[EmbeddedNode.DOWN];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.SOUTH].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.EAST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.SOUTH].nbhd[EmbeddedNode.EAST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.EAST].nbhd[EmbeddedNode.SOUTH];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.EAST].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.SOUTH];
				}
			}
		} else {
			node = node.nbhd[EmbeddedNode.WEST];
			if (j == 1) {
				node = node.nbhd[EmbeddedNode.NORTH];
				if (k == 1) {
					node = node.nbhd[EmbeddedNode.UP];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.NORTH].nbhd[EmbeddedNode.WEST].nbhd[EmbeddedNode.UP];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.NORTH].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.WEST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.NORTH].nbhd[EmbeddedNode.WEST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.WEST].nbhd[EmbeddedNode.NORTH];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.WEST].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.NORTH];
				} else {
					node = node.nbhd[EmbeddedNode.DOWN];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.NORTH].nbhd[EmbeddedNode.WEST].nbhd[EmbeddedNode.DOWN];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.NORTH].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.WEST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.NORTH].nbhd[EmbeddedNode.WEST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.WEST].nbhd[EmbeddedNode.NORTH];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.WEST].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.NORTH];
				}
			} else {
				node = node.nbhd[EmbeddedNode.SOUTH];
				if (k == 1) {
					node = node.nbhd[EmbeddedNode.UP];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.SOUTH].nbhd[EmbeddedNode.WEST].nbhd[EmbeddedNode.UP];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.SOUTH].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.WEST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.SOUTH].nbhd[EmbeddedNode.WEST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.WEST].nbhd[EmbeddedNode.SOUTH];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.WEST].nbhd[EmbeddedNode.UP].nbhd[EmbeddedNode.SOUTH];
				} else {
					node = node.nbhd[EmbeddedNode.DOWN];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.SOUTH].nbhd[EmbeddedNode.WEST].nbhd[EmbeddedNode.DOWN];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.SOUTH].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.WEST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.SOUTH].nbhd[EmbeddedNode.WEST];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.WEST].nbhd[EmbeddedNode.SOUTH];
					if (node == null)
						node = grid[i][j][k].nbhd[EmbeddedNode.WEST].nbhd[EmbeddedNode.DOWN].nbhd[EmbeddedNode.SOUTH];
				}
			}
		}
		return node;
	}
	 */
	/**
	 * Get all nodes within the voxel containing the point p
	 */
	public ArrayList<EmbeddedNode> getNeighbors(Point3f p) {
		ArrayList<EmbeddedNode> nbrs = new ArrayList<EmbeddedNode>();
		int i = (int) Math.floor(p.x);
		int j = (int) Math.floor(p.y);
		int k = (int) Math.floor(p.z);
		if (i == p.x && j == p.y && k == p.z) {
			EmbeddedNode n = getNode(i, j, k);
			nbrs.add(n);
			return nbrs;
		}
		for (int ni = 0; ni < 2; ni++) {
			for (int nj = 0; nj < 2; nj++) {
				for (int nk = 0; nk < 2; nk++) {
					EmbeddedNode n = getNode(i + ni, j + nj, k + nk);
					if (n == null) {
						continue;
					}
					if (n.isInterior()) {
						nbrs.add(n);
					} else {
						for (EmbeddedNode nbr : n.getNeighbors()) {
							if (nbr.isBoundary()) {
								Point3f loc = nbr.getLocation();
								if ((loc.x - i) <= 1 && (loc.y - j) <= 1
										&& (loc.z - k) <= 1 && (loc.x - i) >= 0
										&& (loc.y - j) >= 0 && (loc.z - k) >= 0) {
									nbrs.add(nbr);
								}
							}
						}
					}
				}
			}
		}
		return nbrs;
	}
	/**
	 * Add node to eulerian grid
	 * @param node
	 */
	public void add(GridNode node) {
		if (grid[node.i][node.j][node.k] == null) {
			grid[node.i][node.j][node.k] = node;
			gridNodes.add(node);
		}
		connect(node);
	}

	public void set(GridNode node) {
		grid[node.i][node.j][node.k] = node;
	}
	/**
	 * Rebuild list of nodes from volume of grid nodes
	 */
	public void relist() {
		int i = 0, j = 0, k = 0;
		gridNodes = new LinkedList<GridNode>();
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					if (grid[i][j][k] != null) {
						gridNodes.add(grid[i][j][k]);
					}
				}
			}
		}

	}

	protected int[] nbhdX;
	protected int[] nbhdY;
	protected int[] nbhdZ;
	/**
	 * Connect node to its neighbors
	 * @param node
	 */
	public void connect(EmbeddedNode node) {
		int l, ni, nj, nk;
		int i = node.i;
		int j = node.j;
		int k = node.k;
		for (l = 0; l < 6; l++) {
			ni = i + nbhdX[l];
			nj = j + nbhdY[l];
			nk = k + nbhdZ[l];
			if (ni < 0 || nj < 0 || nk < 0 || ni >= rows || nj >= cols
					|| nk >= slices)
				continue;
			EmbeddedNode neighbor = grid[ni][nj][nk];
			if (neighbor != null) {
				node.connect(neighbor, l);
			}
		}
	}
	/**
	 * Insert surface into grid
	 * @param surf
	 * @return
	 */
	public EmbeddedNode[] insert(EmbeddedSurface surf) {
		// Find inner boundary
		Vector3f v = new Vector3f();
		// Add inner boundary nodes to this
		ExtendedImmersedBoundaryNode n;
		validNodeCount += surf.getVertexCount();
		Vector3f[] norms = NormalGenerator.generate(surf);
		ExtendedImmersedBoundaryNode[] newnodes = new ExtendedImmersedBoundaryNode[surf
				.getVertexCount()];
		double[][] vertDat = surf.getVertexData();
		for (int i = 0; i < surf.getVertexCount(); i++) {
			Point3f p = surf.getVertex(i);
			if (vertDat == null) {
				n = new ExtendedImmersedBoundaryNode(p, new Point3f());
			} else {
				n = new ExtendedImmersedBoundaryNode(p, new Point3f(
						(float) vertDat[i][1], (float) vertDat[i][2],
						(float) vertDat[i][3]));
				n.setLength(vertDat[i][0]);
			}
			n.regionLabel = EmbeddedNode.INNER_BOUNDARY;
			n.setTanget(norms[i]);
			newnodes[i] = n;
			gridNodes.add(n);
			this.insert(n);
			incrementCompletedUnits();
		}

		int nbhdInner[][] = EmbeddedSurface.buildNeighborVertexVertexTable(
				surf, EmbeddedSurface.Direction.COUNTER_CLOCKWISE);
		for (int i = 0; i < surf.getVertexCount(); i++) {
			ExtendedImmersedBoundaryNode node = newnodes[i];
			for (int j = 0; j < nbhdInner[i].length; j++) {
				node.connect(newnodes[nbhdInner[i][j]]);
			}
		}

		int l;
		for (EmbeddedNode node : gridNodes) {
			if (node instanceof ExtendedImmersedBoundaryNode) {
				for (l = 0; l < 6; l++) {
					if (node.nbhd[l] != null)
						((ExtendedGridNode) node).connect(node.nbhd[l]);
				}
			}
		}
		return newnodes;
	}
	/**
	 * Insert boundary node into grid
	 * @param node
	 */
	public void insert(EmbeddedNode node) {
		Point3f p = node.getLocation();
		Point3f p2 = null;
		EmbeddedNode lowerNode = grid[node.i][node.j][node.k];
		//Find orientation of edge based on point location
		int orientation = -1;
		if (p.x > node.i) {
			orientation = EmbeddedNode.EAST;
		} else if (p.y > node.j) {
			orientation = EmbeddedNode.NORTH;
		} else if (p.z > node.k) {
			orientation = EmbeddedNode.UP;
		} else {
			// System.err.println("NOT A VALID ORIENTATION
			// ("+node.i+","+node.j+","+node.k+") "+node.getLocation());
		}
		EmbeddedNode upperNode = lowerNode;
		if (orientation == -1) {
			System.err.println("BOUNDARY NODE FOUND " + node + " "
					+ node.getTanget());
			System.exit(-1);
		} else {
			do {
				//Locate lower and upper node that should be connected to the boundary node
				lowerNode = upperNode;
				if (lowerNode == null) {
					System.err.println("ORIENTATION " + orientation);
					System.err.println("MESH NODE " + node);
					System.err.println("LOWER NODE " + lowerNode);
					System.err.println("UPPER NODE " + upperNode);
					System.err.println("CENTRAL "
							+ EmbeddedNode.getLabelString((byte) labelVol
									.getUByte(node.i, node.j, node.k)));
					System.err.println("NORTH "
							+ EmbeddedNode.getLabelString((byte) labelVol
									.getUByte(node.i, node.j + 1, node.k)));
					System.err.println("SOUTH "
							+ EmbeddedNode.getLabelString((byte) labelVol
									.getUByte(node.i, node.j - 1, node.k)));
					System.err.println("EAST "
							+ EmbeddedNode.getLabelString((byte) labelVol
									.getUByte(node.i + 1, node.j, node.k)));
					System.err.println("WEST "
							+ EmbeddedNode.getLabelString((byte) labelVol
									.getUByte(node.i - 1, node.j, node.k)));
					System.err.println("UP "
							+ EmbeddedNode.getLabelString((byte) labelVol
									.getUByte(node.i, node.j, node.k + 1)));
					System.err.println("DOWN "
							+ EmbeddedNode.getLabelString((byte) labelVol
									.getUByte(node.i, node.j, node.k - 1)));
					System.exit(-1);
				}
				upperNode = lowerNode.nbhd[orientation];
				if (upperNode != null) {
					p2 = upperNode.getLocation();
				} else {
					System.err.println("ORIENTATION " + orientation);
					System.err.println("MESH NODE " + node);
					System.err.println("LOWER NODE " + lowerNode);
					System.err.println("UPPER NODE " + upperNode);
					System.err.println("CENTRAL "
							+ EmbeddedNode.getLabelString((byte) labelVol
									.getUByte(node.i, node.j, node.k)));
					System.err.println("NORTH "
							+ EmbeddedNode.getLabelString((byte) labelVol
									.getUByte(node.i, node.j + 1, node.k)));
					System.err.println("SOUTH "
							+ EmbeddedNode.getLabelString((byte) labelVol
									.getUByte(node.i, node.j - 1, node.k)));
					System.err.println("EAST "
							+ EmbeddedNode.getLabelString((byte) labelVol
									.getUByte(node.i + 1, node.j, node.k)));
					System.err.println("WEST "
							+ EmbeddedNode.getLabelString((byte) labelVol
									.getUByte(node.i - 1, node.j, node.k)));
					System.err.println("UP "
							+ EmbeddedNode.getLabelString((byte) labelVol
									.getUByte(node.i, node.j, node.k + 1)));
					System.err.println("DOWN "
							+ EmbeddedNode.getLabelString((byte) labelVol
									.getUByte(node.i, node.j, node.k - 1)));
					System.exit(-1);
				}
			} while (upperNode != null
					&& ((orientation == EmbeddedNode.EAST && p2.x < p.x)
							|| (orientation == EmbeddedNode.NORTH && p2.y < p.y) || (orientation == EmbeddedNode.UP && p2.z < p.z)));

			if (upperNode == null) {
				System.err.println("UPPER NODE IS NULL");
				System.exit(-1);
			}
			lowerNode.connect(node, orientation);
			node.connect(upperNode, orientation);
		}
	}

	public int getValidNodeCount() {
		return validNodeCount;
	}
	/**
	 * Build embedded grid based on inner and outer level set and inner and outer surface, which may contain embedded data
	 * @param innerVol inner level set
	 * @param outerVol outer level set
	 * @param innerMesh inner iso-surface
	 * @param outerMesh outer iso-surface
	 * @param isoVal 
	 * @param topo
	 */
	public void build(ImageData innerVol, ImageData outerVol,
			EmbeddedSurface innerMesh, EmbeddedSurface outerMesh,
			double isoVal, Topology topo) {
		int i, j, k, l, ni, nj, nk;
		byte tag;
		double val1, val2;
		validNodeCount = 0;
		labelVol = new ImageDataUByte(rows, cols, slices);
		byte[][][] labelMat = labelVol.toArray3d();
		int nodeCount = 0;
		if (topo == Topology.SIMPLE) {
			nbhdX = EmbeddedNode.nbhdX;
			nbhdY = EmbeddedNode.nbhdY;
			nbhdZ = EmbeddedNode.nbhdZ;
		} else {
			nbhdX = ExtendedGridNode.nbhd18X;
			nbhdY = ExtendedGridNode.nbhd18Y;
			nbhdZ = ExtendedGridNode.nbhd18Z;
		}
		// Label voxel regions
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					val1 = outerVol.getDouble(i, j, k);
					val2 = innerVol.getDouble(i, j, k);
					//Label voxels to be turned into voxels later
					if (val1 >= isoVal && val2 >= isoVal) {
						if (val1 < isoVal + NARROW_BAND) {
							labelMat[i][j][k] = OUTSIDE_OUTER;
						} else {
							labelMat[i][j][k] = OUTSIDE_NARROW_BAND;
							nodeCount++;

						}
					} else if (val2 < isoVal && val1 < isoVal) {
						if (val2 >= isoVal - NARROW_BAND) {
							labelMat[i][j][k] = INSIDE_INNER;
						} else {
							labelMat[i][j][k] = INSIDE_NARROW_BAND;
							nodeCount++;
						}
					} else {
						labelMat[i][j][k] = INSIDE;
						validNodeCount++;
					}

				}
			}
		}
		int nbtag;
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					tag = labelMat[i][j][k];
					if (tag != INSIDE) {
						for (l = 0; l < EmbeddedNode.nbhdX.length; l++) {
							ni = i + EmbeddedNode.nbhdX[l];
							nj = j + EmbeddedNode.nbhdY[l];
							nk = k + EmbeddedNode.nbhdZ[l];
							if (ni < 0 || nj < 0 || nk < 0 || ni >= rows
									|| nj >= cols || nk >= slices)
								continue;
							//Use region labels to find nodes attached to shell boundary
							nbtag = labelMat[ni][nj][nk];
							if ((tag == OUTSIDE_OUTER || tag == OUTSIDE_NARROW_BAND)
									&& (nbtag == INSIDE
											|| nbtag == INSIDE_INNER
											|| nbtag == INSIDE_NARROW_BAND || nbtag == JUST_INSIDE_INNER_BOUNDARY)) {
								labelMat[i][j][k] = JUST_OUTSIDE_OUTER_BOUNDARY;
							} else if ((tag == INSIDE_INNER || tag == INSIDE_NARROW_BAND)
									&& (nbtag == INSIDE
											|| nbtag == OUTSIDE_OUTER
											|| nbtag == OUTSIDE_NARROW_BAND || nbtag == JUST_OUTSIDE_OUTER_BOUNDARY)) {
								labelMat[i][j][k] = JUST_INSIDE_INNER_BOUNDARY;
								break;
							}
						}
					}
				}
			}
		}

		// Add interior nodes to grid
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					tag = labelMat[i][j][k];
					if (tag == JUST_OUTSIDE_OUTER_BOUNDARY
							|| tag == JUST_INSIDE_INNER_BOUNDARY) {
						if (topo == Topology.SIMPLE) {
							SimpleOutsideNode node = new SimpleOutsideNode(i,
									j, k, (byte) tag);
							this.add(node);
						} else {
							ExtendedOutsideNode node = new ExtendedOutsideNode(
									i, j, k, (byte) tag);
							this.add(node);
						}
						incrementCompletedUnits();
					} else if (tag == INSIDE || tag == JUST_INSIDE) {
						if (topo == Topology.SIMPLE) {
							SimpleInsideNode node = new SimpleInsideNode(i, j,
									k, (byte) tag);
							this.add(node);
						} else {
							ExtendedInsideNode node = new ExtendedInsideNode(i,
									j, k, (byte) tag);
							this.add(node);
						}
						incrementCompletedUnits();
					}
				}
			}
		}
		SurfaceToComplex map;
		double[][] cmplx = null;

		if (mapInnerToOuterSurface) {
			// No inner map specified, so set coordinates to (0,0,0)
			if (innerMap == null) {
				cmplx = new double[innerMesh.getVertexCount()][3];
			} else {
				int vertCount = innerMap.getVertexCount();
				cmplx = new double[vertCount][3];
				for (i = 0; i < vertCount; i++) {
					double[] pol = innerMap.getVertexData(i);
					// The inner map is a sphere, so use the sphere's
					// coordinates
					if (pol == null || pol.length <= 1) {
						innerMap.getCoordinate(i, cmplx[i]);
					} else {
						// The inner map is a surface with embedded coordinates,
						// so use th embedded coordinates
						if (pol.length == 2) {
							// Coordinates are spherical
							cmplx[i] = new double[] {
									Math.cos(pol[0]) * Math.sin(pol[1]),
									Math.sin(pol[0]) * Math.sin(pol[1]),
									Math.cos(pol[1]) };
						} else if (pol.length == 3) {
							// Coordinates are cartesian
							cmplx[i] = new double[] { pol[0], pol[1], pol[2] };
						}
					}

				}
			}
		}
		// Find inner boundary
		Vector3f v = new Vector3f();
		// Add inner boundary nodes to this
		BoundaryNode n;
		validNodeCount += innerMesh.getVertexCount();
		boundaryNodes = new BoundaryNode[innerMesh.getVertexCount()
				+ outerMesh.getVertexCount()];
		int bcount = 0;
		Vector3f[] norms = NormalGenerator.generate(innerMesh);

		for (i = 0; i < innerMesh.getVertexCount(); i++) {
			Point3f p = new Point3f();
			innerMesh.getCoordinate(i, p);
			if (mapInnerToOuterSurface) {
				double[] pol = innerMesh.getVertexData(i);
				if (mapOuterToInnerSurface) {
					if (pol != null && pol.length > 0) {
						innerMesh.setVertexData(i,
								new double[] {  cmplx[i][0],
										cmplx[i][1], cmplx[i][2],pol[0], 0, 0, 0 });
					} else {
						innerMesh.setVertexData(i,
								new double[] { cmplx[i][0], cmplx[i][1],
										cmplx[i][2],0,  0, 0, 0 });
					}
				} else {
					if (pol != null && pol.length > 0) {
						innerMesh.setVertexData(i, new double[] {
								cmplx[i][0], cmplx[i][1], cmplx[i][2], pol[0] });
					} else {
						innerMesh.setVertexData(i, new double[] { 
								cmplx[i][0], cmplx[i][1], cmplx[i][2],0 });
					}
				}
				//If surface contains correspondence points, embed these in boundary nodes
				if (topo == Topology.SIMPLE) {
					n = new SimpleInnerBoundaryNode(p, new Point3f(
							(float) cmplx[i][0], (float) cmplx[i][1],
							(float) cmplx[i][2]));
				} else {
					n = new ExtendedInnerBoundaryNode(p, new Point3f(
							(float) cmplx[i][0], (float) cmplx[i][1],
							(float) cmplx[i][2]));
				}
			} else {
				innerMesh
						.setVertexData(i, new double[] { 0, 0, 0, 0, 0, 0, 0 });
				if (topo == Topology.SIMPLE) {
					n = new SimpleInnerBoundaryNode(p, new Point3f());
				} else {
					n = new ExtendedInnerBoundaryNode(p, new Point3f());
				}
			}
			n.setTanget(norms[i]);
			this.insert(n);
			boundaryNodes[bcount++] = n;
			incrementCompletedUnits();
			for (EmbeddedNode neighbor : n.nbhd) {
				if (neighbor == null)
					continue;
				if (neighbor.regionLabel == INSIDE) {
					labelMat[neighbor.i][neighbor.j][neighbor.k] = neighbor.regionLabel = JUST_INSIDE;
				}
			}
		}
		System.gc();
		// Find outer boundary
		// Add outer boundary nodes to this
		if (mapOuterToInnerSurface) {
			if (outerMap == null) {
				// Outer parametric map is zero, so set to 0
				cmplx = new double[outerMesh.getVertexCount()][3];
			} else {
				int vertCount = outerMap.getVertexCount();
				cmplx = new double[vertCount][3];
				for (i = 0; i < vertCount; i++) {
					double[] pol = outerMap.getVertexData(i);
					if (pol == null || pol.length <= 1) {
						// Outer map is a sphere, so use the sphere's
						// coordinates
						outerMap.getCoordinate(i, cmplx[i]);
					} else {
						// Outer map is embedded surface, so use embedded
						// coordinates
						if (pol.length == 2) {
							// Coordinates are spherical
							cmplx[i] = new double[] {
									Math.cos(pol[0]) * Math.sin(pol[1]),
									Math.sin(pol[0]) * Math.sin(pol[1]),
									Math.cos(pol[1]) };
						} else if (pol.length == 3) {
							// Coordinates are Cartesian
							cmplx[i] = new double[] { pol[0], pol[1], pol[2] };
						}
					}
				}
			}
		}
		validNodeCount += outerMesh.getVertexCount();
		norms = NormalGenerator.generate(outerMesh);

		for (i = 0; i < outerMesh.getVertexCount(); i++) {
			Point3f p = new Point3f();
			outerMesh.getCoordinate(i, p);
			if (mapOuterToInnerSurface) {
				double[] pol = outerMesh.getVertexData(i);
				if (mapInnerToOuterSurface) {
					if (pol != null && pol.length > 0) {
						outerMesh.setVertexData(i, new double[] { 0, 0,
								0, cmplx[i][0], cmplx[i][1], cmplx[i][2] ,pol[0]});
					} else {
						outerMesh.setVertexData(i, new double[] { 0, 0, 0, 
								cmplx[i][0], cmplx[i][1], cmplx[i][2],0 });
					}
				} else {
					if (pol != null && pol.length > 0) {
						outerMesh.setVertexData(i, new double[] {
								cmplx[i][0], cmplx[i][1], cmplx[i][2] , pol[0]});
					} else {
						outerMesh.setVertexData(i, new double[] {
								cmplx[i][0], cmplx[i][1], cmplx[i][2], 0 });
					}
				}
				//If surface contains correspondence points, embed these in boundary nodes
				if (topo == Topology.SIMPLE) {
					n = new SimpleOuterBoundaryNode(p, new Point3f(
							(float) cmplx[i][0], (float) cmplx[i][1],
							(float) cmplx[i][2]));
				} else {
					n = new ExtendedOuterBoundaryNode(p, new Point3f(
							(float) cmplx[i][0], (float) cmplx[i][1],
							(float) cmplx[i][2]));
				}
			} else {
				outerMesh
						.setVertexData(i, new double[] { 0, 0, 0, 0, 0, 0, 0 });
				if (topo == Topology.SIMPLE) {
					n = new SimpleOuterBoundaryNode(p, new Point3f());
				} else {
					n = new ExtendedOuterBoundaryNode(p, new Point3f());
				}
			}
			n.setTanget(norms[i]);
			this.insert(n);
			boundaryNodes[bcount++] = n;
			incrementCompletedUnits();
			for (EmbeddedNode neighbor : n.nbhd) {
				if (neighbor == null)
					continue;
				if (neighbor.regionLabel == INSIDE) {
					labelMat[neighbor.i][neighbor.j][neighbor.k] = neighbor.regionLabel = JUST_INSIDE;

				}
			}
		}
		if (topo == Topology.EXTENDED)
			extendTopology(innerMesh, outerMesh);
		map = null;
		markCompleted();
		System.gc();
	}
	/**
	 * Embed one surface in grid
	 * @param innerVol level set
	 * @param innerMesh iso-surface
	 * @param isoVal
	 * @param topo
	 */
	public void build(ImageData innerVol, EmbeddedSurface innerMesh,
			double isoVal, Topology topo) {
		int i, j, k, l, ni, nj, nk;
		byte tag;
		double val2;
		validNodeCount = 0;
		labelVol = new ImageDataUByte(rows, cols, slices);
		byte[][][] labelMat = labelVol.toArray3d();
		int nodeCount = 0;
		if (topo == Topology.SIMPLE) {
			nbhdX = EmbeddedNode.nbhdX;
			nbhdY = EmbeddedNode.nbhdY;
			nbhdZ = EmbeddedNode.nbhdZ;
		} else {
			nbhdX = ExtendedGridNode.nbhd18X;
			nbhdY = ExtendedGridNode.nbhd18Y;
			nbhdZ = ExtendedGridNode.nbhd18Z;
		}
		// Label voxel regions
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					val2 = innerVol.getDouble(i, j, k);
					//Label voxel regions to turn into nodes later
					if (val2 >= isoVal) {
						if (val2 >= isoVal + NARROW_BAND) {
							labelMat[i][j][k] = OUTSIDE_NARROW_BAND;
						} else {
							labelMat[i][j][k] = INSIDE;
							nodeCount++;
							validNodeCount++;

						}
					} else {
						if (val2 <= isoVal - NARROW_BAND) {
							labelMat[i][j][k] = OUTSIDE_NARROW_BAND;
						} else {
							labelMat[i][j][k] = INSIDE;
							nodeCount++;
							validNodeCount++;
						}
					}

				}
			}
		}
		if (innerMesh != null) {
			nodeCount += innerMesh.getVertexCount();
		}
		System.out.flush();
		setTotalUnits(nodeCount);
		// Add interior nodes to grid
		for (i = 0; i < rows; i++) {
			for (j = 0; j < cols; j++) {
				for (k = 0; k < slices; k++) {
					tag = labelMat[i][j][k];
					if (tag == INSIDE) {
						if (topo == Topology.SIMPLE) {
							SimpleInsideNode node = new SimpleInsideNode(i, j,
									k, (byte) tag);
							this.add(node);
						} else {
							ExtendedInsideNode node = new ExtendedInsideNode(i,
									j, k, (byte) tag);
							this.add(node);
						}
						incrementCompletedUnits();
					}
				}
			}
		}
		SurfaceToComplex map;
		double[][] cmplx = null;
		System.out.flush();
		if (mapInnerToOuterSurface) {
			// No inner map specified, so set coordinates to (0,0,0)
			if (innerMap == null) {
				cmplx = new double[innerMesh.getVertexCount()][3];
			} else {
				int vertCount = innerMap.getVertexCount();
				cmplx = new double[vertCount][3];
				for (i = 0; i < vertCount; i++) {
					double[] pol = innerMap.getVertexData(i);
					// The inner map is a sphere, so use the sphere's
					// coordinates
					if (pol == null || pol.length <= 1) {
						innerMap.getCoordinate(i, cmplx[i]);
					} else {
						// The inner map is a surface with embedded coordinates,
						// so use th embedded coordinates
						if (pol.length == 2) {
							// Coordinates are spherical
							cmplx[i] = new double[] {
									Math.cos(pol[0]) * Math.sin(pol[1]),
									Math.sin(pol[0]) * Math.sin(pol[1]),
									Math.cos(pol[1]) };
						} else if (pol.length == 3) {
							// Coordinates are cartesian
							cmplx[i] = new double[] { pol[0], pol[1], pol[2] };
						}
					}

				}
			}
		}
		// Add inner boundary nodes to this
		BoundaryNode n;
		validNodeCount += innerMesh.getVertexCount();
		boundaryNodes = new BoundaryNode[innerMesh.getVertexCount()];
		int bcount = 0;
		Vector3f[] norms = NormalGenerator.generate(innerMesh);

		for (i = 0; i < innerMesh.getVertexCount(); i++) {
			Point3f p =innerMesh.getVertex(i);
			if (mapInnerToOuterSurface) {
				//Create nodes with designated correspondence points
				if (topo == Topology.SIMPLE) {
					n = new SimpleInnerBoundaryNode(p, new Point3f(
							(float) cmplx[i][0], (float) cmplx[i][1],
							(float) cmplx[i][2]));
				} else {
					n = new ExtendedInnerBoundaryNode(p, new Point3f(
							(float) cmplx[i][0], (float) cmplx[i][1],
							(float) cmplx[i][2]));
				}
			} else {
				if (topo == Topology.SIMPLE) {
					n = new SimpleInnerBoundaryNode(p, new Point3f());
				} else {
					n = new ExtendedInnerBoundaryNode(p, new Point3f());
				}
			}
			n.setTanget(norms[i]);
			this.insert(n);
			boundaryNodes[bcount++] = n;
			incrementCompletedUnits();
			for (EmbeddedNode neighbor : n.nbhd) {
				if (neighbor == null)
					continue;
				if (neighbor.regionLabel == INSIDE) {
					labelMat[neighbor.i][neighbor.j][neighbor.k] = neighbor.regionLabel = JUST_INSIDE;
				}
			}
		}
		System.out.println("FINISHED CREATING BOUNDARY NODES");
		System.out.flush();
		System.gc();
		if (topo == Topology.EXTENDED)
			extendTopology(innerMesh, null);
		map = null;
		markCompleted();
		System.gc();
	}
	/**
	 * Extend topology of grid from 6 to 18 or 26
	 * @param innerMesh
	 * @param outerMesh
	 */
	public void extendTopology(EmbeddedSurface innerMesh,
			EmbeddedSurface outerMesh) {
		int innerVertCount = 0, outerVertCount = 0;
		int l, ni, nj, nk, n;
		if (innerMesh != null) {
			int nbhdInner[][] = EmbeddedSurface.buildNeighborVertexVertexTable(
					innerMesh, EmbeddedSurface.Direction.COUNTER_CLOCKWISE);
			innerVertCount = innerMesh.getVertexCount();

			for (int i = 0; i < innerVertCount; i++) {
				ExtendedInnerBoundaryNode node = (ExtendedInnerBoundaryNode) boundaryNodes[i];
				for (int j = 0; j < nbhdInner[i].length; j++) {
					node
							.connect((ExtendedInnerBoundaryNode) boundaryNodes[nbhdInner[i][j]]);
				}
			}
			nbhdInner = null;
		}

		if (outerMesh != null) {
			outerVertCount = outerMesh.getVertexCount();
			int nbhdOuter[][] = EmbeddedSurface.buildNeighborVertexVertexTable(
					outerMesh, EmbeddedSurface.Direction.COUNTER_CLOCKWISE);
			for (int i = 0; i < outerVertCount; i++) {
				ExtendedOuterBoundaryNode node = (ExtendedOuterBoundaryNode) boundaryNodes[innerVertCount
						+ i];
				for (int j = 0; j < nbhdOuter[i].length; j++) {
					node
							.connect((ExtendedOuterBoundaryNode) boundaryNodes[innerVertCount
									+ nbhdOuter[i][j]]);
				}
				// Make connections between outer surface and inner surface
				for (l = 0; l < 6; l++) {
					if (node.nbhd[l] != null
							&& node.nbhd[l] instanceof ExtendedInnerBoundaryNode) {
						node.connect(node.nbhd[l]);
					}
				}
			}
		}

		for (EmbeddedNode node : gridNodes) {
			for (l = 0; l < 6; l++) {
				if (node.nbhd[l] != null)
					((ExtendedGridNode) node).connect(node.nbhd[l]);
			}
		}
		for (EmbeddedNode node : gridNodes) {
			if (node.isInterior()) {
				Vector<EmbeddedNode> nbrs = findNeighbors(node);
				for (EmbeddedNode nbr : nbrs) {
					((ExtendedGridNode) node).connect(nbr);
				}
			}
		}
	}
	//Locate all nodes within one voxel of pivot voxel
	public Vector<EmbeddedNode> findNeighbors(EmbeddedNode pivot) {
		Vector<EmbeddedNode> queue = new Vector<EmbeddedNode>();
		Vector<EmbeddedNode> nbhrs = new Vector<EmbeddedNode>();
		queue.add(pivot);
		queue.addAll(pivot.getNeighbors());
		Point3f pt = pivot.getLocation();
		Point3f diff = new Point3f();
		for (int i = 0; i < queue.size(); i++) {
			EmbeddedNode top = queue.get(i);
			for (EmbeddedNode nbr : top.getNeighbors()) {
				if (nbr.isInterior() || nbr.isBoundary()) {
					diff.sub(pt, nbr.getLocation());
					if (Math.abs(diff.x) <= 1 && Math.abs(diff.y) <= 1
							&& Math.abs(diff.z) <= 1) {
						if (!queue.contains(nbr)) {
							nbhrs.add(nbr);
							queue.add(nbr);
						}
					}
				}
			}
		}
		return nbhrs;
	}
	//Get edge that contains point
	public EmbeddedNode[] getEdge(Point3f p) {
		int i0, j0, k0;
		i0 = (int) Math.floor(p.x);
		j0 = (int) Math.floor(p.y);
		k0 = (int) Math.floor(p.z);
		EmbeddedNode lowerNode = grid[i0][j0][k0];

		int orientation = -1;
		if (p.x > i0) {
			orientation = EmbeddedNode.EAST;
		} else if (p.y > j0) {
			orientation = EmbeddedNode.NORTH;
		} else if (p.z > k0) {
			orientation = EmbeddedNode.UP;
		} else {
			// System.err.println("NOT A VALID ORIENTATION
			// ("+node.i+","+node.j+","+node.k+") "+node.getLocation());
		}
		EmbeddedNode upperNode = lowerNode;
		if (lowerNode == null) {
			System.err.println("LOWER NODE IS NULL " + p);
		}
		Point3f p2 = null;
		if (orientation == -1 || lowerNode == null) {
			System.err.println("NODE NOT FOUND " + p + " " + lowerNode);
			return new EmbeddedNode[] { lowerNode, lowerNode };
		} else {
			do {
				lowerNode = upperNode;
				upperNode = lowerNode.getNeighbor(orientation);
				if (upperNode != null) {
					p2 = upperNode.getLocation();
				}
			} while (upperNode != null
					&& ((orientation == EmbeddedNode.EAST && p2.x < p.x)
							|| (orientation == EmbeddedNode.NORTH && p2.y < p.y) || (orientation == EmbeddedNode.UP && p2.z < p.z)));
		}
		return new EmbeddedNode[] { lowerNode, upperNode };
	}
	/*
	public EmbeddedNode[] getEdge(EmbeddedNode c1, EmbeddedNode c2, double lev) {
		boolean flip = false;
		if (c2.getNormalizedLength() < lev) {
			EmbeddedNode tmp = c2;
			c2 = c1;
			c1 = tmp;
			flip = true;
		}
		int orientation = -1;
		if (c2.i > c1.i) {
			orientation = EmbeddedNode.EAST;
		} else if (c2.j > c1.j) {
			orientation = EmbeddedNode.NORTH;
		} else if (c2.k > c1.k) {
			orientation = EmbeddedNode.UP;
		} else {
			// System.err.println("NOT A VALID ORIENTATION
			// ("+node.i+","+node.j+","+node.k+") "+node.getLocation());
		}
		EmbeddedNode lowerNode = c1;
		EmbeddedNode upperNode = c1.nbhd[orientation];
		while (upperNode.getNormalizedLength() < lev) {
			lowerNode = lowerNode.nbhd[orientation];
			upperNode = upperNode.nbhd[orientation];
		}
		if (flip) {
			return new EmbeddedNode[] { upperNode, lowerNode };
		} else {
			return new EmbeddedNode[] { lowerNode, upperNode };
		}
	}
	 */
	/**
	 * Interpolate level set value from grid nodes
	 */
	public Point3f interpolateLevelSet(int i1, int j1, int k1, int i2, int j2,
			int k2, double lev) {
		return interpolateLevelSet(grid[i1][j1][k1], grid[i2][j2][k2], lev);
	}
	/**
	 * Interpolate level set value from grid nodes
	 */
	public Point3f interpolateLevelSet(EmbeddedNode c1, EmbeddedNode c2,
			double lev) {
		if (c1 == null || c2 == null) {
			System.err.println(c1 + " " + c2);
			System.exit(1);
		}
		if (c2.getNormalizedLength() < lev) {
			EmbeddedNode tmp = c2;
			c2 = c1;
			c1 = tmp;
		}
		int orientation = -1;
		if (c2.i != c1.i) {
			orientation = (c2.i > c1.i) ? EmbeddedNode.EAST : EmbeddedNode.WEST;
		} else if (c2.j != c1.j) {
			orientation = (c2.j > c1.j) ? EmbeddedNode.NORTH
					: EmbeddedNode.SOUTH;
		} else if (c2.k != c1.k) {
			orientation = (c2.k > c1.k) ? EmbeddedNode.UP : EmbeddedNode.DOWN;
		} else {
			System.err.println("NOT A VALID ORIENTATION " + c1 + "\n" + c2);
			System.exit(0);
		}
		EmbeddedNode lowerNode = c1;
		EmbeddedNode upperNode = c1.nbhd[orientation];
		while (upperNode.getNormalizedLength() < lev) {
			lowerNode = lowerNode.nbhd[orientation];
			upperNode = upperNode.nbhd[orientation];
		}
		Point3f p = new Point3f();
		Point3f neg = lowerNode.getLocation();
		Point3f pos = upperNode.getLocation();
		// if(lowerNode instanceof OutsideNode || upperNode instanceof
		// OutsideNode) {
		// System.out.println("SHOULD NOT INTERPLOATE
		// "+lowerNode.getNormalizedLength()+"
		// "+upperNode.getNormalizedLength());
		// }
		double v1 = lowerNode.getNormalizedLength() - lev;
		double v2 = upperNode.getNormalizedLength() - lev;
		double tmp = v2 - v1;
		p.x = (float) ((neg.x * v2 - pos.x * v1) / tmp);
		p.y = (float) ((neg.y * v2 - pos.y * v1) / tmp);
		p.z = (float) ((neg.z * v2 - pos.z * v1) / tmp);
		return p;
	}
	/**
	 * Interpolate thickness along edge
	 * @param p
	 * @return
	 */
	public double interpolateThicknessOnEdge(Point3f p) {
		EmbeddedNode[] edge = getEdge(p);
		double d1 = edge[0].getLocation().distance(p);
		double d2 = edge[1].getLocation().distance(p);
		if (d1 + d2 == 0) {
			return edge[0].getThickness();
		} else {
			return (d2 * edge[0].getThickness() + d1 * edge[1].getThickness())
					/ (d1 + d2);
		}
	}
	/**
	 * Interpolate streamline length along edge
	 * @param p
	 * @return
	 */
	public double interpolateLengthOnEdge(Point3f p) {
		EmbeddedNode[] edge = getEdge(p);
		double d1 = edge[0].getLocation().distance(p);
		double d2 = edge[1].getLocation().distance(p);
		if (d1 + d2 == 0) {
			System.err.println("CANNOT FIND EDGE " + p + " "
					+ edge[0].getLocation() + " " + edge[1].getLocation());
			return edge[0].getLength();
		} else {
			double d = (d2 * edge[0].getLength() + d1 * edge[1].getLength())
					/ (d1 + d2);
			return d;
		}
	}
	/**
	 * Interpolate either inner or outer correspondence point along edge using SLERP
	 * @param p
	 * @return
	 */
	public Point3f interpolateCorrespondence(Point3f p) {
		EmbeddedNode[] edge = getEdge(p);
		double d1 = edge[0].getLocation().distance(p);
		double d2 = edge[1].getLocation().distance(p);

		Point3f p1 = edge[0].getCorrespodence();
		Point3f p2 = edge[1].getCorrespodence();
		if (p1 == null && p2 != null) {
			// System.out.println(p1+" "+p2+"
			// "+edge[0].getClass().getCanonicalName()+"
			// "+edge[1].getClass().getCanonicalName());
			return new Point3f(p2);
		} else if (p1 != null && p2 == null) {
			// System.out.println(p1+" "+p2+"
			// "+edge[0].getClass().getCanonicalName()+"
			// "+edge[1].getClass().getCanonicalName());
			return new Point3f(p1);
		} else if (p1 == null && p2 == null) {
			System.err.println("MISSING POINT "
					+ edge[0].getClass().getCanonicalName() + " "
					+ edge[1].getClass().getCanonicalName());
			return new Point3f(0, 0, 0);
		}
		double t = (d1 + d2 > 1E-10) ? d1 / (d1 + d2) : 0;
		Point3f ret = new Point3f(GeometricUtilities.slerp(p1, p2, t));
		if (Float.isNaN(ret.x)) {
			System.out.println("RETURN VALUE NaN " + ret + " " + p1 + " " + p2
					+ " " + d1 + " " + d2 + " "
					+ edge[0].getRegionLabelString() + " "
					+ edge[1].getRegionLabelString() + " "
					+ edge[0].getLength() + " " + edge[0].getLength()
					+ EmbeddedNode.isCompareToInner());
			System.out.println(edge[0]);
			System.out.println(edge[1]);
			System.exit(0);
		}
		return ret;

	}
	/**
	 * Interpolate outer correspondence along edge using SLERP
	 * @param p
	 * @return
	 */
	public Point3f interpolateOuterCorrespondence(Point3f p) {
		EmbeddedNode[] edge = getEdge(p);
		double d1 = edge[0].getLocation().distance(p);
		double d2 = edge[1].getLocation().distance(p);

		Point3f p1 = edge[0].getOuterPoint();
		Point3f p2 = edge[1].getOuterPoint();

		if (p1 == null && p2 != null)
			return p2;
		if (p2 == null && p1 != null)
			return p1;
		if (p1 != null && p2 != null) {
			Matrix A = GeometricUtilities.rotateInverseMatrix(p1);
			Point2d r1 = GeometricUtilities.toUnitSpherical(GeometricUtilities.multMatrix(A, p1));
			Point2d r2 = GeometricUtilities.toUnitSpherical(GeometricUtilities.multMatrix(A, p2));

			Point2d result = new Point2d();
			result.x = ((d2 * r1.x + d1 * r2.x) / (d1 + d2));
			result.y = ((d2 * r1.y + d1 * r2.y) / (d1 + d2));

			return GeometricUtilities.multMatrix(A.inverse(), GeometricUtilities
					.toCartesian(result));
		} else {
			System.out
					.println("Could Not Locate Correspondence Edge for Outer\n"
							+ edge[0] + " " + edge[1]);
			return new Point3f(0, 0, 0);
		}
	}
	/**
	 * Interpoalte inner correspondence along edge using SLERP
	 * @param p
	 * @return
	 */
	public Point3f interpolateInnerCorrespondence(Point3f p) {
		EmbeddedNode[] edge = getEdge(p);
		double d1 = edge[0].getLocation().distance(p);
		double d2 = edge[1].getLocation().distance(p);

		Point3f p1 = edge[0].getInnerPoint();
		Point3f p2 = edge[1].getInnerPoint();
		if (p1 == null && p2 != null)
			return p2;
		if (p2 == null && p1 != null)
			return p1;
		if (p1 != null && p2 != null) {
			Matrix A = GeometricUtilities.rotateInverseMatrix(p1);
			Point2d r1 = GeometricUtilities.toUnitSpherical(GeometricUtilities.multMatrix(A, p1));
			Point2d r2 = GeometricUtilities.toUnitSpherical(GeometricUtilities.multMatrix(A, p2));

			Point2d result = new Point2d();
			result.x = ((d2 * r1.x + d1 * r2.x) / (d1 + d2));
			result.y = ((d2 * r1.y + d1 * r2.y) / (d1 + d2));

			return GeometricUtilities.multMatrix(A.inverse(), GeometricUtilities
					.toCartesian(result));
		} else {
			System.out
					.println("Could Not Locate Correspondence Edge for Inner\n"
							+ edge[0] + " " + edge[1]);
			return new Point3f(0, 0, 0);
		}
	}
	/**
	 * Copy thickness values from grid to meshes
	 * @param innerMesh
	 * @param outerMesh
	 */
	public void updateMeshThickness(EmbeddedSurface innerMesh,
			EmbeddedSurface outerMesh) {
		int innerVertCount = innerMesh.getVertexCount();
		int outerVertCount = outerMesh.getVertexCount();
		for (int i = 0; i < innerVertCount; i++) {
			innerMesh.setVertexData(i, new double[] {boundaryNodes[i]
					.getThickness() });
		}
		for (int i = 0; i < outerVertCount; i++) {
			outerMesh.setVertexData(i,
					new double[] { boundaryNodes[innerVertCount + i]
							.getThickness() });
		}
	}
	/**
	 * Copy thickness values and mask values from grid to meshes 
	 * @param innerMesh
	 * @param outerMesh
	 * @param innerMask
	 * @param outerMask
	 */
	public void updateMeshThickness(EmbeddedSurface innerMesh,
			EmbeddedSurface outerMesh, int[] innerMask, int[] outerMask) {
		int innerVertCount = innerMesh.getVertexCount();
		int outerVertCount = outerMesh.getVertexCount();
		for (int i = 0; i < innerVertCount; i++) {
			innerMesh.setVertexData(i, new double[] {
					boundaryNodes[i].getThickness(), innerMask[i] });
		}
		for (int i = 0; i < outerVertCount; i++) {
			outerMesh.setVertexData(i, new double[] {
					boundaryNodes[innerVertCount + i].getThickness(),
					outerMask[i] });
		}
	}
	/**
	 * Copy thickness and correspondence values to meshes
	 * @param innerMesh
	 * @param outerMesh
	 */
	public void updateMeshThicknessAndCorrespondence(EmbeddedSurface innerMesh,
			EmbeddedSurface outerMesh) {
		updateInnerMeshThicknessAndCorrespondence(innerMesh);
		updateOuterMeshThicknessAndCorrespondence(outerMesh);
	}
	/**
	 * Copy thickness and correspondence to inner mesh
	 * @param innerMesh
	 */
	public void updateInnerMeshThicknessAndCorrespondence(
			EmbeddedSurface innerMesh) {
		int innerVertCount = innerMesh.getVertexCount();
		Point3f pout;
		for (int i = 0; i < innerVertCount; i++) {
			pout = boundaryNodes[i].getOuterPoint();
			/*
			if (GeomUtil.length(pout) < 0.99) {
				System.out.println("INVALID LENGTH LOCATION "
						+ boundaryNodes[i].getLocation() + " Q "
						+ boundaryNodes[i].getCorrespodence() + " LABEL "
						+ boundaryNodes[i].getRegionLabelString() + " "
						+ boundaryNodes[i].getMarchingLabelString() + " "
						+ boundaryNodes[i].getNeighbors().size());
				for (EmbeddedNode nd : boundaryNodes[i].getNeighbors()) {
					System.out.println("NBR " + nd.getMarchingLabelString()
							+ " " + nd.getRegionLabelString() + " "
							+ nd.getLocation() + " " + nd.getCorrespodence());
				}
			}
			*/
			innerMesh.setVertexData(i, new double[] {
					pout.x, pout.y, pout.z,boundaryNodes[i].getThickness() });
		}

	}
	/**
	 * Copy thickness and correspondence to outer mesh
	 * @param outerMesh
	 */
	public void updateOuterMeshThicknessAndCorrespondence(
			EmbeddedSurface outerMesh) {

		int outerVertCount = outerMesh.getVertexCount();
		int innerVertCount = boundaryNodes.length - outerVertCount;
		Point3f pout;
		for (int i = 0; i < outerVertCount; i++) {
			pout = boundaryNodes[innerVertCount + i].getInnerPoint();
			/*
			if (GeomUtil.length(pout) < 0.99) {
				System.out.println("INVALID LENGTH LOCATION "
						+ boundaryNodes[i].getLocation() + " Q "
						+ boundaryNodes[i].getCorrespodence() + " LABEL "
						+ boundaryNodes[i].getRegionLabelString() + " "
						+ boundaryNodes[i].getMarchingLabelString() + " "
						+ boundaryNodes[i].getNeighbors().size());
			}
			*/
			outerMesh.setVertexData(i, new double[] {
					 pout.x,
					pout.y, pout.z ,boundaryNodes[innerVertCount + i].getThickness()});
		}

	}
}
