package edu.jhu.ece.iacl.algorithms.thickness.laplace;

import java.util.LinkedList;
import java.util.List;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.thickness.grid.EmbeddedNode;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import Jama.*;

/**
 * Solve laplace's equation on rectilinear graph using Dirichlet boundary
 * conditions.
 * 
 * @author Blake Lucas
 * 
 */
public class LaplaceSolveOnGraphDirichlet extends LaplaceSolveOnGraph {
	public LaplaceSolveOnGraphDirichlet(AbstractCalculation parent, int rows,
			int cols, int slices) {
		super(parent, rows, cols, slices);
	}

	private static final double LOW_BOUND = 0;
	private static final double HIGH_BOUND = 1;

	protected double computeLaplace(int i, int j, int k) {

		double dxp = 0, dyp = 0, dzp = 0, dxn = 0, dyn = 0, dzn = 0;
		double uxp = 0, uyp = 0, uzp = 0, uxn = 0, uyn = 0, uzn = 0;
		EmbeddedNode neighbor;
		double e = 0;
		EmbeddedNode node = gridMat[i][j][k];
		if (node != null
				&& (node.regionLabel == OUTER_BOUNDARY || node.regionLabel == INNER_BOUNDARY))
			return 0;
		if (i + 1 < rows) {
			neighbor = (node == null) ? null : node.nbhd[EAST];
			if (neighbor != null
					&& (neighbor.regionLabel == INNER_BOUNDARY || neighbor.regionLabel == OUTER_BOUNDARY)) {
				dxp = neighbor.getLocation().x - i;
				uxp = (((neighbor.regionLabel == INNER_BOUNDARY) ? LOW_BOUND
						: HIGH_BOUND) - laplaceMat[i][j][k])
						/ dxp;

			} else {
				uxp = laplaceMat[i + 1][j][k] - laplaceMat[i][j][k];
				dxp = 1;
			}
		}
		if (i > 0) {
			neighbor = (node == null) ? null : node.nbhd[WEST];
			if (neighbor != null
					&& (neighbor.regionLabel == INNER_BOUNDARY || neighbor.regionLabel == OUTER_BOUNDARY)) {
				dxn = i - neighbor.getLocation().x;
				uxn = (laplaceMat[i][j][k] - ((neighbor.regionLabel == INNER_BOUNDARY) ? LOW_BOUND
						: HIGH_BOUND))
						/ dxn;

			} else {
				uxn = laplaceMat[i][j][k] - laplaceMat[i - 1][j][k];
				dxn = 1;
			}
		}
		if (j + 1 < cols) {
			neighbor = (node == null) ? null : node.nbhd[NORTH];
			if (neighbor != null
					&& (neighbor.regionLabel == INNER_BOUNDARY || neighbor.regionLabel == OUTER_BOUNDARY)) {
				dyp = neighbor.getLocation().y - j;
				uyp = (((neighbor.regionLabel == INNER_BOUNDARY) ? LOW_BOUND
						: HIGH_BOUND) - laplaceMat[i][j][k])
						/ dyp;
			} else {
				uyp = laplaceMat[i][j + 1][k] - laplaceMat[i][j][k];
				dyp = 1;
			}
		}
		if (j > 0) {
			neighbor = (node == null) ? null : node.nbhd[SOUTH];
			if (neighbor != null
					&& (neighbor.regionLabel == INNER_BOUNDARY || neighbor.regionLabel == OUTER_BOUNDARY)) {
				dyn = j - neighbor.getLocation().y;
				uyn = (laplaceMat[i][j][k] - ((neighbor.regionLabel == INNER_BOUNDARY) ? LOW_BOUND
						: HIGH_BOUND))
						/ dyn;
			} else {
				uyn = laplaceMat[i][j][k] - laplaceMat[i][j - 1][k];
				dyn = 1;
			}
		}
		if (k + 1 < slices) {
			neighbor = (node == null) ? null : node.nbhd[UP];
			if (neighbor != null
					&& (neighbor.regionLabel == INNER_BOUNDARY || neighbor.regionLabel == OUTER_BOUNDARY)) {
				dzp = neighbor.getLocation().z - k;
				uzp = (((neighbor.regionLabel == INNER_BOUNDARY) ? LOW_BOUND
						: HIGH_BOUND) - laplaceMat[i][j][k])
						/ dzp;
			} else {
				uzp = laplaceMat[i][j][k + 1] - laplaceMat[i][j][k];
				dzp = 1;
			}
		}
		if (k > 0) {
			neighbor = (node == null) ? null : node.nbhd[DOWN];
			if (neighbor != null
					&& (neighbor.regionLabel == INNER_BOUNDARY || neighbor.regionLabel == OUTER_BOUNDARY)) {
				dzn = k - neighbor.getLocation().z;
				uzn = (laplaceMat[i][j][k] - ((neighbor.regionLabel == INNER_BOUNDARY) ? LOW_BOUND
						: HIGH_BOUND))
						/ dzn;
			} else {
				uzn = laplaceMat[i][j][k] - laplaceMat[i][j][k - 1];
				dzn = 1;
			}
		}
		double dx = 0.5 * (dxp + dxn);
		double dy = 0.5 * (dyp + dyn);
		double dz = 0.5 * (dzp + dzn);
		double denom = (1 / dxp + 1 / dxn) / dx + (1 / dyp + 1 / dyn) / dy
				+ (1 / dzp + 1 / dzn) / dz;
		// System.out.printf("(%f,%f,%f) (%f,%f,%f) (%f,%f,%f) (%f,%f,%f)\n",uxp,uyp,uzp,uxn,uyn,uzn,dxp,dyp,dzp,dxn,dyn,dzn);
		e = ((uxp - uxn) / dx + (uyp - uyn) / dy + (uzp - uzn) / dz) / denom;
		// System.out.println("UDPDATE "+laplaceMat[i][j][k]+" "+e+" "+dx+" "+dy+" "+dz);
		// e=-0.5*((uxp-uxn)/(dxp+dxn)+(uyp-uyn)/(dyp+dyn)+(uzp-uzn)/(dzp+dzn))/((dxp+dxn)*(dyp+dyn)*(dzp+dzn));
		return e;

	}

	protected void computeGradient(int i, int j, int k) {
		EmbeddedNode node = grid.getNode(i, j, k);
		if (node == null || node.isOutside())
			return;
		List<EmbeddedNode> upwindNodes = new LinkedList<EmbeddedNode>();
		double up = node.getLength();
		for (EmbeddedNode nd : node.getNeighbors()) {
			if (!nd.isOutside()) {
				upwindNodes.add(nd);
			}
		}
		Matrix D = new Matrix(upwindNodes.size(), 3);
		Matrix W = new Matrix(upwindNodes.size(), upwindNodes.size());
		Matrix GU = new Matrix(upwindNodes.size(), 1);
		Point3f pivot = node.getLocation();
		Vector3f diff = new Vector3f();
		double w;

		for (int n = 0; n < upwindNodes.size(); n++) {
			EmbeddedNode nd = upwindNodes.get(n);
			diff.sub(pivot, nd.getLocation());
			D.set(n, 0, diff.x);
			D.set(n, 1, diff.y);
			D.set(n, 2, diff.z);
			w = diff.length();
			w = (w > 1E-6) ? 1 / w : 1E6;
			W.set(n, n, w);
			GU.set(n, 0, up - nd.getLength());
		}
		Matrix Dsqr = D.transpose().times(W).times(D);
		SingularValueDecomposition svd = new SingularValueDecomposition(Dsqr);
		Matrix S = svd.getS();
		Matrix V = svd.getV();
		Matrix U = svd.getU();
		int zeros = 0;
		for (int n = 0; n < S.getColumnDimension(); n++) {
			if (Math.abs(S.get(n, n)) > 1E-12) {
				S.set(n, n, 1 / S.get(n, n));
			} else {
				zeros++;
				S.set(n, n, 0);
			}
		}
		Matrix Dinv = V.times(S.times(U.transpose())).times(D.transpose())
				.times(W);
		Matrix T = Dinv.times(GU);
		Vector3f tan = new Vector3f((float) T.get(0, 0), (float) T.get(1, 0),
				(float) T.get(2, 0));
		/*
		 * if(tan.length()==0){
		 * System.out.println("TOO SHORT "+node.getRegionLabelString
		 * ()+" "+node.getLength()+" "+node.getLocation()); for(EmbeddedNode
		 * nd:node.getNeighbors()){
		 * System.out.println(nd.getRegionLabelString()+" "+nd.getLength()); } }
		 */
		tangetFieldMat[i][j][k][0] = tan.x;
		tangetFieldMat[i][j][k][1] = tan.y;
		tangetFieldMat[i][j][k][2] = tan.z;
	}
}
