package edu.jhu.ece.iacl.plugins.labeling;


import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;


import edu.jhu.ece.iacl.jist.io.*;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamObject;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamSurface;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamSurfaceCollection;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;


public class MedicAlgorithmSurfaceLabelStatistics extends ProcessingAlgorithm {
	/*
	 * Inputs.
	 */
	protected ParamSurfaceCollection statSurfaces;
	protected ParamSurfaceCollection labelSurfaces;
	protected ParamInteger valueOffset;
	protected ParamBoolean NByM;

	
	/*
	 * Results.
	 */
	protected ParamObject<Object[][]> stats;


	private static final String cvsversion = "$Revision: 1.7 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Computes statistics for each labeled surface region. If multiple surfaces are given the output is written to one file.\nAlgorithm Version: " + revnum + "\n";
	private static final String longDescription = "The 'Surface with Statistic' should have thickness or another measurement embedded in its vertices. The 'Surface with Labels' has a labelled version of the surface. The output csv file can be easily read with Excel or Matlab.";


	protected void createInputParameters(ParamCollection inputParams){
		statSurfaces = new ParamSurfaceCollection("Surface(s) with Statistic");
		statSurfaces.setDescription("Surface(s) with the numeric values on which the statistics are computed.");

		labelSurfaces = new ParamSurfaceCollection("Surface(s) with Labels");
		labelSurfaces.setDescription("Surface(s) with corresponding labelled regions.");

		valueOffset = new ParamInteger("Value to compute statistics", 0, 100, 0);
		valueOffset.setDescription("For surfaces with many values associated with each vertex, we can choose which values the statistics should be computed on.");

		NByM = new ParamBoolean("Compute statistics for all possible combinations", false);
		NByM.setDescription("Compute statistics for each labeled surface against each surface with a statistic. Given N surfaces with labels and M surfaces with statistics generate results for all possible combinations.");



		inputParams.add(statSurfaces);
		inputParams.add(labelSurfaces);
		inputParams.add(valueOffset);
		inputParams.add(NByM);


		inputParams.setPackage("IACL");
		inputParams.setCategory("Measurement.Surface");
		inputParams.setLabel("Surface Label Statistics");
		inputParams.setName("Surface_Label_Statistics");


		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("http://www.iacl.ece.jhu.edu/");
		info.add(new AlgorithmAuthor("Aaron Carass", "aaron_carass@jhu.edu", "http://www.iacl.ece.jhu.edu/"));
		info.setAffiliation("Johns Hopkins University, Department of Electrical and Computer Engineering");
		info.setDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.Release);
		info.setAdditionalDocURL("html/edu/jhu/ece/iacl/plugins/labeling/MedicAlgorithmSurfaceLabelStatistics/index.html");
	}


	protected void createOutputParameters(ParamCollection outputParams){
		stats = new ParamObject<Object[][]>("Statistics", new ArrayObjectTxtReaderWriter());
		stats.setDescription("Output statistics.");


		outputParams.add(stats);
	}


	protected void execute(CalculationMonitor monitor){
		System.out.println("MedicAlgorithmSurfaceLabelStatistics.execute()");
		SurfaceLabelStatistics algorithm = new SurfaceLabelStatistics();


		monitor.observe(algorithm);


		algorithm.solve(statSurfaces, labelSurfaces);
		stats.setObject(algorithm.statistics);
		stats.setFileName("SurfaceLabelStatistics");
	}


	protected class SurfaceLabelStatistics extends AbstractCalculation {
		public SurfaceLabelStatistics(){
		}


	 	Object[][] statistics;
		ArrayList labelArrayList;
		int surface = 0;


		public void labelList(ParamSurfaceCollection labelSurfaces){
			System.out.print("\n\n");
			System.out.print("MedicAlgorithmSurfaceLabelStatistics.labelList()");
			System.out.print("\n\n");

			double label = -1.0;


			/*
			 * Create a list of the labels in the data and
			 * add them to the HashSet labelHashSet.
			 * The HashSet will only have the unique elements.
			 * Then put these into a sorted ArrayList labelArrayList.
			 */
			HashSet labelHashSet = new HashSet<Integer>();
			for (EmbeddedSurface thisLabelSurface : labelSurfaces.getSurfaceList()){
				int i = 0;
				int vertexNumber = thisLabelSurface.getVertexCount();

				for (i = 0; i < vertexNumber; i++) {
					label = thisLabelSurface.getVertexData(i)[0];
					labelHashSet.add(label);
				}
			}


			labelArrayList = new ArrayList(labelHashSet);
			Collections.sort(labelArrayList);
			System.out.println("There are " + labelArrayList.size() + " labels.");
		}


		public void solve(ParamSurfaceCollection statsSurfaces, ParamSurfaceCollection labelSurfaces){
			labelList(labelSurfaces);


			int s = 0, t = 0;
			List<EmbeddedSurface> statsSurfacesList = statsSurfaces.getSurfaceList();
			List<EmbeddedSurface> labelSurfacesList = labelSurfaces.getSurfaceList();

			int S = statsSurfacesList.size();
			int T = labelSurfacesList.size();


			if (S != T && NByM.getValue() == false){
				System.out.print("\n\n");
				System.out.print("statsSurfacesList.size() != labelSurfacesList.size()\n");
				System.out.print("NByM.getValue() == false\n");
				System.out.print("Exiting\n\n");
				System.exit(1);
			}


			if (NByM.getValue() == true){
				statistics = new Object[1 + 5 * S * T][0];
			} else {
				statistics = new Object[1 + 5 * S][0];
			}


			if (labelArrayList.size() != 0){
				int l = 0;
				String tmp_string = "\", ";

				for (l = 0; l < labelArrayList.size() - 1; l++){
					tmp_string += "\"" + labelArrayList.get(l) + "\", ";
				}
				tmp_string += "\"" + labelArrayList.get(l);
				statistics[0] = new Object[] { tmp_string };
			}


			if (NByM.getValue() == true){
				for (s = 0; s < S; s++){
					for (t = 0; t < T; t++){
						EmbeddedSurface statsSurface = statsSurfacesList.get(s);
						EmbeddedSurface labelSurface = labelSurfacesList.get(t);
						solve(statsSurface, labelSurface);
					}
				}
			} else {
				for (s = 0; s < S; s++){
					EmbeddedSurface statsSurface = statsSurfacesList.get(s);
					EmbeddedSurface labelSurface = labelSurfacesList.get(s);
					System.out.print("\n\nProcessing surface = " + surface + "\n\n");
					solve(statsSurface, labelSurface);
				}
			}
		}


		public void solve(EmbeddedSurface statMesh, EmbeddedSurface labelMesh){
			int vertexNumber = statMesh.getVertexCount();

			System.out.println("statMesh.getVertexCount():  " + statMesh.getVertexCount());
			System.out.println("labelMesh.getVertexCount(): " + labelMesh.getVertexCount());


			if (vertexNumber != labelMesh.getVertexCount()) {
				System.err.println("\n\n");
				System.out.println("\n\n");
				for (int i = 0; i < 10; i++){
					System.err.println("MESHES DO NOT HAVE SAME NUMBER OF VERTICES");
					System.out.println("MESHES DO NOT HAVE SAME NUMBER OF VERTICES");
				}
				System.err.println("\n\n");
				System.out.println("\n\n");

				System.exit(1);
			}


			int i = 0, j = 0;
			double label = -1.0;


			if (labelArrayList.size() != 0){
				float[][] values = new float[5][labelArrayList.size() + 1];
				String name = statMesh.getName();

				if (name.indexOf("_") > 0){
					String tmp_name = name.substring(0, name.indexOf("_"));
					name = tmp_name;
				}


				float data = 0.0f, min = 0.0f, max = 0.0f;
				float mean = 0.0f, std = 0.0f, median = 0.0f;
				float tmp = 0.0f;
				int counter = 0;

				ArrayList dataList = new ArrayList();


				for (j = 0;j < labelArrayList.size();j++){
					System.out.println("Label: " + (j + 1) + " / " + labelArrayList.size());
					data = 0.0f;
					mean = 0.0f; std = 0.0f; median = 0.0f;
					dataList.clear();

					System.out.println("\tlabelList.get(" + j + "): " + labelArrayList.get(j));

					for (i = 0; i < vertexNumber; i++) {
						label = labelMesh.getVertexData(i)[0];


						if (((int) label) == (Double) labelArrayList.get(j)){
							data = (float) statMesh.getVertexData(i)[valueOffset.getInt()];
							dataList.add((Float) data);
							mean += data;
						}
					}


					counter += dataList.size();


					if (dataList.size() != 0){
						mean /= dataList.size();
						Collections.sort(dataList);


						for (i = 0;i < dataList.size(); i++){
							std += (mean - (Float) dataList.get(i))*(mean - (Float) dataList.get(i));
						}
						tmp = (std)/(float) (((Integer) dataList.size()) - 1.0f);
						std = (float) Math.sqrt(tmp);


						if (dataList.size() % 2 == 0){
							median = 0.500f * ((Float)dataList.get(dataList.size()/2) + (Float)dataList.get(dataList.size()/2 - 1));
						} else {
							median = (Float) dataList.get(dataList.size()/2);
						}


						min = (Float) dataList.get(0);
						max = (Float) dataList.get(dataList.size() - 1);
						System.out.println("\tMean: " + mean);
						System.out.println("\tMedian: " + median);
						System.out.println("\tStd Dev: " + std);
						System.out.println("\tMin: " + min);
						System.out.println("\tMax: " + max);
						System.out.println("\tdataList.size(): " + dataList.size());
					} else {
						System.out.println("Label not in the data.");
						mean = 0.0f;
						median = 0.0f;
						std = 0.0f;
						min = 0.0f;
						max = 0.0f;
					}


					values[0][j] = mean;
					values[1][j] = median;
					values[2][j] = std;
					values[3][j] = min;
					values[4][j] = max;
				}




				String tmp_string = name + "\", " ;


				for (i = 0; i < 5; i++){
					if (i == 0){
						tmp_string = name + " Mean\", ";
					} else if (i == 1){
						tmp_string = name + " Median\", ";
					} else if (i == 2){
						tmp_string = name + " Std. Dev.\", ";
					} else if (i == 3){
						tmp_string = name + " Min\", ";
					} else if (i == 4){
						tmp_string = name + " Max\", ";
					} else {
						tmp_string = "";
					}


					for (j = 0;j < labelArrayList.size() - 1; j++){
						tmp_string += "\"" + values[i][j] + "\", ";
					}
					tmp_string += "\"" + values[i][j];

					statistics[surface + 1] = new Object[] { tmp_string };
					surface++;
				}
			}
		}
	}
}
