package edu.jhu.ece.iacl.plugins.measure.statistics;

import edu.jhu.ece.iacl.algorithms.PrinceGroupAuthors;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SurfaceIntersector;
import edu.jhu.ece.iacl.jist.io.*;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.*;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedPointSet;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataMipav;

import javax.vecmath.Point3d;
import javax.vecmath.Point3f;

import java.lang.Math;
import java.util.Arrays;

import gov.nih.mipav.model.structures.ModelImage;
import gov.nih.mipav.view.ViewUserInterface;

public class MedicAlgorithmIsoLevelSurfaceLandmarks extends ProcessingAlgorithm {
	ParamVolume volParam;
	ParamSurface surfParam;
	ParamSurface isoSurf;
	ParamDouble maxDist,meanDist,medianDist,stdDist,minDist;

	private static final String cvsversion = "$Revision: 1.4 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	
	protected void createInputParameters(ParamCollection inputParams) {
		inputParams.add(volParam=new ParamVolume("Iso-Level Volume"));
		inputParams.add(surfParam=new ParamSurface("Surface"));
		inputParams.setLabel("Iso-Levels for Surface");
		inputParams.setName("isolevels4surf");

		inputParams.setPackage("IACL");
		inputParams.setCategory("Measurement.Surface");

		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("http://www.iacl.ece.jhu.edu/");
		info.setVersion(revnum);
		info.setEditable(false);
		info.setDescription("Interpolate volumetric data calculated at surface vertices. The data commonly corresponds to geodesic distance, tissue membership, or thickness measurements.");
		info.setLongDescription("This is useful to determine the geodesic distance difference between a FreeSurfer surface and a CRUISE implicit surface, assuming they are in the same coordinate space. To align surfaces in different coordinate spaces, you'll need the reorient surface module and reshape surface module to account for rotation and translation differences.");
		info.setStatus(DevelopmentStatus.RC);
		info.add(PrinceGroupAuthors.blakeLucas);
	}

	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(meanDist=new ParamDouble("Mean Iso-Level"));
		outputParams.add(medianDist=new ParamDouble("Median Iso-Level"));
		outputParams.add(stdDist=new ParamDouble("Std. Dev. Iso-Level"));
		outputParams.add(minDist=new ParamDouble("Min Iso-Level"));
		outputParams.add(maxDist=new ParamDouble("Max Iso-Level"));
		outputParams.add(isoSurf=new ParamSurface("Iso-Level Surface"));
	}
	
	protected void execute(CalculationMonitor monitor) {
		//get vertex points form embedded surface
		
		ImageDataMipav vol=new ImageDataMipav(volParam.getImageData());		
		ModelImage img=vol.getModelImageCopy();
		
		EmbeddedSurface surf=surfParam.getSurface();
		//get landmark landmark points
		Point3f[] landmarks=surf.getVertexCopy();
		double[][] dists=new double[landmarks.length][1];
		double distCopy[]=new double[landmarks.length];
		//compute distance form landmark to surface 
		for (int i = 0; i< landmarks.length; i++){ 
			Point3f p=landmarks[i];
			distCopy[i]=dists[i][0]=img.getFloatTriLinearBounds(p.x,p.y,p.z);
		}
		Arrays.sort(distCopy);
		double median=(distCopy.length%2==0)?0.5*(distCopy[distCopy.length/2]+distCopy[distCopy.length/2-1]):distCopy[distCopy.length/2];
		double min=Double.MAX_VALUE;
		double max=Double.MIN_VALUE;
		double sum=0;
		double sqrs=0;
		int count=dists.length;
		for(int i=0;i<count;i++){
			min=Math.min(min, dists[i][0]);
			max=Math.max(max, dists[i][0]);
			sqrs+=dists[i][0]*dists[i][0];
			sum+=dists[i][0];
		}
		double mean=(sum/count);
		double stdev=Math.sqrt((sqrs-sum*sum/count)/(count-1));
		meanDist.setValue(mean);
		minDist.setValue(min);
		maxDist.setValue(max);
		stdDist.setValue(stdev);
		medianDist.setValue(median);
		
		EmbeddedSurface surfClone=surf.clone();
		surfClone.setName(surf.getName()+"_iso");
		surfClone.setVertexData(dists);
		isoSurf.setValue(surfClone);
		img.disposeLocal();
	}
}
