package edu.jhu.ece.iacl.plugins.measure.statistics;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.StringTokenizer;




import edu.jhmi.rad.medic.utilities.MedicUtilPublic;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.parameter.*;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataMipav;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;
import gov.nih.mipav.view.MipavUtil;



public class MedicAlgorithmSegmentationStatistics extends ProcessingAlgorithm {
	
	private ParamVolumeCollection segImage;
	private ParamVolumeCollection segLesion;
	private	ParamFile 	atlasFile;
	private ParamString statFile;
	private File resultFile;
	private	ParamString omit;
	
	private static final String cvsversion = "$Revision: 1.4 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Computes structure volumes segmented by TOADS/LEsionTOADS.\nAlgorithm Version: " + revnum + "\n";
	private static final String longDescription = "The segmented image is the classification output of TOADS/LesionTOADS. Tha atlas used in the segmentation algorithm should also be provided. The statistic file should be a csv file. If it already contains the statistics from other subjects, the new statistics will be appended to the end of the file.";
		
	protected void createInputParameters(ParamCollection inputParams) {	
	
		inputParams.add(segImage = new ParamVolumeCollection("Structure Segmentation Image",VoxelType.UBYTE));
		segImage.setDescription("Segmentation output from TOADS or LesionTOADS");
		inputParams.add(segLesion = new ParamVolumeCollection("Lesion Segmentation Image",VoxelType.UBYTE));
		segLesion.setDescription("Lesion Segmentation output from LesionTOADS");
		segLesion.setMandatory(false);
		inputParams.add(atlasFile = new ParamFile ("Atlas File"));
		atlasFile.setDescription("Atlas file used in the segmentation algorithm");
		inputParams.add(statFile = new ParamString("Statistic File",""));
		statFile.setMandatory(false);
		inputParams.add(omit = new ParamString("Truncating character", "_"));
		omit.setDescription("Omit the name of the file in statitic report after first occurence of this character");
		omit.setMandatory(false);
		inputParams.setPackage("IACL");
		inputParams.setCategory("Measurement.Statistics");
		inputParams.setName("segmentation statisitics");
		inputParams.setLabel("Segmentation Statistics");
		
		AlgorithmInformation info = getAlgorithmInformation();
		info.add(new AlgorithmAuthor("Navid Shiee", "navid@jhu.edu","http://medic.rad.jhmi.edu/nshiee"));
		info.setAffiliation("Johns Hopkins University, Department of Electrical and Computer Engineering");
		info.setDescription(shortDescription+longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.RC);

	}
	
	protected void createOutputParameters(ParamCollection ouputParams) {
		outputParams.setName("segmentation statistics");
		outputParams.setLabel("Segmentation Statistics");
	}
	
	protected void execute(CalculationMonitor monitor) {
		
		boolean hasLesion =false;
		if (segLesion.getParamVolume(0)!=null)
			if (segLesion.getImageDataList().size() != segImage.getImageDataList().size())
				MedicUtilPublic.displayError("Number of structural segmentation and lesion segmentation images do not match");
			else
				hasLesion =true;
		
			
		// Exact corresponding structure names and labels from atlas file
		try {
			File f = atlasFile.getValue();
			FileReader fr = new FileReader(f);
			BufferedReader br = new BufferedReader(fr);
			String line = br.readLine();
			StringTokenizer st;

			if (!line.equals("Structure Atlas File (edit at your own risks)")) {
				System.out.println("not a proper Structure Atlas file");
				br.close();
				fr.close();
				return;
			}
			do{
				line = br.readLine();
			}while (!line.startsWith("Structure"));
			st = new StringTokenizer(line, "	");
			st.nextToken();
			int classes = MipavUtil.getInt(st);
			byte[] label = new byte[classes+1];
			String info_line = "Subject";
			String temp; 
			byte background = -1;
			byte outerVent = -1;
			byte interVent =-1;
			byte frontal = -1;
			for (int n=0;n<classes;n++) {
				line = br.readLine();
				st = new StringTokenizer(line, "	");
				temp = st.nextToken();
				label[n] = (byte)MipavUtil.getInt(st);
				System.out.println(temp+":"+ label[n]);
				if (!(temp.equals("Background") || temp.equals("InterVentricular-WM") || temp.equals("Frontal-WM"))){
					if (temp.equals("OuterVentricular-WM")){
						info_line += ","+"CerebralWM";
						outerVent = (byte)n;
					}else if (temp.equals("Cerebrum-WM"))
						info_line += ","+"CerebralWM";
					else if (temp.equals("Cerebrum-GM")||temp.equals("CerebralGM"))
						info_line += ","+"CorticalGM";
					else if (temp.equals("Cerebellum-WM"))
						info_line += ","+"CerebellarWM";
					else if (temp.equals("Cerebellum-GM"))
						info_line += ","+"CerebellarGM";
					else if (temp.equals("Sulcal-CSF"))
						info_line += ","+"SulcalCSF";
					else
						info_line += ","+temp;
					
				}else if (temp.equals("Background"))
					background = (byte)n;
				else if (temp.equals("InterVentricular-WM"))
					interVent = (byte)n;
				else if (temp.equals("Frontal-WM"))
					frontal = (byte)n;
			}
			label[classes] = 10; //lesionLabel
			info_line += ","+"Lesion";
			double[] volumes = new double[classes+1];
			
			int n = segImage.getImageDataList().size();
			double voxelVolume;
			String newVolumes ="";
			for (int i=0; i<n; i++){
				ImageData seg = segImage.getParamVolume(i).getImageData();
				voxelVolume = seg.getHeader().getDimResolutions()[0]
				              *seg.getHeader().getDimResolutions()[1]
				              *seg.getHeader().getDimResolutions()[2];
				for (int k=0; k<classes+1; k++) volumes[k]=0.0;				
				for (int x=0; x<seg.getRows();x++)for (int y=0;y<seg.getCols();y++)for(int z=0; z<seg.getSlices();z++){
					for (int k=0; k<classes+1; k++) { 
						if (seg.getByte(x, y, z)==label[k]){
							volumes[k] += 1.0;
							break;
						}
					}
				}
				//If there is an intervenricular WM class, add it to WM
				if (outerVent != -1 && interVent != -1)
					volumes[outerVent] += volumes[interVent];
				if (outerVent != -1 && frontal != -1)
					volumes[outerVent] += volumes[frontal];
				//If lesions included in structure segmentation, add lesion volume to WM volume
				if ( (outerVent != -1) && (volumes[classes] !=0))
					volumes[outerVent]+= volumes[classes];
				if (volumes[classes] ==0 && hasLesion){
					seg = new ImageDataMipav(segLesion.getParamVolume(i).getImageData());
					for (int x=0; x<seg.getRows();x++)for (int y=0;y<seg.getCols();y++)for(int z=0; z<seg.getSlices();z++) if (seg.getByte(x, y, z) != 0)
						volumes[classes] += 1.0;
				}
				//writing stat to file
				String id = segImage.getParamVolume(i).getImageData().getName();
				if (!omit.getValue().equals("") || omit.getValue()!=null)
					id = id.substring(0,id.indexOf(omit.getValue()));
				newVolumes += id;
				for (int k=0; k<classes+1; k++) if (!(k == interVent || k == background || k==frontal)){
					newVolumes += ","+volumes[k]*voxelVolume;
				}
				newVolumes +="\n";
			}
			//Check for statistic file
			//File resultFile = null;
			File destdir = null;
			if (!statFile.getValue().equals("")){
				resultFile = new File(statFile.getValue());
			}else {
				destdir = new File(this.getOutputDirectory().getCanonicalFile()+File.separator+this.getAlgorithmName());
				if(!destdir.isDirectory()){
					(new File(destdir.getCanonicalPath())).mkdir();
				}
				resultFile = new File(destdir+File.separator+"SegmentationStatistics.csv");
				System.out.println("No statistic file is given. A new file is generated: "+destdir+File.separator+"SegmentationStatistics.csv");
			}
			ArrayList<String> previous = loadStatisticsFile(info_line);
			previous.add(newVolumes);
			writeStatisticsFile(info_line,previous);
		}
		
		catch (FileNotFoundException e) {
            System.out.println(e.getMessage());
        }
        catch (IOException e) {
            System.out.println(e.getMessage());
        } 
		catch (OutOfMemoryError e){
			System.out.println(e.getMessage());
		}
		catch (Exception e) {
			System.out.println(e.getMessage());
        }
		
		
	}
	
	
	
	
	private final ArrayList<String> loadStatisticsFile(String first_line) {
		ArrayList<String> list = new ArrayList();
		try {
            System.out.println("reading previous statistic file: "+resultFile.getName());
            File f = new File(resultFile.getAbsolutePath());
            FileReader fr = new FileReader(f);
            BufferedReader br = new BufferedReader(fr);
            //read info line
            String line = br.readLine();
            
         // Exact corresponding template for first line ?
           	if (!line.startsWith(first_line)) {
                System.out.println("not a proper segmentation statistics file");
                br.close();
                fr.close();
                return null;
            }
           	//Now read stats
            line = br.readLine();
           	while (line!=null) {
				list.add(line+"\n");
				line = br.readLine();
			}
			br.close();
            fr.close();
        }
        catch (FileNotFoundException e) {
            System.out.println(e.getMessage());
        }
        catch (IOException e) {
            System.out.println(e.getMessage());
        } 
		catch (OutOfMemoryError e){
			System.out.println(e.getMessage());
		}
		catch (Exception e) {
			System.out.println(e.getMessage());
        }
		return list;	
	}

	private final void writeStatisticsFile(String info_line, ArrayList<String> list) {
		try {
            File f = new File(resultFile.getAbsolutePath());
            FileWriter fw = new FileWriter(f);
            PrintWriter pw = new PrintWriter( fw );
			pw.write(info_line+"\n");
            for (int n=0;n<list.size();n++) {
				pw.write(list.get(n));
			}
			pw.close();
            fw.close();
       }
        catch (FileNotFoundException e) {
            System.out.println(e.getMessage());
        }
        catch (IOException e) {
            System.out.println(e.getMessage());
        } 
		catch (OutOfMemoryError e){
			System.out.println(e.getMessage());
		}
		catch (Exception e) {
			System.out.println(e.getMessage());
        }
	}
}
