package edu.jhu.ece.iacl.plugins.segmentation.cruise;

//import edu.jhu.ece.iacl.algorithms.*;
import gov.nih.mipav.model.file.FileInfoBase;
import gov.nih.mipav.model.structures.ModelImage;
import gov.nih.mipav.model.structures.ModelStorageBase;
import gov.nih.mipav.view.MipavUtil;
import edu.jhu.ece.iacl.algorithms.PrinceGroupAuthors;
import edu.jhu.ece.iacl.algorithms.ReferencedPapers;
import edu.jhu.ece.iacl.algorithms.gvf.GradVecFlow;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.*;
import edu.jhu.ece.iacl.jist.pipeline.view.input.ParamNumberSliderInputView;
import edu.jhu.ece.iacl.jist.structures.image.*;
import edu.jhu.ece.iacl.jist.utility.FileUtil;
/**
 * Gradient Vector Flow Algorithm
 *
 * @author Blake Lucas
 *
 */
public class MedicAlgorithmGradVecFlow extends ProcessingAlgorithm {
	private ParamVolume inputVol;
	private ParamOption method;
	private ParamBoolean normalize;
	private ParamDouble lambda;
	private ParamInteger iterations;
	private ParamVolume outputVol;

	private static final String revnum = GradVecFlow.getVersion();


	/**
	 * Create input parameters for GVF :cubic volume, field method, lambda,
	 * iterations and normalized
	 */
	protected void createInputParameters(ParamCollection inputParams) {
		inputVol = new ParamVolume("Volume", VoxelType.FLOAT);
		method = new ParamOption("Method", new String[] { "Generalized", "Original", "Optimized"});
		lambda = new ParamDouble("Lambda", 0, Double.MAX_VALUE);
		lambda.setValue(0.2);
		normalize = new ParamBoolean("Normalize Vectors");
		iterations = new ParamInteger("Max Iterations", 0, 1000, 200);

		inputParams.setPackage("IACL");
		inputParams.setCategory("Segmentation.CRUISE");

		inputParams.add(method);
		inputParams.add(inputVol);
		inputParams.add(lambda);
		inputParams.add(iterations);
		inputParams.add(normalize);


		inputParams.setLabel("Grad Vector Flow");
		inputParams.setName("gvf");


		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("http://www.iacl.ece.jhu.edu/");
		info.setVersion(revnum);
		info.setDescription("Generate graident vector field.\nIf the input image is a vector field, GVF will operate to smooth the vector field.\nIf the input image is gray scale, an edge map of the gradent is computed and then GVF generates a vector field to that edge map.");
		info.setLongDescription("");
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.Release);
		info.add(ReferencedPapers.gvf);
		info.add(PrinceGroupAuthors.chenyangXu);
    	info.add(PrinceGroupAuthors.blakeLucas);
		info.setAdditionalDocURL("html/edu/jhu/ece/iacl/plugins/segmentation/cruise/MedicAlgorithmGradVecFlow/index.html");
	}


	/**
	 * Create output parameters for GVF: cubic volume
	 */
	protected void createOutputParameters(ParamCollection outputParams) {
		outputVol = new ParamVolume("Vector Field", VoxelType.FLOAT, -1, -1, -1, 3);
//		ImageData outputVol=null;
		outputVol.setName("Vector Field");
		outputParams.add(outputVol);
		outputParams.setLabel("GVF");
		outputParams.setName("gvf");
	}


	/**
	 * Execute GVF
	 */
	protected void execute(CalculationMonitor monitor) {
		GradVecFlow gvf = GradVecFlow.create(method.getValue());
		monitor.observe(gvf);

		ImageDataFloat in = new ImageDataFloat(inputVol.getImageData());
		ImageHeader hdr = in.getHeader();
		int nc = in.getComponents();

		ImageData result = null;

		if (nc == 3){
			System.out.println("\n\nGVF -- Treating input as vector field....\n\n");
			result = gvf.process(in, lambda.getValue().floatValue(),
				normalize.getValue().booleanValue(),
				iterations.getValue().intValue());
		} else if (nc == 1){
			System.out.print("\n\nGVF -- Treating input as gray scale image....\n\n");
			result = gvf.solve(in, lambda.getValue().floatValue(),
				normalize.getValue().booleanValue(),
				iterations.getValue().intValue());
		} else {
			System.err.print("\n\nGVF -- Wrong number of components.\n\n");
		}


		System.out.println("\n\nGVF Output name:  "
			+ result.getName() + "\n\n");

		result.setHeader(hdr);
		outputVol.setValue(result);
	}
}
