package edu.vanderbilt.masi.algorithms.CRUISE.utilities;

import javax.vecmath.Point3d;
import javax.vecmath.Point3f;

import edu.jhu.ece.iacl.algorithms.graphics.intersector.IntersectorTriangle;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SurfaceIntersector;
import edu.jhu.ece.iacl.algorithms.graphics.isosurf.IsoSurfaceOnGrid;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdTriangle;
import edu.jhu.ece.iacl.algorithms.thickness.ThicknessSolver;
import edu.jhu.ece.iacl.algorithms.topology.ConnectivityRule;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;



/**
 * @author Yuankai Huo
 * @email yuankai.huo@vanderbilt.edu
 * @version 1.0
 *
 *Based on Compute thickness by measuring the shortest distance 
 *between corresponding points. by Blake Lucas in CRUISE
 *
 *The difference is that it calculate the average thickness from inner
 *to the closest point in outer and from that point back to inner surf
 *
 *In addition, this file generate the scalar vtk file which project
 *the thickness to central file
 */

public class ClosestPointThicknessForInner extends ThicknessSolver {
	private static final String cvsversion = "$Revision: 1.2 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");


	public static String getVersion(){
		return revnum;
	}


	public ClosestPointThicknessForInner(double lambda, double isoVal,
			double lagrangeDeltaT) {
		this.isoVal = isoVal;
		this.lambda = lambda;
		this.lagrangeDeltaT = lagrangeDeltaT;
		setLabel("Closest Point Thickness");
	}
	
	
	public ClosestPointThicknessForInner() {
		setLabel("Closest Point Thickness");
	}


	public void solve(ImageData innerVol, ImageData outerVol) {
		this.innerVol = innerVol;
		this.outerVol = outerVol;
		
		rows = innerVol.getRows();
		cols = innerVol.getCols();
		slices = innerVol.getSlices();
		
		IsoSurfaceOnGrid surfGen = new IsoSurfaceOnGrid();
		innerMesh = (ADJUST_BOUNDARY) ? surfGen.solveNoMove(innerVol,
				ConnectivityRule.CONNECT_18_6, (float) isoVal, false) : surfGen
				.solveOriginal(innerVol, ConnectivityRule.CONNECT_18_6,
						(float) isoVal, false);
		outerMesh = (ADJUST_BOUNDARY) ? surfGen.solveNoMove(outerVol,
				ConnectivityRule.CONNECT_18_6, (float) isoVal, false) : surfGen
				.solveOriginal(outerVol, ConnectivityRule.CONNECT_18_6,
						(float) isoVal, false);
		
		computeThickness();		
		markCompleted();
	}
	
	//project to central Mesh
	public void solve(EmbeddedSurface innerMesh, EmbeddedSurface outerMesh) {
		this.innerMesh = innerMesh;
		this.outerMesh = outerMesh;
		
		computeThickness();
		markCompleted();
	}

	//project to central Mesh
	public void solve(EmbeddedSurface innerMesh, EmbeddedSurface centralMesh, EmbeddedSurface outerMesh) {
		this.innerMesh = innerMesh;
		this.centralMesh = centralMesh;
		this.outerMesh = outerMesh;
		
		computeThickness();
		centralsurfaceprojection();
		markCompleted();
	}
	
	public double[] solve(Point3f[] xmlPoints,EmbeddedSurface MeshSurf){
		
		int pointCount = xmlPoints.length;
		int i;
		double[] Distance = new double[pointCount];
		Point3f pt;
		
		SurfaceIntersector intersectMeshSurf = new SurfaceIntersector(this, 13, MeshSurf);

		for (i = 0;i<pointCount;i++){		
			pt = xmlPoints[i];				
			Distance[i] = intersectMeshSurf.distance(pt);
			//		intersectInner.distance(lastPointOuter);
		}
		return Distance;
	}

	public SurfaceIntersector solve(EmbeddedSurface MeshSurf){

		SurfaceIntersector intersectMeshSurf = new SurfaceIntersector(this, 13, MeshSurf);

		return intersectMeshSurf;
	}


	protected void computeThickness() {
		Point3f pt;
				
		// distance from outer to inner
		SurfaceIntersector intersectInner = new SurfaceIntersector(this, 13, innerMesh);		
		
		// distance from inner to outer
		SurfaceIntersector intersectOuter = new SurfaceIntersector(this, 13, outerMesh);
		
		int vertCount = innerMesh.getVertexCount();
		double[][] vertData = new double[vertCount][3];
		setTotalUnits(vertCount + outerMesh.getVertexCount());
		for (int id = 0; id < vertCount; id++) {
			pt = innerMesh.getVertex(id);
			double[] stats = vertData[id];
			stats[0] = intersectOuter.distance(pt);
			Point3f lastPointOuter = intersectOuter.getLastIntersectionPoint();
	
			stats[1] = intersectInner.distance(lastPointOuter);
			stats[2] = (stats[0]+stats[1])/2;
			incrementCompletedUnits();
		}
		
		innerMesh.setVertexData(vertData);

		innerMesh.setName(innerMesh.getName() + "_src");
		outerMesh.setName(innerMesh.getName() + "_dest");
	}
	
	protected void centralsurfaceprojection(){
		
		Point3f pt;
		Point3f p = null;
		Point3f[] pts;
		int id1;
		int id2;
		int id3;
		int fid;
		int smallestdistind;
		float smallestdistance;
		
		// distance from central to inner
		SurfaceIntersector intersectInner = new SurfaceIntersector(this, 13, innerMesh);
		
		int vertCount = centralMesh.getVertexCount();
		double[][] vertData = new double[vertCount][4];
		setTotalUnits(vertCount + innerMesh.getVertexCount());
		for (int id = 0; id < vertCount; id++) {
			pt = centralMesh.getVertex(id);
			double[] stats = vertData[id];
			
			// central to inner triangular distance
			stats[1] = intersectInner.distance(pt);
			Point3f lastPointInner = intersectInner.getLastIntersectionPoint();
			KdTriangle lastTriangleInner= intersectInner.getLastIntersectionTriangle();
			IntersectorTriangle TriangleInner = (IntersectorTriangle)lastTriangleInner;
			
			pts = TriangleInner.getPoints();
			id1 = TriangleInner.id1; 
			id2 = TriangleInner.id2; 
			id3 = TriangleInner.id3; 
			fid = TriangleInner.fid;
			
			//find the cloest points among the three vertices in
			//the triangular
			smallestdistind = id1;
			smallestdistance = pt.distance(pts[0]);			
			if (pt.distance(pts[1]) < smallestdistance){
				smallestdistind = id2;
				smallestdistance = 	pt.distance(pts[1]);
			}
			if (pt.distance(pts[2]) < smallestdistance){
				smallestdistind = id3;
				smallestdistance = 	pt.distance(pts[2]);
			}
			
			// inner to outer distance
			stats[0] = innerMesh.getVertexDataAtOffset(smallestdistind,2);
			// central to inner closet point distance
			stats[2] = smallestdistance;
			// central to inner closet point vertice
			stats[3] = (double)smallestdistind;					
		
		}
		
		centralMesh.setVertexData(vertData);
		centralMesh.setName(innerMesh.getName() + "_central");
		incrementCompletedUnits();
		
	}
	

}
