function [card_est,resp_est]=apply_PESTICA2(comps,ep2d_filename,mask_filename,slice_timing,TE)
%function apply_PESTICA(comps,ep2d_filename,mask_filename,slice_timing)
% requires number of components used in ICA decomposision, filename for
% EPI data, and optional parameters for a mask filename and the slice_timing
% which must be a vector of length == number of slices in data/decomposition
% which specifies relative timing (0 to 1) of the slices as indexed
% so for a typical sequential sequence with 30 slices, use slice_timing=linspace(0,1-1/30,30)
% however, you can use the strings 'seq' or 'alt' for sequential or interleaved
% and add the strings 'des' or 'asc' for descending or ascending, with a '-'
% between them such as 'seq-des', 'alt-asc', 'seq-asc', 'alt-des'
% TE is also optional and only relevant for manually specified or siemens slice_timing
%
% 1. applies the transformed PESTICA averaged maps to the independent components
%    spatial maps to find the components most likely to be physiologic noise
% 2. variance-normalizes, detrends, re-orders, and interpolates the independent
%    components according to the slice_timing
% if slice_timing is not specified, uses interleaved ascending
%
% Note: Siemens product ep2d_pace sequence has 20ms pause at end of every volume, even if
% PACE is turned off, this slice_timing can be activated using 'siemens' for the slice_timing
% of an interleaved ascending sequence.  Otherwise, specify your own in fractions from 0 to 1
%
% use accompanying script tfilter_fft for FFT frequency filtering:
% plot(linspace(0,1,length(card_est)),abs(fft(card_est)));
%    then visually look for the cardiac peak, and select percentages surrounding it

mnidir=getenv('PESTICA_DIR');
if (length(mnidir)==0)
  disp('you must have environment variable PESTICA_DIR set to point to PESTICA distribution');
  return
end
mat_afnidir=getenv('MATLAB_AFNI_DIR');
addpath(mat_afnidir);

Opt.Format = 'matrix';
[err, ima, ainfo, ErrMessage]=BrikLoad(ep2d_filename, Opt);
xdim=ainfo.DATASET_DIMENSIONS(1);
ydim=ainfo.DATASET_DIMENSIONS(2);
zdim=ainfo.DATASET_DIMENSIONS(3);
tdim=ainfo.DATASET_RANK(2);
TR=1000*double(ainfo.TAXIS_FLOATS(2));

% open the averaged spatial unmixing matrices
Opt.Format = 'matrix';
[err,resp,ninfo,ErrMessage]=BrikLoad('pestica/resp_PESTICA2.nii',Opt);
[err,card,ninfo,ErrMessage]=BrikLoad('pestica/card_PESTICA2.nii',Opt);

mask_resp=int16(zeros(xdim,ydim,zdim));
mask_resp(find(resp~=0))=1;
mask_card=int16(zeros(xdim,ydim,zdim));
mask_card(find(card~=0))=1;
if (exist(mask_filename)~=0)
  [err,mask,minfo,ErrMessage]=BrikLoad(mask_filename, Opt);
  mask=int16(mask).*mask_card.*mask_resp;
  mask(find(mask~=0))=1;
else
  mask=mask_card.*mask_resp;
end

cd pestica
[w,zstart]=system(sprintf('ls pestica_%dcomps_slice*.mat | sed "s/pestica_%dcomps_slice//" | sed "s/.mat//" | sort -n',comps,comps));
cd ..

zstart=[str2num(zstart)'];
sdim_slices=zeros(zdim,1);
load(sprintf('pestica/pestica_%dcomps_slice%d.mat',comps,zstart(1)),'A');
tdim_ica=size(A,2);
if (tdim_ica<tdim)
  disp('discarded volumes at beginning of scan, correcting tdim to account for these...');
  tdim_skip=tdim-tdim_ica;
  tdim=tdim_ica;
end
slice_timeseries=zeros(zdim,comps,tdim);
comp_r=zeros(zdim,1);
comp_c=zeros(zdim,1);
for z=zstart
  load(sprintf('pestica/pestica_%dcomps_slice%d.mat',comps,z));
  clear D;
  q=0;
  clear sparsed_m sparsed_resp sparsed_card;
  sl_mask=reshape(mask(:,:,z),[xdim*ydim 1]);
  sl_resp=reshape(resp(:,:,z),[xdim*ydim 1]);
  sl_card=reshape(card(:,:,z),[xdim*ydim 1]);
  indices=find(sl_mask==1);
  sdim_slices(z)=length(indices);
  if (length(indices)<5*comps)
    skipped(z)=1;
    comp_r(z)=1;
    max_comp_r(z)=0;
    comp_c(z)=1;
    max_comp_c(z)=0;
    continue;
  end
  sparsed_resp=sl_resp(indices);
  sparsed_card=sl_card(indices);
  for j=1:comps
    sparsed_m(:,j)=m(indices,j);
  end
  for j=1:comps
    timeseries=squeeze(A(j,:));
    % de-mean
    timeseries=timeseries-mean(timeseries);
    % variance normalize
    D(j,:)=timeseries/std(timeseries);
    r_r(j)=corr(sparsed_resp,sparsed_m(:,j));
    r_c(j)=corr(sparsed_card,sparsed_m(:,j));
  end
  r_r(isnan(r_r))=0;
  r_c(isnan(r_c))=0;
  slice_timeseries(z,:,:)=D;
  % use the maximum absolute coupling component, since sign of components is arbitrary
  comp_r(z)=find(abs(r_r)==max(abs(r_r)),1);
  max_comp_r(z)=r_r(comp_r(z));
  comp_c(z)=find(abs(r_c)==max(abs(r_c)),1);
  max_comp_c(z)=r_c(comp_c(z));
  corr_r(z,:)=r_r;
  corr_c(z,:)=r_c;
end

% take maximum coupling components
skip=zeros(zdim,1);
for z=1:zdim
  if (numel(find(zstart==z))==0)
    slice_m_r(:,z)=zeros(xdim*ydim,1);
    slice_m_c(:,z)=zeros(xdim*ydim,1);
    temporal_c(z,:)=zeros(tdim,1);
    temporal_r(z,:)=zeros(tdim,1);
    comp_r(z)=1;
    max_comp_r(z)=0;
    comp_c(z)=1;
    max_comp_c(z)=0;
    skip(z)=1;
    slice_timeseries(z,:,:)=zeros(comps,tdim);
    continue;
  end
  temporal_r(z,:)=squeeze(slice_timeseries(z,comp_r(z),:));
  if (temporal_r(z,:)==zeros(1,tdim))
    % do nothing
  else
    temporal_r(z,:)=temporal_r(z,:)/std(temporal_r(z,[5:end]));
  end
  temporal_c(z,:)=squeeze(slice_timeseries(z,comp_c(z),:));
  if (temporal_c(z,:)==zeros(1,tdim))
    % do nothing
  else
    temporal_c(z,:)=temporal_c(z,:)/std(temporal_c(z,:));
  end
  if (max_comp_r(z)<0)
    temporal_r(z,:)=-1*temporal_r(z,:);
  end
  if (max_comp_c(z)<0)
    temporal_c(z,:)=-1*temporal_c(z,:);
  end
  load(sprintf('pestica/pestica_%dcomps_slice%d.mat',comps,z),'m');
  slice_m_r(:,z)=m(:,comp_r(z));
  slice_m_c(:,z)=m(:,comp_c(z));
end

% detrend estimators
temporal_r=detrend(temporal_r')';
temporal_c=detrend(temporal_c')';

% find cutoff for the interpolation over poorly-identified slices
keep_card_slices=zeros(zdim,1);
keep_resp_slices=zeros(zdim,1);
for z=1:zdim
  cutoff(z)=get_correlation_threshold(comps,sdim_slices(z));
  if (abs(max_comp_c(z))>=cutoff(z))
    keep_card_slices(z)=1;
  end
  if (abs(max_comp_r(z))>=cutoff(z))
    keep_resp_slices(z)=1;
  end
end

if (exist('TE')==0)
  TE=29;
end
TE=double(TE);
zdim=double(zdim);
if (length(find(keep_card_slices==1))==0)
  disp('no slices can be trusted for cardiac (below significance), taking those above avg corr');
  keep_card_slices(find(abs(max_comp_c)>mean(abs(max_comp_c))))=1;
end
if (length(find(keep_resp_slices==1))==0)
  disp('no slices can be trusted for respiratory (below significance), taking those above avg corr');
  keep_resp_slices(find(abs(max_comp_r)>mean(abs(max_comp_r))))=1;
end
% re-order by slice-acquisition
card_est=assemble_slices_to_timeseries(temporal_c,zdim,tdim,TR,TE,slice_timing,keep_card_slices);
resp_est=assemble_slices_to_timeseries(temporal_r,zdim,tdim,TR,TE,slice_timing,keep_resp_slices);

save(sprintf('pestica/pestica2_matrices_%dcomps.mat',comps),'slice_m_r','slice_m_c','keep_card_slices','keep_resp_slices','skip','corr_c','corr_r','max_comp_r','max_comp_c','cutoff');

