function matlab_retroicor(ep2d_filename,card,resp,M,slice_timing,mask_filename)
% function matlab_retroicor(ep2d_filename,card,resp,M,slice_timing,mask_filename)
% This function performs the 2nd-order RETROICOR algorithm on an epi dataset, using
% input vectors for physiologic data, and returns coupling coefficients and the corrected image
% created for PESTICA distribution
% variance normalization prior to calculation results in statistic values
% use stat=sqrt(sum(im_ca.^2+im_cb.^2,4)), which follows a
% 2*M-th order Chi-square distribution (where M is order of correction, here M=2)

%for simulated data: cardph=unifrnd(0,3.14159*2,zdim,tdim); card=sin(cardph);
if (exist('card')==0 | exist('resp')==0 | exist('ep2d_filename')==0)
  disp('must input three parameters: filename of EPI data, cardiac, and respiratory traces');
  disp('     fourth optional paramter = order of correction (default=2, which is sin(ph)+sin(2*ph)+cos...)');
  disp('     fifth optional paramter = slice acquisition timing string (''alt-asc'' is typical)');
  disp('     sixth optional paramter  = 3D mask file for EPI data');
  return;
end
mnidir=getenv('PESTICA_DIR');
mat_afnidir=getenv('MATLAB_AFNI_DIR');
if (length(mnidir)==0)
  disp('you must have environment variable PESTICA_DIR set to point to PESTICA distribution');
  return
end
addpath(mnidir);
addpath(mat_afnidir);

Opt.Format = 'matrix';
[err, ima, ainfo, ErrMessage]=BrikLoad(ep2d_filename, Opt);
xdim=ainfo.DATASET_DIMENSIONS(1);
ydim=ainfo.DATASET_DIMENSIONS(2);
zdim=ainfo.DATASET_DIMENSIONS(3);
tdim=ainfo.DATASET_RANK(2);
TR=1000*double(ainfo.TAXIS_FLOATS(2));
if (exist('mask_filename')~=0)
  [err,mask,minfo,ErrMessage]=BrikLoad(mask_filename, Opt);
  mask(find(mask~=0))=1;
else
  mask=ones(xdim,ydim,zdim);
end
ima=double(reshape(ima,[xdim ydim zdim tdim]));
mask=double(reshape(mask,[xdim ydim zdim]));

% variance normalize the timeseries
variance=squeeze(std(ima,0,4));
variance=variance.*mask;
ima=ima./repmat(variance,[1 1 1 tdim]);

if (exist('M')==0)
  M=2;
  disp(sprintf('setting default RETROICOR model order to %d',M));
end

im_ca=zeros(xdim,xdim,zdim,M);
im_cb=zeros(xdim,xdim,zdim,M);
im_ra=zeros(xdim,xdim,zdim,M);
im_rb=zeros(xdim,xdim,zdim,M);
retima=zeros(xdim,xdim,zdim,tdim);
tim_ca=zeros(xdim,xdim,zdim,M);
tim_cb=zeros(xdim,xdim,zdim,M);
tim_ra=zeros(xdim,xdim,zdim,M);
tim_rb=zeros(xdim,xdim,zdim,M);

cdim=size(card);
if (cdim(1)>cdim(2))
  card=card';
end
cdim=size(resp);
if (cdim(1)>cdim(2))
  resp=resp';
end
% convert physio data input into phase before proceeding
%cardph=convert_physio_into_phase(card);
%respph=convert_physio_into_phase(resp);
cardph=card;
respph=resp;

if (exist('slice_timing')==0)
  slice_timing='siemens-alt-asc'
  disp('using interleaved ascending ''alt-asc'' for the slice timing');
end
if (exist('TR')==0)
  TR=1;
end
TE=35;
% setup slice timing of physiologic data
cph_slice=disassemble_timeseries_to_slices(cardph,zdim,tdim,TR,TE,slice_timing);
rph_slice=disassemble_timeseries_to_slices(respph,zdim,tdim,TR,TE,slice_timing);
% degrees of freedom for test-statistic calculation
dof=double(tdim-(M*2*2+1));

for z=1:zdim
  A=[sin((1:M)'*cph_slice(z,:))' cos((1:M)'*cph_slice(z,:))' sin((1:M)'*rph_slice(z,:))' cos((1:M)'*rph_slice(z,:))' ones(tdim,1)];
  % QR-decomposition
  [Q,R] = qr(A,0);
  for y=1:xdim
    for x=1:xdim
      vox=squeeze(ima(x,y,z,:));
      if (mask(x,y,z)==0)
        retima(x,y,z,:)=vox;
        continue;
      end
      % matrix division to get amplitudes of fits
      p = R\(Q'*vox);
      im_ca(x,y,z,:)=p(1:M);
      im_cb(x,y,z,:)=p(M+1:M*2);
      im_ra(x,y,z,:)=p(M*2+1:M*3);
      im_rb(x,y,z,:)=p(M*3+1:M*4);
      % residuals
      res=vox-A*p;
      % mean square error
      mse=res'*res./(dof+1);
      Rinv=pinv(R);
      % error covariance matrix
      covb=(Rinv*Rinv')*mse;
      % get standardized error (variance that is not explained by fit to design matrix)
      stand_err=sqrt(diag(covb));
      % test statistics
      tim_ca(x,y,z,:)=p(1:M)./stand_err(1:M);
      tim_cb(x,y,z,:)=p(M+1:M*2)./stand_err(M+1:M*2);
      tim_ra(x,y,z,:)=p(M*2+1:M*3)./stand_err(M*2+1:M*3);
      tim_rb(x,y,z,:)=p(M*3+1:M*4)./stand_err(M*3+1:M*4);
      % don't remove the mean from voxel timeseries
      p(end)=0;
      retima(x,y,z,:)=vox-A*p;
    end
  end
end

% sum squares of coupling coefficients
im_card=sqrt(sum(im_ca.^2+im_cb.^2,4));
im_resp=sqrt(sum(im_ra.^2+im_rb.^2,4));
% sum squares of t-statistics
tim_card=sqrt(sum(tim_ca.^2+tim_cb.^2,4));
tim_resp=sqrt(sum(tim_ra.^2+tim_rb.^2,4));
% re-introduce the image variance
retima=retima.*repmat(variance,[1 1 1 tdim]);

ainfo.BRICK_TYPES=3*ones(1,tdim); %1=short, 3=float
ainfo.BRICK_STATS = []; %automatically set
ainfo.BRICK_FLOAT_FACS = [];%automatically set
OptOut.Scale = 1;
OptOut.Prefix = 'retroicor';
OptOut.verbose = 0;
!rm -f irfretroicor+orig.*
[err,ErrMessage,InfoOut]=WriteBrik(retima,ainfo,OptOut);

ainfo.BRICK_TYPES = [3]; %1=short, 3=float
ainfo.BRICK_STATS = []; %automatically set
ainfo.BRICK_FLOAT_FACS = [];%automatically set
% write out cardiac coupling maps
OptOut.Prefix = 'coupling_ret_card';
!rm -f coupling_irfret_card+orig.*
[err,ErrMessage,InfoOut]=WriteBrik(tim_card,ainfo,OptOut);
% write out respiratory coupling maps (2 IRFs)
OptOut.Prefix = 'coupling_ret_resp';
!rm -f coupling_irfret_resp+orig.*
[err,ErrMessage,InfoOut]=WriteBrik(tim_resp,ainfo,OptOut);

