function ima_reg=removephysionoise(ep2d_filename,cardph,respph,mask_filename,M,slicereg,retreg,slice_timing,TR)
%function ima_reg=removephysionoise(ep2d_filename,cardph,respph,mask_filename,M,slicereg,retreg,slice_timing,TR)
% This function performs the 2nd-order RETROICOR algorithm on an epi dataset, using
% input vectors for phase of physiologic data, and writes coupling coefficients and the corrected image
% t-statistic images written for noise regression coefficients, for RETROICOR coefficients, I have summed
% them using sqrt(sum(im_ca.^2+im_cb.^2)), so they will follow a
% 2*M-th order Chi-square distribution (where M is order of correction, here M=2)
%
% example of input if you wish to see simulated noise coupling: cardph=unwrap(unifrnd(0,3.14159*2,zdim*tdim));
% this is useful for assessing null distribution: the image of this should look largely flat unless you're unlucky
% (in which case, simply run again, this should be rare occurence), and can be useful to compare with true physio coupling
if (exist('cardph')==0 | exist('respph')==0 | exist('ep2d_filename')==0)
  disp('must input three parameters: filename of EPI data, cardiac phase, and respiratory phase (make using RETROICOR)');
  disp('     first optional parameter   = name of maskfile (also in ANALYZE format)');
  disp('     second optional parameter  = M, order of correction (default=2, which is sin(ph)+sin(2*ph)+cos...)');
  disp('     third optional parameter   = run slice-common noise regression (i.e. the average of all signal per slice)');
  disp('     fourth optional parameter  = run physiologic noise regression (i.e. RETROICOR noise model using phase input)');
  disp('     fifth optional parameter   = slice acquisition timing string (''alt-asc'' is typical)');
  disp('     sixth optional parameter   = ANALYZE format 3D mask file for EPI data');
  disp('     seventh optional parameter = TR repetition time (in milliseconds)');
  return;
end
mnidir=getenv('PESTICA_DIR');
if (length(mnidir)==0)
  disp('you must have environment variable PESTICA_DIR set to point to PESTICA distribution');
  return
end
addpath(mnidir);
TE=35;

mat_afnidir=getenv('MATLAB_AFNI_DIR');
addpath(mat_afnidir);

Opt.Format = 'matrix';
[err, ima, ainfo, ErrMessage]=BrikLoad(ep2d_filename, Opt);
xdim=ainfo.DATASET_DIMENSIONS(1);
ydim=ainfo.DATASET_DIMENSIONS(2);
zdim=ainfo.DATASET_DIMENSIONS(3);
tdim=ainfo.DATASET_RANK(2);
if (exist('mask_filename')~=0)
  [err,mask,minfo,ErrMessage]=BrikLoad(mask_filename, Opt);
  mask(find(mask~=0))=1;
else
  mask=ones(xdim,ydim,zdim);
end
ima=double(ima);
mask=double(mask);

if (exist('slicereg')==0)
  slicereg=1;
end
if (exist('retreg')==0)
  retreg=1;
end

if (slicereg==1)
 for k=1:zdim
  for q=1:tdim
    % now average signal inside mask in each slice at each time point
    % but only that signal inside the volume mask, ignore noise outside head (ideally, ignore skull also)
    slice_signal(k,q)=sum(sum(ima(:,:,k,q).*mask(:,:,k)))/sum(sum(mask(:,:,k)));
  end
  % de-mean the signal
  slice_signal(k,:)=slice_signal(k,:)-mean(slice_signal(k,:));
 end
end

if (exist('slice_timing')==0)
  slice_timing='alt-asc';
end

% prepare matrices for storing results
ima_reg=zeros(xdim,ydim,zdim,tdim);
if (slicereg==1)
  im_slicenoise=zeros(xdim,ydim,zdim);
  tim_slicenoise=zeros(xdim,ydim,zdim);
end
if (retreg==1)
  % convert the phase input files to slice-sampled data
  %cph_sl=disassemble_timeseries_to_slices(cardph,zdim,tdim,slice_timing,TR);
  cph_sl=disassemble_timeseries_to_slices(cardph,zdim,tdim,TR,TE,slice_timing);
  rph_sl=disassemble_timeseries_to_slices(respph,zdim,tdim,TR,TE,slice_timing);
  im_ca=zeros(xdim,ydim,zdim,M);
  im_cb=zeros(xdim,ydim,zdim,M);
  im_ra=zeros(xdim,ydim,zdim,M);
  im_rb=zeros(xdim,ydim,zdim,M);
  tim_ca=zeros(xdim,ydim,zdim,M);
  tim_cb=zeros(xdim,ydim,zdim,M);
  tim_ra=zeros(xdim,ydim,zdim,M);
  tim_rb=zeros(xdim,ydim,zdim,M);
end
% order of RETROICOR model, default=2
if (exist('M')==0)
  M=2;
end
% Degrees-of-freedom following regressions equals tdim minus each regression
dof=double(tdim-(1+M*4*retreg+slicereg));

for k=1:zdim
  % design matrix: RETROICOR model terms: M*4 terms of sin and cos of physiologic phases, one term of mean signal level.  OPTIONAL: slice-common noise as additional regressor
  if (slicereg==1 & retreg==1)
    A=[sin((1:M)'*cph_sl(k,:))' cos((1:M)'*cph_sl(k,:))' sin((1:M)'*rph_sl(k,:))' cos((1:M)'*rph_sl(k,:))' slice_signal(k,:)' ones(tdim,1)];
  elseif (slicereg==0 & retreg==1)
    A=[sin((1:M)'*cph_sl(k,:))' cos((1:M)'*cph_sl(k,:))' sin((1:M)'*rph_sl(k,:))' cos((1:M)'*rph_sl(k,:))' ones(tdim,1)];
  elseif (slicereg==1 & retreg==0)
    A=[slice_signal(k,:)' ones(tdim,1)];
  else
    disp('must use slice-noise and/or RETROICOR-noise regression');
    return;
  end
  % QR-decomposition
  [Q,R] = qr(A,0);
  for i=1:xdim
    for j=1:ydim
      vox=double(squeeze(ima(i,j,k,:)));
      if (mask(i,j,k)==0)
        for q=1:tdim
          ima_reg(i,j,k,q)=vox(q);
        end
        continue;
      end
      % remove correlations between voxel and RETROICOR physiologic noise model
      % matrix division to get amplitudes of fits
      p = R\(Q'*vox);
      % residuals (must calculate before we do anything to p)
      res=vox-A*p;
      % don't remove mean signal level, so set amplitude of mean signal fit to 0
      p(end)=0;
      % and save this corrected voxel into ima_reg 4-D volume
      ima_reg(i,j,k,:)=vox-A*p;
      % mean square error
      mse=res'*res./(dof+1);
      Rinv=pinv(R);
      % error covariance matrix
      covb=(Rinv*Rinv')*mse;
      % get standardized error (variance that is not explained by fit to design matrix)
      stand_err=sqrt(diag(covb));
      % don't remove the mean from voxel timeseries
      if (retreg==1)
        % store also the coupling coefficients to RETROICOR model
        im_ca(i,j,k,:)=p(1:M);
        im_cb(i,j,k,:)=p(M+1:M*2);
        im_ra(i,j,k,:)=p(M*2+1:M*3);
        im_rb(i,j,k,:)=p(M*3+1:M*4);
        % test statistics
        tim_ca(i,j,k,:)=p(1:M)./stand_err(1:M);
        tim_cb(i,j,k,:)=p(M+1:M*2)./stand_err(M+1:M*2);
        tim_ra(i,j,k,:)=p(M*2+1:M*3)./stand_err(M*2+1:M*3);
        tim_rb(i,j,k,:)=p(M*3+1:M*4)./stand_err(M*3+1:M*4);
      end
      if (slicereg==1)
        % coupling to slice-common signal
        im_slicenoise(i,j,k)=p(end-1);
        tim_slicenoise(i,j,k)=p(end-1)/stand_err(end-1);
      end
    end
  end
end

% sum squares of coupling coefficients
im_card=sqrt(sum(im_ca.^2+im_cb.^2,4));
im_resp=sqrt(sum(im_ra.^2+im_rb.^2,4));
% sum squares of t-statistics
tim_card=sqrt(sum(tim_ca.^2+tim_cb.^2,4));
tim_resp=sqrt(sum(tim_ra.^2+tim_rb.^2,4));

ainfo.BRICK_TYPES = [3]; %1=short, 3=float
ainfo.BRICK_STATS = []; %automatically set
ainfo.BRICK_FLOAT_FACS = [];%automatically set
OptOut.Scale = 1;
OptOut.Prefix = 'retroicor';
OptOut.verbose = 0;
!rm retroicor.nii
[err,ErrMessage,InfoOut]=WriteBrik(retima,ainfo,OptOut);

ainfo.DATASET_DIMENSIONS(4)=1;
if (slicereg==1)
  disp('writing output file: tim_slice.nii is slice coupling coefficients - float (4byte) format');
  OptOut.Prefix = 'tim_slice';
  [err,ErrMessage,InfoOut]=WriteBrik(tim_resp,ainfo,OptOut);
end
if (retreg==1)
  disp('writing output files: tim_ret_card.nii, tim_ret_resp.nii are physio coupling coefficients - float (4byte) format');
  OptOut.Prefix = 'tim_ret_card';
  [err,ErrMessage,InfoOut]=WriteBrik(tim_card,ainfo,OptOut);
  OptOut.Prefix = 'tim_ret_resp';
  [err,ErrMessage,InfoOut]=WriteBrik(tim_resp,ainfo,OptOut);
end



