function [tim_card,tim_resp,retima]=irf_retroicor(ep2d_filename,cardph,respph,coeffs_card,coeffs_resp,slice_timing,mask_filename)
%function [tim_card,tim_resp,retima]=irf_retroicor(ep2d_filename,cardph,respph,coeffs_card,coeffs_resp,slice_timing,mask_filename)
% Fits Impulse Response Function (IRF) derived from data to the data, so as to perform
% the RETROICOR method without spurious removal of degrees of freedom
% Output is coupling maps (t-maps) and corrected data

%for simulated data: cardph=unifrnd(0,3.14159*2,zdim,tdim); card=sin(cardph);
if (exist('cardph')==0 | exist('respph')==0 | exist('ep2d_filename')==0)
  disp('must input three parameters: filename of EPI data, phases of cardiac and respiratory traces');
  disp('     fourth optional paramter = order of correction (default=5, or sin([1:M]*phase)+cos([1:M]*phase)');
  disp('     fifth optional paramter = slice acquisition timing string (''alt-asc'' is typical)');
  disp('     sixth optional paramter  = ANALYZE format 3D mask file for EPI data');
  return;
end
mnidir=getenv('PESTICA_DIR');
mat_afnidir=getenv('MATLAB_AFNI_DIR');
if (length(mnidir)==0)
  disp('you must have environment variable PESTICA_DIR set to point to PESTICA distribution');
  return
end
addpath(mnidir);
addpath(strcat(mnidir,'/eeglab'));
addpath(mat_afnidir);

Opt.Format = 'matrix';
[err, ima, ainfo, ErrMessage]=BrikLoad(ep2d_filename, Opt);
xdim=ainfo.DATASET_DIMENSIONS(1);
ydim=ainfo.DATASET_DIMENSIONS(2);
zdim=ainfo.DATASET_DIMENSIONS(3);
tdim=ainfo.DATASET_RANK(2);
TR=1000*double(ainfo.TAXIS_FLOATS(2));
if (exist('mask_filename')~=0)
  [err,mask,minfo,ErrMessage]=BrikLoad(mask_filename, Opt);
  mask(find(mask~=0))=1;
else
  mask=ones(xdim,ydim,zdim);
end
ima=double(reshape(ima,[ydim xdim zdim tdim]));
mask=double(reshape(mask,[ydim xdim zdim]));

cdim=size(cardph);
if (cdim(1)>cdim(2))
  cardph=cardph';
end
cdim=size(respph);
if (cdim(1)>cdim(2))
  respph=respph';
end

% define order of Fourier-decomposition
if (exist('M')==0)
  M=5;
  disp(sprintf('setting default order of decomp to %d',M));
end

% slice-acquisition sampling of physiologic phases
cardph=disassemble_timeseries_to_slices(cardph,zdim,tdim,TR,29,slice_timing);
respph=disassemble_timeseries_to_slices(respph,zdim,tdim,TR,29,slice_timing);

im_card=zeros(xdim,ydim,zdim,4);
im_resp=zeros(xdim,ydim,zdim,2);
tim_card=zeros(xdim,ydim,zdim,4);
tim_resp=zeros(xdim,ydim,zdim,2);
% dof is now 6 for 4 card and 2 resp IRFs
dof=double(tdim-(6+1));
retima=zeros(xdim,ydim,zdim,tdim);

for z=1:zdim
  c=cardph(z,:); r=respph(z,:);
  A=[(coeffs_card(1,1:5)*sin((1:5)'*c)+coeffs_card(1,6:end)*cos((1:5)'*c))' (coeffs_card(2,1:5)*sin((1:5)'*c)+coeffs_card(2,6:end)*cos((1:5)'*c))' (coeffs_card(3,1:5)*sin((1:5)'*c)+coeffs_card(3,6:end)*cos((1:5)'*c))' (coeffs_card(4,1:5)*sin((1:5)'*c)+coeffs_card(4,6:end)*cos((1:5)'*c))' (coeffs_resp(1,1:5)*sin((1:5)'*r)+coeffs_resp(1,6:end)*cos((1:5)'*r))' (coeffs_resp(2,1:5)*sin((1:5)'*r)+coeffs_resp(2,6:end)*cos((1:5)'*r))' ones(tdim,1)];
  % QR-decomposition
  [Q,R] = qr(A,0);
  for x=1:xdim
    for y=1:ydim
      if (mask(x,y,z)==0)
        continue;
      end
      vox=squeeze(ima(x,y,z,:));
      if (mask(x,y,z)==0)
        retima(x,y,z,:)=vox;
        continue;
      end
      % matrix division to get amplitudes of fits
      p = R\(Q'*vox);
      im_card(x,y,z,:)=p(1:4);
      im_resp(x,y,z,:)=p(5:6);
      % and t-scores
      % residuals
      res=vox-A*p;
      % mean square error
      ms=res'*res./(dof+1);
      Rinv=pinv(R);
      % error covariance matrix
      covb=(Rinv*Rinv')*ms;
      % get standardized error (variance that is not explained by fit to design matrix)
      stand_err=sqrt(diag(covb));
      tim_card(x,y,z,:)=p(1:4)./stand_err(1:4);
      tim_resp(x,y,z,:)=p(5:6)./stand_err(5:6);
      % do not remove mean from data
      p(end)=0;
      % corrected data
      retima(x,y,z,:)=vox-A*p;
    end
  end
end

ainfo.BRICK_TYPES=3*ones(1,tdim); %1=short, 3=float
ainfo.BRICK_STATS = []; %automatically set
ainfo.BRICK_FLOAT_FACS = [];%automatically set
OptOut.Scale = 1;
OptOut.Prefix = 'irfretroicor';
OptOut.verbose = 0;
!rm -f irfretroicor+orig.*
[err,ErrMessage,InfoOut]=WriteBrik(retima,ainfo,OptOut);

ainfo.BRICK_STATS = []; %automatically set
ainfo.BRICK_FLOAT_FACS = [];%automatically set
% write out cardiac coupling maps (4 IRFs)
ainfo.BRICK_TYPES = [3 3 3 3]; %1=short, 3=float
ainfo.DATASET_RANK(2)=4;
ainfo.TAXIS_NUMS(1) = 4;
OptOut.Prefix = 'coupling_irfret_card';
!rm -f coupling_irfret_card+orig.*
[err,ErrMessage,InfoOut]=WriteBrik(tim_card,ainfo,OptOut);
% write out respiratory coupling maps (2 IRFs)
ainfo.BRICK_TYPES = [3 3]; %1=short, 3=float
ainfo.DATASET_RANK(2)=2;
ainfo.TAXIS_NUMS(1) = 2;
OptOut.Prefix = 'coupling_irfret_resp';
!rm -f coupling_irfret_resp+orig.*
[err,ErrMessage,InfoOut]=WriteBrik(tim_resp,ainfo,OptOut);

