function [irf_resp,irf_card,coeffs_card,coeffs_resp]=retroicor_get_irf(ep2d_filename,cardph,respph,M,slice_timing,mask_filename)
%function [irf_resp,irf_card,coeffs_card,coeffs_resp]=retroicor_get_irf(ep2d_filename,cardph,respph,M,slice_timing,mask_filename)
% new matlab version of retroicor which tries to determine impulse response functions to 
% (1) heart beat and (2) respiration.  It does this by fitting sines and cosines of phase
% to raw data and finding the optimal combination as the limit of abs(tscore) increases
%
%  Output is M*2 timeseries of IRF shapes for card and resp each and the coefficients needed to
%   reproduce these using the parallel measured, RETROICOR physiologic phases
%
% The intent is to run this iteratively after regressing out the detected noise to get smaller 
% and smaller components (which will likely have different i.r.f.'s) of the physiologic noise

%for simulated data: cardph=unifrnd(0,3.14159*2,zdim*tdim,1); card=sin(cardph);
if (exist('cardph')==0 | exist('respph')==0 | exist('ep2d_filename')==0)
  disp('must input three parameters: filename of EPI data, phases of cardiac and respiratory traces');
  disp('     fourth optional paramter = order of correction (default=5, or sin([1:M]*phase)+cos([1:M]*phase)');
  disp('     fifth optional paramter = slice acquisition timing string (''alt-asc'' is typical)');
  disp('     sixth optional paramter  = mask file for EPI data');
  return;
end
mnidir=getenv('PESTICA_DIR');
mat_afnidir=getenv('MATLAB_AFNI_DIR');
if (length(mnidir)==0)
  disp('you must have environment variable PESTICA_DIR set to point to PESTICA distribution');
  return
end
addpath(mnidir);
addpath(strcat(mnidir,'/eeglab'));
addpath(mat_afnidir);

Opt.Format = 'matrix';
[err, ima, ainfo, ErrMessage]=BrikLoad(ep2d_filename, Opt);
xdim=ainfo.DATASET_DIMENSIONS(1);
ydim=ainfo.DATASET_DIMENSIONS(2);
zdim=ainfo.DATASET_DIMENSIONS(3);
tdim=ainfo.DATASET_RANK(2);
TR=1000*double(ainfo.TAXIS_FLOATS(2));
if (exist('mask_filename')~=0)
  [err,mask,minfo,ErrMessage]=BrikLoad(mask_filename, Opt);
  mask(find(mask~=0))=1;
else
  mask=ones(xdim,ydim,zdim);
end
ima=double(reshape(ima,[ydim xdim zdim tdim]));
mask=double(reshape(mask,[ydim xdim zdim]));

% variance normalizing is unnecessary, although distributions will be easier to understand
variance=squeeze(std(ima,0,4));
ima=ima./repmat(variance,[1 1 1 tdim]);

cdim=size(cardph);
if (cdim(1)>cdim(2))
  cardph=cardph';
end
cdim=size(respph);
if (cdim(1)>cdim(2))
  respph=respph';
end

% define order of Fourier-decomposition
if (exist('M')==0)
  M=5;
  disp(sprintf('setting default order of decomp to %d',M));
end

% modify phase data
[card_peaks,card_avg_period,cardph,cardphm]=get_phase_period(cardph,'card');
%disp('not modifying respiratory input phase data');
disp('normalizing respiratory input phase data');
[resp_peaks,resp_avg_period,respph,respphm]=get_phase_period(respph,'resp');

% set TE for 3T, GRE
TE=29;

% slice-acquisition sampling of physiologic phases
cardph=disassemble_timeseries_to_slices(cardph,zdim,tdim,TR,TE,slice_timing);
respph=disassemble_timeseries_to_slices(respph,zdim,tdim,TR,TE,slice_timing);

im_ca=zeros(xdim,ydim,zdim,M);
im_cb=zeros(xdim,ydim,zdim,M);
im_ra=zeros(xdim,ydim,zdim,M);
im_rb=zeros(xdim,ydim,zdim,M);
tim_ca=zeros(xdim,ydim,zdim,M);
tim_cb=zeros(xdim,ydim,zdim,M);
tim_ra=zeros(xdim,ydim,zdim,M);
tim_rb=zeros(xdim,ydim,zdim,M);

im_card=zeros(xdim,ydim,zdim);
im_resp=zeros(xdim,ydim,zdim);
tim_card=zeros(xdim,ydim,zdim);
tim_resp=zeros(xdim,ydim,zdim);
dof=double(tdim-(M*2*2+1));

for z=1:zdim
  A=[sin((1:M)'*cardph(z,:))' cos((1:M)'*cardph(z,:))' sin((1:M)'*respph(z,:))' cos((1:M)'*respph(z,:))' ones(tdim,1)];
  % QR-decomposition
  [Q,R] = qr(A,0);
  for x=1:xdim
    for y=1:ydim
      if (mask(x,y,z)==0)
        continue;
      end
      vox=squeeze(ima(x,y,z,:));
      % detrending voxel timecourse prior to regression increases tscore
      vox=detrend(vox);
      % matrix division to get amplitudes of fits
      p = R\(Q'*vox);
      im_ca(x,y,z,:)=p(1:M);
      im_cb(x,y,z,:)=p(M+1:M*2);
      im_ra(x,y,z,:)=p(M*2+1:M*3);
      im_rb(x,y,z,:)=p(M*3+1:M*4);
      % and t-scores
      % residuals
      res=vox-A*p;
      % mean square error
      ms=res'*res./(dof+1);
      Rinv=pinv(R);
      % error covariance matrix
      covb=(Rinv*Rinv')*ms;
      % get standardized error (variance that is not explained by fit to design matrix)
      stand_err=sqrt(diag(covb));
      tim_ca(x,y,z,:)=p(1:M)./stand_err(1:M);
      tim_cb(x,y,z,:)=p(M+1:M*2)./stand_err(M+1:M*2);
      tim_ra(x,y,z,:)=p(M*2+1:M*3)./stand_err(M*2+1:M*3);
      tim_rb(x,y,z,:)=p(M*3+1:M*4)./stand_err(M*3+1:M*4);
    end
  end
end

% identify M*2 shapes of IRFs for each of card,resp using the following procedure:
% 1. tscore threshold, the distribution of tim_card,tim_resp look like M*2th order chi
%    in order to identify the threshold, fit to uniform dist of rand phase data and plot these 
%    distributions to determine the null hypothesis and an appropriate threshold for significance
%   RESULTS: Tscore threshold is order dependent, 
%       for M=3, use 4.6, for M=4, use 5.3, for M=5, use 5.6
%       for M between 2 and 10, good threshold approximately equiv to 0.52*M+3.03
% 2. In the thresholded voxels, create shapes based on the fit im_ca, im_cb, im_ra, im_rb values,
%    weighting is unimportant since voxels were all variance normalized, so fit coefficients
%    are the appropriate weights
% 3. PCA separately for each slice
% 4. PCA across the slice-determined shapes
% PCA is appropriate since these are constructed from orthogonal components of Fourier series

tim_card=sqrt(sum(tim_ca.^2+tim_cb.^2,4));
tim_resp=sqrt(sum(tim_ra.^2+tim_rb.^2,4));
cmask=zeros(xdim,ydim,zdim);
cmask(find(tim_card>M*0.5+3))=1;
rmask=zeros(xdim,ydim,zdim);
rmask(find(tim_resp>M*0.5+3))=1;

% find the impulse response
cdata=linspace(0,2*pi,100);
rdata=linspace(-pi,pi,100);
pct=zeros(z,M*2,length(cdata));
prt=zeros(z,M*2,length(cdata));
for z=1:zdim
  % average the impulse response functions per slice
  clear ct rt
  c=0;
  r=0;
  for x=1:xdim
    for y=1:ydim
      if (cmask(x,y,z)==1)
        c=c+1;
        ct(c,:)=squeeze(im_ca(x,y,z,:))'*sin((1:M)'*cdata)+squeeze(im_cb(x,y,z,:))'*cos((1:M)'*cdata);
      end
      if (rmask(x,y,z)==1)
        r=r+1;
        rt(r,:)=squeeze(im_ra(x,y,z,:))'*sin((1:M)'*rdata)+squeeze(im_rb(x,y,z,:))'*cos((1:M)'*rdata);
      end
    end
  end
  % check for underfill (not enough significant voxels on this slice)
  if (r<M*2)
    rt(r+1:M*2,:)=zeros(M*2-r,length(rdata));
  end
  if (c<M*2)
    ct(c+1:M*2,:)=zeros(M*2-c,length(cdata));
  end
  % respiration is simple, use mean and assume one impulse only
  resp_impulse(z,:)=mean(rt);
  % resp is made ambiguous by the respiratory phase issue which will have to be dealt with)
  % PCA the impulses across the slice, saving only M*2 comps (orthogonal fourier)
  [eg,ev,comps]=pcsquash(ct,M*2);
  pct(z,:,:)=comps;
  [eg,ev,comps]=pcsquash(rt,M*2);
  prt(z,:,:)=comps;
end
t=reshape(pct,[zdim*M*2 100]);
[eg,ev,comps]=pcsquash(t,M*2);
irf_card=comps;
t=reshape(prt,[zdim*M*2 100]);
[eg,ev,comps]=pcsquash(t,M*2);
irf_resp=comps;

% find coefficients
clear coeffs_resp coeffs_card
for i=1:M*2
  A=[sin((1:M)'*cdata)' cos((1:M)'*cdata)'];
  [Q,R] = qr(A,0);
  p = R\(Q'*irf_card(i,:)');
  coeffs_card(i,:)=p;
  A=[sin((1:M)'*rdata)' cos((1:M)'*rdata)'];
  [Q,R] = qr(A,0);
  p = R\(Q'*irf_resp(i,:)');
  coeffs_resp(i,:)=p;
end

cardph=assemble_slices_to_timeseries(cardph,zdim,tdim,TR,TE,slice_timing,ones(zdim,1));
respph=assemble_slices_to_timeseries(respph,zdim,tdim,TR,TE,slice_timing,ones(zdim,1));

% store impulses and phase data for later use with the impulses, 
% along with mask of significantly coupled voxels (useful for later amplitude fitting)
save impulse_responses.mat cmask rmask respph cardph irf_resp irf_card coeffs_card coeffs_resp

disp(sprintf('Number of significant voxels for cardiac: %d, respiratory: %d',length(find(cmask==1)),length(find(rmask==1))));
% Based on group analysis of above (see IRF-RET paper), cardiac 4 impulse responses present, respiration has 2

