function data=assemble_slices_to_timeseries(slicedata,zdim,tdim,TR,TE,slice_mode,keep_slices)
%function data=assemble_slices_to_timeseries(slicedata,zdim,tdim,TR,TE,slice_mode,keep_slices)
% this function takes in slices of data, assumes it has the same timing mesh
% as the acquired data (i.e. the slices may not be evenly spaced in time)
% and re-assembles it as a long timeseries, with an evenly spaced timing mesh
% the purpose of this script is to produce physiologic timeseries. Temporal 
% filtering can only be done on evenly-spaced mesh data.

zdim=double(zdim);
tdim=double(tdim);
slicedata=double(slicedata);
if (size(slicedata,1)==tdim)
  % dimensions should be slices X volumes
  slicedata=slicedata';
end
if (size(slicedata,1)~=zdim | size(slicedata,2)~=tdim)
  disp('size of slicedata incorrect');
  size(slicedata)
  return
end
if (exist('keep_slices')==0)
  keep_slices=ones(zdim,1);
end
if (length(keep_slices)~=zdim)
  disp('length of keep_slices not equal to zdim:');
  size(keep_slices)
end
if (exist('TR')==0)
  % most common TR, however, a guess is worse than actually inputting the correct value
  TR=2000;
end
if (exist('TE')==0)
  % if GRE 3T, this is a good guess, otherwise actual TE is likely to be higher
  % suggested: 1.5T GRE TE=50, SE TE=100, 3T GRE TE=29, SE TE=75
  TE=29;
end
TR=double(TR);
TE=double(TE);
if (TR<30)
  % assume TR has been given in seconds, rather than milliseconds, adjust
  TR=TR*1000;
  if (isnumeric(slice_mode))
    slice_mode=slice_mode*1000;
  end
end
[acquisition_timing,slice_timing]=get_slice_timing(slice_mode,zdim,TR,TE);
sorted_slice_timing=sort(slice_timing);

for z=1:zdim
  % slice_index tells us which slice is chronologically acquired as z increases
  slice_index=find(slice_timing==sorted_slice_timing(z));
  data(z:zdim:zdim*tdim-zdim+z)=slicedata(slice_index,:);
  indices(z:zdim:zdim*tdim-zdim+z)=keep_slices(slice_index);
end

for i=1:tdim
  inputmesh(((i-1)*zdim)+1:(i*zdim))=((i-1)*TR)+acquisition_timing;
end
% interpolate to evenly spaced mesh (keep offset of TE since drawbacks 
% are minimal compared with interpolating an endpoint to infinity)
outputmesh=linspace(acquisition_timing(1),TR*tdim-(TR/zdim)+acquisition_timing(1),tdim*zdim);

% use only indices of slices that were significantly identified
inputmesh=inputmesh(find(indices==1));
data=data(find(indices==1))';

% interpolate over poorly-identified slices, keeping same mesh points as input
data=pchip([(2*inputmesh(1)-inputmesh(2)) inputmesh (2*inputmesh(end)-inputmesh(end-1))],[data(1); data; data(end)],[(2*outputmesh(1)-outputmesh(2)) outputmesh (2*outputmesh(end)-outputmesh(end-1))]);
data=data(2:end-1);

