function [tdim,fcard,fresp,fecg,fext]=fix_siemens_pmu(fname,TR,cardflag,respflag,ecgflag,truncatedflag,realtdim)
% this function applies the PMU sampling algorithm found in Beall and Lowe, Neuroimage 37/4 2007
% Siemens PMU data files consist of evenly sampled data, with an extra data point signifying 
% triggered events, such as a heart beat, breath or MCPU external output.  This extra data
% point is near 5000 in value.  All sampled PMU values fit in 12-bit range: 0-4092. Due to set-up
% and stopping time, there are additional samples at the start and end of acquisition.
% INPUT: TR (s or ms), the flags refer to whether or not .puls and .resp files exist, truncatedflag
% and realtdim are if the file was truncated (PMU has buffer so if saved very quickly, the file
% will be truncated, in this case, give the realtdim here and the produced file will be correct)
% 1. Remove datapoints with values greater than 4900
% 2. Find trigger events in external trigger file, count them to produce the tdim (assume
%    one trigger is output per volume, TR must be given in secs or millisecs)
% 3. use the trigger events to produce an acquisition start percentage and stop percentage
% 4. These percentages are applied to the pulse and respiration files assuming only that 
%    the starts and stops of all files are synchronous.
% 5. output is tdim, pulse, resp and external trigger synchronized to volumetric acquisition

if (TR>29)
  % assume TR is in seconds, but was given in milliseconds
  TR=TR/1000;
end

if (exist(sprintf('%s.ext',fname))==0)
  disp('trigger file does not exist, consider copying a trigger file from a similar scan');
  disp('ie, make sure that the number of volumes taken is the same for the scan you copy');
  disp('the trigger file from, or else this script will truncate to that number of vols');
  disp('make note of this (thus far, the number of pmu ticks prior to first trigger has');
  disp('been very stable, making this a viable method, so long as you copy from identical sequence)');
  return
end

% optional arguments if not given resolve to defaults of .puls and .resp existing and not truncated
if (exist('truncatedflag')==0)
  truncatedflag=0;
end
if (exist('cardflag')==0)
  cardflag=1;
end
if (exist('respflag')==0)
  respflag=1;
end
if (exist('ecgflag')==0)
  ecgflag=0;
end

% for all files that are assumed to exist, read in, remove first and last values
% and remove all trigger datapoints

fnamer=sprintf('%s.resp',fname);
if (exist(fnamer)==0)
  respflag=0;
end
if (respflag)
 temp=textread(fnamer,'%s', 'delimiter','\n', 'bufsize', 800000);
 resp=str2num(temp{1});
 resp=resp(5:end-1);
 resp=resp(find(resp<4900));
end

fnamec=sprintf('%s.puls',fname);
if (exist(fnamec)==0)
  cardflag=0;
end
if (cardflag)
 temp=textread(fnamec,'%s', 'delimiter','\n', 'bufsize', 800000);
 card=str2num(temp{1});
 card=card(5:end-1);
 card=card(find(card<4900));
end

fnamec=sprintf('%s.ecg',fname);
if (exist(fnamec)==0)
  ecgflag=0;
end
if (ecgflag)
 temp=textread(fnamec,'%s', 'delimiter','\n', 'bufsize', 80000000);
 a=temp{1};
 % find last non-numeric string and then following whitespace afterwards
 a=a(strfind(a,'maxLimitAVF:')+length('maxLimitAVF: '):end);
 ecg=str2num(a);
 % remove first two numbers (second number should be 6002, last should be 5003: both special numbers for pmu)
 ecg=ecg(3:end-1);
 % split into the two differential signals
 v1=ecg(find(ecg>5000));
 v2=ecg(find(ecg<5000));
 % find trigger pulses in v1
 ecg_trigs=find(v1<7000);
 trig_from_ecg=zeros(length(v1),1);
 trig_from_ecg(ecg_trigs+1)=1;
 trig_from_ecg(ecg_trigs)=-1;
 trig_from_ecg=trig_from_ecg(find(trig_from_ecg~=-1));
 % remove trigger pulses from v1
 v1=v1(find(v1>7000));
 clear ecg;
 % shorten length to that of shorter vector (sometimes there is an extra sample in v1)
 if (length(v1)>length(v2))
   v1=v1(1:length(v2));
   trig_from_ecg=trig_from_ecg(1:length(v2));
 elseif (length(v2)>length(v1))
   v2=v2(1:length(v1));
 end
 ecg(1,:)=v1;
 ecg(2,:)=v2;
 ecg(3,:)=trig_from_ecg;
 clear v1 v2 trig_from_ecg
end

% trigger file always must exist
temp=textread(sprintf('%s.ext',fname),'%s', 'delimiter','\n', 'bufsize', 80000000);
ext=str2num(temp{1});
% remove first five words and last word
ext=ext(5:end-1);
% set trigger point (so we can find it after we remove spurious trigger words)
ext(find(ext>4900)-1)=10;
% remove artificial trigs
ext=ext(find(ext<4900));

% find the trigger data points (sampling mesh)
xtrigs=find(ext(1:end)==10);
% where was first trigger relative to beginning of file
pctstart=xtrigs(1)/length(ext);
% set tdim
tdim=length(xtrigs);
% since we didn't mark end, have to determine it from xtrigs. Trigger is at volume start, so
% we simply divide by tdim-1 to get xlen per trigger
xlen=length(ext(xtrigs(1):xtrigs(end)))/(tdim-1);
% where was last sampling point (one volume after last trigger) relative to beginning of file
pctstop=((xtrigs(end)+round(xlen))/length(ext));
if (truncatedflag==1)
  % if truncated, we have less info, so we have to later on zerofill the data based on xlen (determined from
  % available triggers, which will be accurate for the lost triggers)
  tdim=realtdim;
  pctstop=1;
end

% trigger sampling: 200Hz, .resp and .puls are 50Hz, .ecg is 400Hz
% total length of trigger timepoints between first vol and end of last vol
extlen=TR*tdim*200;
% determine ppulen, resplen
ppulen=TR*tdim*50;
ecglen=TR*tdim*400;
resplen=TR*tdim*50;
% if truncated, zerofill to assumed end
if (truncatedflag==1)
  ext(end+1:1+round(pctstart*length(ext))+extlen)=0;
  card(end+1:1+round(pctstart*length(card))+ppulen)=2000;
  ecg(:,end+1:1+round(pctstart*length(ecg))+ecglen)=2000;
  resp(end+1:1+round(pctstart*length(resp))+resplen)=2000;
end
% take the triggers only from first volume start to last volume END
fext=ext(round(pctstart*length(ext)):round(pctstop*length(ext)));
% sanity checking
if (extlen>length(fext))
  disp('length not as calculated');
  extlen=length(fext)
end
fext=fext(1:extlen);

% resample to the volumetric triggers, producing something that is sampled exactly 50Hz
% with pchip(inputmesh,data,outputmesh)
if (cardflag)
 temp=card(round(pctstart*length(card)):round(pctstop*length(card)));
 fcard=pchip(linspace(1,ppulen,length(temp)),temp,1:ppulen);
else
 fcard=zeros(1,ppulen);
end
if (respflag)
 temp=resp(round(pctstart*length(resp)):round(pctstop*length(resp)));
 fresp=pchip(linspace(1,resplen,length(temp)),temp,1:resplen);
else
 fresp=zeros(1,resplen);
end
if (ecgflag)
 temp=ecg(1,round(pctstart*length(ecg)):round(pctstop*length(ecg)));
 fecg(1,:)=pchip(linspace(1,ecglen,length(temp)),temp,1:ecglen)-8200;
 temp=ecg(2,round(pctstart*length(ecg)):round(pctstop*length(ecg)));
 fecg(2,:)=pchip(linspace(1,ecglen,length(temp)),temp,1:ecglen);
 temp=ecg(3,round(pctstart*length(ecg)):round(pctstop*length(ecg)));
 temp=pchip(linspace(1,ecglen,length(temp)),temp,1:ecglen);
 % pchip will interpolate the triggers in most cases, so they have to be turned back into ones and zeros
 temp2=ceil(temp);
 inttrigs=find(temp2==1);
 doubles=find(inttrigs(2:end)-inttrigs(1:end-1)==1);
 for i=1:length(doubles)
   if (temp(inttrigs(doubles(i)))>temp(inttrigs(doubles(i))+1))
     temp2(inttrigs(doubles(i))+1)=0;
   else
     temp2(inttrigs(doubles(i)))=0;
   end
 end
 fecg(3,:)=temp2;
else
 fecg=zeros(3,ecglen);
end

