#include "GAMBITComputation.h"

#define DEBUG 0

Computation::Computation(Parameters _Parameters)
{
  m_Parameters = &_Parameters;
}

void Computation::Compute()
{
  ReadInputFile();
  CheckInputFileParameters();
  if (DEBUG) 
    Debug();
  SetBMSParameterFile();
  SetBMSComputationFile();
  WriteBMSParameterFile();  
  WriteBMSComputationFile();    
  ExecuteBatchMakeScript(GetBMSComputationFile());
  SetBMSMRMLCreationFile();
  WriteBMSMRMLCreationFile();
  ExecuteBatchMakeScript(GetBMSMRMLCreationFile());
}

void Computation::Debug()
{
  std::cout<<"\nDebug..."<<std::endl<<std::endl;
  
  std::cout<<"ModulePath: "<<m_Parameters->GetModulePath()<<std::endl;
  // IO
  std::cout<<"InputGroupFile: "<<m_Parameters->GetInputGroupFile()<<std::endl;
  std::cout<<"NbData: "<<GetNbData()<<std::endl;
  std::cout<<"T1 InputData:"<<std::endl;
  for (int i = 0; i < GetNbData(); i++)
    std::cout<<GetInputData(i,m_Parameters->GetT1ColumnNumber())<<" ";
  std::cout<<std::endl;
  std::cout<<"OutputDirectory: "<<m_Parameters->GetOutputDirectory()<<std::endl;
  std::cout<<"Data Subdirectory: "<<m_Parameters->GetDataSubdir()<<std::endl;
  if (m_Parameters->GetIsInputT2Image())
    {
      std::cout<<"T2 InputData:"<<std::endl;
      for (int i = 0; i < GetNbData(); i++)
	std::cout<<GetInputData(i,m_Parameters->GetT2ColumnNumber())<<" ";
      std::cout<<std::endl;
    }
  
  // Tissue segmentation
  std::cout<<"Tissue Segmentation: "<<m_Parameters->GetTissueSegmentation()<<std::endl;
  std::cout<<"SegAtlasDir: "<<m_Parameters->GetSegAtlasDir()<<std::endl;
  std::cout<<"SegAtlasType: "<<m_Parameters->GetSegAtlasType()<<std::endl;
  // Skull-stripping
  std::cout<<"Skull-stripping: "<<m_Parameters->GetSkullStripping()<<std::endl;
  std::cout<<"MaskDilation: "<<m_Parameters->GetMaskDilation()<<std::endl;
  // Atlas registration
  std::cout<<"RegistrationType: "<<m_Parameters->GetRegistrationType()<<std::endl;
  std::cout<<"ROIAtlasImage: "<<m_Parameters->GetROIAtlasImage()<<std::endl;
  std::cout<<"AtlasAbsoluteWMMaskImage: "<<m_Parameters->GetAtlasAbsoluteWMMaskImage()<<std::endl;
  std::cout<<"AtlasCSFToWMMaskImage: "<<m_Parameters->GetAtlasCSFToWMMaskImage()<<std::endl;
  std::cout<<"AtlasRemoveGMMaskImage: "<<m_Parameters->GetAtlasRemoveGMMaskImage()<<std::endl;
  std::cout<<"AtlasExclusionMaskImage: "<<m_Parameters->GetAtlasExclusionMaskImage()<<std::endl;
  std::cout<<"AtlasParcellationImage: "<<m_Parameters->GetAtlasParcellationImage()<<std::endl;
  std::cout<<"AtlasParticleInitializationImage: "<<m_Parameters->GetAtlasParticleInitializationImage()<<std::endl;
  // Advanced input file parameters
  std::cout<<"T1ManualSelection: "<<m_Parameters->GetT1ManualSelection()<<std::endl;
  std::cout<<"T1ColumnNumber: "<<m_Parameters->GetT1ColumnNumber()<<std::endl;
  std::cout<<"IsInputT2Image: "<<m_Parameters->GetIsInputT2Image()<<std::endl;
  std::cout<<"T2ColumnNumber: "<<m_Parameters->GetT2ColumnNumber()<<std::endl;
  std::cout<<"IsInputPDImage: "<<m_Parameters->GetIsInputPDImage()<<std::endl;
  std::cout<<"PDColumnNumber: "<<m_Parameters->GetPDColumnNumber()<<std::endl;
  // Advanced tissue segmentation parameters
  std::cout<<"ABCMaxBiasDegree: "<<m_Parameters->GetABCMaxBiasDegree()<<std::endl;
  std::cout<<"ABCPriors: "<<m_Parameters->GetABCPriors()<<std::endl;
  //for (unsigned int i = 0; i < m_Parameters->GetABCPriors().size(); i++)
  //  std::cout<<"ABCPriors: "<<m_Parameters->GetABCPriors()[i]<<std::endl;
  //std::cout<<"ABCAtlasWarpingOff: "<<m_Parameters->GetABCAtlasWarpingOff()<<std::endl;
  std::cout<<"ABCAtlasWarpIteration: "<<m_Parameters->GetABCAtlasWarpIteration()<<std::endl;
  std::cout<<"ABCAtlasLinearMapType: "<<m_Parameters->GetABCAtlasLinearMapType()<<std::endl;
  std::cout<<"ABCImageLinearMapType: "<<m_Parameters->GetABCImageLinearMapType()<<std::endl;
  // Advanced surface inflation parameters
  std::cout<<"InflationMaxIterationsBeforeFixing: "<<m_Parameters->GetInflationMaxIterationsBeforeFixing()<<std::endl;
  std::cout<<"InflationMaxCurvature: "<<m_Parameters->GetInflationMaxCurvature()<<std::endl;
  std::cout<<"InflationMaxIteration: "<<m_Parameters->GetInflationMaxIteration()<<std::endl;
  std::cout<<"InflationMeanCurvature: "<<m_Parameters->GetInflationMeanCurvature()<<std::endl;
  std::cout<<"InflationSmoothing: "<<m_Parameters->GetInflationSmoothing()<<std::endl;
  // Advanced correspondence parameters
  std::cout<<"CorrespondencePreprocessingSmoothing: "<<m_Parameters->GetCorrespondencePreprocessingSmoothing()<<std::endl;
  std::cout<<"CorrespondenceNbParticles: "<<m_Parameters->GetCorrespondenceNbParticles()<<std::endl;
  std::cout<<"CorrespondenceIterationsPerSplit: "<<m_Parameters->GetCorrespondenceIterationsPerSplit()<<std::endl;
  std::cout<<"CorrespondenceStartingRegularization: "<<m_Parameters->GetCorrespondenceStartingRegularization()<<std::endl;
  std::cout<<"CorrespondenceEndingRegularization: "<<m_Parameters->GetCorrespondenceEndingRegularization()<<std::endl;
  std::cout<<"CorrespondenceOptimizationIterations: "<<m_Parameters->GetCorrespondenceOptimizationIterations()<<std::endl;
  std::cout<<"CorrespondenceCheckPointingIntervals: "<<m_Parameters->GetCorrespondenceCheckPointingIntervals()<<std::endl;
  std::cout<<"CorrespondenceRelativeWeighting: "<<m_Parameters->GetCorrespondenceRelativeWeighting()<<std::endl;
  std::cout<<"CorrespondenceProcrustesOff: "<<m_Parameters->GetCorrespondenceProcrustesOff()<<std::endl;
  std::cout<<"CorrespondenceProcrustesInterval: "<<m_Parameters->GetCorrespondenceProcrustesInterval()<<std::endl;
  std::cout<<"CorrespondenceProcrustesScalingOff: "<<m_Parameters->GetCorrespondenceProcrustesScalingOff()<<std::endl;
  std::cout<<"CorrespondencePostProcessingProjectToSurfaceOff: "<<m_Parameters->GetCorrespondencePostProcessingProjectToSurfaceOff()<<std::endl;
  std::cout<<"CorrespondencePostProcessingProjectionFactor: "<<m_Parameters->GetCorrespondencePostProcessingProjectionFactor()<<std::endl;
  
  std::cout<<"Debug: Done!"<<std::endl<<std::endl;
}

void Computation::SetBMSParameterFile()
{
  m_BMSParameterFile = m_Parameters->GetOutputDirectory();
  m_BMSParameterFile.append("GAMBIT_Parameters.bms");
}

void Computation::SetBMSComputationFile()
{
  m_BMSComputationFile = m_Parameters->GetOutputDirectory();
  m_BMSComputationFile.append("GAMBIT_Computation.bms");
}

void Computation::SetBMSMRMLCreationFile()
{
  m_BMSMRMLCreationFile = m_Parameters->GetOutputDirectory();
  m_BMSMRMLCreationFile.append("GAMBIT_MRMLCreation.bms");
}

void Computation::ReadInputFile()
{
  std::cout<<"Reading input file..."<<std::endl;

  std::ifstream File(m_Parameters->GetInputGroupFile().c_str());
  std::string Line;
  bool T1VolumeFileFound = false;
  std::vector<std::vector<std::string> > vInputData;
  std::vector<std::string> vInputHeaders;
  unsigned int ColumnNb = 0, LineNb = 0, NbColumns = 0;

  if (File.is_open())
    {
      while (!File.eof() )
	{
	  getline(File,Line);
	  if(!Line.empty())
	    {
	      std::string Value;
	      std::vector<std::string> vData;
	      vData.clear();
	      std::istringstream StringValue(Line);
	      while(getline(StringValue,Value,','))
		{
		  if (LineNb == 0)
		    {
		      vInputHeaders.push_back(Value);
		      NbColumns++;
		    }
		  else
		    vData.push_back(Value);

		  if (LineNb == 1)
		    {
		      // One sets manually the input column of the T1 input images
		      if (m_Parameters->GetT1ManualSelection())
			{
			  if (ColumnNb == (unsigned int)m_Parameters->GetT1ColumnNumber() && !CheckImageFormat(Value))
			    DisplayErrorAndExit("T1w",0);
			}
		      //One doesn't set manually the input column of the T1 input images
		      else if (T1VolumeFileFound == false)
			{
			  if (CheckImageFormat(Value))
			    {
			      m_Parameters->SetT1ColumnNumber(ColumnNb);
			      T1VolumeFileFound = true;
			    }
			  else if (ColumnNb == NbColumns)
			    DisplayErrorAndExit("T1w",0);
			}
		      if (m_Parameters->GetIsInputT2Image() && ColumnNb == (unsigned int)m_Parameters->GetT2ColumnNumber() && !CheckImageFormat(Value))
			DisplayErrorAndExit("T2w",0);
		      else if (m_Parameters->GetIsInputPDImage() && ColumnNb == (unsigned int)m_Parameters->GetPDColumnNumber() && !CheckImageFormat(Value))
			DisplayErrorAndExit("PDw",0);
		      else if (m_Parameters->GetIsInputTissueLabelImage()  && ColumnNb == (unsigned int)m_Parameters->GetTissueLabelColumnNumber() && !CheckImageFormat(Value))
			DisplayErrorAndExit("tissue label",0);
		    }
		  ColumnNb++;
		}
	      ColumnNb = 0;
	      if (LineNb != 0)
		vInputData.push_back(vData);
	      LineNb++;
	    }
	}
      File.close();
      SetInputHeaders(vInputHeaders);
      SetInputData(vInputData);
      SetNbData(LineNb-1);
    }
  else
    {
      std::cerr<<"Error: cannot read input file!"<<std::endl;
      exit(1);
    }
}

bool Computation::CheckImageFormat(std::string _FileName)
{
  bool ImageFormatOk = false;
  if (_FileName.rfind(".gipl.gz") != std::string::npos || _FileName.rfind(".gipl") != std::string::npos || _FileName.rfind(".mha") != std::string::npos || _FileName.rfind(".mhd") != std::string::npos \
      || _FileName.rfind(".nrrd") != std::string::npos || _FileName.rfind(".nhdr") != std::string::npos || _FileName.rfind(".hdr") != std::string::npos || _FileName.rfind(".mnc") != std::string::npos \
      || _FileName.rfind(".nii") != std::string::npos || _FileName.rfind(".nii.gz") != std::string::npos)
    ImageFormatOk = true;
  else 
    ImageFormatOk = false;
  return ImageFormatOk;
}

void Computation::DisplayErrorAndExit(std::string _ImageModality, bool _Mode)
{
  if (_Mode == 0)
    std::cerr<<"Error: "<<_ImageModality<<" input image format not recognized! Please, double check input file or selected column number."<<std::endl;
  else
    std::cerr<<"Error: column number of "<<_ImageModality<<" input images too large!"<<std::endl;
  exit(1);
}

// Check if input images have unique column numbers
void Computation::CheckInputFileParameters()
{
  std::cout<<"Checking input file parameters..."<<std::endl;

  // To perform the skull-stripping, check if input tissue segmentation label map exists when tissue segmentation step is disabled
  if (m_Parameters->GetSkullStripping() && !m_Parameters->GetTissueSegmentation() && !m_Parameters->GetIsInputTissueLabelImage())
    {
      std::cerr<<"Error: Cannot perform skull-stripping if tissue segmentation step is disabled and input tissue segmentation label images are not provided!"<<std::endl;
      std::cerr<<"To perform a skull-stripping: enable tissue segmentation step OR provide input tissue segmentation label images."<<std::endl;      
      exit(1);      
    }

  // Check if input images have unique column numbers
  int NbDataTypes = GetInputHeaders().size();
  std::vector<int> v_ColumnNumber(NbDataTypes,0);
  if (m_Parameters->GetT1ColumnNumber() >= NbDataTypes)
    DisplayErrorAndExit("T1w",1);
  else
    v_ColumnNumber[m_Parameters->GetT1ColumnNumber()]++;

  if (m_Parameters->GetIsInputT2Image())
    {
      if (m_Parameters->GetT2ColumnNumber() >= NbDataTypes)
    DisplayErrorAndExit("T2w",1);
      else
	v_ColumnNumber[m_Parameters->GetT2ColumnNumber()]++;
    }
  if (m_Parameters->GetIsInputPDImage())
    {
      if (m_Parameters->GetPDColumnNumber() >= NbDataTypes)
	DisplayErrorAndExit("PDw",1);
      else
	v_ColumnNumber[m_Parameters->GetPDColumnNumber()]++;
    }
  if (m_Parameters->GetIsInputTissueLabelImage())
    {
      if (m_Parameters->GetTissueLabelColumnNumber() >= NbDataTypes)
	DisplayErrorAndExit("tissue label",1);
      else
	v_ColumnNumber[m_Parameters->GetTissueLabelColumnNumber()]++;
    }

  int i = 0; 
  while (i < NbDataTypes)
    {
      if (v_ColumnNumber[i] > 1)
	{
	  std::cerr<<"Error: At least two input image modalities have the same column number!"<<std::endl;
	  exit(1);
	}
      else 
	i++;
    }
}

void Computation::ExecuteBatchMakeScript(std::string _FileName)
{
  std::cout<<"Executing BatchMake..."<<std::endl;

  bm::ScriptParser Parser;
  char *envpath = getenv("BatchmakeWrapper_Dir");
  
  // Module path containing bmm files exists
  if(!chdir(m_Parameters->GetModulePath().c_str()))
    Parser.LoadWrappedApplication(m_Parameters->GetModulePath());
  // Environnment variable 'BatchmakeWrapper_Dir' has been set
  else if (envpath) 
    Parser.LoadWrappedApplication(envpath);
  else
    {
      std::cerr<<"The environment variable 'BatchmakeWrapper_Dir' needs to be set"<<std::endl;
      std::cerr<<"bash usage : export BatchmakeWrapper_Dir=<Batchmake Wrapper Directory>"<<std::endl;
      std::cerr<<"tcsh usage : setenv BatchmakeWrapper_Dir <Batchmake Wrapper Directory>"<<std::endl;
      exit(1);
    }  
  
  bool ComputationSuccess = Parser.Execute(_FileName);
  if (!ComputationSuccess)
    {
      std::cerr<<"\tExecuting BatchMake: Error!"<<std::endl;
      exit(1);
    }
}

void Computation::WriteBMSParameterFile()
{
  std::cout<<"Writing Parameter BatchMake script..."<<std::endl;

  std::ofstream BatchMakeScript(GetBMSParameterFile().c_str());

  BatchMakeScript<<"#BatchMake Parameter File"<<std::endl;
  BatchMakeScript<<"\n\n# Input Data"<<std::endl;
  BatchMakeScript<<"set(NbData "<<GetNbData()<<")"<<std::endl;

  BatchMakeScript<<"set(OrigT1CasesList "<<GetInputData(0,m_Parameters->GetT1ColumnNumber())<<")"<<std::endl;
  for (int Case = 1; Case < GetNbData(); Case++)
    BatchMakeScript<<"set(OrigT1CasesList ${OrigT1CasesList} "<<GetInputData(Case,m_Parameters->GetT1ColumnNumber())<<")"<<std::endl;

  if (m_Parameters->GetIsInputT2Image())
    {
      BatchMakeScript<<"\nset(OrigT2CasesList "<<GetInputData(0,m_Parameters->GetT2ColumnNumber())<<")"<<std::endl;
      for (int Case = 1; Case < GetNbData(); Case++)
	BatchMakeScript<<"set(OrigT2CasesList ${OrigT2CasesList} "<<GetInputData(Case,m_Parameters->GetT2ColumnNumber())<<")"<<std::endl;
    }
  if (m_Parameters->GetIsInputPDImage())
    {
      BatchMakeScript<<"\nset(OrigPDCasesList "<<GetInputData(0,m_Parameters->GetPDColumnNumber())<<")"<<std::endl;
      for (int Case = 1; Case < GetNbData(); Case++)
	BatchMakeScript<<"set(OrigPDCasesList ${OrigPDCasesList} "<<GetInputData(Case,m_Parameters->GetPDColumnNumber())<<")"<<std::endl;
    }
  if (m_Parameters->GetIsInputTissueLabelImage())
    {
      BatchMakeScript<<"\nset(OrigTissueLabelCasesList "<<GetInputData(0,m_Parameters->GetTissueLabelColumnNumber())<<")"<<std::endl;
      for (int Case = 1; Case < GetNbData(); Case++)
	BatchMakeScript<<"set(OrigTissueLabelCasesList ${OrigTissueLabelCasesList} "<<GetInputData(Case,m_Parameters->GetTissueLabelColumnNumber())<<")"<<std::endl;
    }

  BatchMakeScript<<"\n# IO"<<std::endl;
  BatchMakeScript<<"set (OutputDir "<<m_Parameters->GetOutputDirectory()<<")"<<std::endl;
  BatchMakeScript<<"set (DataSubdir "<<m_Parameters->GetDataSubdir()<<")"<<std::endl;
  if (m_Parameters->GetTissueSegmentation())
    {
      BatchMakeScript<<"\n# Tissue segmentation"<<std::endl;
      BatchMakeScript<<"set (SegAtlasDir "<<m_Parameters->GetSegAtlasDir()<<")"<<std::endl;
      BatchMakeScript<<"set (ABCPriors "<<m_Parameters->GetABCPriors()<<")"<<std::endl;
      BatchMakeScript<<"set (ABCMaxBiasDegree "<<m_Parameters->GetABCMaxBiasDegree()<<")"<<std::endl;
      BatchMakeScript<<"set (ABCAtlasLinearMapType "<<m_Parameters->GetABCAtlasLinearMapType()<<")"<<std::endl;
      BatchMakeScript<<"set (ABCImageLinearMapType "<<m_Parameters->GetABCImageLinearMapType()<<")"<<std::endl;
      BatchMakeScript<<"set (ABCAtlasWarpIteration "<<m_Parameters->GetABCAtlasWarpIteration()<<")"<<std::endl;
    }
  if (m_Parameters->GetSkullStripping())
    {
      BatchMakeScript<<"\n# Skull Stripping"<<std::endl;
      BatchMakeScript<<"set (MaskDilation "<<m_Parameters->GetMaskDilation()<<")"<<std::endl;
    }
  BatchMakeScript<<"\n# Atlas registration"<<std::endl;
  BatchMakeScript<<"set (RegistrationType "<<m_Parameters->GetRegistrationType()<<")"<<std::endl;
  BatchMakeScript<<"set (ROIAtlasImage "<<m_Parameters->GetROIAtlasImage()<<")"<<std::endl;
  BatchMakeScript<<"set (AtlasAbsoluteWMMaskImage "<<m_Parameters->GetAtlasAbsoluteWMMaskImage()<<")"<<std::endl;
  BatchMakeScript<<"set (AtlasCSFToWMMaskImage "<<m_Parameters->GetAtlasCSFToWMMaskImage()<<")"<<std::endl;
  BatchMakeScript<<"set (AtlasRemoveGMMaskImage "<<m_Parameters->GetAtlasRemoveGMMaskImage()<<")"<<std::endl;
  BatchMakeScript<<"set (AtlasExclusionMaskImage "<<m_Parameters->GetAtlasExclusionMaskImage()<<")"<<std::endl;
  BatchMakeScript<<"set (AtlasParcellationImage "<<m_Parameters->GetAtlasParcellationImage()<<")"<<std::endl;
  BatchMakeScript<<"set (AtlasParticleInitializationImage "<<m_Parameters->GetAtlasParticleInitializationImage()<<")"<<std::endl;
  BatchMakeScript<<"\n# Advanced surface inflation parameters"<<std::endl;
  BatchMakeScript<<"set (InflationMaxIterationsBeforeFixing "<<m_Parameters->GetInflationMaxIterationsBeforeFixing()<<")"<<std::endl;
  BatchMakeScript<<"set (InflationMaxCurvature "<<m_Parameters->GetInflationMaxCurvature()<<")"<<std::endl;
  BatchMakeScript<<"set (InflationMaxIteration "<<m_Parameters->GetInflationMaxIteration()<<")"<<std::endl;
  BatchMakeScript<<"set (InflationMeanCurvature "<<m_Parameters->GetInflationMeanCurvature()<<")"<<std::endl;
  BatchMakeScript<<"set (InflationSmoothing "<<m_Parameters->GetInflationSmoothing()<<")"<<std::endl;
  BatchMakeScript<<"\n# Advanced correspondence parameters"<<std::endl;
  BatchMakeScript<<"set (CorrespondencePreprocessingSmoothing "<<m_Parameters->GetCorrespondencePreprocessingSmoothing()<<")"<<std::endl;
  BatchMakeScript<<"set (CorrespondenceNbParticles "<<m_Parameters->GetCorrespondenceNbParticles()<<")"<<std::endl;
  BatchMakeScript<<"set (CorrespondenceIterationsPerSplit "<<m_Parameters->GetCorrespondenceIterationsPerSplit()<<")"<<std::endl;
  BatchMakeScript<<"set (CorrespondenceStartingRegularization "<<m_Parameters->GetCorrespondenceStartingRegularization()<<")"<<std::endl;
  BatchMakeScript<<"set (CorrespondenceEndingRegularization "<<m_Parameters->GetCorrespondenceEndingRegularization()<<")"<<std::endl;
  BatchMakeScript<<"set (CorrespondenceOptimizationIterations "<<m_Parameters->GetCorrespondenceOptimizationIterations()<<")"<<std::endl;
  BatchMakeScript<<"set (CorrespondenceCheckPointingIntervals "<<m_Parameters->GetCorrespondenceCheckPointingIntervals()<<")"<<std::endl;
  BatchMakeScript<<"set (CorrespondenceRelativeWeighting "<<m_Parameters->GetCorrespondenceRelativeWeighting()<<")"<<std::endl;
  BatchMakeScript<<"set (CorrespondenceProcrustesOff "<<m_Parameters->GetCorrespondenceProcrustesOff()<<")"<<std::endl;
  BatchMakeScript<<"set (CorrespondenceProcrustesInterval "<<m_Parameters->GetCorrespondenceProcrustesInterval()<<")"<<std::endl;
  BatchMakeScript<<"set (CorrespondenceProcrustesScalingOff "<<m_Parameters->GetCorrespondenceProcrustesScalingOff()<<")"<<std::endl;
  BatchMakeScript<<"set (CorrespondencePostProcessingProjectToSurfaceOff "<<m_Parameters->GetCorrespondencePostProcessingProjectToSurfaceOff()<<")"<<std::endl;
  BatchMakeScript<<"set (CorrespondencePostProcessingProjectionFactor "<<m_Parameters->GetCorrespondencePostProcessingProjectionFactor()<<")"<<std::endl;
  
  BatchMakeScript.close();

  std::cout<<"Writing Parameter BatchMake script: Done!"<<std::endl;
}

void Computation::WriteBMSComputationFile()
{
  std::cout<<"Writing Computation BatchMake script..."<<std::endl;

  std::ofstream BatchMakeScript(GetBMSComputationFile().c_str());

  BatchMakeScript<<"#---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"#---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"#--------------------- GAMBIT -----------------------------------------"<<std::endl;
  BatchMakeScript<<"#---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"#---------------------------------------------------------------------"<<std::endl;

  BatchMakeScript<<"Include("<<GetBMSParameterFile()<<")"<<std::endl;

  BatchMakeScript<<"\n# Applications"<<std::endl;
  BatchMakeScript<<"setApp(segmentation @'ABC (Atlas Based Classification)')"<<std::endl;
  BatchMakeScript<<"setApp(biasfieldcorrection @'N4ITK MRI Bias correction')"<<std::endl;
  BatchMakeScript<<"setApp(skullstripping @SegPostProcess)"<<std::endl;
  BatchMakeScript<<"setApp(BrainsfitRegistration @BRAINSFit)"<<std::endl;
  BatchMakeScript<<"setApp(DemonsRegistration @BRAINSDemonWarp)"<<std::endl;
  BatchMakeScript<<"setApp(Transformation @BRAINSResample)"<<std::endl;
  BatchMakeScript<<"setApp(cortthick @CortThick)"<<std::endl;
  BatchMakeScript<<"setApp(WMSegPostProcessing @WMSegPostProcess)"<<std::endl;
  BatchMakeScript<<"setApp(LabelSegPostProcessing @LabelSegPostProcess)"<<std::endl;
  BatchMakeScript<<"setApp(GenusZero @'Genus 0 Image Filter')"<<std::endl;
  BatchMakeScript<<"setApp(meshinflation @MeshInflation)"<<std::endl;
  BatchMakeScript<<"setApp(correspondencepreprocessing @'Particle Correspondence Preprocessing')"<<std::endl;
  BatchMakeScript<<"setApp(correspondencepostprocessing @'Particle Correspondence Postprocessing using Thin Plate Splines')"<<std::endl;

  BatchMakeScript<<"\nset (ImageSuffix '')"<<std::endl;
  BatchMakeScript<<"set (LabelImageSuffix '')"<<std::endl;

  BatchMakeScript<<"\n# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# 1. Bias field correction "<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"echo (*************************************************)"<<std::endl;
  BatchMakeScript<<"echo ('BIAS FIELD CORRECTION...')"<<std::endl;
  BatchMakeScript<<"echo ( )"<<std::endl;
  BatchMakeScript<<"set (CaseNumber 0)"<<std::endl;
  BatchMakeScript<<"set (BiasFieldCorrectedSuffix '_bias')"<<std::endl;
  BatchMakeScript<<"set (NewImageSuffix ${ImageSuffix}${BiasFieldCorrectedSuffix})"<<std::endl;
  BatchMakeScript<<"set (T1CasesList ${OrigT1CasesList})"<<std::endl;
  BatchMakeScript<<"ForEach (T1Case ${T1CasesList})"<<std::endl;
  BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
  BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
  BatchMakeScript<<"  echo ( )"<<std::endl;
  BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
  BatchMakeScript<<"  echo ( )"<<std::endl;
  BatchMakeScript<<"  # Creating GAMBIT Directory if necessary"<<std::endl;
  BatchMakeScript<<"  set (GAMBITPath ${T1Path}/${DataSubdir}/)"<<std::endl;
  BatchMakeScript<<"  FileExists(IsGAMBITPath ${GAMBITPath})"<<std::endl;
  BatchMakeScript<<"  If (${IsGAMBITPath} == 0)"<<std::endl;
  BatchMakeScript<<"     MakeDirectory(${GAMBITPath})"<<std::endl;
  BatchMakeScript<<"  EndIf (${IsGAMBITPath})"<<std::endl;
  BatchMakeScript<<"  # Creating  Bias Field Correction directory if necessary"<<std::endl;
  BatchMakeScript<<"  set (BiasFieldCorrectedPath ${T1Path}/${DataSubdir}/BiasFieldCorrected/)"<<std::endl;
  BatchMakeScript<<"  FileExists(IsBiasFieldCorrectedPath ${BiasFieldCorrectedPath})"<<std::endl;
  BatchMakeScript<<"  If (${IsBiasFieldCorrectedPath} == 0)"<<std::endl;
  BatchMakeScript<<"     MakeDirectory(${BiasFieldCorrectedPath})"<<std::endl;
  BatchMakeScript<<"  EndIf (${IsBiasFieldCorrectedPath})"<<std::endl;
  BatchMakeScript<<"  set (BiasFieldCorrectedT1Case ${BiasFieldCorrectedPath}${T1CaseHead}${NewImageSuffix}.nrrd)"<<std::endl;
  BatchMakeScript<<"  FileExists(IsBiasFieldCorrectedT1Case ${BiasFieldCorrectedT1Case})"<<std::endl;
  BatchMakeScript<<"  If (${IsBiasFieldCorrectedT1Case} == 0)"<<std::endl;
  BatchMakeScript<<"    SetAppOption(biasfieldcorrection.inputImageName.inputImageName ${T1Case})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(biasfieldcorrection.outputImageName.outputImageName ${BiasFieldCorrectedT1Case})"<<std::endl;
  BatchMakeScript<<"    Run(outputbiasfieldcorrection ${biasfieldcorrection} errorbiasfieldcorrection)"<<std::endl;
  BatchMakeScript<<"    If (${errorbiasfieldcorrection} != '')"<<std::endl;
  BatchMakeScript<<"      echo ('Bias field correction error: ' ${errorbiasfieldcorrection})"<<std::endl;
  BatchMakeScript<<"      exit()"<<std::endl;
  BatchMakeScript<<"    EndIf (${errorbiasfieldcorrection})"<<std::endl;
  BatchMakeScript<<"  Else ()"<<std::endl;
  BatchMakeScript<<"    echo ('T1 bias-field correction already done!')"<<std::endl;
  BatchMakeScript<<"  EndIf  (${IsBiasFieldCorrectedT1Case})"<<std::endl;

  if (m_Parameters->GetIsInputT2Image())
    {
      BatchMakeScript<<"  If (${OrigT2CasesList} != '')"<<std::endl;
      BatchMakeScript<<"    GetParam (T2Case ${OrigT2CasesList} ${CaseNumber})"<<std::endl;
      BatchMakeScript<<"  EndIf (${OrigT2CasesList})"<<std::endl;  
      BatchMakeScript<<"  GetFilename (T2CaseHead ${T2Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
      BatchMakeScript<<"  set (BiasFieldCorrectedT2Case ${BiasFieldCorrectedPath}${T2CaseHead}${NewImageSuffix}.nrrd)"<<std::endl;
      BatchMakeScript<<"  FileExists(IsBiasFieldCorrectedT2Case ${BiasFieldCorrectedT2Case})"<<std::endl;
      BatchMakeScript<<"  If (${IsBiasFieldCorrectedT2Case} == 0)"<<std::endl;
      BatchMakeScript<<"    SetAppOption(biasfieldcorrection.inputImageName.inputImageName ${T2Case})"<<std::endl;
      BatchMakeScript<<"    SetAppOption(biasfieldcorrection.outputImageName.outputImageName ${BiasFieldCorrectedT2Case})"<<std::endl;
      BatchMakeScript<<"    Run(outputbiasfieldcorrection ${biasfieldcorrection} errorbiasfieldcorrection)"<<std::endl;
      BatchMakeScript<<"    If (${errorbiasfieldcorrection} != '')"<<std::endl;
      BatchMakeScript<<"      echo ('Bias field correction error: ' ${errorbiasfieldcorrection})"<<std::endl;
      BatchMakeScript<<"      exit()"<<std::endl;
      BatchMakeScript<<"    EndIf (${errorbiasfieldcorrection})"<<std::endl;
      BatchMakeScript<<"  Else ()"<<std::endl;
      BatchMakeScript<<"    echo ('T2 bias-field correction already done!')"<<std::endl;
      BatchMakeScript<<"  EndIf  (${IsBiasFieldCorrectedT2Case})"<<std::endl;
    }
  if (m_Parameters->GetIsInputPDImage())
    {
      BatchMakeScript<<"  If (${OrigPDCasesList} != '')"<<std::endl;
      BatchMakeScript<<"    GetParam (PDCase ${OrigPDCasesList} ${CaseNumber})"<<std::endl;
      BatchMakeScript<<"  EndIf (${OrigPDCasesList})"<<std::endl;  
      BatchMakeScript<<"  GetFilename (PDCaseHead ${PDCase} NAME_WITHOUT_EXTENSION)"<<std::endl;
      BatchMakeScript<<"  set (BiasFieldCorrectedPDCase ${BiasFieldCorrectedPath}${PDCaseHead}${NewImageSuffix}.nrrd)"<<std::endl;
      BatchMakeScript<<"  FileExists(IsBiasFieldCorrectedPDCase ${BiasFieldCorrectedPDCase})"<<std::endl;
      BatchMakeScript<<"  If (${IsBiasFieldCorrectedPDCase} == 0)"<<std::endl;
      BatchMakeScript<<"    SetAppOption(biasfieldcorrection.inputImageName.inputImageName ${PDCase})"<<std::endl;
      BatchMakeScript<<"    SetAppOption(biasfieldcorrection.outputImageName.outputImageName ${BiasFieldCorrectedPDCase})"<<std::endl;
      BatchMakeScript<<"    Run(outputbiasfieldcorrection ${biasfieldcorrection} errorbiasfieldcorrection)"<<std::endl;
      BatchMakeScript<<"    If (${errorbiasfieldcorrection} != '')"<<std::endl;
      BatchMakeScript<<"      echo ('Bias field correction error: ' ${errorbiasfieldcorrection})"<<std::endl;
      BatchMakeScript<<"      exit()"<<std::endl;
      BatchMakeScript<<"    EndIf (${errorbiasfieldcorrection})"<<std::endl;
      BatchMakeScript<<"  Else ()"<<std::endl;
      BatchMakeScript<<"    echo ('PD bias-field correction already done!')"<<std::endl;
      BatchMakeScript<<"  EndIf  (${IsBiasFieldCorrectedPDCase})"<<std::endl;
    }
  BatchMakeScript<<"  Inc (${CaseNumber} 1)"<<std::endl;
  BatchMakeScript<<"  Int (${CaseNumber})"<<std::endl;
  BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
  BatchMakeScript<<"set (ImageSuffix ${NewImageSuffix})"<<std::endl;
  BatchMakeScript<<"echo ( )"<<std::endl;
  BatchMakeScript<<"echo ('BIAS FIELD CORRECTION: DONE!')"<<std::endl;
  BatchMakeScript<<"echo ( )"<<std::endl;

  if (m_Parameters->GetTissueSegmentation())
    {
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# 2. Tissue Segmentation "<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"echo (*************************************************)"<<std::endl;
  BatchMakeScript<<"echo ('TISSUE SEGMENTATION...')"<<std::endl;
  BatchMakeScript<<"echo ( )"<<std::endl;
  BatchMakeScript<<"set (CaseNumber 0)"<<std::endl;
  BatchMakeScript<<"set (TissueSegmentationSuffix '_corrected_EMS')"<<std::endl;
  BatchMakeScript<<"set (TissueSegmentationLabelSuffix '_labels_EMS')"<<std::endl;
  BatchMakeScript<<"set (NewImageSuffix ${ImageSuffix}${TissueSegmentationSuffix})"<<std::endl;
  BatchMakeScript<<"set (NewLabelImageSuffix ${ImageSuffix}${TissueSegmentationLabelSuffix})"<<std::endl;
  BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
  BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
  BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
  BatchMakeScript<<"  echo ( )"<<std::endl;
  BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
  BatchMakeScript<<"  echo ( )"<<std::endl;
  BatchMakeScript<<"  # Creating EMS Directory if necessary"<<std::endl;
  BatchMakeScript<<"  set (EMSPath ${T1Path}/${DataSubdir}/TissueSegmentation/)"<<std::endl;
  BatchMakeScript<<"  FileExists(IsEMSPath ${EMSPath})"<<std::endl;
  BatchMakeScript<<"  If (${IsEMSPath} == 0)"<<std::endl;
  BatchMakeScript<<"     MakeDirectory(${EMSPath})"<<std::endl;
  BatchMakeScript<<"  EndIf (${IsEMSPath})"<<std::endl;
  BatchMakeScript<<"  set (BiasFieldCorrectedPath ${T1Path}/${DataSubdir}/BiasFieldCorrected/)"<<std::endl;
  BatchMakeScript<<"  set (BiasFieldCorrectedCase ${BiasFieldCorrectedPath}${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;  
  BatchMakeScript<<"  FileExists(IsEMSFile ${EMSPath}${T1CaseHead}${NewImageSuffix}.nrrd)"<<std::endl;
  BatchMakeScript<<"  If (${IsEMSFile} == 0)"<<std::endl;
  BatchMakeScript<<"    SetAppOption(segmentation.inputImage1.inputImage1 ${BiasFieldCorrectedCase})"<<std::endl;
  if (m_Parameters->GetIsInputT2Image())
    {
      BatchMakeScript<<"    If (${OrigT2CasesList} != '')"<<std::endl;
      BatchMakeScript<<"      GetParam (T2Case ${OrigT2CasesList} ${CaseNumber})"<<std::endl;
      BatchMakeScript<<"    EndIf (${OrigT2CasesList})"<<std::endl;  
      BatchMakeScript<<"    GetFilename (T2CaseHead ${T2Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
      BatchMakeScript<<"    set (BiasFieldCorrectedT2Case ${BiasFieldCorrectedPath}${T2CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
      BatchMakeScript<<"    SetAppOption(segmentation.inputImage2.inputImage2 ${BiasFieldCorrectedT2Case})"<<std::endl;
    }
  if (m_Parameters->GetIsInputPDImage())
    {
      BatchMakeScript<<"    If (${OrigPDCasesList} != '')"<<std::endl;
      BatchMakeScript<<"      GetParam (PDCase ${OrigPDCasesList} ${CaseNumber})"<<std::endl;
      BatchMakeScript<<"    EndIf (${OrigPDCasesList})"<<std::endl;  
      BatchMakeScript<<"    GetFilename (PDCaseHead ${PDCase} NAME_WITHOUT_EXTENSION)"<<std::endl;
      BatchMakeScript<<"    set (BiasFieldCorrectedPDCase ${BiasFieldCorrectedPath}${PDCaseHead}${ImageSuffix}.nrrd)"<<std::endl;
      BatchMakeScript<<"    SetAppOption(segmentation.inputImage3.inputImage3 ${BiasFieldCorrectedPDCase})"<<std::endl;
    }  
  BatchMakeScript<<"    SetAppOption(segmentation.atlasDir.atlasDir ${SegAtlasDir})"<<std::endl;

  /*std::stringstream Prior1(stringstream::in | stringstream::out);
  Prior1 << m_Parameters->GetABCPriors()[0];
  std::cout<<"Prior1: "<<Prior1<<std::endl;
  std::cout<<"Prior1: "<<m_Parameters->GetABCPriors()[0]<<std::endl;
  std::string Priors = Prior1.str();
  std::cout<<"Priors: "<<Priors<<std::endl;
  for (unsigned int i = 0; i < m_Parameters->GetABCPriors().size(); i++)
    {
      Priors.append(",");
      std::stringstream Prior(stringstream::in | stringstream::out);
      Prior << m_Parameters->GetABCPriors()[i];
      Priors.append(Prior.str());
    }
    BatchMakeScript<<"    SetAppOption(segmentation.priorAdjustVec.priorAdjustVec "<<Priors<<")"<<std::endl;*/

  BatchMakeScript<<"    SetAppOption(segmentation.priorAdjustVec.priorAdjustVec ${ABCPriors})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(segmentation.biasDegree.biasDegree ${ABCMaxBiasDegree})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(segmentation.atlasMapType.atlasMapType ${ABCAtlasLinearMapType})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(segmentation.coregMapType.coregMapType ${ABCImageLinearMapType})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(segmentation.atlasFluidIters.atlasFluidIters ${ABCAtlasWarpIteration})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(segmentation.outputImage1.outputImage1 ${EMSPath}${T1CaseHead}${NewImageSuffix}.nrrd)"<<std::endl;
  BatchMakeScript<<"    SetAppOption(segmentation.labelImage.labelImage ${EMSPath}${T1CaseHead}${NewLabelImageSuffix}.nrrd)"<<std::endl;
  if (m_Parameters->GetIsInputT2Image())
    BatchMakeScript<<"    SetAppOption(segmentation.outputImage2.outputImage2 ${EMSPath}${T2CaseHead}${NewImageSuffix}.nrrd)"<<std::endl;
  if (m_Parameters->GetIsInputPDImage())
    BatchMakeScript<<"    SetAppOption(segmentation.outputImage3.outputImage3 ${EMSPath}${PDCaseHead}${NewImageSuffix}.nrrd)"<<std::endl;  
  BatchMakeScript<<"    Run(outputsegmentation ${segmentation} errorsegmentation)"<<std::endl;
  BatchMakeScript<<"    If (${errorsegmentation} != '')"<<std::endl;
  BatchMakeScript<<"      echo ('Tissue segmentation error: ' ${errorsegmentation})"<<std::endl;
  BatchMakeScript<<"      exit()"<<std::endl;
  BatchMakeScript<<"    EndIf (${errorsegmentation})"<<std::endl;
  BatchMakeScript<<"  Else ()"<<std::endl;
  BatchMakeScript<<"    echo ('Tissue segmentation already done!')"<<std::endl;
  BatchMakeScript<<"  EndIf  (${IsEMSFile})"<<std::endl;
  BatchMakeScript<<"  Inc (${CaseNumber} 1)"<<std::endl;
  BatchMakeScript<<"  Int (${CaseNumber})"<<std::endl;
  BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
  BatchMakeScript<<"set (ImageSuffix ${NewImageSuffix})"<<std::endl;
  BatchMakeScript<<"set (LabelImageSuffix ${NewLabelImageSuffix})"<<std::endl;
  BatchMakeScript<<"echo ( )"<<std::endl;
  BatchMakeScript<<"echo ('EMS SEGMENTATION: DONE!')"<<std::endl;
  BatchMakeScript<<"echo ( )"<<std::endl;
    }

  if (m_Parameters->GetSkullStripping())
    {
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# 3. Atlas ROI segmentation"<<std::endl;
  BatchMakeScript<<"# 3.1. Skull-stripping"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"echo (*************************************************)"<<std::endl;
  BatchMakeScript<<"echo ('SKULL-STRIPPING...')"<<std::endl;
  BatchMakeScript<<"echo ( )"<<std::endl;
  BatchMakeScript<<"set (SkullStrippingSuffix '_stripped')"<<std::endl;
  BatchMakeScript<<"set (NewImageSuffix ${ImageSuffix}${SkullStrippingSuffix})"<<std::endl;
  BatchMakeScript<<"set (CasesList ${OrigT1CasesList})"<<std::endl;
  BatchMakeScript<<"set (CaseNumber 0)"<<std::endl;
  BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
  BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
  BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
  BatchMakeScript<<"  echo ( )"<<std::endl;
  BatchMakeScript<<"  echo ('T1Case Number: '${T1CaseHead})"<<std::endl;
  BatchMakeScript<<"  echo ( )"<<std::endl;  
  BatchMakeScript<<"  set (SkullStrippedPath ${T1Path}/${DataSubdir}/SkullStripped/)"<<std::endl;
  BatchMakeScript<<"  FileExists(IsSkullStrippedPath ${SkullStrippedPath})"<<std::endl;
  BatchMakeScript<<"  If (${IsSkullStrippedPath} == 0)"<<std::endl;
  BatchMakeScript<<"     MakeDirectory(${SkullStrippedPath})"<<std::endl;
  BatchMakeScript<<"  EndIf (${IsSkullStrippedPath})"<<std::endl;

  // Tissue segmentation is performed
  if (m_Parameters->GetTissueSegmentation())
    {
      BatchMakeScript<<"  set (InputImagePath ${T1Path}/${DataSubdir}/TissueSegmentation/)"<<std::endl;
      BatchMakeScript<<"  set (LabelImage ${InputImagePath}${T1CaseHead}${LabelImageSuffix}.nrrd)"<<std::endl;
    }
  // Tissue segmentation is not performed: label images are provided as inputs
  else
    {
      BatchMakeScript<<"  set (InputImagePath ${T1Path}/${DataSubdir}/BiasFieldCorrected/)"<<std::endl;
      BatchMakeScript<<"  If (${OrigTissueLabelCasesList} != '')"<<std::endl;
      BatchMakeScript<<"    GetParam (LabelImage ${OrigTissueLabelCasesList} ${CaseNumber})"<<std::endl;
      BatchMakeScript<<"  EndIf (${OrigTissueLabelCasesList})"<<std::endl;
    }
  BatchMakeScript<<"  set (T1InputImage ${InputImagePath}${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
  BatchMakeScript<<"  GetFilename (LabelImageHead ${LabelImage} NAME_WITHOUT_EXTENSION)"<<std::endl;
  BatchMakeScript<<"  set (MaskImage ${SkullStrippedPath}${LabelImageHead}_mask.nrrd)"<<std::endl;
  BatchMakeScript<<"  FileExists(IsMaskImage ${MaskImage})"<<std::endl;
  BatchMakeScript<<"  If (${IsMaskImage} == 0)"<<std::endl;
  BatchMakeScript<<"    set (StrippedT1Case ${SkullStrippedPath}${T1CaseHead}${NewImageSuffix}.nrrd)"<<std::endl;
  BatchMakeScript<<"    SetAppOption(skullstripping.fileName ${LabelImage})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(skullstripping.outfileName ${StrippedT1Case})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(skullstripping.notstrippedfileName.notstrippedfileName ${T1InputImage})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(skullstripping.maskfileName.maskfileName ${MaskImage})"<<std::endl;
  BatchMakeScript<<"    If (${MaskDilation} == 1)"<<std::endl;
  BatchMakeScript<<"      SetAppOption(skullstripping.dilate ${MaskDilation})"<<std::endl;
  BatchMakeScript<<"    Endif (${MaskDilation})"<<std::endl; 
  BatchMakeScript<<"    Run(outputskullstripping ${skullstripping} errorskullstripping)"<<std::endl;
  BatchMakeScript<<"    If (${errorskullstripping} != '')"<<std::endl;
  BatchMakeScript<<"      echo ('Skull-stripping error: ' ${errorskullstripping})"<<std::endl;
  BatchMakeScript<<"      exit()"<<std::endl;
  BatchMakeScript<<"    EndIf (${errorskullstripping})"<<std::endl;
  BatchMakeScript<<"  Else (${IsMaskImage})"<<std::endl;
  BatchMakeScript<<"    echo ('Skull-stripping already done!')"<<std::endl;
  BatchMakeScript<<"  EndIf (${IsMaskImage})"<<std::endl;
  if (m_Parameters->GetIsInputT2Image())
    {
      BatchMakeScript<<"  If (${OrigT2CasesList} != '')"<<std::endl;
      BatchMakeScript<<"    GetParam (T2Case ${OrigT2CasesList} ${CaseNumber})"<<std::endl;
      BatchMakeScript<<"  EndIf (${OrigT2CasesList})"<<std::endl;
      BatchMakeScript<<"  GetFilename (T2CaseHead ${T2Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
      BatchMakeScript<<"  set (T2InputImage ${InputImagePath}${T2CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
      BatchMakeScript<<"  set (StrippedT2Case ${SkullStrippedPath}${T2CaseHead}${NewImageSuffix}.nrrd)"<<std::endl;
      BatchMakeScript<<"  FileExists(IsStrippedT2Case ${StrippedT2Case})"<<std::endl;
      BatchMakeScript<<"  If (${IsStrippedT2Case} == 0)"<<std::endl;
      BatchMakeScript<<"    Run(output 'ImageMath '${T2InputImage}' -mask '${MaskImage}' -outfile '${StrippedT2Case} error)"<<std::endl;
      BatchMakeScript<<"    If (${error} != '')"<<std::endl;
      BatchMakeScript<<"      echo ('ImageMath error: ' ${error})"<<std::endl;
      BatchMakeScript<<"      exit()"<<std::endl;
      BatchMakeScript<<"    EndIf (${error})"<<std::endl;
      BatchMakeScript<<"  Else (${IsStrippedT2Case})"<<std::endl;
      BatchMakeScript<<"    echo ('T2 Skull-stripping already done!')"<<std::endl;
      BatchMakeScript<<"  EndIf (${IsStrippedT2Case})"<<std::endl;
    }
  if (m_Parameters->GetIsInputPDImage())
    {
      BatchMakeScript<<"  If (${OrigPDCasesList} != '')"<<std::endl;
      BatchMakeScript<<"    GetParam (PDCase ${OrigPDCasesList} ${CaseNumber})"<<std::endl;
      BatchMakeScript<<"  EndIf (${OrigPDCasesList})"<<std::endl;  
      BatchMakeScript<<"  GetFilename (PDCaseHead ${PDCase} NAME_WITHOUT_EXTENSION)"<<std::endl;
      BatchMakeScript<<"  set (PDInputImage ${InputImagePath}${PDCaseHead}${ImageSuffix}.nrrd)"<<std::endl;
      BatchMakeScript<<"  set (StrippedPDCase ${SkullStrippedPath}${PDCaseHead}${NewImageSuffix}.nrrd)"<<std::endl;
      BatchMakeScript<<"  FileExists(IsStrippedPDCase ${StrippedPDCase})"<<std::endl;
      BatchMakeScript<<"  If (${IsStrippedPDCase} == 0)"<<std::endl;
      BatchMakeScript<<"    Run(output 'ImageMath '${PDInputImage}' -mask '${MaskImage}' -outfile '${StrippedPDCase} error)"<<std::endl;
      BatchMakeScript<<"    If (${error} != '')"<<std::endl;
      BatchMakeScript<<"      echo ('ImageMath error: ' ${error})"<<std::endl;
      BatchMakeScript<<"      exit()"<<std::endl;
      BatchMakeScript<<"    EndIf (${error})"<<std::endl;
      BatchMakeScript<<"  Else (${IsStrippedPDCase})"<<std::endl;
      BatchMakeScript<<"    echo ('PD Skull-stripping already done!')"<<std::endl;
      BatchMakeScript<<"  EndIf (${IsStrippedPDCase})"<<std::endl;
    }

  BatchMakeScript<<"  Inc (${CaseNumber} 1)"<<std::endl;
  BatchMakeScript<<"  Int (${CaseNumber})"<<std::endl;
  BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
  BatchMakeScript<<"set (ImageSuffix ${NewImageSuffix})"<<std::endl;
  BatchMakeScript<<"echo ( )"<<std::endl;
  BatchMakeScript<<"echo ('SKULL-STRIPPING: DONE!')"<<std::endl;
  BatchMakeScript<<"echo ( )"<<std::endl;
    }

  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# 3.2. Atlas registration"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"echo (*************************************************)"<<std::endl;
  BatchMakeScript<<"echo ('Atlas registration...')"<<std::endl;
  BatchMakeScript<<"echo ( )"<<std::endl;
  BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
  BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
  BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
  BatchMakeScript<<"  echo ( )"<<std::endl;
  BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
  BatchMakeScript<<"  echo ( )"<<std::endl;
  BatchMakeScript<<"  set (ROISegmentationPath ${T1Path}/${DataSubdir}/ROISegmentation/)"<<std::endl;
  BatchMakeScript<<"  FileExists(IsROISegmentationPath ${ROISegmentationPath})"<<std::endl;
  BatchMakeScript<<"  If (${IsROISegmentationPath} == 0)"<<std::endl;
  BatchMakeScript<<"     MakeDirectory(${ROISegmentationPath})"<<std::endl;
  BatchMakeScript<<"  EndIf (${IsROISegmentationPath})"<<std::endl;  

  if (!m_Parameters->GetSkullStripping())
    BatchMakeScript<<"  set (InputImagePath ${T1Path}/${DataSubdir}/BiasFieldCorrected/)"<<std::endl;
  else
    BatchMakeScript<<"  set (InputImagePath ${T1Path}/${DataSubdir}/SkullStripped/)"<<std::endl;
  BatchMakeScript<<"  set (StrippedCase ${InputImagePath}${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
  BatchMakeScript<<"  set (RegisteredAtlas ${ROISegmentationPath}AtlasRegistered_${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
  
  if (!std::strcmp(m_Parameters->GetRegistrationType().c_str(),"BSpline"))
    BatchMakeScript<<"  set (TransformFile ${ROISegmentationPath}AtlasBSplineTransform_${T1CaseHead}${ImageSuffix}.txt)"<<std::endl;
  else
    BatchMakeScript<<"  set (TransformFile ${ROISegmentationPath}AtlasAffineTransform_${T1CaseHead}${ImageSuffix}.txt)"<<std::endl;
  BatchMakeScript<<"  FileExists(IsTransformFile ${TransformFile})"<<std::endl;
  BatchMakeScript<<"  If (${IsTransformFile} == 0)"<<std::endl;
  BatchMakeScript<<"    SetAppOption(BrainsfitRegistration.fixedVolume.fixedVolume ${StrippedCase})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(BrainsfitRegistration.movingVolume.movingVolume ${ROIAtlasImage})"<<std::endl;  
  if (!std::strcmp(m_Parameters->GetRegistrationType().c_str(),"BSpline"))
    {
      BatchMakeScript<<"    echo ('Atlas BSpline registration...')"<<std::endl;
      BatchMakeScript<<"    SetAppOption(BrainsfitRegistration.transformType.transformType 'Rigid,Affine,BSpline')"<<std::endl;
      BatchMakeScript<<"    SetAppOption(BrainsfitRegistration.outputVolume.outputVolume ${RegisteredAtlas})"<<std::endl;
      BatchMakeScript<<"    SetAppOption(BrainsfitRegistration.outputVolumePixelType.outputVolumePixelType 'ushort')"<<std::endl;
    }
  else
    {
      BatchMakeScript<<"    echo ('Atlas affine registration...')"<<std::endl;
      BatchMakeScript<<"    SetAppOption(BrainsfitRegistration.transformType.transformType 'Rigid,Affine')"<<std::endl;
    }
  BatchMakeScript<<"    SetAppOption(BrainsfitRegistration.useCenterOfHeadAlign 1)"<<std::endl;
  BatchMakeScript<<"    SetAppOption(BrainsfitRegistration.outputTransform.outputTransform ${TransformFile})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(BrainsfitRegistration.interpolationMode.interpolationMode 'BSpline')"<<std::endl;
  BatchMakeScript<<"    Run(outputBrainsfitRegistration ${BrainsfitRegistration} errorBrainsfitRegistration)"<<std::endl;
  //  BatchMakeScript<<"    If (${errorBrainsfitRegistration} != '0')"<<std::endl;
    //  BatchMakeScript<<"      echo ('Brainsfit registration error: ' ${errorBrainsfitRegistration})"<<std::endl;
  //BatchMakeScript<<"      exit()"<<std::endl;
  //  BatchMakeScript<<"    EndIf (${errorBrainsfitRegistration})"<<std::endl;
  BatchMakeScript<<"  Else (${IsTransformFile})"<<std::endl;
  if (!std::strcmp(m_Parameters->GetRegistrationType().c_str(),"BSpline"))
    BatchMakeScript<<"    echo ('BSpline registration already done!')"<<std::endl;
  else
    BatchMakeScript<<"    echo ('Affine registration already done!')"<<std::endl;
  BatchMakeScript<<"  EndIf (${IsTransformFile})"<<std::endl;

  if (std::strcmp(m_Parameters->GetRegistrationType().c_str(),"BSpline"))
    {
      BatchMakeScript<<"  set (InitTransformFile ${TransformFile})"<<std::endl;
      BatchMakeScript<<"  set (TransformFile ${ROISegmentationPath}AtlasDemonsTransform_${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
      BatchMakeScript<<"  FileExists(IsRegisteredAtlas ${RegisteredAtlas})"<<std::endl;
      BatchMakeScript<<"  If (${IsRegisteredAtlas} == 0)"<<std::endl;
      BatchMakeScript<<"    echo ('Atlas demons registration...')"<<std::endl;      
      BatchMakeScript<<"    SetAppOption(DemonsRegistration.fixedVolume.fixedVolume ${StrippedCase})"<<std::endl;
      BatchMakeScript<<"    SetAppOption(DemonsRegistration.movingVolume.movingVolume ${ROIAtlasImage})"<<std::endl;
      BatchMakeScript<<"    SetAppOption(DemonsRegistration.initializeWithTransform.initializeWithTransform ${InitTransformFile})"<<std::endl;
      BatchMakeScript<<"    SetAppOption(DemonsRegistration.outputDeformationFieldVolume.outputDeformationFieldVolume ${TransformFile})"<<std::endl;
      BatchMakeScript<<"    SetAppOption(DemonsRegistration.outputVolume.outputVolume ${RegisteredAtlas})"<<std::endl;
      if (!std::strcmp(m_Parameters->GetRegistrationType().c_str(),"Demons"))
	BatchMakeScript<<"    SetAppOption(DemonsRegistration.registrationFilterType.registrationFilterType 'Demons')"<<std::endl;
      else if (!std::strcmp(m_Parameters->GetRegistrationType().c_str(),"Diffeomorphic"))
	BatchMakeScript<<"    SetAppOption(DemonsRegistration.registrationFilterType.registrationFilterType 'Diffeomorphic')"<<std::endl;
      else if (!std::strcmp(m_Parameters->GetRegistrationType().c_str(),"LogDemons"))
	BatchMakeScript<<"    SetAppOption(DemonsRegistration.registrationFilterType.registrationFilterType 'LogDemons')"<<std::endl;
      else if (!std::strcmp(m_Parameters->GetRegistrationType().c_str(),"SymmetricLogDemons"))
	BatchMakeScript<<"    SetAppOption(DemonsRegistration.registrationFilterType.registrationFilterType 'SymmetricLogDemons')"<<std::endl; 	
      else
	{
	  std::cerr<<"Error: registration type!"<<std::endl;
	  exit(1);
	}
      BatchMakeScript<<"    SetAppOption(DemonsRegistration.interpolationMode.interpolationMode 'BSpline')"<<std::endl;
      BatchMakeScript<<"    SetAppOption(DemonsRegistration.histogramMatch 1)"<<std::endl;
      BatchMakeScript<<"    SetAppOption(DemonsRegistration.numberOfHistogramBins.numberOfHistogramBins 1024)"<<std::endl;
      BatchMakeScript<<"    SetAppOption(DemonsRegistration.numberOfMatchPoints.numberOfMatchPoints 50)"<<std::endl;
      BatchMakeScript<<"    SetAppOption(DemonsRegistration.outputPixelType.outputPixelType 'uchar')"<<std::endl;
      BatchMakeScript<<"    Run(outputDemonsRegistration ${DemonsRegistration} errorDemonsRegistration)"<<std::endl;
      BatchMakeScript<<"    If (${errorDemonsRegistration} != '')"<<std::endl;
      BatchMakeScript<<"      echo ('Demons registration error: ' ${errorDemonsRegistration})"<<std::endl;
      BatchMakeScript<<"      exit()"<<std::endl;
      BatchMakeScript<<"    EndIf (${errorDemonsRegistration})"<<std::endl;
      BatchMakeScript<<"  Else (${IsRegisteredAtlas})"<<std::endl;
      BatchMakeScript<<"    echo ('Atlas demons registration already done!')"<<std::endl;
      BatchMakeScript<<"  EndIf (${IsRegisteredAtlas})"<<std::endl;
    }
  BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
  BatchMakeScript<<"echo ( )"<<std::endl;
  BatchMakeScript<<"echo ('ATLAS REGISTRATION: DONE!')"<<std::endl;
  BatchMakeScript<<"echo ( )"<<std::endl;

  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# 3.3. Parcellation and ROI segmentation"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"echo (*************************************************)"<<std::endl;
  BatchMakeScript<<"echo ('PARCELLATION AND ROI SEGMENTATION...')"<<std::endl;
  BatchMakeScript<<"echo ( )"<<std::endl;
  BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
  BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
  BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
  BatchMakeScript<<"  echo ( )"<<std::endl;
  BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
  BatchMakeScript<<"  echo ( )"<<std::endl;
  BatchMakeScript<<"  set (ROISegmentationPath ${T1Path}/${DataSubdir}/ROISegmentation/)"<<std::endl;
  BatchMakeScript<<"  set (SkullStrippedPath ${T1Path}/${DataSubdir}/SkullStripped/)"<<std::endl;
  BatchMakeScript<<"  set (StrippedCase ${SkullStrippedPath}${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
  if (!std::strcmp(m_Parameters->GetRegistrationType().c_str(),"BSpline"))
    BatchMakeScript<<"  set (TransformFile ${ROISegmentationPath}AtlasBSplineTransform_${T1CaseHead}${ImageSuffix}.txt)"<<std::endl;
  else
    BatchMakeScript<<"  set (DeformationVolume ${ROISegmentationPath}AtlasDemonsTransform_${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
  BatchMakeScript<<"  set (RegisteredAbsoluteWMMaskImage ${ROISegmentationPath}AbsoluteWMMaskImageRegistered_${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
  BatchMakeScript<<"  FileExists(IsRegisteredAbsoluteWMMaskImage ${RegisteredAbsoluteWMMaskImage})"<<std::endl;
  BatchMakeScript<<"  If (${IsRegisteredAbsoluteWMMaskImage} == 0)"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.inputVolume.inputVolume ${AtlasAbsoluteWMMaskImage})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.outputVolume.outputVolume ${RegisteredAbsoluteWMMaskImage})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.referenceVolume.referenceVolume ${StrippedCase})"<<std::endl;
  if (!std::strcmp(m_Parameters->GetRegistrationType().c_str(),"BSpline"))
    BatchMakeScript<<"    SetAppOption(Transformation.warpTransform.warpTransform ${TransformFile})"<<std::endl;
  else
    BatchMakeScript<<"    SetAppOption(Transformation.deformationVolume.deformationVolume ${DeformationVolume})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.interpolationMode.interpolationMode 'NearestNeighbor')"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.pixelType.pixelType 'binary')"<<std::endl;
  BatchMakeScript<<"    Run(outputTransformation ${Transformation} errorTransformation)"<<std::endl;
  BatchMakeScript<<"    If (${errorTransformation} != '')"<<std::endl;
  BatchMakeScript<<"      echo ('AbsoluteWMMaskImage Demons registration error: ' ${errorTransformation})"<<std::endl;
  BatchMakeScript<<"      exit()"<<std::endl;
  BatchMakeScript<<"    EndIf (${errorTransformation})"<<std::endl;
  BatchMakeScript<<"  Else (${IsRegisteredAbsoluteWMMaskImage})"<<std::endl;
  BatchMakeScript<<"    echo ('AbsoluteWMMaskImage Demons registration already done!')"<<std::endl;
  BatchMakeScript<<"  EndIf (${IsRegisteredAbsoluteWMMaskImage})"<<std::endl;
  BatchMakeScript<<"  set (RegisteredCSFToWMMaskImage ${ROISegmentationPath}CSFToWMMaskImageRegistered_${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
  BatchMakeScript<<"  FileExists(IsRegisteredCSFToWMMaskImage ${RegisteredCSFToWMMaskImage})"<<std::endl;
  BatchMakeScript<<"  If (${IsRegisteredCSFToWMMaskImage} == 0)"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.inputVolume.inputVolume ${AtlasCSFToWMMaskImage})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.outputVolume.outputVolume ${RegisteredCSFToWMMaskImage})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.referenceVolume.referenceVolume ${StrippedCase})"<<std::endl;
  if (!std::strcmp(m_Parameters->GetRegistrationType().c_str(),"BSpline"))
    BatchMakeScript<<"    SetAppOption(Transformation.warpTransform.warpTransform ${TransformFile})"<<std::endl;
  else
    BatchMakeScript<<"    SetAppOption(Transformation.deformationVolume.deformationVolume ${DeformationVolume})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.interpolationMode.interpolationMode 'NearestNeighbor')"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.pixelType.pixelType 'binary')"<<std::endl;
  BatchMakeScript<<"    Run(outputTransformation ${Transformation} errorTransformation)"<<std::endl;
  BatchMakeScript<<"    If (${errorTransformation} != '')"<<std::endl;
  BatchMakeScript<<"      echo ('CSFToWMMaskImage Demons registration error: ' ${errorTransformation})"<<std::endl;
  BatchMakeScript<<"      exit()"<<std::endl;
  BatchMakeScript<<"    EndIf (${errorTransformation})"<<std::endl;
  BatchMakeScript<<"  Else (${IsRegisteredCSFToWMMaskImage})"<<std::endl;
  BatchMakeScript<<"    echo ('CSFToWMMaskImage Demons registration already done!')"<<std::endl;
  BatchMakeScript<<"  EndIf (${IsRegisteredCSFToWMMaskImage})"<<std::endl;
  BatchMakeScript<<"  set (RegisteredRemoveGMMaskImage ${ROISegmentationPath}RemoveGMMaskImageRegistered_${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
  BatchMakeScript<<"  FileExists(IsRegisteredRemoveGMMaskImage ${RegisteredRemoveGMMaskImage})"<<std::endl;
  BatchMakeScript<<"  If (${IsRegisteredRemoveGMMaskImage} == 0)"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.inputVolume.inputVolume ${AtlasRemoveGMMaskImage})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.outputVolume.outputVolume ${RegisteredRemoveGMMaskImage})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.referenceVolume.referenceVolume ${StrippedCase})"<<std::endl;
  if (!std::strcmp(m_Parameters->GetRegistrationType().c_str(),"BSpline"))
    BatchMakeScript<<"    SetAppOption(Transformation.warpTransform.warpTransform ${TransformFile})"<<std::endl;
  else
    BatchMakeScript<<"    SetAppOption(Transformation.deformationVolume.deformationVolume ${DeformationVolume})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.interpolationMode.interpolationMode 'NearestNeighbor')"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.pixelType.pixelType 'binary')"<<std::endl;
  BatchMakeScript<<"    Run(outputTransformation ${Transformation} errorTransformation)"<<std::endl;
  BatchMakeScript<<"    If (${errorTransformation} != '')"<<std::endl;
  BatchMakeScript<<"      echo ('RemoveGMMaskImage Demons registration error: ' ${errorTransformation})"<<std::endl;
  BatchMakeScript<<"      exit()"<<std::endl;
  BatchMakeScript<<"    EndIf (${errorTransformation})"<<std::endl;
  BatchMakeScript<<"  Else (${IsRegisteredRemoveGMMaskImage})"<<std::endl;
  BatchMakeScript<<"    echo ('RemoveGMMaskImage Demons registration already done!')"<<std::endl;
  BatchMakeScript<<"  EndIf (${IsRegisteredRemoveGMMaskImage})"<<std::endl;
  BatchMakeScript<<"  set (RegisteredExclusionMaskImage ${ROISegmentationPath}ExclusionMaskImageRegistered_${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
  BatchMakeScript<<"  FileExists(IsRegisteredExclusionMaskImage ${RegisteredExclusionMaskImage})"<<std::endl;
  BatchMakeScript<<"  If (${IsRegisteredExclusionMaskImage} == 0)"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.inputVolume.inputVolume ${AtlasExclusionMaskImage})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.outputVolume.outputVolume ${RegisteredExclusionMaskImage})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.referenceVolume.referenceVolume ${StrippedCase})"<<std::endl;
  if (!std::strcmp(m_Parameters->GetRegistrationType().c_str(),"BSpline"))
    BatchMakeScript<<"    SetAppOption(Transformation.warpTransform.warpTransform ${TransformFile})"<<std::endl;
  else
    BatchMakeScript<<"    SetAppOption(Transformation.deformationVolume.deformationVolume ${DeformationVolume})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.interpolationMode.interpolationMode 'NearestNeighbor')"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.pixelType.pixelType 'binary')"<<std::endl;
  BatchMakeScript<<"    Run(outputTransformation ${Transformation} errorTransformation)"<<std::endl;
  BatchMakeScript<<"    If (${errorTransformation} != '')"<<std::endl;
  BatchMakeScript<<"      echo ('ExclusionMaskImage Demons registration error: ' ${errorTransformation})"<<std::endl;
  BatchMakeScript<<"      exit()"<<std::endl;
  BatchMakeScript<<"    EndIf (${errorTransformation})"<<std::endl;
  BatchMakeScript<<"  Else (${IsRegisteredExclusionMaskImage})"<<std::endl;
  BatchMakeScript<<"    echo ('ExclusionMaskImage Demons registration already done!')"<<std::endl;
  BatchMakeScript<<"  EndIf (${IsRegisteredExclusionMaskImage})"<<std::endl;
  BatchMakeScript<<"  set (RegisteredParcellation ${ROISegmentationPath}ParcellationRegistered_${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
  BatchMakeScript<<"  FileExists(IsRegisteredParcellation ${RegisteredParcellation})"<<std::endl;
  BatchMakeScript<<"  If (${IsRegisteredParcellation} == 0)"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.inputVolume.inputVolume ${AtlasParcellationImage})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.outputVolume.outputVolume ${RegisteredParcellation})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.referenceVolume.referenceVolume ${StrippedCase})"<<std::endl;
  if (!std::strcmp(m_Parameters->GetRegistrationType().c_str(),"BSpline"))
    BatchMakeScript<<"    SetAppOption(Transformation.warpTransform.warpTransform ${TransformFile})"<<std::endl;
  else
    BatchMakeScript<<"    SetAppOption(Transformation.deformationVolume.deformationVolume ${DeformationVolume})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.interpolationMode.interpolationMode 'NearestNeighbor')"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.pixelType.pixelType 'short')"<<std::endl;
  BatchMakeScript<<"    Run(outputTransformation ${Transformation} errorTransformation)    "<<std::endl;
  BatchMakeScript<<"    If (${errorTransformation} != '')"<<std::endl;
  BatchMakeScript<<"      echo ('Parcellation map Demons registration error: ' ${errorTransformation})"<<std::endl;
  BatchMakeScript<<"      exit()"<<std::endl;
  BatchMakeScript<<"    EndIf (${errorTransformation})"<<std::endl;
  BatchMakeScript<<"  Else (${IsRegisteredParcellation})"<<std::endl;
  BatchMakeScript<<"    echo ('Parcellation map Demons registration already done!')"<<std::endl;
  BatchMakeScript<<"  EndIf (${IsRegisteredParcellation})"<<std::endl;
  BatchMakeScript<<"  set (RegisteredParticleInitialization ${ROISegmentationPath}ParticleInitializationRegistered_${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
  BatchMakeScript<<"  FileExists(IsRegisteredParticleInitialization ${RegisteredParticleInitialization})"<<std::endl;
  BatchMakeScript<<"  If (${IsRegisteredParticleInitialization} == 0)"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.inputVolume.inputVolume ${AtlasParticleInitializationImage})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.outputVolume.outputVolume ${RegisteredParticleInitialization})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.referenceVolume.referenceVolume ${StrippedCase})"<<std::endl;
  if (!std::strcmp(m_Parameters->GetRegistrationType().c_str(),"BSpline"))
    BatchMakeScript<<"    SetAppOption(Transformation.warpTransform.warpTransform ${TransformFile})"<<std::endl;
  else
    BatchMakeScript<<"    SetAppOption(Transformation.deformationVolume.deformationVolume ${DeformationVolume})"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.interpolationMode.interpolationMode 'NearestNeighbor')"<<std::endl;
  BatchMakeScript<<"    SetAppOption(Transformation.pixelType.pixelType 'short')"<<std::endl;
  BatchMakeScript<<"    Run(outputTransformation ${Transformation} errorTransformation)"<<std::endl;
  BatchMakeScript<<"    If (${errorTransformation} != '')"<<std::endl;
  BatchMakeScript<<"      echo ('ParticleInitialization map Demons registration error: ' ${errorTransformation})"<<std::endl;
  BatchMakeScript<<"      exit()"<<std::endl;
  BatchMakeScript<<"    EndIf (${errorTransformation})"<<std::endl;
  BatchMakeScript<<"  Else (${IsRegisteredParticleInitialization})"<<std::endl;
  BatchMakeScript<<"    echo ('ParticleInitialization Demons registration already done!')"<<std::endl;
  BatchMakeScript<<"  EndIf (${IsRegisteredParticleInitialization})"<<std::endl;
  BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
  BatchMakeScript<<"echo ( )"<<std::endl;
  BatchMakeScript<<"echo ('PARCELLATION AND ROI SEGMENTATION: DONE!')"<<std::endl;
  BatchMakeScript<<"echo ( )"<<std::endl;

BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 3.4. Lateral Ventricles segmentation"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('LATERAL VENTRICLES SEGMENTATION...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"set (CaseNumber 0)"<<std::endl;
BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  set (ROISegmentationPath ${T1Path}/${DataSubdir}/ROISegmentation/)"<<std::endl;
BatchMakeScript<<"  set (RegisteredCSFToWMMaskImage ${ROISegmentationPath}CSFToWMMaskImageRegistered_${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
BatchMakeScript<<"  set (LatVent ${ROISegmentationPath}${T1CaseHead}${ImageSuffix}_latVentricles.nrrd)"<<std::endl;
BatchMakeScript<<"  set (EMSPath ${T1Path}/${DataSubdir}/TissueSegmentation/)"<<std::endl;
// Tissue segmentation is performed
 if (m_Parameters->GetTissueSegmentation())
   {
     BatchMakeScript<<"  set (LabelMap ${EMSPath}${T1CaseHead}${LabelImageSuffix}.nrrd)"<<std::endl;
     BatchMakeScript<<"  set (CSF ${EMSPath}${T1CaseHead}${LabelImageSuffix}_CSF.nrrd)"<<std::endl;
   }
  // Tissue segmentation is not performed: label images are provided as inputs
 else
   {
     BatchMakeScript<<"  If (${OrigTissueLabelCasesList} != '')"<<std::endl;
     BatchMakeScript<<"    GetParam (LabelMap ${OrigTissueLabelCasesList} ${CaseNumber})"<<std::endl;
     BatchMakeScript<<"  EndIf (${OrigTissueLabelCasesList})"<<std::endl;
     BatchMakeScript<<"  GetFilename (LabelMapHead ${LabelMap} NAME_WITHOUT_EXTENSION)"<<std::endl;
     BatchMakeScript<<"  set (CSF ${ROISegmentationPath}${LabelMapHead}_CSF.nrrd)"<<std::endl;
   }
BatchMakeScript<<"  FileExists(IsCSF ${CSF})"<<std::endl;
BatchMakeScript<<"  If (${IsCSF} == 0)"<<std::endl;
BatchMakeScript<<"    Run(output 'ImageMath '${LabelMap}' -extractLabel 3 -outfile '${CSF} error)"<<std::endl;
BatchMakeScript<<"    If (${error} != '')"<<std::endl;
BatchMakeScript<<"      echo ('ImageMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${error})"<<std::endl;
BatchMakeScript<<"  Endif (${IsCSF})"<<std::endl;
BatchMakeScript<<"  FileExists(IsLatVent ${LatVent})"<<std::endl;
BatchMakeScript<<"  If (${IsLatVent} == 0)"<<std::endl;
BatchMakeScript<<"    Run(output 'ImageMath '${CSF}' -mask '${RegisteredCSFToWMMaskImage}' -outfile '${LatVent} error)"<<std::endl;
BatchMakeScript<<"    If (${error} != '')"<<std::endl;
BatchMakeScript<<"      echo ('ImageMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${error})"<<std::endl;
BatchMakeScript<<"  Else (${IsLatVent})"<<std::endl;
 BatchMakeScript<<"    echo ('Lateral ventricles segmentation already done!')"<<std::endl;
BatchMakeScript<<"  Endif (${IsLatVent})"<<std::endl;
 BatchMakeScript<<"  Inc (${CaseNumber} 1)"<<std::endl;
 BatchMakeScript<<"  Int (${CaseNumber})"<<std::endl;
 BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
 BatchMakeScript<<"echo ('LATERAL VENTRICLES SEGMENTATION: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;

BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 4. Label map creation"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('LABEL MAP CREATION...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
 BatchMakeScript<<"set (CaseNumber 0)"<<std::endl;
BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  set (LabelMapPath ${T1Path}/${DataSubdir}/LabelMap/)"<<std::endl;
BatchMakeScript<<"  FileExists(IsLabelMapPath ${LabelMapPath})"<<std::endl;
BatchMakeScript<<"  If (${IsLabelMapPath} == 0)"<<std::endl;
BatchMakeScript<<"    MakeDirectory(${LabelMapPath})"<<std::endl;
BatchMakeScript<<"  EndIf (${LabelMapPath})"<<std::endl;
BatchMakeScript<<"  set (ROISegmentationPath ${T1Path}/${DataSubdir}/ROISegmentation/)"<<std::endl;
BatchMakeScript<<"  set (LabelMap ${LabelMapPath}${T1CaseHead}${ImageSuffix}_LabelMap.nrrd)"<<std::endl;
BatchMakeScript<<"  FileExists(IsLabelMap ${LabelMap})"<<std::endl;
BatchMakeScript<<"  If (${IsLabelMap} == 0)"<<std::endl;
BatchMakeScript<<"    set (EMSPath ${T1Path}/${DataSubdir}/TissueSegmentation/)"<<std::endl;

// Tissue segmentation is performed
 if (m_Parameters->GetTissueSegmentation())
   BatchMakeScript<<"    set (InitLabelMap ${EMSPath}${T1CaseHead}${LabelImageSuffix}.nrrd)"<<std::endl;
  // Tissue segmentation is not performed: label images are provided as inputs
 else
   {
     BatchMakeScript<<"    If (${OrigTissueLabelCasesList} != '')"<<std::endl;
     BatchMakeScript<<"      GetParam (InitLabelMap ${OrigTissueLabelCasesList} ${CaseNumber})"<<std::endl;
     BatchMakeScript<<"    EndIf (${OrigTissueLabelCasesList})"<<std::endl;
   }
BatchMakeScript<<"    set (FinalAbsoluteWMMask ${ROISegmentationPath}${T1CaseHead}${ImageSuffix}_AbsoluteWMMask.nrrd)"<<std::endl;
BatchMakeScript<<"    FileExists(IsFinalAbsoluteWMMask ${FinalAbsoluteWMMask})"<<std::endl;
BatchMakeScript<<"    If (${IsFinalAbsoluteWMMask} == 0)"<<std::endl;
BatchMakeScript<<"      set (LatVent ${ROISegmentationPath}${T1CaseHead}${ImageSuffix}_latVentricles.nrrd)"<<std::endl;
BatchMakeScript<<"      set (RegisteredAbsoluteWMMaskImage ${ROISegmentationPath}AbsoluteWMMaskImageRegistered_${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;  
BatchMakeScript<<"      Run (output 'ImageMath '${RegisteredAbsoluteWMMaskImage}' -combine '${LatVent}' -outfile '${FinalAbsoluteWMMask} error)"<<std::endl;
BatchMakeScript<<"      If (${error} != '')"<<std::endl;
BatchMakeScript<<"        echo ('ImageMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"        exit()"<<std::endl;
BatchMakeScript<<"      EndIf (${error})"<<std::endl;
BatchMakeScript<<"    EndIf (${IsFinalAbsoluteWMMask})"<<std::endl;
BatchMakeScript<<"    set (RegisteredRemoveGMMaskImage ${ROISegmentationPath}RemoveGMMaskImageRegistered_${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
BatchMakeScript<<"    set (RegisteredExclusionMaskImage ${ROISegmentationPath}ExclusionMaskImageRegistered_${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
BatchMakeScript<<"    set (WMMap ${LabelMapPath}${T1CaseHead}${ImageSuffix}_WMMap.nrrd)"<<std::endl;
BatchMakeScript<<"    set (GMMap ${LabelMapPath}${T1CaseHead}${ImageSuffix}_GMMap.nrrd)"<<std::endl;
BatchMakeScript<<"    set (CSFMap ${LabelMapPath}${T1CaseHead}${ImageSuffix}_CSFMap.nrrd)"<<std::endl;  
BatchMakeScript<<"    SetAppOption(LabelSegPostProcessing.InputLabel ${InitLabelMap})"<<std::endl;
BatchMakeScript<<"    SetAppOption(LabelSegPostProcessing.OutputLabel ${LabelMap})"<<std::endl;
BatchMakeScript<<"    SetAppOption(LabelSegPostProcessing.AbsoluteWMMaskImage.AbsoluteWMMaskImage ${FinalAbsoluteWMMask})"<<std::endl;
BatchMakeScript<<"    SetAppOption(LabelSegPostProcessing.RemoveGMMaskImage.RemoveGMMaskImage ${RegisteredRemoveGMMaskImage})"<<std::endl;
BatchMakeScript<<"    SetAppOption(LabelSegPostProcessing.ExclusionMaskImage.ExclusionMaskImage ${RegisteredExclusionMaskImage})"<<std::endl;
BatchMakeScript<<"    SetAppOption(LabelSegPostProcessing.OutputWM.OutputWM ${WMMap})"<<std::endl;
BatchMakeScript<<"    SetAppOption(LabelSegPostProcessing.OutputGM.OutputGM ${GMMap})"<<std::endl;
BatchMakeScript<<"    SetAppOption(LabelSegPostProcessing.OutputCSF.OutputCSF ${CSFMap})"<<std::endl;
BatchMakeScript<<"    Run (outputlabelsegpostprocessing ${LabelSegPostProcessing} errorlabelsegpostprocessing)"<<std::endl;
BatchMakeScript<<"    If (${errorlabelsegpostprocessing} != '')"<<std::endl;
BatchMakeScript<<"      echo ('Label segmentation post-processing error: ' ${errorlabelsegpostprocessing})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${errorlabelsegpostprocessing})"<<std::endl;
BatchMakeScript<<"  Else (${IsLabelMap})"<<std::endl;
BatchMakeScript<<"    echo ('Label map creation already done!')"<<std::endl;
BatchMakeScript<<"  EndIf (${IsLabelMap})"<<std::endl;
 BatchMakeScript<<"  Inc (${CaseNumber} 1)"<<std::endl;
 BatchMakeScript<<"  Int (${CaseNumber})"<<std::endl;
BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('LABEL MAP CREATION: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 5. Cortical thickness computation"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('CORTICAL THICKNESS COMPUTATION...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
 BatchMakeScript<<"set (CaseNumber 0)"<<std::endl;
BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  set (CortThickPath ${T1Path}/${DataSubdir}/CorticalThickness/)  "<<std::endl;
BatchMakeScript<<"  FileExists(IsCortThickPath ${CortThickPath})"<<std::endl;
BatchMakeScript<<"  If (${IsCortThickPath} == 0)"<<std::endl;
BatchMakeScript<<"    MakeDirectory(${CortThickPath})"<<std::endl;
BatchMakeScript<<"  EndIf (${IsCortThickPath})  "<<std::endl;
BatchMakeScript<<"  set (WMCortThickImage ${CortThickPath}${T1CaseHead}_AvgCortThickOnWMBoundary.nrrd)"<<std::endl;
BatchMakeScript<<"  FileExists(IsWMCortThickImage ${WMCortThickImage})"<<std::endl;
BatchMakeScript<<"  If (${IsWMCortThickImage} == 0)"<<std::endl;
BatchMakeScript<<"    set (LabelMapPath ${T1Path}/${DataSubdir}/LabelMap/)"<<std::endl;
BatchMakeScript<<"    set (LabelMap ${LabelMapPath}${T1CaseHead}${ImageSuffix}_LabelMap.nrrd)"<<std::endl;
BatchMakeScript<<"    set (ROISegmentationPath ${T1Path}/${DataSubdir}/ROISegmentation/)"<<std::endl;
BatchMakeScript<<"    set (RegisteredParcellation ${ROISegmentationPath}ParcellationRegistered_${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
BatchMakeScript<<"    set (GMCortThickImage ${CortThickPath}${T1CaseHead}_AvgCortThickOnGMBoundary.nrrd)"<<std::endl;
BatchMakeScript<<"    SetAppOption(cortthick.Output ${CortThickPath})"<<std::endl;
BatchMakeScript<<"    SetAppOption(cortthick.SegmentationInput.SegmentationInput ${LabelMap})"<<std::endl;
BatchMakeScript<<"    SetAppOption(cortthick.ParcellationOption.ParcellationOption ${RegisteredParcellation})"<<std::endl;
BatchMakeScript<<"    SetAppOption(cortthick.WhiteMatterFile.WhiteMatterFile ${WMCortThickImage})"<<std::endl;
BatchMakeScript<<"    SetAppOption(cortthick.GMMapsAvgDist.GMMapsAvgDist ${GMCortThickImage})"<<std::endl;
BatchMakeScript<<"    SetAppOption(cortthick.Interp 1)"<<std::endl;
BatchMakeScript<<"    SetAppOption(cortthick.Threshold.Threshold 1.8)"<<std::endl;
BatchMakeScript<<"    Run (outputcortthick ${cortthick} errorcortthick)"<<std::endl;
BatchMakeScript<<"    If (${errorcortthick} != '')"<<std::endl;
BatchMakeScript<<"      echo ('Cortical thickness error: ' ${errorcortthick})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${errorcortthick})"<<std::endl;
BatchMakeScript<<"    set (ParcellationOnWMImage ${CortThickPath}parcellationOnWhite.nrrd)"<<std::endl;
BatchMakeScript<<"    set (ParcellationOnWMInterpImage ${CortThickPath}parcellationOnWhiteInterp.nrrd)"<<std::endl;
BatchMakeScript<<"    set (FinalParcellationOnWMInterpImage ${CortThickPath}${T1CaseHead}_parcellationOnWMBoundary.nrrd)"<<std::endl;
BatchMakeScript<<"    CopyFile(${ParcellationOnWMInterpImage} ${FinalParcellationOnWMInterpImage})"<<std::endl;
BatchMakeScript<<"    DeleteFile(${ParcellationOnWMInterpImage})"<<std::endl;
BatchMakeScript<<"    DeleteFile(${ParcellationOnWMImage})"<<std::endl;
BatchMakeScript<<"  Else (${IsWMCortThickImage})"<<std::endl;
BatchMakeScript<<"    echo ('Cortical thickness computation already done!')"<<std::endl;
BatchMakeScript<<"  EndIf (${IsWMCortThickImage})"<<std::endl;
 BatchMakeScript<<"  Inc (${CaseNumber} 1)"<<std::endl;
 BatchMakeScript<<"  Int (${CaseNumber})"<<std::endl;
BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('CORTICAL THICKNESS COMPUTATION: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 6. White matter cortical image post-processing"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('WHITE MATTER MAP POST-PROCESSING...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  set (WMCorticalImagePath ${T1Path}/${DataSubdir}/WMCorticalImage/)"<<std::endl;
BatchMakeScript<<"  FileExists(IsWMCorticalImagePath ${WMCorticalImagePath})"<<std::endl;
BatchMakeScript<<"  If (${IsWMCorticalImagePath} == 0)"<<std::endl;
BatchMakeScript<<"    MakeDirectory(${WMCorticalImagePath})"<<std::endl;
BatchMakeScript<<"  EndIf (${IsWMCorticalImagePath})  "<<std::endl;
BatchMakeScript<<"  set (LabelMapPath ${T1Path}/${DataSubdir}/LabelMap/)"<<std::endl;
BatchMakeScript<<"  set (PostProcessedWMMapImage ${WMCorticalImagePath}${T1CaseHead}${ImageSuffix}_WMMap_post_init.nrrd)"<<std::endl;
BatchMakeScript<<"  FileExists(IsPostProcessedWMMapImage ${PostProcessedWMMapImage})"<<std::endl;
BatchMakeScript<<"  If (${IsPostProcessedWMMapImage} == 0)"<<std::endl;
BatchMakeScript<<"    set (WMMap ${LabelMapPath}${T1CaseHead}${ImageSuffix}_WMMap.nrrd)"<<std::endl;
BatchMakeScript<<"    SetAppOption(WMSegPostProcessing.InputImage ${WMMap})"<<std::endl;
BatchMakeScript<<"    SetAppOption(WMSegPostProcessing.OutputImage ${PostProcessedWMMapImage})"<<std::endl;
BatchMakeScript<<"    Run (outputpostprocessing ${WMSegPostProcessing} errorpostprocessing)"<<std::endl;
BatchMakeScript<<"    If (${errorpostprocessing} != '')"<<std::endl;
BatchMakeScript<<"      echo ('WM cortical image post-processing error: ' ${errorpostprocessing})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${errorpostprocessing})"<<std::endl;
BatchMakeScript<<"  Else (${IsPostProcessedWMMapImage})"<<std::endl;
BatchMakeScript<<"    echo ('White matter map post-processing already done!')"<<std::endl;
BatchMakeScript<<"  EndIf (${IsPostProcessedWMMapImage})"<<std::endl;
BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('WHITE MATTER MAP POST-PROCESSING: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 7. Genus zero surface creation and inflation"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('GENUS-ZERO SURFACE CREATION AND INFLATION...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"set (InflationMaxIterationsBeforeFixing2 ${InflationMaxIteration})"<<std::endl;
BatchMakeScript<<"Inc(${InflationMaxIterationsBeforeFixing2} 1)"<<std::endl;
BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  set (InflationPath ${T1Path}/${DataSubdir}/MeshInflation/)"<<std::endl;
BatchMakeScript<<"  FileExists(IsInflationPath ${InflationPath})"<<std::endl;
BatchMakeScript<<"  If (${IsInflationPath} == 0)"<<std::endl;
BatchMakeScript<<"    MakeDirectory(${InflationPath})"<<std::endl;
BatchMakeScript<<"  EndIf (${IsInflationPath})"<<std::endl;
BatchMakeScript<<"  set (GenusZeroPath ${T1Path}/${DataSubdir}/GenusZero/)  "<<std::endl;
BatchMakeScript<<"  FileExists(IsGenusZeroPath ${GenusZeroPath})"<<std::endl;
BatchMakeScript<<"  If (${IsGenusZeroPath} == 0)"<<std::endl;
BatchMakeScript<<"    MakeDirectory(${GenusZeroPath})"<<std::endl;
BatchMakeScript<<"  EndIf (${IsGenusZeroPath})"<<std::endl;
BatchMakeScript<<"  set (WMCorticalImagePath ${T1Path}/${DataSubdir}/WMCorticalImage/)"<<std::endl;
BatchMakeScript<<"  set (PostProcessedWMMapImage ${WMCorticalImagePath}${T1CaseHead}${ImageSuffix}_WMMap_post_init.nrrd)"<<std::endl;
BatchMakeScript<<"  set (NewPostProcessedWMMapImage ${WMCorticalImagePath}${T1CaseHead}${ImageSuffix}_WMMap_post_tmp.nrrd)"<<std::endl;
BatchMakeScript<<"  set (FinalPostProcessedWMMapImage ${WMCorticalImagePath}${T1CaseHead}${ImageSuffix}_WMMap_post.nrrd)"<<std::endl;
BatchMakeScript<<"  set (GenusZeroWMMapImage ${GenusZeroPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus.nrrd)"<<std::endl;
BatchMakeScript<<"  set (GenusZeroWMMapSurface ${GenusZeroPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus.vtk)"<<std::endl;
BatchMakeScript<<"  set (WMMapInflatedSurface ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl.vtk)"<<std::endl;
BatchMakeScript<<"  set (WMMapCurvatureInfo ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl_curv.txt)"<<std::endl;
BatchMakeScript<<"  set (WMMapLogFile ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl.log)"<<std::endl;
BatchMakeScript<<" FileExists(IsWMMapInflatedSurface ${WMMapInflatedSurface})"<<std::endl;
BatchMakeScript<<" If (${IsWMMapInflatedSurface} == 0)"<<std::endl;
BatchMakeScript<<"  set (IsIntermediateInflation 1)"<<std::endl;
BatchMakeScript<<"  sequence (IterationSeq 1 5 1)"<<std::endl;
BatchMakeScript<<"  ForEach(Iteration ${IterationSeq})"<<std::endl;
BatchMakeScript<<"   echo('Iteration ' ${Iteration})"<<std::endl;
BatchMakeScript<<"    # GenusZero WMMap image and surface creation"<<std::endl;
BatchMakeScript<<"    FileExists(IsGenusZeroWMMapImage ${GenusZeroWMMapImage})"<<std::endl;
BatchMakeScript<<"    If (${IsGenusZeroWMMapImage} == 0)"<<std::endl;
BatchMakeScript<<"      echo('Genus-zero image and surface creation...')"<<std::endl;
BatchMakeScript<<"      SetAppOption(GenusZero.inputVolume ${PostProcessedWMMapImage})"<<std::endl;
BatchMakeScript<<"      SetAppOption(GenusZero.outputVolume ${GenusZeroWMMapImage})"<<std::endl;
BatchMakeScript<<"      SetAppOption(GenusZero.vtkOutput ${GenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"      SetAppOption(GenusZero.biggestComponent 1)"<<std::endl;
BatchMakeScript<<"      SetAppOption(GenusZero.connectivityModel.connectivityModel 6)"<<std::endl;
BatchMakeScript<<"      SetAppOption(GenusZero.computeSurface 1)"<<std::endl;
BatchMakeScript<<"      Run (outputGenusZero ${GenusZero} errorGenusZero)"<<std::endl;
BatchMakeScript<<"      If (${errorGenusZero} != '')"<<std::endl;
BatchMakeScript<<"        echo ('GenusZero error: ' ${errorGenusZero})"<<std::endl;
BatchMakeScript<<"        exit()"<<std::endl;
BatchMakeScript<<"      EndIf (${errorGenusZero})"<<std::endl;
BatchMakeScript<<"      Run (output 'ImageMath '${GenusZeroWMMapImage}' -conComp 1 -outfile '${GenusZeroWMMapImage} error)"<<std::endl;
BatchMakeScript<<"    Else (${IsGenusZeroWMMapImage})"<<std::endl;
BatchMakeScript<<"      echo ('Genus zero surface creation already done!')"<<std::endl;
BatchMakeScript<<"    EndIf (${IsGenusZeroWMMapImage})"<<std::endl;
BatchMakeScript<<"   If (${IsIntermediateInflation} == 1 && ${Iteration} < 5)"<<std::endl;

BatchMakeScript<<"    # WM cortical surface inflation"<<std::endl;
//BatchMakeScript<<"    FileExists(IsWMMapInflatedSurface ${WMMapInflatedSurface})"<<std::endl;
//BatchMakeScript<<"    If (${IsWMMapInflatedSurface} == 0)"<<std::endl;
BatchMakeScript<<"      echo('Cortex inflation...')"<<std::endl;
BatchMakeScript<<"      AppendFile(${WMMapLogFile} 'ITERATION: ' ${Iteration}'\\n')"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.InputMesh ${GenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.OutputMesh ${WMMapInflatedSurface})"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.CurvatureFile.CurvatureFile ${WMMapCurvatureInfo})"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.MaxCurvature.MaxCurvature ${InflationMaxCurvature})"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.MaxIterationsBeforeFixing.MaxIterationsBeforeFixing ${InflationMaxIterationsBeforeFixing})"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.MeanCurvature.MeanCurvature ${InflationMeanCurvature})"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.MaxIterations.MaxIterations ${InflationMaxIteration})"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.Smooth.Smooth ${InflationSmoothing})"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.IsIntermediateMesh 1)"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.IntermediateMeshIteration.IntermediateMeshIteration 2)"<<std::endl;
BatchMakeScript<<"      Run (outputmeshinflation ${meshinflation} errormeshinflation)"<<std::endl;
BatchMakeScript<<"      AppendFile(${WMMapLogFile} ${outputmeshinflation}'\\n')"<<std::endl;
BatchMakeScript<<"      If (${errormeshinflation} != '')"<<std::endl;
BatchMakeScript<<"        echo ('Mesh inflation error: ' ${errormeshinflation})"<<std::endl;
BatchMakeScript<<"        exit()"<<std::endl;
BatchMakeScript<<"      EndIf (${errormeshinflation})"<<std::endl;
//BatchMakeScript<<"    EndIf (${IsWMMapInflatedSurface})"<<std::endl;
BatchMakeScript<<"    # WM image fixing"<<std::endl;
BatchMakeScript<<"    set(BadVerticesFile ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl_badVertices.txt)"<<std::endl;
BatchMakeScript<<"    echo('BadVerticesFile: ' ${BadVerticesFile})"<<std::endl;
BatchMakeScript<<"    set(CorrectionGenusZeroWMMapImage ${WMCorticalImagePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_correction.nrrd)"<<std::endl;
BatchMakeScript<<"    FileExists(IsBadVerticesFile ${BadVerticesFile})"<<std::endl;
BatchMakeScript<<"    If (${IsBadVerticesFile} == 0)"<<std::endl;
BatchMakeScript<<"      echo('No bad vertices after inflation!')"<<std::endl;
BatchMakeScript<<"      CopyFile(${PostProcessedWMMapImage} ${FinalPostProcessedWMMapImage})"<<std::endl;
BatchMakeScript<<"      set (IsIntermediateInflation 0)"<<std::endl;
BatchMakeScript<<"    Else (${IsBadVerticesFile})"<<std::endl;
BatchMakeScript<<"      echo ('Bad vertices after inflation found!')"<<std::endl;
BatchMakeScript<<"      echo ('WM Image fixing...')"<<std::endl;
BatchMakeScript<<"      set (WMImageFixingCommand 'WMImageFixing ' ${GenusZeroWMMapImage} ${GenusZeroWMMapSurface} ${BadVerticesFile} ${NewPostProcessedWMMapImage}' --neighborhood --saveCorrIm ' ${CorrectionGenusZeroWMMapImage})"<<std::endl;
BatchMakeScript<<"      Run (output ${WMImageFixingCommand} error)"<<std::endl;
BatchMakeScript<<"      FileExists(IsNewPostProcessedWMMapImage ${NewPostProcessedWMMapImage})"<<std::endl;
BatchMakeScript<<"      If(${IsNewPostProcessedWMMapImage} == 0)"<<std::endl;
BatchMakeScript<<"        echo ('  Nothing to be fixed!')"<<std::endl;

BatchMakeScript<<"        set (IsIntermediateInflation 0)"<<std::endl;

//BatchMakeScript<<"        echo ('Final cortex inflation...')"<<std::endl;
//BatchMakeScript<<"        AppendFile(${WMMapLogFile} 'ITERATION: ' ${Iteration}'\\n')"<<std::endl;
//BatchMakeScript<<"        If(${Iteration} == 1)"<<std::endl;
BatchMakeScript<<"          CopyFile(${PostProcessedWMMapImage} ${FinalPostProcessedWMMapImage})"<<std::endl;
//BatchMakeScript<<"        EndIf(${Iteration})"<<std::endl;
//BatchMakeScript<<"        SetAppOption(meshinflation.InputMesh ${GenusZeroWMMapSurface})"<<std::endl;
//BatchMakeScript<<"        SetAppOption(meshinflation.OutputMesh ${WMMapInflatedSurface})"<<std::endl;
//BatchMakeScript<<"        SetAppOption(meshinflation.CurvatureFile.CurvatureFile ${WMMapCurvatureInfo})"<<std::endl;
//BatchMakeScript<<"        SetAppOption(meshinflation.MaxIterationsBeforeFixing.MaxIterationsBeforeFixing ${InflationMaxIterationsBeforeFixing2})"<<std::endl;
//BatchMakeScript<<"        SetAppOption(meshinflation.MeanCurvature.MeanCurvature ${InflationMeanCurvature})"<<std::endl;
//BatchMakeScript<<"        SetAppOption(meshinflation.MaxIterations.MaxIterations ${InflationMaxIteration})"<<std::endl;
//BatchMakeScript<<"        SetAppOption(meshinflation.Smooth.Smooth ${InflationSmoothing})"<<std::endl;
//BatchMakeScript<<"        SetAppOption(meshinflation.IsIntermediateMesh 1)"<<std::endl;
//BatchMakeScript<<"        SetAppOption(meshinflation.IntermediateMeshIteration.IntermediateMeshIteration 2)"<<std::endl;
//BatchMakeScript<<"        Run (outputmeshinflation ${meshinflation} errormeshinflation)"<<std::endl;
//BatchMakeScript<<"        AppendFile(${WMMapLogFile} ${outputmeshinflation}'\\n')"<<std::endl;
//BatchMakeScript<<"        If (${errormeshinflation} != '')"<<std::endl;
//BatchMakeScript<<"          echo ('Mesh inflation error: ' ${errormeshinflation})"<<std::endl;
//BatchMakeScript<<"          exit()"<<std::endl;
//BatchMakeScript<<"        EndIf (${errormeshinflation})"<<std::endl;
BatchMakeScript<<"      Else(${IsNewPostProcessedWMMapImage})"<<std::endl;
BatchMakeScript<<"        echo ('  WM Image was fixed!')"<<std::endl;
BatchMakeScript<<"        CopyFile(${NewPostProcessedWMMapImage} ${FinalPostProcessedWMMapImage})"<<std::endl;
BatchMakeScript<<"        DeleteFile(${NewPostProcessedWMMapImage})"<<std::endl;
BatchMakeScript<<"        set(PostProcessedWMMapImage ${FinalPostProcessedWMMapImage})"<<std::endl;
BatchMakeScript<<"        echo ('Delete temporary GenusZeroWMMapImage...')"<<std::endl;
BatchMakeScript<<"        DeleteFile(${GenusZeroWMMapImage})"<<std::endl;
BatchMakeScript<<"        echo ('Delete temporary GenusZeroWMMapSurface...')"<<std::endl;
BatchMakeScript<<"        DeleteFile(${GenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"        echo ('Delete temporary WMMapInflatedSurface...')"<<std::endl;
BatchMakeScript<<"        DeleteFile(${WMMapInflatedSurface})"<<std::endl;
BatchMakeScript<<"        echo ('Delete temporary WMMapCurvatureInfo...')"<<std::endl;
BatchMakeScript<<"        DeleteFile(${WMMapCurvatureInfo})"<<std::endl;
BatchMakeScript<<"        echo ('Delete temporary BadVerticesFile...')"<<std::endl;
BatchMakeScript<<"        DeleteFile(${BadVerticesFile})"<<std::endl;
BatchMakeScript<<"        echo ('Delete temporary CorrectionGenusZeroWMMapImage...')"<<std::endl;
BatchMakeScript<<"        DeleteFile(${CorrectionGenusZeroWMMapImage})"<<std::endl;
BatchMakeScript<<"      EndIf(${IsNewPostProcessedWMMapImage})"<<std::endl;
BatchMakeScript<<"    EndIf(${IsBadVerticesFile})"<<std::endl;
BatchMakeScript<<"   EndIf (${IsIntermediateInflation})"<<std::endl;
BatchMakeScript<<"   If(${Iteration} == 5)"<<std::endl;
BatchMakeScript<<"      echo('Final Cortex inflation...')"<<std::endl;
BatchMakeScript<<"      AppendFile(${WMMapLogFile} 'ITERATION: ' ${Iteration}'\\n')"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.InputMesh ${GenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.OutputMesh ${WMMapInflatedSurface})"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.CurvatureFile.CurvatureFile ${WMMapCurvatureInfo})"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.MaxIterationsBeforeFixing.MaxIterationsBeforeFixing ${InflationMaxIterationsBeforeFixing2})"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.MeanCurvature.MeanCurvature ${InflationMeanCurvature})"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.MaxIterations.MaxIterations ${InflationMaxIteration})"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.Smooth.Smooth ${InflationSmoothing})"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.IsIntermediateMesh 1)"<<std::endl;
BatchMakeScript<<"      SetAppOption(meshinflation.IntermediateMeshIteration.IntermediateMeshIteration 2)"<<std::endl;
BatchMakeScript<<"      Run (outputmeshinflation ${meshinflation} errormeshinflation)"<<std::endl;
BatchMakeScript<<"      AppendFile(${WMMapLogFile} ${outputmeshinflation}'\\n')"<<std::endl;
BatchMakeScript<<"      If (${errormeshinflation} != '')"<<std::endl;
BatchMakeScript<<"        echo ('Mesh inflation error: ' ${errormeshinflation})"<<std::endl;
BatchMakeScript<<"        exit()"<<std::endl;
BatchMakeScript<<"      EndIf (${errormeshinflation})"<<std::endl;
BatchMakeScript<<"   EndIf(${Iteration})"<<std::endl;
BatchMakeScript<<"  EndForEach(Iteration)"<<std::endl;
BatchMakeScript<<" Else (${IsWMMapInflatedSurface})"<<std::endl;
BatchMakeScript<<"   echo('WM cortical surface inflation already done!')"<<std::endl;
BatchMakeScript<<" EndIf (${IsWMMapInflatedSurface})"<<std::endl;
BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('GENUS-ZERO SURFACE CREATION AND INFLATION: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 8. Cortical thickness attribute computation"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('CORTICAL THICKNESS ATTRIBUTE COMPUTATION...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  set (MeshAttributePath ${T1Path}/${DataSubdir}/MeshAttribute/)"<<std::endl;
BatchMakeScript<<"  FileExists(IsMeshAttributePath ${MeshAttributePath})"<<std::endl;
BatchMakeScript<<"  If (${IsMeshAttributePath} == 0)"<<std::endl;
BatchMakeScript<<"    MakeDirectory(${MeshAttributePath})"<<std::endl;
BatchMakeScript<<"  EndIf (${IsMeshAttributePath})"<<std::endl;
BatchMakeScript<<"  set (CortThickPath ${T1Path}/${DataSubdir}/CorticalThickness/)"<<std::endl;
BatchMakeScript<<"  set (WMCortThickImage ${CortThickPath}${T1CaseHead}_AvgCortThickOnWMBoundary.nrrd)"<<std::endl;
BatchMakeScript<<"  set (GenusMeshCortThickAttributeFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_CortThick.txt) "<<std::endl;
BatchMakeScript<<"  FileExists(IsGenusMeshCortThickAttributeFile ${GenusMeshCortThickAttributeFile})"<<std::endl;
BatchMakeScript<<"  If (${IsGenusMeshCortThickAttributeFile} == 0)"<<std::endl;
BatchMakeScript<<"    set (GenusZeroPath ${T1Path}/${DataSubdir}/GenusZero/)"<<std::endl;
BatchMakeScript<<"    set (GenusZeroWMMapSurface ${GenusZeroPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus.vtk)"<<std::endl;
BatchMakeScript<<"    set (MeshCortThickCmd 'MeshCortThick -m '${GenusZeroWMMapSurface} ${WMCortThickImage}' -signVector -1 -1 1')"<<std::endl;
BatchMakeScript<<"    Run (output ${MeshCortThickCmd} error)"<<std::endl;
//BatchMakeScript<<"    If (${error} != '')"<<std::endl;
//BatchMakeScript<<"      echo ('MeshCortThick error: ' ${error})"<<std::endl;
//BatchMakeScript<<"      exit()"<<std::endl;
//BatchMakeScript<<"    EndIf (${error})"<<std::endl;
BatchMakeScript<<"    set (TmpFile ${CortThickPath}${T1CaseHead}_AvgCortThickOnWMBoundary_attr.txt)"<<std::endl;
BatchMakeScript<<"    CopyFile(${TmpFile} ${GenusMeshCortThickAttributeFile})"<<std::endl;
BatchMakeScript<<"    DeleteFile(${TmpFile})"<<std::endl;
BatchMakeScript<<"    set (MeshMathCmd 'MeshMath '${GenusZeroWMMapSurface} ${GenusZeroWMMapSurface}' -KWMtoPolyData '${GenusMeshCortThickAttributeFile}' CortThick')"<<std::endl;
BatchMakeScript<<"    Run (output ${MeshMathCmd} error)"<<std::endl;
BatchMakeScript<<"    If (${error} != '')"<<std::endl;
BatchMakeScript<<"      echo ('MeshMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${error})"<<std::endl;
BatchMakeScript<<"    set (InflationPath ${T1Path}/${DataSubdir}/MeshInflation/)"<<std::endl;
BatchMakeScript<<"    set (InflatedGenusZeroWMMapSurface ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl.vtk)"<<std::endl;
BatchMakeScript<<"    set (MeshMathCmd 'MeshMath '${InflatedGenusZeroWMMapSurface} ${InflatedGenusZeroWMMapSurface}' -KWMtoPolyData '${GenusMeshCortThickAttributeFile}' CortThick')"<<std::endl;
BatchMakeScript<<"    Run (output ${MeshMathCmd} error)"<<std::endl;
BatchMakeScript<<"    If (${error} != '')"<<std::endl;
BatchMakeScript<<"      echo ('MeshMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${error})"<<std::endl;
BatchMakeScript<<"  Else (${IsGenusMeshCortThickAttributeFile})"<<std::endl;
BatchMakeScript<<"    echo ('Cortical thickness attibute computation already done!')"<<std::endl;
BatchMakeScript<<"  EndIf (${IsGenusMeshCortThickAttributeFile})"<<std::endl;
BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('CORTICAL THICKNESS ATTRIBUTE COMPUTATION: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 9. Sulcal depth computation"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('SULCAL DEPTH COMPUTATION...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  set (MeshAttributePath ${T1Path}/${DataSubdir}/MeshAttribute/)"<<std::endl;
BatchMakeScript<<"  set (SulcalDepth ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_SulcalDepth.txt)"<<std::endl;
BatchMakeScript<<"  set (SulcalDepthCentered ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_SulcalDepth_centered.txt)"<<std::endl;
BatchMakeScript<<"  FileExists(IsSulcalDepth ${SulcalDepth})"<<std::endl;
BatchMakeScript<<"  If (${IsSulcalDepth} == 0)"<<std::endl;
BatchMakeScript<<"    set (GenusZeroPath ${T1Path}/${DataSubdir}/GenusZero/)"<<std::endl;
BatchMakeScript<<"    set (GenusZeroWMMapSurface ${GenusZeroPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus.vtk)"<<std::endl;
BatchMakeScript<<"    set (MetaGenusZeroWMMapSurface ${GenusZeroPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus.meta)"<<std::endl;
BatchMakeScript<<"    set (InflationPath ${T1Path}/${DataSubdir}/MeshInflation/)"<<std::endl;
BatchMakeScript<<"    set (InflatedGenusZeroWMMapSurface ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl.vtk)"<<std::endl;
BatchMakeScript<<"    set (MetaInflatedGenusZeroWMMapSurface ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl.meta)"<<std::endl;
BatchMakeScript<<"    set (DifferenceVectorFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_DifferenceVector.txt)"<<std::endl;
BatchMakeScript<<"    set (MagDifferenceVectorFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_DifferenceVector.mag.txt)"<<std::endl;
BatchMakeScript<<"    FileExists(IsMetaGenusZeroWMMapSurface ${MetaGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"    If(${IsMetaGenusZeroWMMapSurface} == 0)"<<std::endl;
BatchMakeScript<<"      set (VTK2MetaCommand VTK2Meta ${GenusZeroWMMapSurface} ${MetaGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"      Run (output ${VTK2MetaCommand} error)"<<std::endl;
BatchMakeScript<<"    EndIf(${IsMetaGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"    FileExists(IsMetaInflatedGenusZeroWMMapSurface ${MetaInflatedGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"    If(${IsMetaInflatedGenusZeroWMMapSurface} == 0)"<<std::endl;
BatchMakeScript<<"      set (VTK2MetaCommand VTK2Meta ${InflatedGenusZeroWMMapSurface} ${MetaInflatedGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"      Run (output ${VTK2MetaCommand} error)"<<std::endl;
BatchMakeScript<<"    EndIf(${IsMetaInflatedGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"    set (MeshMathCmd 'MeshMath '${MetaGenusZeroWMMapSurface} ${DifferenceVectorFile}' -subtract '${MetaInflatedGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"    Run (output ${MeshMathCmd} error)"<<std::endl;
BatchMakeScript<<"    If (${error} != '')"<<std::endl;
BatchMakeScript<<"      echo ('MeshMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${error})"<<std::endl;
BatchMakeScript<<"    set (MeshMathCmd 'MeshMath '${MetaGenusZeroWMMapSurface} ${SulcalDepth}' -magNormDir '${DifferenceVectorFile})"<<std::endl;
BatchMakeScript<<"    Run (output ${MeshMathCmd} error)"<<std::endl;
BatchMakeScript<<"    If (${error} != '')"<<std::endl;
BatchMakeScript<<"      echo ('MeshMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${error})"<<std::endl;
BatchMakeScript<<"    CopyFile(${SulcalDepthCentered} ${SulcalDepth})"<<std::endl;
BatchMakeScript<<"    DeleteFile(${SulcalDepthCentered})"<<std::endl;
BatchMakeScript<<"    DeleteFile(${DifferenceVectorFile})"<<std::endl;
BatchMakeScript<<"    DeleteFile(${MagDifferenceVectorFile})"<<std::endl;
BatchMakeScript<<"    DeleteFile(${MetaInflatedGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"    set (MeshMathCmd 'MeshMath '${GenusZeroWMMapSurface} ${GenusZeroWMMapSurface}' -KWMtoPolyData '${SulcalDepth} 'SulcalDepth')"<<std::endl;
BatchMakeScript<<"    Run (output ${MeshMathCmd} error)"<<std::endl;
BatchMakeScript<<"    If (${error} != '')"<<std::endl;
BatchMakeScript<<"      echo ('MeshMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${error})"<<std::endl;
BatchMakeScript<<"    set (MeshMathCmd 'MeshMath '${InflatedGenusZeroWMMapSurface} ${InflatedGenusZeroWMMapSurface}' -KWMtoPolyData '${SulcalDepth} 'SulcalDepth')"<<std::endl;
BatchMakeScript<<"    Run (output ${MeshMathCmd} error)"<<std::endl;
BatchMakeScript<<"    If (${error} != '')"<<std::endl;
BatchMakeScript<<"      echo ('MeshMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${error})"<<std::endl;
BatchMakeScript<<"  Else (${IsSulcalDepth})"<<std::endl;
BatchMakeScript<<"    echo ('Sulcal depth computation already done!')"<<std::endl;
BatchMakeScript<<"  EndIf (${IsSulcalDepth})"<<std::endl;
BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('SULCAL DEPTH COMPUTATION: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 10. Particles initialization"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('PARTICLES INITIALIZATION...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  set (MeshAttributePath ${T1Path}/${DataSubdir}/MeshAttribute/)"<<std::endl;
BatchMakeScript<<"  set (GenusMeshParticleInitAttributeFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_InitParticles.txt)"<<std::endl;
BatchMakeScript<<"  set (GenusMeshParticleInitPointsFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_InitParticles.lpts)"<<std::endl;
BatchMakeScript<<"  set (InflatedMeshParticleInitPointsFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl_InitParticles.lpts)"<<std::endl;
BatchMakeScript<<"  FileExists(IsInflatedMeshParticleInitPointsFile ${InflatedMeshParticleInitPointsFile})"<<std::endl;
BatchMakeScript<<"  If (${IsInflatedMeshParticleInitPointsFile} == 0)"<<std::endl;
BatchMakeScript<<"    set (GenusZeroPath ${T1Path}/${DataSubdir}/GenusZero/)"<<std::endl;
BatchMakeScript<<"    set (GenusZeroWMMapSurface ${GenusZeroPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus.vtk)"<<std::endl;
BatchMakeScript<<"    set (MetaGenusZeroWMMapSurface ${GenusZeroPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus.meta)"<<std::endl;
BatchMakeScript<<"    set (InflationPath ${T1Path}/${DataSubdir}/MeshInflation/)"<<std::endl;
BatchMakeScript<<"    set (InflatedGenusZeroWMMapSurface ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl.vtk)"<<std::endl;
BatchMakeScript<<"    set (ROISegmentationPath ${T1Path}/${DataSubdir}/ROISegmentation/)"<<std::endl;
BatchMakeScript<<"    set (RegisteredParticleInitialization ${ROISegmentationPath}ParticleInitializationRegistered_${T1CaseHead}${ImageSuffix}.nrrd)"<<std::endl;
BatchMakeScript<<"    set (GenusMeshParcellationAttributeFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_Parcellation.txt)"<<std::endl;
BatchMakeScript<<"    FileExists(IsGenusMeshParcellationAttributeFile ${GenusMeshParcellationAttributeFile})"<<std::endl;
BatchMakeScript<<"    If (${IsGenusMeshParcellationAttributeFile} == 0)"<<std::endl;
BatchMakeScript<<"      FileExists(Is{MetaGenusZeroWMMapSurface ${MetaGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"      If(${IsMetaGenusZeroWMMapSurface} == 0)"<<std::endl;
BatchMakeScript<<"        set (VTK2MetaCommand VTK2Meta ${GenusZeroWMMapSurface} ${MetaGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"        Run (output ${VTK2MetaCommand} error)"<<std::endl;
BatchMakeScript<<"      EndIf(${IsMetaGenusZeroWMMapSurface})      "<<std::endl;
BatchMakeScript<<"      set (MeshMathCmd 'MeshMath '${MetaGenusZeroWMMapSurface} ${GenusMeshParcellationAttributeFile}' -extraction '${RegisteredParticleInitialization}' -extractClosest -nn')"<<std::endl;
BatchMakeScript<<"      Run (output ${MeshMathCmd} error)"<<std::endl;
BatchMakeScript<<"      If (${error} != '')"<<std::endl;
BatchMakeScript<<"        echo ('MeshMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"        exit()"<<std::endl;
BatchMakeScript<<"      EndIf (${error})"<<std::endl;
BatchMakeScript<<"      DeleteFile(${MetaGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"    EndIf (${IsGenusMeshParcellationAttributeFile})"<<std::endl;
BatchMakeScript<<"    FileExists(IsGenusMeshParticleInitAttributeFile ${GenusMeshParticleInitAttributeFile})"<<std::endl;
BatchMakeScript<<"    If (${IsGenusMeshParticleInitAttributeFile} == 0)"<<std::endl;
BatchMakeScript<<"      set(ParticleInitializerCmd ParticleInitializer ${InflatedGenusZeroWMMapSurface} ${GenusMeshParticleInitAttributeFile} -meanVertex ${GenusMeshParcellationAttributeFile})"<<std::endl;
BatchMakeScript<<"      Run (output ${ParticleInitializerCmd} error)"<<std::endl;
BatchMakeScript<<"      If (${error} != '')"<<std::endl;
BatchMakeScript<<"        echo ('ParticleInitializer error: ' ${error})"<<std::endl;
BatchMakeScript<<"        exit()"<<std::endl;
BatchMakeScript<<"      EndIf (${error})"<<std::endl;
BatchMakeScript<<"    EndIf (${IsGenusMeshParticleInitAttributeFile})"<<std::endl;
BatchMakeScript<<"    FileExists(IsGenusMeshParticleInitPointsFile ${GenusMeshParticleInitPointsFile})"<<std::endl;
BatchMakeScript<<"    If (${IsGenusMeshParticleInitPointsFile} == 0)"<<std::endl;
BatchMakeScript<<"      set(ParticleInitializerCmd ParticleInitializer ${GenusZeroWMMapSurface} ${GenusMeshParticleInitPointsFile} -savePoints ${GenusMeshParticleInitAttributeFile})"<<std::endl;
BatchMakeScript<<"      Run (output ${ParticleInitializerCmd} error)"<<std::endl;
BatchMakeScript<<"      If (${error} != '')"<<std::endl;
BatchMakeScript<<"        echo ('ParticleInitializer error: ' ${error})"<<std::endl;
BatchMakeScript<<"        exit()"<<std::endl;
BatchMakeScript<<"      EndIf (${error})"<<std::endl;
BatchMakeScript<<"    EndIf (${IsGenusMeshParticleInitPointsFile})"<<std::endl;
BatchMakeScript<<"    set(ParticleInitializerCmd ParticleInitializer ${InflatedGenusZeroWMMapSurface} ${InflatedMeshParticleInitPointsFile} -savePoints ${GenusMeshParticleInitAttributeFile})"<<std::endl;
BatchMakeScript<<"    Run (output ${ParticleInitializerCmd} error)"<<std::endl;
BatchMakeScript<<"    If (${error} != '')"<<std::endl;
BatchMakeScript<<"      echo ('ParticleInitializer error: ' ${error})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${error})"<<std::endl;
BatchMakeScript<<"  Else (${IsInflatedMeshParticleInitPointsFile})"<<std::endl;
BatchMakeScript<<"    echo ('Particles initialization already done!')"<<std::endl;
BatchMakeScript<<"  EndIf (${IsInflatedMeshParticleInitPointsFile})"<<std::endl;
BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('PARTICLES INITIALIZATION: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 11. Surface area computation"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('SURFACE AREA COMPUTATION...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  set (MeshAttributePath ${T1Path}/${DataSubdir}/MeshAttribute/)"<<std::endl;
BatchMakeScript<<"  set (SurfaceArea ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_SurfaceArea.txt)"<<std::endl;
BatchMakeScript<<"  FileExists(IsSurfaceArea ${SurfaceArea})"<<std::endl;
BatchMakeScript<<"  If (${IsSurfaceArea} == 0)"<<std::endl;
BatchMakeScript<<"    set (InflationPath ${T1Path}/${DataSubdir}/MeshInflation/)"<<std::endl;
BatchMakeScript<<"    set (SmoothedGenusZeroWMMapSurface ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl_It2.vtk)"<<std::endl;
BatchMakeScript<<"    set (MetaSmoothedGenusZeroWMMapSurface ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl_It2.meta)"<<std::endl;
BatchMakeScript<<"    set (VTK2MetaCommand VTK2Meta ${SmoothedGenusZeroWMMapSurface} ${MetaSmoothedGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"    Run (output ${VTK2MetaCommand} error)"<<std::endl;
BatchMakeScript<<"    set (MeshMathCmd 'MeshMath '${MetaSmoothedGenusZeroWMMapSurface} ${SurfaceArea}' -surfaceArea')"<<std::endl;
BatchMakeScript<<"    Run (output ${MeshMathCmd} error)"<<std::endl;
BatchMakeScript<<"    If (${error} != '')"<<std::endl;
BatchMakeScript<<"      echo ('MeshMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${error})"<<std::endl;
BatchMakeScript<<"    DeleteFile(${MetaSmoothedGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"  Else (${IsSurfaceArea})"<<std::endl;
BatchMakeScript<<"    echo ('Surface area computation already done!')"<<std::endl;
BatchMakeScript<<"  EndIf (${IsSurfaceArea})"<<std::endl;
BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('SURFACE AREA COMPUTATION: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 11_bis. Lobar surface area computation"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('LOBAR SURFACE AREA COMPUTATION...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  set (MeshAttributePath ${T1Path}/${DataSubdir}/MeshAttribute/)"<<std::endl;
BatchMakeScript<<"  set (SurfaceAreaPath ${T1Path}/${DataSubdir}/SurfaceArea/)"<<std::endl;
BatchMakeScript<<"  FileExists(IsSurfaceAreaPath ${SurfaceAreaPath})"<<std::endl;
BatchMakeScript<<"  If (${IsSurfaceAreaPath} == 0)"<<std::endl;
BatchMakeScript<<"    MakeDirectory(${SurfaceAreaPath})"<<std::endl;
BatchMakeScript<<"  EndIf (${IsSurfaceAreaPath})  "<<std::endl;
BatchMakeScript<<"  set (LobarSurfaceArea ${SurfaceAreaPath}${T1CaseHead}${ImageSuffix}_LobarSurfaceArea.csv)"<<std::endl;
BatchMakeScript<<"  FileExists(IsLobarSurfaceArea ${LobarSurfaceArea})"<<std::endl;
BatchMakeScript<<"  If (${IsLobarSurfaceArea} == 0)"<<std::endl;
BatchMakeScript<<"    set (InflationPath ${T1Path}/${DataSubdir}/MeshInflation/)"<<std::endl;
BatchMakeScript<<"    set (SmoothedGenusZeroWMMapSurface ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl_It2.vtk)"<<std::endl;
BatchMakeScript<<"    set (MetaSmoothedGenusZeroWMMapSurface ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl_It2.meta)"<<std::endl;
BatchMakeScript<<"    set (GenusMeshParcellationAttributeFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_Parcellation.txt)"<<std::endl;
BatchMakeScript<<"    set (VTK2MetaCommand VTK2Meta ${SmoothedGenusZeroWMMapSurface} ${MetaSmoothedGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"    Run (output ${VTK2MetaCommand} error)"<<std::endl;
BatchMakeScript<<"    set (MeshMathCmd 'MeshMath '${MetaSmoothedGenusZeroWMMapSurface} ${LobarSurfaceArea}' -lobarSurfaceArea '${GenusMeshParcellationAttributeFile})"<<std::endl;
BatchMakeScript<<"    Run (output ${MeshMathCmd} error)"<<std::endl;
BatchMakeScript<<"    If (${error} != '')"<<std::endl;
BatchMakeScript<<"      echo ('MeshMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${error})"<<std::endl;
BatchMakeScript<<"    DeleteFile(${MetaSmoothedGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"  Else (${IsLobarSurfaceArea})"<<std::endl;
BatchMakeScript<<"    echo ('Surface area computation already done!')"<<std::endl;
BatchMakeScript<<"  EndIf (${IsLobarSurfaceArea})"<<std::endl;
BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('LOBAR SURFACE AREA COMPUTATION: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 12. Spatial location attribute computation"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('SPATIAL LOCATION ATTRIBUTE COMPUTATION...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  set (MeshAttributePath ${T1Path}/${DataSubdir}/MeshAttribute/)"<<std::endl;
BatchMakeScript<<"  set (InflationPath ${T1Path}/${DataSubdir}/MeshInflation/)"<<std::endl;
BatchMakeScript<<"  set (InflatedGenusZeroWMMapSurface ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl.vtk)"<<std::endl;
BatchMakeScript<<"  set (XCoordinatesAttributeFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl_Coordinates_X.txt)"<<std::endl;
BatchMakeScript<<"  FileExists(IsXCoordinatesAttributeFile ${XCoordinatesAttributeFile})"<<std::endl;
BatchMakeScript<<"  If (${IsXCoordinatesAttributeFile} == 0)"<<std::endl;
BatchMakeScript<<"    set (YCoordinatesAttributeFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl_Coordinates_Y.txt)"<<std::endl;
BatchMakeScript<<"    set (ZCoordinatesAttributeFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl_Coordinates_Z.txt)"<<std::endl;
BatchMakeScript<<"    set (MeshMathCmd 'MeshMath '${InflatedGenusZeroWMMapSurface} ${XCoordinatesAttributeFile}' -extractVertices '${YCoordinatesAttributeFile} ${ZCoordinatesAttributeFile})"<<std::endl;
BatchMakeScript<<"    Run (output ${MeshMathCmd} error)"<<std::endl;
BatchMakeScript<<"    If (${error} != '')"<<std::endl;
BatchMakeScript<<"      echo ('MeshMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${error})"<<std::endl;
BatchMakeScript<<"  Else (${IsXCoordinatesAttributeFile})"<<std::endl;
BatchMakeScript<<"    echo ('Spatial location computation already done!')"<<std::endl;
BatchMakeScript<<"  EndIf (${IsXCoordinatesAttributeFile})"<<std::endl;
BatchMakeScript<<"EndForEach(T1Case)"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('SPATIAL LOCATION ATTRIBUTE COMPUTATION: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 13. Correspondence preprocessing"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('CORRESPONDENCE PREPROCESSING...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"set (CorrespondencePath ${OutputDir}Correspondence/)"<<std::endl;
BatchMakeScript<<"FileExists(IsCorrespondencePath ${CorrespondencePath})"<<std::endl;
BatchMakeScript<<"If (${IsCorrespondencePath} == 0)"<<std::endl;
BatchMakeScript<<"  MakeDirectory(${CorrespondencePath})"<<std::endl;
BatchMakeScript<<"EndIf (${IsCorrespondencePath})"<<std::endl;
BatchMakeScript<<"set (CorrespondencePreprocessingPath ${CorrespondencePath}CorrespondencePreProcessing/)"<<std::endl;
BatchMakeScript<<"FileExists(IsCorrespondencePreprocessingPath ${CorrespondencePreprocessingPath})"<<std::endl;
BatchMakeScript<<"If (${IsCorrespondencePreprocessingPath} == 0)"<<std::endl;
BatchMakeScript<<"  MakeDirectory(${CorrespondencePreprocessingPath})"<<std::endl;
BatchMakeScript<<"EndIf (${IsCorrespondencePreprocessingPath})"<<std::endl;

BatchMakeScript<<"set (CoordinateList X Y Z)"<<std::endl;
BatchMakeScript<<"ForEach (Coordinate ${CoordinateList})"<<std::endl;
BatchMakeScript<<"  set (CoordinatesVarianceFile ${CorrespondencePreprocessingPath}Variance_${Coordinate}.txt)"<<std::endl;
BatchMakeScript<<"  FileExists(IsCoordinatesVarianceFile ${CoordinatesVarianceFile})"<<std::endl;
BatchMakeScript<<"  If (${IsCoordinatesVarianceFile} == 0)"<<std::endl;
BatchMakeScript<<"    set (CaseNumber 0)"<<std::endl;
BatchMakeScript<<"    set (MeshMathCmd '')"<<std::endl;
BatchMakeScript<<"    ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"      GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"      GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"      set (MeshAttributePath ${T1Path}/${DataSubdir}/MeshAttribute/)"<<std::endl;
BatchMakeScript<<"      set (CoordinatesAttributeFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl_Coordinates_${Coordinate}.txt)"<<std::endl;
BatchMakeScript<<"      If (${CaseNumber} == 0)"<<std::endl;
BatchMakeScript<<"        set (MeshMathCmd 'MeshMath '${CoordinatesAttributeFile} ${CoordinatesVarianceFile}' -variance ')"<<std::endl;
BatchMakeScript<<"      Else (${CaseNumber})"<<std::endl;
BatchMakeScript<<"        set (MeshMathCmd ${MeshMathCmd} ${CoordinatesAttributeFile})"<<std::endl;
BatchMakeScript<<"      EndIf (${CaseNumber})"<<std::endl;
BatchMakeScript<<"      Inc (${CaseNumber} 1)"<<std::endl;
BatchMakeScript<<"      Int (${CaseNumber})"<<std::endl;
BatchMakeScript<<"    EndForEach (T1Case)"<<std::endl;
BatchMakeScript<<"    Run (output ${MeshMathCmd} error)"<<std::endl;
BatchMakeScript<<"    If (${error} != '')"<<std::endl;
BatchMakeScript<<"      echo ('MeshMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${error})"<<std::endl;
BatchMakeScript<<"  Else (${IsCoordinatesVarianceFile})"<<std::endl;
BatchMakeScript<<"    echo ('Spatial location '${Coordinate}' variance computation already done!')"<<std::endl;
BatchMakeScript<<"  EndIf (${IsCoordinatesVarianceFile})"<<std::endl;
BatchMakeScript<<"EndForEach (Coordinate)"<<std::endl;

BatchMakeScript<<"set (SulcalDepthVarianceFile ${CorrespondencePreprocessingPath}Variance_SulcalDepth.txt)"<<std::endl;
BatchMakeScript<<"FileExists(IsSulcalDepthVarianceFile ${SulcalDepthVarianceFile})"<<std::endl;
BatchMakeScript<<"If (${IsSulcalDepthVarianceFile} == 0)"<<std::endl;
BatchMakeScript<<"  set (CaseNumber 0)"<<std::endl;
BatchMakeScript<<"  set (MeshMathCmd '')"<<std::endl;
BatchMakeScript<<"  ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"    GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"    GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"    set (MeshAttributePath ${T1Path}/${DataSubdir}/MeshAttribute/)"<<std::endl;
BatchMakeScript<<"    set (SulcalDepthAttributeFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_SulcalDepth.txt)"<<std::endl;
BatchMakeScript<<"    If (${CaseNumber} == 0)"<<std::endl;
BatchMakeScript<<"      set (MeshMathCmd 'MeshMath '${SulcalDepthAttributeFile} ${SulcalDepthVarianceFile}' -variance ')"<<std::endl;
BatchMakeScript<<"    Else (${CaseNumber})"<<std::endl;
BatchMakeScript<<"      set (MeshMathCmd ${MeshMathCmd} ${SulcalDepthAttributeFile})"<<std::endl;
BatchMakeScript<<"    EndIf (${CaseNumber})"<<std::endl;
BatchMakeScript<<"    Inc (${CaseNumber} 1)"<<std::endl;
BatchMakeScript<<"    Int (${CaseNumber})"<<std::endl;
BatchMakeScript<<"  EndForEach (T1Case)"<<std::endl;
BatchMakeScript<<"  Run (output ${MeshMathCmd} error)"<<std::endl;
BatchMakeScript<<"  If (${error} != '')"<<std::endl;
BatchMakeScript<<"    echo ('MeshMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"    exit()"<<std::endl;
BatchMakeScript<<"  EndIf (${error})"<<std::endl;
BatchMakeScript<<"Else (${IsSulcalDepthVarianceFile})"<<std::endl;
BatchMakeScript<<"  echo ('Sulcal depth variance computation already done!')"<<std::endl;
BatchMakeScript<<"EndIf (${IsSulcalDepthVarianceFile})"<<std::endl;
BatchMakeScript<<"set (VarianceSulcalDepth ${output})"<<std::endl;

BatchMakeScript<<"set (PreprocessingFile ${CorrespondencePath}GAMBIT_CorrespondencePreprocessing.params)"<<std::endl;
BatchMakeScript<<"FileExists(IsPreprocessingFile ${PreprocessingFile})"<<std::endl;
BatchMakeScript<<"If (${IsPreprocessingFile} == 0)"<<std::endl;
BatchMakeScript<<"  WriteFile(${PreprocessingFile} 'NUMBER_OF_SHAPES='${NbData}'\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${PreprocessingFile} 'NUMBER_OF_ATTRIBUTES=4\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${PreprocessingFile} 'TEMPLATE_INDEX=0\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${PreprocessingFile} 'WEIGHTS=1 1 1 1\\n')"<<std::endl;
BatchMakeScript<<"  ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"    GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"    GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"    set (InflationPath ${T1Path}/${DataSubdir}/MeshInflation/)"<<std::endl;
BatchMakeScript<<"    set (InflatedGenusZeroWMMapSurface ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl.vtk)"<<std::endl;
BatchMakeScript<<"    set (MeshAttributePath ${T1Path}/${DataSubdir}/MeshAttribute/)"<<std::endl;
BatchMakeScript<<"    set (SulcalDepthAttributeFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_SulcalDepth.txt)"<<std::endl;
BatchMakeScript<<"    set (XCoordinatesAttributeFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl_Coordinates_X.txt)"<<std::endl;
BatchMakeScript<<"    set (YCoordinatesAttributeFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl_Coordinates_Y.txt)"<<std::endl;
BatchMakeScript<<"    set (ZCoordinatesAttributeFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl_Coordinates_Z.txt)"<<std::endl;
BatchMakeScript<<"    AppendFile(${PreprocessingFile} ${InflatedGenusZeroWMMapSurface}' '${SulcalDepthAttributeFile}' '${XCoordinatesAttributeFile}' '${YCoordinatesAttributeFile}' '${ZCoordinatesAttributeFile}'\\n')"<<std::endl;
BatchMakeScript<<"  EndForEach(T1Case)"<<std::endl;
BatchMakeScript<<"Else (${IsPreprocessingFile})"<<std::endl;
BatchMakeScript<<"  echo('Correspondence preprocessing file already exists!')"<<std::endl;
BatchMakeScript<<"Endif (${IsPreprocessingFile})"<<std::endl;
BatchMakeScript<<"GetParam (FirstT1Case ${OrigT1CasesList} 0)"<<std::endl;
BatchMakeScript<<"GetFilename (FirstT1CaseHead ${FirstT1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"set(FirstDistanceMap ${CorrespondencePreprocessingPath}${FirstT1CaseHead}_WMMap_infl_DistanceMap.mha)"<<std::endl;
BatchMakeScript<<"FileExists(IsFirstDistanceMap ${FirstDistanceMap})"<<std::endl;
BatchMakeScript<<"If (${IsFirstDistanceMap} == 0)"<<std::endl;
BatchMakeScript<<"  SetAppOption(correspondencepreprocessing.parameterFileName.parameterFileName ${PreprocessingFile})"<<std::endl;
BatchMakeScript<<"  SetAppOption(correspondencepreprocessing.outputDirectory.outputDirectory ${CorrespondencePreprocessingPath})"<<std::endl;
BatchMakeScript<<"  SetAppOption(correspondencepreprocessing.smoothing.smoothing ${CorrespondencePreprocessingSmoothing})"<<std::endl;
BatchMakeScript<<"  Run (outputcorrespondencepreprocessing ${correspondencepreprocessing} errorcorrespondencepreprocessing)"<<std::endl;
BatchMakeScript<<"  If (${errorcorrespondencepreprocessing} != '')"<<std::endl;
BatchMakeScript<<"    echo ('Correspondence preprocessing error: ' ${errorcorrespondencepreprocessing})"<<std::endl;
BatchMakeScript<<"    exit()"<<std::endl;
BatchMakeScript<<"  EndIf (${errorcorrespondencepreprocessing})"<<std::endl;
BatchMakeScript<<"  set(FileNumber 0)"<<std::endl;
BatchMakeScript<<"  ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"    GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"    If (${FileNumber} < 10)"<<std::endl;
BatchMakeScript<<"      set (DistanceMap ${CorrespondencePreprocessingPath}subject_00${FileNumber}_DistanceMap.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap0 ${CorrespondencePreprocessingPath}subject_00${FileNumber}_Attribute_00.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap1 ${CorrespondencePreprocessingPath}subject_00${FileNumber}_Attribute_01.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap2 ${CorrespondencePreprocessingPath}subject_00${FileNumber}_Attribute_02.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap3 ${CorrespondencePreprocessingPath}subject_00${FileNumber}_Attribute_03.mha)"<<std::endl;
BatchMakeScript<<"    EndIf (${FileNumber})"<<std::endl;
BatchMakeScript<<"    If (${FileNumber} >= 10 && ${FileNumber} < 100)"<<std::endl;
BatchMakeScript<<"      set (DistanceMap ${CorrespondencePreprocessingPath}subject_0${FileNumber}_DistanceMap.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap0 ${CorrespondencePreprocessingPath}subject_0${FileNumber}_Attribute_00.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap1 ${CorrespondencePreprocessingPath}subject_0${FileNumber}_Attribute_01.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap2 ${CorrespondencePreprocessingPath}subject_0${FileNumber}_Attribute_02.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap3 ${CorrespondencePreprocessingPath}subject_0${FileNumber}_Attribute_03.mha)"<<std::endl;
BatchMakeScript<<"    EndIf (${FileNumber})"<<std::endl;
BatchMakeScript<<"    If (${FileNumber} >= 100)"<<std::endl;
BatchMakeScript<<"      set (DistanceMap ${CorrespondencePreprocessingPath}subject_${FileNumber}_DistanceMap.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap0 ${CorrespondencePreprocessingPath}subject_${FileNumber}_Attribute_00.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap1 ${CorrespondencePreprocessingPath}subject_${FileNumber}_Attribute_01.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap2 ${CorrespondencePreprocessingPath}subject_${FileNumber}_Attribute_02.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap3 ${CorrespondencePreprocessingPath}subject_${FileNumber}_Attribute_03.mha)"<<std::endl;
BatchMakeScript<<"    EndIf (${FileNumber})"<<std::endl;
BatchMakeScript<<"    set (NewDistanceMap ${CorrespondencePreprocessingPath}${T1CaseHead}_WMMap_infl_DistanceMap.mha)"<<std::endl;
BatchMakeScript<<"    set (NewAttributeMap0 ${CorrespondencePreprocessingPath}${T1CaseHead}_WMMap_infl_Attribute_00.mha)"<<std::endl;
BatchMakeScript<<"    set (NewAttributeMap1 ${CorrespondencePreprocessingPath}${T1CaseHead}_WMMap_infl_Attribute_01.mha)"<<std::endl;
BatchMakeScript<<"    set (NewAttributeMap2 ${CorrespondencePreprocessingPath}${T1CaseHead}_WMMap_infl_Attribute_02.mha)"<<std::endl;
BatchMakeScript<<"    set (NewAttributeMap3 ${CorrespondencePreprocessingPath}${T1CaseHead}_WMMap_infl_Attribute_03.mha)"<<std::endl;
BatchMakeScript<<"    CopyFile(${DistanceMap} ${NewDistanceMap})"<<std::endl;
BatchMakeScript<<"    DeleteFile(${DistanceMap})"<<std::endl;
BatchMakeScript<<"    CopyFile(${AttributeMap0} ${NewAttributeMap0})"<<std::endl;
BatchMakeScript<<"    DeleteFile(${AttributeMap0})"<<std::endl;
BatchMakeScript<<"    CopyFile(${AttributeMap1} ${NewAttributeMap1})"<<std::endl;
BatchMakeScript<<"    DeleteFile(${AttributeMap1})"<<std::endl;
BatchMakeScript<<"    CopyFile(${AttributeMap2} ${NewAttributeMap2})"<<std::endl;
BatchMakeScript<<"    DeleteFile(${AttributeMap2})"<<std::endl;
BatchMakeScript<<"    CopyFile(${AttributeMap3} ${NewAttributeMap3})"<<std::endl;
BatchMakeScript<<"    DeleteFile(${AttributeMap3})"<<std::endl;
BatchMakeScript<<"    Inc(${FileNumber} 1)"<<std::endl;
BatchMakeScript<<"    Int(${FileNumber})"<<std::endl;
BatchMakeScript<<"  EndForEach (T1Case)"<<std::endl;
BatchMakeScript<<"Else (${IsFirstDistanceMap})"<<std::endl;
BatchMakeScript<<"  echo('Correspondence preprocessing already done!')"<<std::endl;
BatchMakeScript<<"Endif (${IsFirstDistanceMap})"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('CORRESPONDENCE PREPROCESSING: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"#---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 14. Particle correspondence "<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('PARTICLE CORRESPONDENCE...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"set (CorrespondenceExecPath ${CorrespondencePath}Correspondence/)"<<std::endl;
BatchMakeScript<<"FileExists(IsCorrespondenceExecPath ${CorrespondenceExecPath})"<<std::endl;
BatchMakeScript<<"If (${IsCorrespondenceExecPath} == 0)"<<std::endl;
BatchMakeScript<<"  MakeDirectory(${CorrespondenceExecPath})"<<std::endl;
BatchMakeScript<<"EndIf (${IsCorrespondenceExecPath})"<<std::endl;
BatchMakeScript<<"GetParam (FirstT1Case ${OrigT1CasesList} 0)"<<std::endl;
BatchMakeScript<<"GetFilename (FirstT1CaseHead ${FirstT1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"set(FirstParticleFile ${CorrespondenceExecPath}${FirstT1CaseHead}_WMMap_infl_particles.lpts)"<<std::endl;
BatchMakeScript<<"FileExists(IsFirstParticleFile ${FirstParticleFile})"<<std::endl;
BatchMakeScript<<"If (${IsFirstParticleFile} == 0)"<<std::endl;
BatchMakeScript<<"  set (CorrespondenceLogFile ${CorrespondencePath}GAMBIT_Correspondence.log)"<<std::endl;
BatchMakeScript<<"  set (CorrespondenceFile ${CorrespondencePath}GAMBIT_Correspondence.params)"<<std::endl;
BatchMakeScript<<"  FileExists(IsCorrespondenceFile ${CorrespondenceFile})"<<std::endl;
BatchMakeScript<<"  If (${IsCorrespondenceFile} == 0)"<<std::endl;
BatchMakeScript<<"    set (SulcalDepthVarianceFile ${CorrespondencePreprocessingPath}Variance_SulcalDepth.txt)"<<std::endl;
BatchMakeScript<<"    Run (output 'more '${SulcalDepthVarianceFile} error)"<<std::endl;
BatchMakeScript<<"    set (VarianceSulcalDepth ${output})"<<std::endl;
BatchMakeScript<<"    set (XCoordinatesVarianceFile ${CorrespondencePreprocessingPath}Variance_X.txt)"<<std::endl;
BatchMakeScript<<"    set (YCoordinatesVarianceFile ${CorrespondencePreprocessingPath}Variance_Y.txt)"<<std::endl;
BatchMakeScript<<"    set (ZCoordinatesVarianceFile ${CorrespondencePreprocessingPath}Variance_Z.txt)"<<std::endl;
BatchMakeScript<<"    Run (output 'more '${XCoordinatesVarianceFile} error)"<<std::endl;
BatchMakeScript<<"    set (VarianceX ${output})"<<std::endl;
BatchMakeScript<<"    Run (output 'more '${YCoordinatesVarianceFile} error)"<<std::endl;
BatchMakeScript<<"    set (VarianceY ${output})"<<std::endl;
BatchMakeScript<<"    Run (output 'more '${ZCoordinatesVarianceFile} error)"<<std::endl;
BatchMakeScript<<"    set (VarianceZ ${output})"<<std::endl;
BatchMakeScript<<"    Math (MeanCoordinateVariance ${VarianceX} + ${VarianceY})"<<std::endl;
BatchMakeScript<<"    Math (MeanCoordinateVariance ${MeanCoordinateVariance} + ${VarianceZ})"<<std::endl;
BatchMakeScript<<"    Math (MeanCoordinateVariance ${MeanCoordinateVariance} / 3)"<<std::endl;
BatchMakeScript<<"    Math (WeightCoordinates 1 / ${MeanCoordinateVariance})"<<std::endl;
BatchMakeScript<<"    Math (WeightSulcalDepth 1 / ${VarianceSulcalDepth})"<<std::endl;
BatchMakeScript<<"    set(FileNumber 0)"<<std::endl;
BatchMakeScript<<"    ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"      GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"      GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"      set (MeshAttributePath ${T1Path}/${DataSubdir}/MeshAttribute/)"<<std::endl;
BatchMakeScript<<"      set (InflatedMeshParticleInitPointsFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl_InitParticles.lpts)"<<std::endl;
BatchMakeScript<<"      set (DistanceMap ${CorrespondencePreprocessingPath}${T1CaseHead}_WMMap_infl_DistanceMap.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap0 ${CorrespondencePreprocessingPath}${T1CaseHead}_WMMap_infl_Attribute_00.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap1 ${CorrespondencePreprocessingPath}${T1CaseHead}_WMMap_infl_Attribute_01.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap2 ${CorrespondencePreprocessingPath}${T1CaseHead}_WMMap_infl_Attribute_02.mha)"<<std::endl;
BatchMakeScript<<"      set (AttributeMap3 ${CorrespondencePreprocessingPath}${T1CaseHead}_WMMap_infl_Attribute_03.mha)"<<std::endl;
BatchMakeScript<<"      If(${FileNumber} == 0)"<<std::endl;
BatchMakeScript<<"        set(InputList '\"'${DistanceMap}'\"')"<<std::endl;
BatchMakeScript<<"        set(PointList '\"'${InflatedMeshParticleInitPointsFile}'\"')"<<std::endl;
BatchMakeScript<<"        set(AttributeList '\"'${AttributeMap0}'\" \"'${AttributeMap1}'\" \"'${AttributeMap2}'\" \"'${AttributeMap3}'\"')"<<std::endl;
BatchMakeScript<<"      Else (${FileNumber})"<<std::endl;
BatchMakeScript<<"        set(InputList ${InputList} '\"'${DistanceMap}'\"')"<<std::endl;
BatchMakeScript<<"        set(PointList ${PointList} '\"'${InflatedMeshParticleInitPointsFile}'\"')"<<std::endl;
BatchMakeScript<<"        set(AttributeList ${AttributeList} '\"'${AttributeMap0}'\" \"'${AttributeMap1}'\" \"'${AttributeMap2}'\" \"'${AttributeMap3}'\"')"<<std::endl;
BatchMakeScript<<"      EndIf(${FileNumber})"<<std::endl;
BatchMakeScript<<"      Inc(${FileNumber} 1)"<<std::endl;
BatchMakeScript<<"      Int(${FileNumber})"<<std::endl;
BatchMakeScript<<"    EndForEach(T1Case)"<<std::endl;
BatchMakeScript<<"    WriteFile(${CorrespondenceFile} '(inputs '${InputList}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceFile} '(point_files '${PointList}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceFile} '(attributes_per_domain 4)\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceFile} '(attribute_files '${AttributeList}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceFile} '(attribute_scales '${WeightSulcalDepth}' '${WeightCoordinates}' '${WeightCoordinates}' '${WeightCoordinates}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceFile} '(number_of_particles '${CorrespondenceNbParticles}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceFile} '(iterations_per_split '${CorrespondenceIterationsPerSplit}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceFile} '(starting_regularization '${CorrespondenceStartingRegularization}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceFile} '(ending_regularization '${CorrespondenceEndingRegularization}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceFile} '(optimization_iterations '${CorrespondenceOptimizationIterations}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceFile} '(checkpointing_interval '${CorrespondenceCheckPointingIntervals}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceFile} '(output_points_prefix \"'${CorrespondenceExecPath}'Particles\")\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceFile} '(relative_weighting '${CorrespondenceRelativeWeighting}')\\n')"<<std::endl;
BatchMakeScript<<"    If(${CorrespondenceProcrustesOff} == 0)"<<std::endl;
BatchMakeScript<<"      AppendFile(${CorrespondenceFile} '(procrustes_interval '${CorrespondenceProcrustesInterval}')\\n')"<<std::endl;
BatchMakeScript<<"      If(${CorrespondenceProcrustesScalingOff} == 0)"<<std::endl;
BatchMakeScript<<"        AppendFile(${CorrespondenceFile} '(procrustes_scaling 1)\\n')"<<std::endl;
BatchMakeScript<<"      EndIf(${CorrespondenceProcrustesScalingOff})"<<std::endl;
BatchMakeScript<<"    EndIf(${CorrespondenceProcrustesOff})  "<<std::endl;
BatchMakeScript<<"  Else (${IsCorrespondenceFile})"<<std::endl;
BatchMakeScript<<"    echo('Particle correspondence file already exists!')"<<std::endl;
BatchMakeScript<<"  Endif (${IsCorrespondenceFile})"<<std::endl;
BatchMakeScript<<"  Run (outputcorrespondence 'ShapeWorksRun '${CorrespondenceFile} errorcorrespondence)"<<std::endl;
BatchMakeScript<<"  WriteFile(${CorrespondenceLogFile} ${outputcorrespondence})"<<std::endl;
BatchMakeScript<<"  If (${errorcorrespondence} != '')"<<std::endl;
BatchMakeScript<<"    echo ('Particle correspondence error: ' ${errorcorrespondence})"<<std::endl;
BatchMakeScript<<"    exit()"<<std::endl;
BatchMakeScript<<"  EndIf (${errorcorrespondence})"<<std::endl;
BatchMakeScript<<"  set(TotalCases 0)"<<std::endl;
BatchMakeScript<<"  ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"    Inc(${TotalCases} 1)"<<std::endl;
BatchMakeScript<<"    Int(${TotalCases})"<<std::endl;
BatchMakeScript<<"  EndForEach (T1Case)  "<<std::endl;
BatchMakeScript<<"  set(FileNumber 0)"<<std::endl;
BatchMakeScript<<"  ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"    GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"    If (${TotalCases} < 10)"<<std::endl;
BatchMakeScript<<"      set (ParticleFile ${CorrespondenceExecPath}Particles.${FileNumber}.lpts)"<<std::endl;
BatchMakeScript<<"    EndIf (${TotalCases})"<<std::endl;
BatchMakeScript<<"    If (${TotalCases} >= 10 && ${TotalCases} < 100)"<<std::endl;
BatchMakeScript<<"      If (${FileNumber} < 10)"<<std::endl;
BatchMakeScript<<"        set (ParticleFile ${CorrespondenceExecPath}Particles.0${FileNumber}.lpts)"<<std::endl;
BatchMakeScript<<"      Else (${FileNumber})"<<std::endl;
BatchMakeScript<<"        set (ParticleFile ${CorrespondenceExecPath}Particles.${FileNumber}.lpts)"<<std::endl;
BatchMakeScript<<"      EndIf (${FileNumber})"<<std::endl;
BatchMakeScript<<"    EndIf (${TotalCases})"<<std::endl;
BatchMakeScript<<"    If (${TotalCases} >= 100 && ${TotalCases} < 1000)"<<std::endl;
BatchMakeScript<<"      If (${FileNumber} < 10)"<<std::endl;
BatchMakeScript<<"        set (ParticleFile ${CorrespondenceExecPath}Particles.00${FileNumber}.lpts)"<<std::endl;
BatchMakeScript<<"      EndIf (${FileNumber})"<<std::endl;
BatchMakeScript<<"      If (${FileNumber} >= 10 && ${FileNumber} < 100)"<<std::endl;
BatchMakeScript<<"        set (ParticleFile ${CorrespondenceExecPath}Particles.0${FileNumber}.lpts)"<<std::endl;
BatchMakeScript<<"      EndIf (${FileNumber})"<<std::endl;
BatchMakeScript<<"      If (${FileNumber} >= 100)"<<std::endl;
BatchMakeScript<<"        set (ParticleFile ${CorrespondenceExecPath}Particles.${FileNumber}.lpts)"<<std::endl;
BatchMakeScript<<"      EndIf (${FileNumber})"<<std::endl;
BatchMakeScript<<"    EndIf (${TotalCases})"<<std::endl;
BatchMakeScript<<"    set (NewParticleFile ${CorrespondenceExecPath}${T1CaseHead}_WMMap_infl_particles.lpts)"<<std::endl;
BatchMakeScript<<"    CopyFile(${ParticleFile} ${NewParticleFile})"<<std::endl;
//BatchMakeScript<<"    DeleteFile(${ParticleFile})"<<std::endl;
BatchMakeScript<<"    Inc(${FileNumber} 1)"<<std::endl;
BatchMakeScript<<"    Int(${FileNumber})"<<std::endl;
BatchMakeScript<<"  EndForEach (T1Case)"<<std::endl;
BatchMakeScript<<"Else (${IsFirstParticleFile})"<<std::endl;
BatchMakeScript<<"  echo('Particle Correspondence already done!')"<<std::endl;
BatchMakeScript<<"EndIf (${IsFirstParticleFile})"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('PARTICLE CORRESPONDENCE: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"#---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 14_bis. Particle correspondence Quality Control"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('PARTICLE CORRESPONDENCE QC...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"set (CorrespondenceQCFile ${CorrespondencePath}GAMBIT_Correspondence_QC.params)"<<std::endl;
BatchMakeScript<<"FileExists(IsCorrespondenceQCFile ${CorrespondenceQCFile})"<<std::endl;
BatchMakeScript<<"If (${IsCorrespondenceQCFile} == 0)"<<std::endl;
BatchMakeScript<<"    set(FileNumber 0)"<<std::endl;
BatchMakeScript<<"    ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"      GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"      GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"      set (MeshAttributePath ${T1Path}/${DataSubdir}/MeshAttribute/)"<<std::endl;
BatchMakeScript<<"      set (ParticleFile ${CorrespondenceExecPath}${T1CaseHead}_WMMap_infl_particles.lpts)"<<std::endl;
BatchMakeScript<<"      set (DistanceMap ${CorrespondencePreprocessingPath}${T1CaseHead}_WMMap_infl_DistanceMap.mha)"<<std::endl;
BatchMakeScript<<"      If(${FileNumber} == 0)"<<std::endl;
BatchMakeScript<<"        set(InputList '\"'${DistanceMap}'\"')"<<std::endl;
BatchMakeScript<<"        set(PointList '\"'${ParticleFile}'\"')"<<std::endl;
BatchMakeScript<<"      Else (${FileNumber})"<<std::endl;
BatchMakeScript<<"        set(InputList ${InputList} '\"'${DistanceMap}'\"')"<<std::endl;
BatchMakeScript<<"        set(PointList ${PointList} '\"'${ParticleFile}'\"')"<<std::endl;
BatchMakeScript<<"      EndIf(${FileNumber})"<<std::endl;
BatchMakeScript<<"      Inc(${FileNumber} 1)"<<std::endl;
BatchMakeScript<<"      Int(${FileNumber})"<<std::endl;
BatchMakeScript<<"    EndForEach(T1Case)"<<std::endl;
BatchMakeScript<<"    WriteFile(${CorrespondenceQCFile} '(inputs '${InputList}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceQCFile} '(point_files '${PointList}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceQCFile} '(number_of_particles '${CorrespondenceNbParticles}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceQCFile} '(iterations_per_split '${CorrespondenceIterationsPerSplit}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceQCFile} '(starting_regularization '${CorrespondenceStartingRegularization}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceQCFile} '(ending_regularization '${CorrespondenceEndingRegularization}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceQCFile} '(optimization_iterations '${CorrespondenceOptimizationIterations}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceQCFile} '(checkpointing_interval '${CorrespondenceCheckPointingIntervals}')\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceQCFile} '(output_points_prefix \"'${CorrespondenceExecPath}'Particles\")\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondenceQCFile} '(relative_weighting '${CorrespondenceRelativeWeighting}')\\n')"<<std::endl;
BatchMakeScript<<"    If(${CorrespondenceProcrustesOff} == 0)"<<std::endl;
BatchMakeScript<<"      AppendFile(${CorrespondenceQCFile} '(procrustes_interval '${CorrespondenceProcrustesInterval}')\\n')"<<std::endl;
BatchMakeScript<<"      If(${CorrespondenceProcrustesScalingOff} == 0)"<<std::endl;
BatchMakeScript<<"        AppendFile(${CorrespondenceQCFile} '(procrustes_scaling 1)\\n')"<<std::endl;
BatchMakeScript<<"      EndIf(${CorrespondenceProcrustesScalingOff})"<<std::endl;
BatchMakeScript<<"    EndIf(${CorrespondenceProcrustesOff})  "<<std::endl;
BatchMakeScript<<"Else (${IsCorrespondenceQCFile})"<<std::endl;
BatchMakeScript<<"  echo('Particle correspondence QC file already exists!')"<<std::endl;
BatchMakeScript<<"Endif (${IsCorrespondenceQCFile})"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('PARTICLE CORRESPONDENCE QC: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 15. Correspondence postprocessing"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('CORRESPONDENCE POSTPROCESSING...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"set (CorrespondencePostProcessingPath ${CorrespondencePath}CorrespondencePostProcessing/)"<<std::endl;
BatchMakeScript<<"FileExists(IsCorrespondencePostProcessingPath ${CorrespondencePostProcessingPath})"<<std::endl;
BatchMakeScript<<"If (${IsCorrespondencePostProcessingPath} == 0)"<<std::endl;
BatchMakeScript<<"  MakeDirectory(${CorrespondencePostProcessingPath})"<<std::endl;
BatchMakeScript<<"EndIf (${IsCorrespondencePostProcessingPath})"<<std::endl;
BatchMakeScript<<"set (CorrespondencePostProcessingLogFile ${CorrespondencePostProcessingPath}GAMBIT_CorrespondencePostProcessing.log)"<<std::endl;
BatchMakeScript<<"set (CorrespondencePostProcessingFile ${CorrespondencePath}GAMBIT_CorrespondencePostProcessing.params)"<<std::endl;
BatchMakeScript<<"FileExists(IsCorrespondencePostProcessingFile ${CorrespondencePostProcessingFile})"<<std::endl;
BatchMakeScript<<"If (${IsCorrespondencePostProcessingFile} == 0)"<<std::endl;
BatchMakeScript<<"  WriteFile(${CorrespondencePostProcessingFile} 'NUMBER_OF_SHAPES='${NbData}'\\n')"<<std::endl;
BatchMakeScript<<"  ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"    GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"    GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"    set (InflationPath ${T1Path}/${DataSubdir}/MeshInflation/)"<<std::endl;
BatchMakeScript<<"    set (InflatedGenusZeroWMMapSurface ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl.vtk)"<<std::endl;
BatchMakeScript<<"    AppendFile(${CorrespondencePostProcessingFile} ${CorrespondenceExecPath}${T1CaseHead}'_WMMap_infl_particles.lpts '${InflatedGenusZeroWMMapSurface}' '${CorrespondencePostProcessingPath}${T1CaseHead}'_WMMap_infl.vtk \\n')"<<std::endl;
BatchMakeScript<<"  EndForEach (T1Case)"<<std::endl;
BatchMakeScript<<"Else (${IsCorrespondencePostProcessingFile})"<<std::endl;
BatchMakeScript<<"  echo('Correspondence postprocessing file already exists!')"<<std::endl;
BatchMakeScript<<"Endif (${IsCorrespondencePostProcessingFile})"<<std::endl;
BatchMakeScript<<"GetParam (FirstT1Case ${OrigT1CasesList} 0)"<<std::endl;
BatchMakeScript<<"GetFilename (FirstT1CaseHead ${FirstT1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"set(FirstPostProcessedVTKFile ${CorrespondencePostProcessingPath}${FirstT1CaseHead}_WMMap_infl.vtk)"<<std::endl;
BatchMakeScript<<"FileExists(IsFirstPostProcessedVTKFile ${FirstPostProcessedVTKFile})"<<std::endl;
BatchMakeScript<<"If (${IsFirstPostProcessedVTKFile} == 0)"<<std::endl;
BatchMakeScript<<"  SetAppOption(correspondencepostprocessing.parameterFileName.parameterFileName ${CorrespondencePostProcessingFile})"<<std::endl;
BatchMakeScript<<"  SetAppOption(correspondencepostprocessing.outputDirectory.outputDirectory ${CorrespondencePostProcessingPath})"<<std::endl;
BatchMakeScript<<"  If(${CorrespondencePostProcessingProjectToSurfaceOff} == 0)"<<std::endl;
BatchMakeScript<<"    SetAppOption(correspondencepostprocessing.projectToSurface 1)"<<std::endl;
BatchMakeScript<<"    SetAppOption(correspondencepostprocessing.projectionFactor.projectionFactor ${CorrespondencePostProcessingProjectionFactor})"<<std::endl;
BatchMakeScript<<"  EndIf(${CorrespondencePostProcessingProjectToSurfaceOff})"<<std::endl;
BatchMakeScript<<"  Run (outputcorrespondencepostprocessing ${correspondencepostprocessing} errorcorrespondencepostprocessing)"<<std::endl;
BatchMakeScript<<"  WriteFile(${CorrespondencePostProcessingLogFile} ${outputcorrespondencepostprocessing})"<<std::endl;
BatchMakeScript<<"  If (${errorcorrespondencepostprocessing} != '')"<<std::endl;
BatchMakeScript<<"    echo ('Correspondence post-processing error: ' ${errorcorrespondencepostprocessing})"<<std::endl;
BatchMakeScript<<"    exit()"<<std::endl;
BatchMakeScript<<"  EndIf (${errorcorrespondencepostprocessing})"<<std::endl;
BatchMakeScript<<"Else (${IsFirstPostProcessedVTKFile})"<<std::endl;
BatchMakeScript<<"  echo('Correspondence postprocessing already done!')"<<std::endl;
BatchMakeScript<<"EndIf (${IsFirstPostProcessedVTKFile})"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('CORRESPONDENCE POSTPROCESSING: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 16. Surface measurements interpolation (cortical thickness, sulcal depth, surface area)"<<std::endl;
BatchMakeScript<<"# and VTK Mesh creation with scalar field for QC"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('SURFACE MEASUREMENT INTERPOLATION...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"  GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  echo ('Case Number: '${T1CaseHead})"<<std::endl;
BatchMakeScript<<"  echo ( )"<<std::endl;
BatchMakeScript<<"  set (InflationPath ${T1Path}/${DataSubdir}/MeshInflation/)"<<std::endl;
BatchMakeScript<<"  set (MeshAttributePath ${T1Path}/${DataSubdir}/MeshAttribute/)"<<std::endl;
BatchMakeScript<<"  set (WMMapInflatedSurface ${InflationPath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_infl.vtk)"<<std::endl;
BatchMakeScript<<"  set (PostProcessedVTKFile ${CorrespondencePostProcessingPath}${T1CaseHead}_WMMap_infl.vtk)"<<std::endl;
BatchMakeScript<<"  set (MeasurementList 'SulcalDepth' 'CortThick' 'SurfaceArea')"<<std::endl;
BatchMakeScript<<"  ForEach (Measurement ${MeasurementList})"<<std::endl;
BatchMakeScript<<"    set (MeasurementAttributeFile ${MeshAttributePath}${T1CaseHead}${ImageSuffix}_WMMap_post_genus_${Measurement}.txt)"<<std::endl;
BatchMakeScript<<"    set (InterpolatedMeasurement ${CorrespondencePostProcessingPath}${T1CaseHead}_WMMap_infl_${Measurement}.txt)"<<std::endl;
BatchMakeScript<<"    FileExists(IsInterpolatedMeasurement ${InterpolatedMeasurement})"<<std::endl;
BatchMakeScript<<"    If (${IsInterpolatedMeasurement} == 0)"<<std::endl;
BatchMakeScript<<"      set (MeshMathCmd 'MeshMath '${WMMapInflatedSurface} ${InterpolatedMeasurement}' -closestPoint '${MeasurementAttributeFile} ${PostProcessedVTKFile})"<<std::endl;
BatchMakeScript<<"      Run (output ${MeshMathCmd} error)"<<std::endl;
BatchMakeScript<<"      If (${error} != '')"<<std::endl;
BatchMakeScript<<"        echo ('MeshMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"        exit()"<<std::endl;
BatchMakeScript<<"      EndIf (${error})"<<std::endl;
BatchMakeScript<<"      set (MeshMathCmd 'MeshMath '${PostProcessedVTKFile} ${PostProcessedVTKFile}' -KWMtoPolyData '${InterpolatedMeasurement} ${Measurement})"<<std::endl;
BatchMakeScript<<"      Run (output ${MeshMathCmd} error)"<<std::endl;
BatchMakeScript<<"      If (${error} != '')"<<std::endl;
BatchMakeScript<<"        echo ('MeshMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"        exit()"<<std::endl;
BatchMakeScript<<"      EndIf (${error})"<<std::endl;
BatchMakeScript<<"    Else (${IsInterpolatedMeasurement})"<<std::endl;
BatchMakeScript<<"      echo (${Measurement}' interpolation already done!')"<<std::endl;
BatchMakeScript<<"    EndIf(${IsInterpolatedMeasurement})"<<std::endl;
BatchMakeScript<<"  EndForEach (Measurement)"<<std::endl;
BatchMakeScript<<"EndForEach (T1Case)"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('SURFACE MEASUREMENT INTERPOLATION: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;

BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# 17. QC: Mean interpolated measurements computation"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"# ---------------------------------------------------------------------"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('QC: MEAN INTERPOLATED MEASUREMENTS COMPUTATION...')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"set (MeasurementList 'SulcalDepth' 'CortThick' 'SurfaceArea')"<<std::endl;
BatchMakeScript<<"ForEach (Measurement ${MeasurementList})"<<std::endl;
BatchMakeScript<<"  set (MeanInterpolatedMeasurement ${CorrespondencePostProcessingPath}Mean_${Measurement}.txt)"<<std::endl;
BatchMakeScript<<"  FileExists(IsMeanInterpolatedMeasurement ${MeanInterpolatedMeasurement})"<<std::endl;
BatchMakeScript<<"  If (${IsMeanInterpolatedMeasurement} == 0)"<<std::endl;
BatchMakeScript<<"    set(FileNumber 0)"<<std::endl;
BatchMakeScript<<"    ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"      GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"      GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"      set (InterpolatedMeasurement ${CorrespondencePostProcessingPath}${T1CaseHead}_WMMap_infl_${Measurement}.txt)"<<std::endl;
BatchMakeScript<<"      If(${FileNumber} == 0)"<<std::endl;
BatchMakeScript<<"        set(FirstAttribute ${InterpolatedMeasurement})"<<std::endl;
BatchMakeScript<<"      EndIf(${FileNumber})"<<std::endl;
BatchMakeScript<<"      If(${FileNumber} == 1)"<<std::endl;
BatchMakeScript<<"        set(AttributeList ${InterpolatedMeasurement})"<<std::endl;
BatchMakeScript<<"      EndIf(${FileNumber})"<<std::endl;
BatchMakeScript<<"      If(${FileNumber} > 1)"<<std::endl;
BatchMakeScript<<"        set(AttributeList ${AttributeList} ${InterpolatedMeasurement})"<<std::endl;
BatchMakeScript<<"      EndIf(${FileNumber})"<<std::endl;
BatchMakeScript<<"      Inc(${FileNumber} 1)"<<std::endl;
BatchMakeScript<<"      Int(${FileNumber})"<<std::endl;
BatchMakeScript<<"    EndForEach (T1Case)"<<std::endl;
BatchMakeScript<<"    set (MeshMathCmd 'MeshMath '${FirstAttribute} ${MeanInterpolatedMeasurement}' -mean '${AttributeList})"<<std::endl;
BatchMakeScript<<"    Run (output ${MeshMathCmd} error)"<<std::endl;
BatchMakeScript<<"    If (${error} != '')"<<std::endl;
BatchMakeScript<<"      echo ('MeshMath error: ' ${error})"<<std::endl;
BatchMakeScript<<"      exit()"<<std::endl;
BatchMakeScript<<"    EndIf (${error})"<<std::endl;
BatchMakeScript<<"  Else (${IsMeanInterpolatedMeasurement})"<<std::endl;
BatchMakeScript<<"    echo ('Mean '${Measurement}' already done!')"<<std::endl;
BatchMakeScript<<"  EndIf(${IsMeanInterpolatedMeasurement})"<<std::endl;
BatchMakeScript<<"EndForEach (Measurement)"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('QC: MEAN INTERPOLATED MEASUREMENTS COMPUTATION: DONE!')"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;

  BatchMakeScript.close();

  std::cout<<"Writing Computation BatchMake script: Done!"<<std::endl;
}

void Computation::WriteBMSMRMLCreationFile()
{
  std::cout<<"Writing Slicer3 MRML scenes creation script..."<<std::endl;

  std::ofstream BatchMakeScript(GetBMSMRMLCreationFile().c_str());

  BatchMakeScript<<"#---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"#---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"#--------------------- GAMBIT -----------------------------------------"<<std::endl;
  BatchMakeScript<<"#-------------Slicer3 MRML Scenes creation----------------------------"<<std::endl;
  BatchMakeScript<<"#---------------------------------------------------------------------"<<std::endl;
  BatchMakeScript<<"#---------------------------------------------------------------------"<<std::endl;

  BatchMakeScript<<"Include("<<GetBMSParameterFile()<<")"<<std::endl;

BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ('MRML SCENE CREATION...')"<<std::endl;
BatchMakeScript<<"# Creating GAMBIT Directory if necessary"<<std::endl;
BatchMakeScript<<"set (MRMLPath ${OutputDir}/QC_MRML/)"<<std::endl;
BatchMakeScript<<"FileExists(IsMRMLPath ${MRMLPath})"<<std::endl;
BatchMakeScript<<"If (${IsMRMLPath} == 0)"<<std::endl;
BatchMakeScript<<"  MakeDirectory(${MRMLPath})"<<std::endl;
BatchMakeScript<<"EndIf (${IsMRMLPath})"<<std::endl;

BatchMakeScript<<"set (CaseNb 0)"<<std::endl;
BatchMakeScript<<"ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"  GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"  If (${CaseNb} == 0)"<<std::endl;
BatchMakeScript<<"    set (LabelList ${T1CaseHead})"<<std::endl;
BatchMakeScript<<"  Else (${CaseNb})"<<std::endl;
BatchMakeScript<<"    set (LabelList ${LabelList} ${T1CaseHead})"<<std::endl;
BatchMakeScript<<"  Endif (${CaseNb})"<<std::endl;
BatchMakeScript<<"  Inc (${CaseNb} 1)"<<std::endl;
BatchMakeScript<<"  Int (${CaseNb})"<<std::endl;
BatchMakeScript<<"EndForEach (T1Case)"<<std::endl;
BatchMakeScript<<"Set (CorrespondencePath ${OutputDir}Correspondence/)"<<std::endl;
BatchMakeScript<<"Set (CorrespondencePostProcessingPath ${CorrespondencePath}CorrespondencePostProcessing/)"<<std::endl;
BatchMakeScript<<"Set (SulcalDepthCustomLUTFileHead customLUT_SulcalDepth.txt)"<<std::endl;
BatchMakeScript<<"Set (SulcalDepthCustomLUTFile ${CorrespondencePostProcessingPath}${SulcalDepthCustomLUTFileHead})"<<std::endl;
BatchMakeScript<<"Set (CorticalThicknessCustomLUTFileHead customLUT_CortThick.txt)"<<std::endl;
BatchMakeScript<<"Set (CorticalThicknessCustomLUTFile ${CorrespondencePostProcessingPath}${CorticalThicknessCustomLUTFileHead})"<<std::endl;
BatchMakeScript<<"Set (MRMLSceneList ${MRMLPath}SlicerScene_CorrespondingMesh.mrml ${MRMLPath}SlicerScene_SmoothedGenusZeroMesh.mrml ${MRMLPath}SlicerScene_InflatedMesh.mrml)"<<std::endl;
BatchMakeScript<<"Set (MRMLSceneNb 0)"<<std::endl;
BatchMakeScript<<"ForEach(MRMLScene ${MRMLSceneList})"<<std::endl;
BatchMakeScript<<"  If (${MRMLSceneNb} == 0)"<<std::endl;
BatchMakeScript<<"    set (FileNb 0)"<<std::endl;
BatchMakeScript<<"    ListFileInDir (FileList ${CorrespondencePostProcessingPath} *_WMMap_infl.vtk)"<<std::endl;
BatchMakeScript<<"    Foreach (File ${FileList})"<<std::endl;
BatchMakeScript<<"      If(${FileNb} == 0)"<<std::endl;
BatchMakeScript<<"        set(MeshList ${CorrespondencePostProcessingPath}${File})"<<std::endl;
BatchMakeScript<<"      Else (${FileNb})"<<std::endl;
BatchMakeScript<<"        set(MeshList ${MeshList} ${CorrespondencePostProcessingPath}${File})"<<std::endl;
BatchMakeScript<<"      EndIf(${FileNb})"<<std::endl;
BatchMakeScript<<"      Inc(${FileNb} 1)"<<std::endl;
BatchMakeScript<<"      Int(${FileNb})"<<std::endl;
BatchMakeScript<<"    EndForeach (File)"<<std::endl;
BatchMakeScript<<"  EndIf(${MRMLSceneNb})"<<std::endl;
BatchMakeScript<<"  If (${MRMLSceneNb} == 1)"<<std::endl;
BatchMakeScript<<"    set (FileNb 0)"<<std::endl;
BatchMakeScript<<"    ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"      GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"      GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"      set (InflationPath ${T1Path}/${DataSubdir}/MeshInflation/)"<<std::endl;
BatchMakeScript<<"      ListFileInDir (SmoothedGenusZeroWMMapSurfaceTail ${InflationPath} ${T1CaseHead}*_WMMap_post_genus_infl_It2.vtk)"<<std::endl;
BatchMakeScript<<"      set (SmoothedGenusZeroWMMapSurface ${InflationPath}${SmoothedGenusZeroWMMapSurfaceTail})"<<std::endl;
BatchMakeScript<<"      If(${FileNb} == 0)"<<std::endl;
BatchMakeScript<<"        set(MeshList ${SmoothedGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"      Else (${FileNb})"<<std::endl;
BatchMakeScript<<"        set(MeshList ${MeshList} ${SmoothedGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"      EndIf(${FileNb})"<<std::endl;
BatchMakeScript<<"      Inc(${FileNb} 1)"<<std::endl;
BatchMakeScript<<"      Int(${FileNb})"<<std::endl;
BatchMakeScript<<"    EndForEach (T1Case)"<<std::endl;
BatchMakeScript<<"  EndIf (${MRMLSceneNb})"<<std::endl;
BatchMakeScript<<"  If (${MRMLSceneNb} == 2)"<<std::endl;
BatchMakeScript<<"    set (FileNb 0)"<<std::endl;
BatchMakeScript<<"    ForEach (T1Case ${OrigT1CasesList})"<<std::endl;
BatchMakeScript<<"      GetFilename (T1Path ${T1Case} PATH)"<<std::endl;
BatchMakeScript<<"      GetFilename (T1CaseHead ${T1Case} NAME_WITHOUT_EXTENSION)"<<std::endl;
BatchMakeScript<<"      set (InflationPath ${T1Path}/${DataSubdir}/MeshInflation/)"<<std::endl;
BatchMakeScript<<"      ListFileInDir (InflatedGenusZeroWMMapSurfaceTail ${InflationPath} ${T1CaseHead}*_WMMap_post_genus_infl.vtk)"<<std::endl;
BatchMakeScript<<"      set (InflatedGenusZeroWMMapSurface ${InflationPath}${InflatedGenusZeroWMMapSurfaceTail})"<<std::endl;
BatchMakeScript<<"      If(${FileNb} == 0)"<<std::endl;
BatchMakeScript<<"        set(MeshList ${InflatedGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"      Else (${FileNb})"<<std::endl;
BatchMakeScript<<"        set(MeshList ${MeshList} ${InflatedGenusZeroWMMapSurface})"<<std::endl;
BatchMakeScript<<"      EndIf(${FileNb})"<<std::endl;
BatchMakeScript<<"      Inc(${FileNb} 1)"<<std::endl;
BatchMakeScript<<"      Int(${FileNb})"<<std::endl;
BatchMakeScript<<"    EndForEach (T1Case)"<<std::endl;
BatchMakeScript<<"  EndIf (${MRMLSceneNb})"<<std::endl;

BatchMakeScript<<" Set(dim1 200)"<<std::endl;
BatchMakeScript<<" Set(dim2 200)"<<std::endl;
BatchMakeScript<<" WriteFile(${MRMLScene} '<MRML userTags=\"\">\\n')"<<std::endl;
BatchMakeScript<<" Set (SnapshotList 0 1 2 3)"<<std::endl;
BatchMakeScript<<" ForEach(Snapshot ${SnapshotList})"<<std::endl;
BatchMakeScript<<"  If (${Snapshot} == 1)"<<std::endl;
BatchMakeScript<<"   AppendFile(${MRMLScene} '<SceneSnapshot\\n')"<<std::endl;
BatchMakeScript<<"   AppendFile(${MRMLScene} 'id=\"vtkMRMLSceneSnapshotNode1\" name=\"Default\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\">\\n')"<<std::endl;
BatchMakeScript<<"  EndIf (${Snapshot})"<<std::endl;
BatchMakeScript<<"  If (${Snapshot} == 2)"<<std::endl;
BatchMakeScript<<"   AppendFile(${MRMLScene} '<SceneSnapshot\\n')"<<std::endl;
BatchMakeScript<<"   AppendFile(${MRMLScene} 'id=\"vtkMRMLSceneSnapshotNode1\" name=\"CorticalThickness\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\">\\n')"<<std::endl;
BatchMakeScript<<"  EndIf (${Snapshot})"<<std::endl;
BatchMakeScript<<"  If (${Snapshot} == 3)"<<std::endl;
BatchMakeScript<<"   AppendFile(${MRMLScene} '<SceneSnapshot\\n')"<<std::endl;
BatchMakeScript<<"   AppendFile(${MRMLScene} 'id=\"vtkMRMLSceneSnapshotNode1\" name=\"SulcalDepth\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\">\\n')"<<std::endl;
BatchMakeScript<<"  EndIf (${Snapshot})"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<Selection\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLSelectionNode1\" name=\"vtkMRMLSelectionNode1\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" activeVolumeID=\"NULL\" secondaryVolumeID=\"NULL\" activeLabelVolumeID=\"NULL\" activeFiducialListID=\"NULL\" activeROIListID=\"NULL\" activeCameraID=\"NULL\" activeViewID=\"NULL\" activeLayoutID=\"vtkMRMLLayoutNode1\"></Selection>\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<Interaction\\n' )"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLInteractionNode1\" name=\"vtkMRMLInteractionNode1\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" currentInteractionMode=\"ViewTransform\" lastInteractionMode=\"ViewTransform\"></Interaction>\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<Layout\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLLayoutNode1\" name=\"vtkMRMLLayoutNode1\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" currentViewArrangement=\"2\" guiPanelVisibility=\"1\" bottomPanelVisibility =\"1\" guiPanelLR=\"0\" numberOfCompareViewRows=\"0\" numberOfCompareViewColumns=\"0\" numberOfLightboxRows=\"1\" numberOfLightboxColumns=\"1\" mainPanelSize=\"400\" secondaryPanelSize=\"400\"></Layout>\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<View\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLViewNode1\" name=\"vtkMRMLViewNode1\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" active=\"false\" fieldOfView=\"200\" letterSize=\"0.05\" boxVisible=\"true\" fiducialsVisible=\"true\" fiducialLabelsVisible=\"true\" axisLabelsVisible=\"true\" backgroundColor=\"0.70196 0.70196 0.90588\" animationMode=\"Off\" viewAxisMode=\"LookFrom\" spinDegrees=\"2\" spinMs=\"5\" spinDirection=\"YawLeft\" rotateDegrees=\"5\" rockLength=\"200\" rockCount=\"0\" stereoType=\"NoStereo\" renderMode=\"Perspective\"></View>\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<Camera\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLCameraNode1\" name=\"vtkMRMLCameraNode1\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" position=\"-1554.04 129.958 449.973\" focalPoint=\"-129.434 153.443 240.212\" viewUp=\"0 0 1\" parallelProjection=\"false\" parallelScale=\"1\" active=\"false\"></Camera>\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<TGParameters\\n')"<<std::endl;
 BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLChangeTrackerNode1\" name=\"vtkMRMLChangeTrackerNode1\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" ROIMin=\"-1 -1 -1\" ROIMax=\"-1 -1 -1\" SegmentThresholdMin=\"-1\" SegmentThresholdMax=\"-1\" Analysis_Intensity_Flag=\"0\" Analysis_Deformable_Flag=\"0\" UseITK=\"1\" RegistrationChoice=\"3\"  ROIRegistration=\"1\"  ResampleChoice=\"3\"  ResampleConst=\"0.5\"></TGParameters>\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<VolumeRenderingSelection\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLVolumeRenderingSelectionNode1\" name=\"vtkMRMLVolumeRenderingSelectionNode1\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" activeVolumeID=\"NULL\" activeVolumeRenderingID=\"NULL\"></VolumeRenderingSelection>\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<Slice\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLSliceNode1\" name=\"Green\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" fieldOfView=\"400 250 1\" dimensions=\"500 320 1\" activeSlice=\"0\" layoutGridRows=\"1\" layoutGridColumns=\"1\" sliceToRAS=\"-1 0 0 0 0 0 1 0 0 1 0 0 0 0 0 1\" layoutName=\"Green\" orientation=\"Coronal\" jumpMode=\"1\" sliceVisibility=\"false\" widgetVisibility=\"false\" useLabelOutline=\"false\" sliceSpacingMode=\"0\" prescribedSliceSpacing=\"1 1 1\"></Slice>\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<SliceComposite\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLSliceCompositeNode1\" name=\"vtkMRMLSliceCompositeNode1\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" backgroundVolumeID=\"\" foregroundVolumeID=\"\" labelVolumeID=\"\" compositing=\"0\" labelOpacity=\"1\" linkedControl=\"0\" foregroundGrid=\"0\" backgroundGrid=\"0\" labelGrid=\"1\" fiducialVisibility=\"1\" fiducialLabelVisibility=\"1\" sliceIntersectionVisibility=\"0\" layoutName=\"Green\" annotationMode=\"All\" doPropagateVolumeSelection=\"1\"></SliceComposite>\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<Slice\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLSliceNode2\" name=\"Red\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" fieldOfView=\"400 250 1\" dimensions=\"500 320 1\" activeSlice=\"0\" layoutGridRows=\"1\" layoutGridColumns=\"1\" sliceToRAS=\"-1 0 0 0 0 1 0 0 0 0 1 0 0 0 0 1\" layoutName=\"Red\" orientation=\"Axial\" jumpMode=\"1\" sliceVisibility=\"false\" widgetVisibility=\"false\" useLabelOutline=\"false\" sliceSpacingMode=\"0\" prescribedSliceSpacing=\"1 1 1\"></Slice>\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<SliceComposite\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLSliceCompositeNode2\" name=\"vtkMRMLSliceCompositeNode2\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" backgroundVolumeID=\"\" foregroundVolumeID=\"\" labelVolumeID=\"\" compositing=\"0\" labelOpacity=\"1\" linkedControl=\"0\" foregroundGrid=\"0\" backgroundGrid=\"0\" labelGrid=\"1\" fiducialVisibility=\"1\" fiducialLabelVisibility=\"1\" sliceIntersectionVisibility=\"0\" layoutName=\"Red\" annotationMode=\"All\" doPropagateVolumeSelection=\"1\"></SliceComposite>\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<Slice\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLSliceNode3\" name=\"Yellow\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" fieldOfView=\"400 250 1\" dimensions=\"500 320 1\" activeSlice=\"0\" layoutGridRows=\"1\" layoutGridColumns=\"1\" sliceToRAS=\"0 0 1 0 -1 0 0 0 0 1 0 0 0 0 0 1\" layoutName=\"Yellow\" orientation=\"Sagittal\" jumpMode=\"1\" sliceVisibility=\"false\" widgetVisibility=\"false\" useLabelOutline=\"false\" sliceSpacingMode=\"0\" prescribedSliceSpacing=\"1 1 1\"></Slice>\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<SliceComposite\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLSliceCompositeNode3\" name=\"vtkMRMLSliceCompositeNode3\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" backgroundVolumeID=\"\" foregroundVolumeID=\"\" labelVolumeID=\"\" compositing=\"0\" labelOpacity=\"1\" linkedControl=\"0\" foregroundGrid=\"0\" backgroundGrid=\"0\" labelGrid=\"1\" fiducialVisibility=\"1\" fiducialLabelVisibility=\"1\" sliceIntersectionVisibility=\"0\" layoutName=\"Yellow\" annotationMode=\"All\" doPropagateVolumeSelection=\"1\"></SliceComposite>\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<Crosshair\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLCrosshairNode1\" name=\"vtkMRMLCrosshairNode1\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" crosshairMode=\"NoCrosshair\" crosshairBehavior=\"Normal\" crosshairThickness=\"Fine\" crosshairRAS=\"0 0 0\"></Crosshair>\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<ClipModels\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLClipModelsNode1\" name=\"vtkMRMLClipModelsNode1\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" clipType=\"0\" redSliceClipState=\"0\" yellowSliceClipState=\"0\" greenSliceClipState=\"0\"></ClipModels>\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} '<ScriptedModule\\n')"<<std::endl;
BatchMakeScript<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLScriptedModuleNode1\" name=\"vtkMRMLScriptedModuleNode1\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" ModuleName =\"Editor\" parameter0= \"label 1\"></ScriptedModule>\\n')"<<std::endl;
BatchMakeScript<<"  Set(count 1)"<<std::endl;
BatchMakeScript<<"  Set(count2 0)"<<std::endl;
BatchMakeScript<<"  Set(res 1)"<<std::endl;
BatchMakeScript<<"  Set(Dim1 0)"<<std::endl;
BatchMakeScript<<"  Set(count3 1)"<<std::endl;


BatchMakeScript<<"  ForEach(Mesh ${MeshList})"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} '<ModelStorage\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} 'id=\"vtkMRMLModelStorageNode'${count}'\" name=\"vtkMRMLModelStorageNode'${count}'\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" fileName=\"'${Mesh}'\" useCompression=\"1\" readState=\"0\" writeState=\"0\"></ModelStorage>\\n')"<<std::endl;
//BatchMakeScript<<"    Randomize(Rvalue uniform 0 1) "<<std::endl;
//BatchMakeScript<<"    Randomize(Gvalue uniform 0 1) "<<std::endl;
//BatchMakeScript<<"    Randomize(Bvalue uniform 0 1) "<<std::endl;
BatchMakeScript<<"    Set(Rvalue 0.9) "<<std::endl;
BatchMakeScript<<"    Set(Gvalue 0.9) "<<std::endl;
BatchMakeScript<<"    Set(Bvalue 0.9) "<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} '<ModelDisplay\\n')"<<std::endl;

BatchMakeScript<<"  If (${Snapshot} == 0)"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} 'id=\"vtkMRMLModelDisplayNode'${count}'\" name=\"vtkMRMLModelDisplayNode'${count}'\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" color=\"'${Rvalue}' '${Gvalue}' '${Bvalue}'\" selectedColor=\"1 0 0\" selectedAmbient=\"0.4\" ambient=\"0.18\" diffuse=\"0.82\" selectedSpecular=\"0.5\" specular=\"0.16\" power=\"1\" opacity=\"1\" visibility=\"true\" clipping=\"false\" sliceIntersectionVisibility=\"false\" backfaceCulling=\"true\" scalarVisibility=\"false\" vectorVisibility=\"false\" tensorVisibility=\"false\" autoScalarRange=\"true\" scalarRange=\"0 100\" colorNodeRef=\"vtkMRMLColorTableNodeFullRainbow\" activeScalarName=\"\" ></ModelDisplay>\\n')"<<std::endl;
BatchMakeScript<<"  EndIf (${Snapshot})"<<std::endl;
BatchMakeScript<<"  If (${Snapshot} == 1)"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} 'id=\"vtkMRMLModelDisplayNode'${count}'\" name=\"vtkMRMLModelDisplayNode'${count}'\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" color=\"'${Rvalue}' '${Gvalue}' '${Bvalue}'\" selectedColor=\"1 0 0\" selectedAmbient=\"0.4\" ambient=\"0.18\" diffuse=\"0.82\" selectedSpecular=\"0.5\" specular=\"0.16\" power=\"1\" opacity=\"1\" visibility=\"true\" clipping=\"false\" sliceIntersectionVisibility=\"false\" backfaceCulling=\"true\" scalarVisibility=\"false\" vectorVisibility=\"false\" tensorVisibility=\"false\" autoScalarRange=\"true\" scalarRange=\"0 100\" colorNodeRef=\"vtkMRMLColorTableNodeFullRainbow\" activeScalarName=\"\" ></ModelDisplay>\\n')"<<std::endl;
BatchMakeScript<<"  EndIf (${Snapshot})"<<std::endl;
BatchMakeScript<<"  If (${Snapshot} == 2)"<<std::endl;
//BatchMakeScript<<"    AppendFile(${MRMLScene} 'id=\"vtkMRMLModelDisplayNode'${count}'\" name=\"vtkMRMLModelDisplayNode'${count}'\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" color=\"'${Rvalue}' '${Gvalue}' '${Bvalue}'\" selectedColor=\"1 0 0\" selectedAmbient=\"0.4\" ambient=\"0.18\" diffuse=\"0.82\" selectedSpecular=\"0.5\" specular=\"0.16\" power=\"1\" opacity=\"1\" visibility=\"true\" clipping=\"false\" sliceIntersectionVisibility=\"false\" backfaceCulling=\"true\" scalarVisibility=\"true\" vectorVisibility=\"false\" tensorVisibility=\"false\" autoScalarRange=\"true\" scalarRange=\"0 100\" colorNodeRef=\"vtkMRMLColorTableNodeFullRainbow\" activeScalarName=\"CorticalThickness\" ></ModelDisplay>\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} 'id=\"vtkMRMLModelDisplayNode'${count}'\" name=\"vtkMRMLModelDisplayNode'${count}'\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" color=\"'${Rvalue}' '${Gvalue}' '${Bvalue}'\" selectedColor=\"1 0 0\" selectedAmbient=\"0.4\" ambient=\"0.18\" diffuse=\"0.82\" selectedSpecular=\"0.5\" specular=\"0.16\" power=\"1\" opacity=\"1\" visibility=\"true\" clipping=\"false\" sliceIntersectionVisibility=\"false\" backfaceCulling=\"true\" scalarVisibility=\"true\" vectorVisibility=\"false\" tensorVisibility=\"false\" autoScalarRange=\"true\" scalarRange=\"0 100\" colorNodeRef=\"vtkMRMLColorTableNodeFile'${CorticalThicknessCustomLUTFileHead}'\" activeScalarName=\"CortThick\" ></ModelDisplay>\\n')"<<std::endl;

BatchMakeScript<<"  EndIf (${Snapshot})"<<std::endl;
BatchMakeScript<<"  If (${Snapshot} == 3)"<<std::endl;
//BatchMakeScript<<"    AppendFile(${MRMLScene} 'id=\"vtkMRMLModelDisplayNode'${count}'\" name=\"vtkMRMLModelDisplayNode'${count}'\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" color=\"'${Rvalue}' '${Gvalue}' '${Bvalue}'\" selectedColor=\"1 0 0\" selectedAmbient=\"0.4\" ambient=\"0.18\" diffuse=\"0.82\" selectedSpecular=\"0.5\" specular=\"0.16\" power=\"1\" opacity=\"1\" visibility=\"true\" clipping=\"false\" sliceIntersectionVisibility=\"false\" backfaceCulling=\"true\" scalarVisibility=\"true\" vectorVisibility=\"false\" tensorVisibility=\"false\" autoScalarRange=\"true\" scalarRange=\"0 100\" colorNodeRef=\"vtkMRMLColorTableNodeFullRainbow\" activeScalarName=\"SulcalDepth\" ></ModelDisplay>\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} 'id=\"vtkMRMLModelDisplayNode'${count}'\" name=\"vtkMRMLModelDisplayNode'${count}'\" hideFromEditors=\"true\" selectable=\"true\" selected=\"false\" color=\"'${Rvalue}' '${Gvalue}' '${Bvalue}'\" selectedColor=\"1 0 0\" selectedAmbient=\"0.4\" ambient=\"0.18\" diffuse=\"0.82\" selectedSpecular=\"0.5\" specular=\"0.16\" power=\"1\" opacity=\"1\" visibility=\"true\" clipping=\"false\" sliceIntersectionVisibility=\"false\" backfaceCulling=\"true\" scalarVisibility=\"true\" vectorVisibility=\"false\" tensorVisibility=\"false\" autoScalarRange=\"true\" scalarRange=\"0 100\" colorNodeRef=\"vtkMRMLColorTableNodeFile'${SulcalDepthCustomLUTFileHead}'\" activeScalarName=\"SulcalDepth\" ></ModelDisplay>\\n')"<<std::endl;


BatchMakeScript<<"  EndIf (${Snapshot})"<<std::endl;

BatchMakeScript<<"    AppendFile(${MRMLScene} '<Model\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} 'id=\"vtkMRMLModelNode'${count}'\" name=\"'${Mesh}'\" hideFromEditors=\"false\" selectable=\"true\" selected=\"false\" transformNodeRef=\"vtkMRMLLinearTransformNode'${count}'\" storageNodeRef=\"vtkMRMLModelStorageNode'${count}'\" userTags=\"\" displayNodeRef=\"vtkMRMLModelDisplayNode'${count}'\"></Model>\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} '<LinearTransform\\n')"<<std::endl;
BatchMakeScript<<"    Math(Dim1 ${dim1} * ${count2})"<<std::endl;
BatchMakeScript<<"    Math(Dim2 ${dim2} * ${count3} )"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} 'id=\"vtkMRMLLinearTransformNode'${count}'\" name=\"vtkMRMLLinearTransformNode'${count}'\" hideFromEditors=\"false\" selectable=\"true\" selected=\"false\" userTags=\"\" matrixTransformToParent=\"1 0 0 0  0 1 0 '${Dim1}'  0 0 1 '${Dim2}'  0 0 0 1\"></LinearTransform>\\n')"<<std::endl;

BatchMakeScript<<"     Math(Dim1Fiducial ${Dim1} - ${dim1})"<<endl;
 BatchMakeScript<<"    If(${count} == 1)"<<std::endl;
 BatchMakeScript<<"     Set(fiducialCoordY ${Dim1Fiducial})"<<endl; 
 BatchMakeScript<<"     Set(fiducialCoordZ ${Dim2})"<<endl;
 BatchMakeScript<<"    EndIf(${count})"<<std::endl;
 BatchMakeScript<<"    If(${count} > 1)"<<std::endl;
 BatchMakeScript<<"     Set(fiducialCoordY  ${fiducialCoordY} ${Dim1Fiducial})"<<endl;
 BatchMakeScript<<"     Set(fiducialCoordZ ${fiducialCoordZ} ${Dim2})"<<endl;
 BatchMakeScript<<"    EndIf(${count})"<<std::endl;
	
BatchMakeScript<<"    If( ${res} != 5)"<<std::endl;
BatchMakeScript<<"      Inc(${count2} 1)"<<std::endl;
BatchMakeScript<<"    EndIf( ${res} )"<<std::endl;
BatchMakeScript<<"    If( ${res} == 5)"<<std::endl;
BatchMakeScript<<"      Math(Dim2 ${dim2} * ${count3} )"<<std::endl;
BatchMakeScript<<"      Set(count2 0)"<<std::endl;
BatchMakeScript<<"      Inc(${count3} 1)"<<std::endl;
BatchMakeScript<<"      Set(res 0)"<<std::endl;
BatchMakeScript<<"    EndIf( ${res} )"<<std::endl;
BatchMakeScript<<"    Inc(${count} 1)"<<std::endl;
BatchMakeScript<<"    Inc(${res} 1)"<<std::endl;
BatchMakeScript<<"   EndForEach(Mesh)"<<std::endl;


BatchMakeScript<<"    AppendFile(${MRMLScene} '<ColorTableStorage\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} 'id=\"vtkMRMLColorTableStorageNode1\"  name=\"vtkMRMLColorTableStorageNode1\"  hideFromEditors=\"true\"  selectable=\"true\"  selected=\"false\"  fileName=\"'${CorticalThicknessCustomLUTFile}'\"  useCompression=\"1\"  readState=\"0\"  writeState=\"0\" ></ColorTableStorage>\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} '<ColorTable\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} 'id=\"vtkMRMLColorTableNodeFile'${CorticalThicknessCustomLUTFileHead}'\"  name=\"'${CorticalThicknessCustomLUTFileHead}'\"  description=\"A color table read in from a text file, each line of the format: IntegerLabel  Name  R  G  B  Alpha\"  hideFromEditors=\"false\"  selectable=\"true\"  selected=\"false\"  storageNodeRef=\"vtkMRMLColorTableStorageNode1\"  userTags=\"\" type=\"14\" numcolors=\"256\"></ColorTable>\\n')"<<std::endl;

BatchMakeScript<<"    AppendFile(${MRMLScene} '<ColorTableStorage\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} 'id=\"vtkMRMLColorTableStorageNode2\"  name=\"vtkMRMLColorTableStorageNode2\"  hideFromEditors=\"true\"  selectable=\"true\"  selected=\"false\"  fileName=\"'${SulcalDepthCustomLUTFile}'\"  useCompression=\"1\"  readState=\"0\"  writeState=\"0\" ></ColorTableStorage>\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} '<ColorTable\\n')"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} 'id=\"vtkMRMLColorTableNodeFile'${SulcalDepthCustomLUTFileHead}'\"  name=\"'${SulcalDepthCustomLUTFileHead}'\"  description=\"A color table read in from a text file, each line of the format: IntegerLabel  Name  R  G  B  Alpha\"  hideFromEditors=\"false\"  selectable=\"true\"  selected=\"false\"  storageNodeRef=\"vtkMRMLColorTableStorageNode2\"  userTags=\"\" type=\"14\" numcolors=\"256\"></ColorTable>\\n')"<<std::endl;


BatchMakeScript	<<"  AppendFile(${MRMLScene} '<FiducialList\\n')"<<std::endl;
BatchMakeScript	<<"  AppendFile(${MRMLScene} 'id=\"vtkMRMLFiducialListNode1\" name=\"label: \" hideFromEditors=\"false\" selectable=\"true\" selected=\"false\" storageNodeRef=\"vtkMRMLFiducialListStorageNode1\" userTags=\"\" symbolScale=\"0\" symbolType=\"1\" textScale=\"10\" visibility=\"1\" color=\"0 0 0\" selectedcolor=\"0.0117647 0.00784314 0.00784314\" ambient=\"0\" diffuse=\"1\" specular=\"0\" power=\"1\" locked=\"0\" opacity=\"1\" fiducials=\"\\n')"<<std::endl;

BatchMakeScript<<"  set (MeshNb 0)"<<std::endl;
BatchMakeScript<<"  ForEach(Mesh ${MeshList})"<<std::endl;
BatchMakeScript	<<"  GetParam(Y ${fiducialCoordY} ${MeshNb})"<<std::endl;
BatchMakeScript	<<"  Inc(${Y} 100)"<<std::endl;
BatchMakeScript	<<"  GetParam(Z ${fiducialCoordZ} ${MeshNb} )"<<std::endl;
BatchMakeScript	<<"  GetParam(Label ${LabelList} ${MeshNb} )"<<std::endl;
BatchMakeScript	<<"  Inc(${Z} 70)"<<std::endl;
BatchMakeScript	<<"  AppendFile(${MRMLScene} 'id '${Label}' labeltext '${Label}' xyz ')"<<std::endl;
BatchMakeScript	<<"  AppendFile(${MRMLScene} '-130 '${Y}' '${Z}' orientationwxyz 0 0 0 1 selected 1 visibility 1\\n')"<<std::endl;
BatchMakeScript<<"  Inc(${MeshNb} 1)"<<std::endl;
BatchMakeScript<<"  Int(${MeshNb})"<<std::endl;
BatchMakeScript<<" EndForEach(Mesh)"<<std::endl;

BatchMakeScript <<"  AppendFile(${MRMLScene} '\"></FiducialList>\\n')"<<std::endl;

BatchMakeScript<<"  If (${Snapshot} != 0)"<<std::endl;
BatchMakeScript<<"    AppendFile(${MRMLScene} '</SceneSnapshot>\\n')"<<std::endl;
BatchMakeScript<<"  EndIf (${Snapshot})"<<std::endl;
BatchMakeScript<<"  EndForEach(Snapshot)"<<std::endl;

BatchMakeScript<<"  AppendFile(${MRMLScene} '</MRML>\\n')"<<std::endl;
BatchMakeScript<<"  Inc (${MRMLSceneNb} 1)"<<std::endl;
BatchMakeScript<<"  Int (${MRMLSceneNb})"<<std::endl;
BatchMakeScript<<"EndForEach(MRMLScene)"<<std::endl;
BatchMakeScript<<"echo ('MRML SCENE CREATION: DONE!')"<<std::endl;
BatchMakeScript<<"echo (*************************************************)"<<std::endl;
BatchMakeScript<<"echo ( )"<<std::endl;
BatchMakeScript<<"echo ('END OF GAMBIT EXECUTION!')"<<std::endl;

  BatchMakeScript.close();

  std::cout<<"Writing Slicer3 MRML scenes creation script: Done!"<<std::endl;
}
