/*=========================================================================

  Program:   MeshCortThick
  Module:    $RCSfile: Write_Mesh.cxx,v $
  Language:  C++
  Date:      $Date: 2011/04/28 13:01:39 $
  Version:   $Revision: 1.5 $
  Author:    Delphine Ribes

  Copyright (c) 2004 NeuroImaging Lab @ UNC. All rights reserved.
  See NeuroLibCopyright.txt for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/

#if defined(_MSC_VER)
#pragma warning ( disable : 4786 )
#pragma warning ( disable : 4284 )
#pragma warning ( disable : 4018 )
#endif

#include "Write_Mesh.h"
#include "CommandLineArgumentParser.h"
#include <iostream>

#define W_VERSION 1.1

void Help()
{
  std::cout<<" MeshCortThick "<<W_VERSION<<" - Compiled on: " << __DATE__ << " - " <<__TIME__ <<std::endl;
  std::cout<<"                 ------------------------------- "<<std::endl;
  std::cout<<"usage: MeshCortThick -i <ImageToMeshFileName> <ThicknessFileName> "<<std::endl;
  std::cout<<"or "<<std::endl;
  std::cout<<"       MeshCortThick -m <MeshFileName> <ThicknessFileName> "<<std::endl;

  
  std::cout << std::endl;
  std::cout << "Input:" << std::endl;
  std::cout << "-i <ImageToMeshFileName> <ThicknessFileName>"<<std::endl;
  std::cout << "-m <MeshFileName> <ThicknessFileName>"<<std::endl<<endl;
  std::cout << "Options:" << std::endl;
  std::cout << "-varI <value> : set the variance of the gaussian filter applied on the Input image"<<std::endl;
  std::cout << "default value of varI : 1.5 x max_InputImageSpacing"<<std::endl;
  std::cout << "-varT <value> : set the variance of the gaussian filter applied on the Thickness image"<<std::endl;
  std::cout << "default value of varT : 0.3 x max_ThicknessImageSpacing"<<std::endl;
  std::cout << "-signVector <X> <Y> <Z> : X,Y,Z sign coordinates of the input mesh (e.g for negative coordinates)"<<std::endl;  
  std::cout << "default value of signVector : 1 1 1"<<std::endl;
  std::cout << "-saveSmooth : save smoothed cortical thickness map"<<std::endl;
  std::cout<< std::endl;
  std::cout<<"MeshCortThick -version : Version "<<std::endl;
  std::cout<< std::endl;
  
}  

int main (int argc, char * argv[])
{
	if(argc==1)
	{
		Help();
                return 0;
        }        	
	if ((argc==2) && (!strcmp(argv[1],"-version")))
 	{
    		std::cout << " MeshCortThick " << W_VERSION<< " - Compiled on: " << __DATE__ << " - " 
					<<__TIME__  << std::endl;
    		return 0;
  	}
	else
        {
		CommandLineArgumentParser parser;
                parser.AddOption("-i",2);
                parser.AddOption("-h",0);
                parser.AddOption("-m",2);
                parser.AddOption("-varI",1);
                parser.AddOption("-varT",1);
		parser.AddOption("-signVector",3);
		parser.AddOption("-saveSmooth",0);
		
		CommandLineArgumentParseResult parseResult;        
                if(!parser.TryParseCommandLine(argc,argv,parseResult))
                    {
                      Help();
                      return -1;
                    }
                    else
                    {
                      if (parseResult.IsOptionPresent("-h"))
                      {
                        Help();
                        return 0;
                      }         
                    }
                
                    const char* InputFileName;
                    const char* ThicknessFileName;
                    const char * VarianceI;
                    const char * VarianceT;
                    Write_Mesh mesh;
                    
                    if(parseResult.IsOptionPresent("-varI"))
                    {
                      VarianceI = (parseResult.GetOptionParameter("-varI",0));
                      mesh.Set_VarianceI( VarianceI );
                    }
                    else
                      mesh.Bool_VarianceI();
                    
                    if(parseResult.IsOptionPresent("-varT"))
                    {
                      VarianceT = (parseResult.GetOptionParameter("-varT",0));
                      mesh.Set_VarianceT( VarianceT );
                    }
                    else
                      mesh.Bool_VarianceT();
		    
		    if(parseResult.IsOptionPresent("-signVector"))
                      mesh.Set_SignVector(parseResult.GetOptionParameter("-signVector",0),parseResult.GetOptionParameter("-signVector",1),parseResult.GetOptionParameter("-signVector",2));
		    else
		      mesh.Set_SignVector("1","1","1");

		    if(parseResult.IsOptionPresent("-saveSmooth"))
		      mesh.SaveSmoothOn(true);
		    else
		      mesh.SaveSmoothOn(false);

                    if(parseResult.IsOptionPresent("-i"))
                    {
                      InputFileName = (parseResult.GetOptionParameter("-i",0));
                      ThicknessFileName = (parseResult.GetOptionParameter("-i",1));
	              mesh.Set_FileName(InputFileName);
		      mesh.Set_AttrFileName(ThicknessFileName);
		
                      mesh.generateMesh();
                    }
                    
                    else if(parseResult.IsOptionPresent("-m"))
                    {
                      InputFileName = (parseResult.GetOptionParameter("-m",0));
                      ThicknessFileName = (parseResult.GetOptionParameter("-m",1));
                      mesh.Set_FileName(InputFileName);
		      mesh.Set_AttrFileName(ThicknessFileName);

		      std::string inputfilename = InputFileName;
		      int lastPoint = inputfilename.rfind('.');
		      std::string fileExtension = inputfilename.substr(lastPoint);
		      if (fileExtension.compare(".meta") == 0 )
			mesh.readMetaFile();
		      else if (fileExtension.compare(".vtk") == 0 )
			{
			  mesh.readVTKFile();
			  mesh.convertVTKToMeta();
			}
		      else
			{
			  std::cerr<<"Error: cannot read input mesh file (should be .meta or .vtk)"<<std::endl;
			  exit(1);
			}
                    }
                    mesh.generateMeshAttr();
                    mesh.writeAttr();
	}
	return 0;
}

void Write_Mesh::readVTKFile()
{
  m_VTKreader = vtkPolyDataReader::New();
  m_VTKreader->SetFileName(Get_FileName());
  m_VTKreader->Update();
  m_polydata = m_VTKreader->GetOutput () ; 
}

void Write_Mesh::convertVTKToMeta()
{
  // convert to itk mesh data structure
  vtkPolyDataToitkMesh *vtkItkConverter = new vtkPolyDataToitkMesh () ;
  vtkItkConverter->SetInput ( m_polydata ) ;

  MeshSOType::Pointer inputMeshSO = MeshSOType::New () ;
  inputMeshSO->SetMesh ( vtkItkConverter->GetOutput () ) ;
  m_inputMesh = inputMeshSO->GetMesh();
}

void Write_Mesh::readMetaFile()
{
  MeshConverterType *converter = new MeshConverterType();
  MeshSOType::Pointer inputMeshSO = converter->ReadMeta(Get_FileName());
  m_inputMesh = inputMeshSO->GetMesh();
  delete(converter);
}


void Write_Mesh::generateMesh()
{
  // load the image
  ImageType::Pointer image ;
  ReaderType::Pointer imageReader = ReaderType::New() ;
  try
  {
    imageReader->SetFileName(Get_FileName()) ;
    imageReader->Update() ;
    image = imageReader->GetOutput() ;
  }
  catch( itk::ExceptionObject exp )
  {
    std::cerr << "Exception caught !" << std::endl;
    std::cerr <<     exp    << std::endl;
  }

  //Make Sure background is 0 and object is 255
  IteratorType inputIt( image, image->GetRequestedRegion() );
  
  for ( inputIt.GoToBegin() ; !inputIt.IsAtEnd(); ++inputIt)
    {
      if( inputIt.Get()> 0)
	{
	  inputIt.Set(255);
	}
      else inputIt.Set(0);
    }
  
  // gaussian smoothing
  gaussFilterType::Pointer smoothFilter = gaussFilterType::New();  
  smoothFilter->SetInput(image);
  if(m_VarI)
    smoothFilter->SetVariance(m_VarianceI);
  else
  {
    const ImageType::SpacingType& sp = image->GetSpacing();
    double sp_max=sp[0];
    for(int i=1;i<3;i++)
    {
      if(sp[i]>sp[i-1])
        sp_max = sp[i];
    }
    smoothFilter->SetVariance(1.5*sp_max);
  }  
    
  smoothFilter->Update () ;
  image = smoothFilter->GetOutput () ;

  // go to vtk 
  Itk2VtkType::Pointer  m_Itk2Vtk = Itk2VtkType::New();
  
  m_Itk2Vtk->SetInput(image);  
  m_Itk2Vtk->Update();

  std::cout<<" Marching cube "<<std::endl;

  // create mesh using marching cubes
  vtkImageMarchingCubes *marcher = vtkImageMarchingCubes::New();
  marcher->SetInput(m_Itk2Vtk->GetOutput());
  marcher->SetValue(0, 128);
  marcher->ComputeScalarsOff();
  marcher->ComputeGradientsOff();
  marcher->Update() ;  

  m_polydata = marcher->GetOutput();

  // write out the vtk mesh
  std::string outfileName(Get_FileName()) ;
  int i ;
  i = outfileName.find_last_of ('.') ;
  outfileName.replace ( i+1, 4, "vtk" ) ;
  
  vtkPolyDataWriter *writer = vtkPolyDataWriter::New() ;
  writer->SetInput (m_polydata) ;
  writer->SetFileName ( outfileName.c_str() ) ;
  writer->Update () ;
  writer->Delete () ;

  // convert to itk mesh data structure
  vtkPolyDataToitkMesh *vtkItkConverter = new vtkPolyDataToitkMesh () ;
  vtkItkConverter->SetInput ( m_polydata ) ;

  // write out the itk meta mesh file
  MeshSOType::Pointer inputMeshSO = MeshSOType::New () ;
  inputMeshSO->SetMesh ( vtkItkConverter->GetOutput () ) ;
  m_inputMesh = inputMeshSO->GetMesh();  
}

void Write_Mesh::writeAttr()
{
  unsigned int i;

  std::ofstream outfile;
  std::string outfileName( Get_AttrFileName() );

  i = outfileName.find_last_of ('.') ;
  outfileName.replace ( i, 5, "_attr.txt" ) ;
  outfile.open ( outfileName.c_str() ) ;

  // output header
  outfile << "NUMBER_OF_POINTS="<<m_inputMesh->GetNumberOfPoints()<<std::endl<<"DIMENSION=1"<<std::endl<<"TYPE=Scalar"<<std::endl;
  
  for ( i = 0 ; i < m_inputMesh->GetNumberOfPoints() ; i++ )
    outfile << thick[i] << std::endl;
  outfile.close () ;
}

void Write_Mesh::generateMeshAttr()
{
  float min=99999.0, max=-99999.0;
  float interpolatedValue;
  	
  //load Thick image
  ImageType::Pointer ThickImage ;
  ReaderType::Pointer segImageReader = ReaderType::New() ;
  try
  {  
    segImageReader->SetFileName(Get_AttrFileName()) ;    
    segImageReader->Update() ;
    ThickImage = segImageReader->GetOutput() ;
  }
  catch( itk::ExceptionObject exp )
  {
    std::cerr << "Exception caughinterpolatort !" << std::endl;
    std::cerr <<     exp    << std::endl;
  }

  //Origin & Spacing
  ImageType::SpacingType  spacing = ThickImage->GetSpacing();
  ImageType::PointType origin = ThickImage->GetOrigin();

  ThicknessSmoother filter;
  filter.SetInput(ThickImage);
  if(m_VarT)
    filter.SetVarianceT(m_VarianceT);
  else      
    filter.BoolVarianceT();
  
  filter.ThickSmoother();
 
  ThickImage = filter.GetOutput();

  if (IsSaveSmoothOn())
    {
      std::string outImageName(Get_AttrFileName()) ;
      int i;
      i = outImageName.find_last_of ('.') ;
      outImageName.replace ( i, 5, "_smooth.nrrd" ) ;
      WriterType::Pointer writer = WriterType::New();
      writer->UseCompressionOn();
      writer->SetFileName(outImageName.c_str()); 
      writer->SetInput(ThickImage);
      writer->Write();
    }

  InterpolatorType::Pointer Interpolator =  InterpolatorType::New();
  Interpolator->SetInputImage(ThickImage);

  MeshType::PointsContainerPointer points = m_inputMesh->GetPoints();

  for ( unsigned int pointID = 0 ; pointID < m_inputMesh->GetNumberOfPoints()  ; pointID++ )
    {
      MeshPointType curVert =  points->GetElement(pointID);
      PointType point;
      
      point[0] = m_SignVector[0]* curVert[0];
      point[1] = m_SignVector[1]* curVert[1];
      point[2] = m_SignVector[2]* curVert[2];
      interpolatedValue = Interpolator->Evaluate(point);
      
      thick.push_back ( interpolatedValue );
      if(interpolatedValue < min) min = interpolatedValue;
      if(interpolatedValue > max) max = interpolatedValue;
    }
  std::cout << "Minimum attribute=" << min << "  " << "Maximum attribute=" << max << endl ;
}
