/*=========================================================================
  Program:   MeshInflation
  Module:    $RCSfile: ComputeInflation.cxx,v $
  Language:  C++
  Date:      $Date: 2010/06/07 15:56:36 $
  Version:   $Revision: 1.6 $
  Author:    Clement Vachet (cvachet@email.unc.edu)

  Copyright (c) Clement Vachet. All rights reserved.
  See NeuroLibCopyright.txt or http://www.ia.unc.edu/dev/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/

#include "ComputeInflation.h"

ComputeInflation::ComputeInflation(std::string _InputMesh, std::string _OutputMesh, float _Smoothing, float _MeanCurvature, int _MaxIterations, int _MaxIterationsBeforeFixing, float _MaxCurvature, bool _IsIntermediateMesh, int _IntermediateMeshIteration)
{
  SetInputMesh(_InputMesh);
  SetOutputMesh(_OutputMesh);
  SetSmoothing(_Smoothing);
  SetMeanCurvature(_MeanCurvature);
  SetMaxIterations(_MaxIterations);
  SetMaxIterationsBeforeFixing(_MaxIterationsBeforeFixing);
  SetMaxCurvature(_MaxCurvature);
  SetIsIntermediateMesh(_IsIntermediateMesh);
  SetIntermediateMeshIteration(_IntermediateMeshIteration);

  m_vNeighborVertices = NULL;
  m_vNeighborTriangles = NULL;
}

ComputeInflation::~ComputeInflation()
{
  if ( m_vNeighborVertices != NULL ) 
    {
      delete[] m_vNeighborVertices;
      m_vNeighborVertices = NULL;
    }
  if ( m_vNeighborTriangles != NULL ) 
    {
      delete[] m_vNeighborTriangles;
      m_vNeighborTriangles = NULL;
    }
}

void ComputeInflation::ReadMesh()
{
  std::cout<<"\nReading Mesh..."<<std::endl;
  
  std::ifstream Infile;
  std::string fileExtension;
  char Line[40], NbVertices[10], NbTriangles[10];
  std::string line;
  std::size_t found1, found2, length;
  TriangleType Triangle;
  CovariantVectorType Vertex1, Vertex2, Vertex3;
  int CurrentPoint, CurrentTriangle;
   
  m_vVertices.clear();
  m_vTriangles.clear();
  
  Infile.open(GetInputMesh().c_str());

  int lastPoint = GetInputMesh().rfind('.');
  fileExtension = GetInputMesh().substr(lastPoint);

  if (fileExtension.compare(".meta") == 0 )
    {
      //Skips the header and gets the number of points
      while ( strncmp (Line, "NPoints =", 9) && strncmp (Line, "NPoints=", 8))
	Infile.getline (Line, 40);    
      SetNbVertices(atoi(strrchr(Line,' ')));
      
      Infile.getline ( Line, 40);
      
      //read the points in the file and set them as vertices
      for (int i = 0; i < GetNbVertices(); i++ )
	{
	  Infile >> CurrentPoint >> Vertex1[0] >> Vertex1[1] >> Vertex1[2];
	  m_vVertices.push_back(Vertex1);
	}
      
      while ( strncmp (Line, "NCells =", 8) && strncmp (Line, "NCells=", 7))
	Infile.getline (Line, 40);    
      SetNbTriangles(atoi(strrchr(Line,' ')));
      
      Infile.getline ( Line, 40);
      
      //read the triangles in the file and set them as triangles
      for (int i = 0; i < GetNbTriangles(); i++ )
	{
	  Infile >> CurrentTriangle >> Triangle[0] >> Triangle[1] >> Triangle[2];
	  m_vTriangles.push_back(Triangle);
	}  
    }
  else if (fileExtension.compare(".vtk") == 0 )
    {
      while ( strncmp (Line, "POINTS", 6))
	Infile.getline (Line, 40);
      line = Line;
      found1 = line.find(' ');
      found2 = line.find(' ',found1+1);
      length = line.copy(NbVertices,found2-found1-1,found1+1);
      NbVertices[length]='\0';
      SetNbVertices(atoi(NbVertices));
      
      for (int i = 0; i < GetNbVertices()/3; i++ )
	{
	  Infile >> Vertex1[0] >> Vertex1[1] >> Vertex1[2] >> Vertex2[0] >> Vertex2[1] >> Vertex2[2] >> Vertex3[0] >> Vertex3[1] >> Vertex3[2];
	  m_vVertices.push_back(Vertex1);
	  m_vVertices.push_back(Vertex2);
	  m_vVertices.push_back(Vertex3);	  
	}
      if ((GetNbVertices() % 3) == 1)
	{
	  Infile >> Vertex1[0] >> Vertex1[1] >> Vertex1[2];
	  m_vVertices.push_back(Vertex1);
	}
      else if ((GetNbVertices() % 3) == 2)
	{
	  Infile >> Vertex1[0] >> Vertex1[1] >> Vertex1[2] >> Vertex2[0] >> Vertex2[1] >> Vertex2[2];
	  m_vVertices.push_back(Vertex1);
	  m_vVertices.push_back(Vertex2);
	}
      
      while ( strncmp (Line, "POLYGONS", 8))
	Infile.getline (Line, 40);
      line.clear();
      line = Line;
      found1 = line.find(' ');
      found2 = line.find(' ',found1+1);
      length = line.copy(NbTriangles,found2-found1-1,found1+1);
      NbTriangles[length]='\0';
      SetNbTriangles(atoi(NbTriangles));
      
      //read the triangles in the file and set them as triangles
      for (int i = 0; i < GetNbTriangles(); i++ )
	{
	  Infile >> CurrentTriangle >> Triangle[0] >> Triangle[1] >> Triangle[2];
	  m_vTriangles.push_back(Triangle);
	}
      
    }
  //close file
  Infile.close();
}


void ComputeInflation::WriteMesh(std::string _OutputMesh)
{
  std::cout<<"Writing mesh..."<<std::endl;
  
  std::ofstream outfile;
  std::string fileExtension;
  int i ;
  
  outfile.open(_OutputMesh.c_str());

  int lastPoint = _OutputMesh.rfind('.');
  fileExtension = _OutputMesh.substr(lastPoint);

  if (fileExtension.compare(".meta") == 0 )
    {
      // output header
      outfile<<"ObjectType = Mesh"<<std::endl<<"NDims = 3"<<std::endl<<"ID = 0"<<std::endl ;
      outfile<<"TransformMatrix = 1 0 0 0 1 0 0 0 1"<<std::endl<<"Offset = 0 0 0"<<std::endl<<"CenterOfRotation = 0 0 0"<<std::endl ;
      outfile<<"ElementSpacing = 1 1 1"<<std::endl<<"PointType = MET_FLOAT"<<std::endl<<"PointDataType = MET_FLOAT"<<std::endl ;
      outfile<<"CellDataType = MET_FLOAT"<<std::endl<<"NCellTypes = 1"<<std::endl<<"PointDim = ID x y ..."<<std::endl ;
      outfile<<"NPoints = "<<GetNbVertices()<<std::endl ;
      
      outfile<<"Points = "<<std::endl ;
      
      for ( i = 0 ; i < GetNbVertices() ; i++ )
	outfile<<i<<" "<<m_vVertices[i][0]<<" "<<m_vVertices[i][1]<<" "<<m_vVertices[i][2]<<std::endl;
      
      outfile<<std::endl<<"CellType = TRI"<<std::endl<<"NCells = "<<GetNbTriangles()<<std::endl<<"Cells = "<<std::endl ;
      
      for ( i = 0 ; i < GetNbTriangles() ; i++)
	outfile<<i<<" "<<m_vTriangles[i][0]<<" "<<m_vTriangles[i][1]<<" "<<m_vTriangles[i][2]<<std::endl;
    }
  else if (fileExtension.compare(".vtk") == 0 )
    {
      // output header
      outfile<<"# vtk DataFile Version 3.0"<<std::endl<<"vtk output"<<std::endl<<"ASCII"<<std::endl<<"DATASET POLYDATA"<<std::endl;
      outfile<<"POINTS "<<m_NbVertices<<" float"<<std::endl;
      for (int i = 0; i <= m_NbVertices-3; i+=3 )
	outfile<<m_vVertices[i][0]<<" "<<m_vVertices[i][1]<<" "<<m_vVertices[i][2]<<" "<<m_vVertices[i+1][0]<<" "<<m_vVertices[i+1][1]<<" "<<m_vVertices[i+1][2] \
	       <<" "<<m_vVertices[i+2][0]<<" "<<m_vVertices[i+2][1]<<" "<<m_vVertices[i+2][2]<<" "<<std::endl;
      if ((m_NbVertices % 3) == 1)
	outfile<<m_vVertices[m_NbVertices-1][0]<<" "<<m_vVertices[m_NbVertices-1][1]<<" "<<m_vVertices[m_NbVertices-1][2]<<" "<<std::endl;
      else if ((m_NbVertices % 3) == 2)
	outfile<<m_vVertices[m_NbVertices-2][0]<<" "<<m_vVertices[m_NbVertices-2][1]<<" "<<m_vVertices[m_NbVertices-2][2]<<" "<<m_vVertices[m_NbVertices-1][0]<<" "<<m_vVertices[m_NbVertices-1][1]<<" "<<m_vVertices[m_NbVertices-1][2]<<" "<<std::endl;
      outfile<<"POLYGONS "<<m_NbTriangles<<" "<<4*m_NbTriangles<<std::endl;
      for (int i = 0 ; i < m_NbTriangles ; i++)
	outfile<<"3 "<<m_vTriangles[i][0]<<" "<<m_vTriangles[i][1]<<" "<<m_vTriangles[i][2]<<" "<<std::endl;
      outfile<<std::endl;
    }
  outfile.close () ;
}

void ComputeInflation::WriteCurvatureAttributeFile(std::string _CurvatureAttributeFile)
{
  int i;
  std::ofstream outfile;
  
  outfile.open(_CurvatureAttributeFile.c_str());

  outfile<<"NUMBER_OF_POINTS="<<GetNbVertices()<<std::endl<<"DIMENSION=1"<<std::endl<<"TYPE=Scalar"<<std::endl;
  for ( i = 0; i < GetNbVertices(); i++)
    outfile<<m_vCurvatures[i]<<std::endl;
  outfile.close();
}

void ComputeInflation::WriteBadVerticesInfoFile()
{
  std::cout<<"Writing badVertices info file..."<<std::endl;
  
  std::ofstream outfile;
  std::string FixInfoFile = GetOutputMesh();
  size_t found;
  std::string fileExtension;

  int lastPoint = GetInputMesh().rfind('.');
  fileExtension = GetInputMesh().substr(lastPoint);

  if (fileExtension.compare(".meta") == 0 )
    {
      found = FixInfoFile.find(".meta");
      FixInfoFile.replace(found,5,"_badVertices.txt");  
    }
  else if (fileExtension.compare(".vtk") == 0 )
    {
      found = FixInfoFile.find(".vtk");
      FixInfoFile.replace(found,4,"_badVertices.txt");  
    }
  outfile.open(FixInfoFile.c_str());

  std::vector<int> BadVertices;
  for (int i = 0; i < GetNbVertices(); i++)
    {
      if (m_vCurvatures[i] > GetMaxCurvature())
	BadVertices.push_back(i);
    }

  outfile<<"NUMBER_OF_POINTS="<<BadVertices.size()<<std::endl<<"DIMENSION=1"<<std::endl<<"TYPE=Scalar"<<std::endl;
  for (unsigned int i = 0; i < BadVertices.size(); i++)
    outfile<<BadVertices[i]<<std::endl;
  outfile.close();
}

void ComputeInflation::InflateMesh()
{
  std::cout<<"Inflation..."<<std::endl;
  
  int IterationNumber = 0;
  bool StopInflation = false;
  long double CurrentMaxCurvature;

  ComputeNeighborhood();
  do
    {
      CurrentMaxCurvature = 0.0;
      
      if (IterationNumber > 0)
	UpdateMesh();
      
      ComputeMeshInfo();
      ComputeCurrentMeanCurvature();
      CurrentMaxCurvature = ComputeCurrentMaxCurvature();

      std::cout<<"\nIteration: "<<IterationNumber<<std::endl;
      std::cout<<"CurrentMeanCurvature: "<<GetCurrentMeanCurvature()<<std::endl;

      if (GetIsIntermediateMesh() && IterationNumber == GetIntermediateMeshIteration())
	{
	  size_t found;
	  std::stringstream stream;
	  std::string fileExtension;
	  stream << IterationNumber;
	  std::string Mesh = GetOutputMesh();
	  int lastPoint = GetOutputMesh().rfind('.');
	  fileExtension = GetOutputMesh().substr(lastPoint);
	  if (fileExtension.compare(".meta") == 0 )
	    {
	      found = Mesh.find(".meta");
	      Mesh.replace(found,5,"_It");
	      Mesh.append(stream.str());
	      Mesh.append(".meta");
	    }
	  else if (fileExtension.compare(".vtk") == 0 )
	    {
	      found = Mesh.find(".vtk");
	      Mesh.replace(found,4,"_It");
	      Mesh.append(stream.str());
	      Mesh.append(".vtk");
	    }
	  std::cout<<"IntermediateMesh: "<<Mesh<<std::endl;
	  WriteMesh(Mesh);
	}
      
      IterationNumber++;

      if ( (IterationNumber == GetMaxIterationsBeforeFixing()) && (CurrentMaxCurvature > GetMaxCurvature()) )
	{
	  WriteBadVerticesInfoFile();
	  StopInflation = true;
	}
      if ( (IterationNumber == GetMaxIterations()) || (GetCurrentMeanCurvature() <= GetMeanCurvature()) )
	StopInflation = true;
    }
  while (!StopInflation);
}

void ComputeInflation::ComputeNeighborhood() 
{
  int Vertex0, Vertex1, Vertex2;

  m_vNeighborVertices = new std::vector<int>[GetNbVertices()];
  m_vNeighborTriangles = new std::vector<int>[GetNbVertices()];  
  
  for (int iT=0; iT < GetNbTriangles(); iT++ ) 
    {
      Vertex0 = m_vTriangles[iT][0];
      Vertex1 = m_vTriangles[iT][1];
      Vertex2 = m_vTriangles[iT][2];
      // Filling neighbor vertices vector
      FillNeighborsVector(Vertex0, Vertex1, Vertex2);
      FillNeighborsVector(Vertex1, Vertex0, Vertex2);
      FillNeighborsVector(Vertex2, Vertex0, Vertex1);
      
      // Filling neighbor triangles vector
      m_vNeighborTriangles[Vertex0].push_back(iT);
      m_vNeighborTriangles[Vertex1].push_back(iT);
      m_vNeighborTriangles[Vertex2].push_back(iT);      
    }
}

void ComputeInflation::FillNeighborsVector(int _VertexId, int _Neighbor1, int _Neighbor2)
{
  bool IsNeighbor1, IsNeighbor2;

  IsNeighbor1 = false;
  IsNeighbor2 = false;
  for (unsigned int iN = 0 ;  iN < m_vNeighborVertices[_VertexId].size(); iN++)
    {
      if (m_vNeighborVertices[_VertexId][iN] == _Neighbor1)
	IsNeighbor1 = true;
      if (m_vNeighborVertices[_VertexId][iN] == _Neighbor2)
	IsNeighbor2 = true;
    }
  if (!IsNeighbor1)
    m_vNeighborVertices[_VertexId].push_back(_Neighbor1);
  if (!IsNeighbor2)
    m_vNeighborVertices[_VertexId].push_back(_Neighbor2);      
}

void ComputeInflation::UpdateMesh()
{
  for (int VertexId = 0; VertexId < GetNbVertices(); VertexId++)
    m_vVertices[VertexId] = m_vVertices[VertexId] * (1 - m_Smoothing) + m_vAverageVertices[VertexId] * m_Smoothing;
}

void ComputeInflation::ComputeMeshInfo()
{
  m_vCurvatures.clear();
  m_vAverageVertices.clear();
  m_vArea.clear();
  
  for (int VertexId = 0; VertexId < GetNbVertices(); VertexId++)
    {
      SVertexDataType vertexData;
      ComputeWeights(VertexId, vertexData);
      ComputeVertexInfo(VertexId, vertexData);
      ComputeCurvature(VertexId, vertexData);
      ComputeAverageVertex(VertexId, vertexData);
      m_vCurvatures.push_back(vertexData.Curvature);
      m_vAverageVertices.push_back(vertexData.AverageVertex);
      m_vArea.push_back(vertexData.dOneRingArea);
    }
}

long double ComputeInflation::ComputeCurrentMaxCurvature()
{
  long double MaxCurvature = 0.0;
  
  for (int VertexId = 0; VertexId < GetNbVertices(); VertexId++)
    {
      if (m_vCurvatures[VertexId] > MaxCurvature)
	MaxCurvature = m_vCurvatures[VertexId];
    }
  return MaxCurvature;
}

void ComputeInflation::ComputeCurrentMeanCurvature()
{
  long double MeanCurvature = 0.0;

  for (int VertexId = 0; VertexId < GetNbVertices() ; VertexId++)
    MeanCurvature += pow(m_vCurvatures[VertexId],2) * m_vArea[VertexId];
  MeanCurvature /= 4 * PI;
  MeanCurvature = sqrt(MeanCurvature);

  SetCurrentMeanCurvature(MeanCurvature);
}

void ComputeInflation::ComputeWeights(int _VertexId, SVertexDataType & _vertexData)
{
  int iT0, iT1, iCurrentNB;
  long double dWeight,dAngle0,dAngle1;
         
  int NbNeighbors = m_vNeighborVertices[_VertexId].size();
  
  for ( int iN=0; iN<NbNeighbors; iN++ ) 
    {
      iCurrentNB = m_vNeighborVertices[_VertexId][iN];
      FindTrianglesSharingEdge( _VertexId, iCurrentNB, iT0, iT1 );
      
      dAngle0 = ComputeAngleWrtSide( iT0, _VertexId, iCurrentNB );
      dAngle1 = ComputeAngleWrtSide( iT1, _VertexId, iCurrentNB );
      
      if ( dAngle0==0 || dAngle1==0 )
	dWeight = 0.0;
      else 
	dWeight = 1/tan(dAngle0) + 1/tan(dAngle1);
      _vertexData.vWeights.push_back( dWeight );
    }
}

void ComputeInflation::FindTrianglesSharingEdge( int iI, int iJ, int &iT0, int &iT1 )
{
  int iFound = 0;
  int NbNeighborTriangles = m_vNeighborTriangles[iI].size();

  for ( int iK=0; iK<NbNeighborTriangles; iK++ )
    {
      TriangleType triangle = m_vTriangles[ m_vNeighborTriangles[iI][iK] ];
      if ( triangle[0]==iJ || triangle[1]==iJ || triangle[2]==iJ )
	{
	  if ( iFound==0 ) 
	    iT0 = m_vNeighborTriangles[iI][iK];
	  if ( iFound==1 ) 
	    iT1 = m_vNeighborTriangles[iI][iK];
	  iFound++;
	}
      if ( iFound>=2 ) return;
    }
  if ( iFound<2 )
    std::cerr<<"WARNING: could not find triangles that share the edge "<<iI<<" "<<iJ<<std::endl;
}

long double ComputeInflation::ComputeAngleWrtSide( int iT, int iI, int iN )
{
  TriangleType triangle = m_vTriangles[iT];
  
  CovariantVectorType Vertex0;
  CovariantVectorType Vertex1;
  CovariantVectorType Vertex2;

  // find the point that is not on edge iI, iN and compute the angle there  
  if ( triangle[0]!=iI && triangle[0]!=iN ) 
    {
      Vertex0 = m_vVertices[ triangle[0] ];
      Vertex1 = m_vVertices[ triangle[1] ];
      Vertex2 = m_vVertices[ triangle[2] ];
    } 
  else if ( triangle[1]!=iI && triangle[1]!=iN )
    {
      Vertex0 = m_vVertices[ triangle[1] ];
      Vertex1 = m_vVertices[ triangle[2] ];
      Vertex2 = m_vVertices[ triangle[0] ];
    } 
  else if ( triangle[2]!=iI && triangle[2]!=iN )
    {
      Vertex0 = m_vVertices[ triangle[2] ];
      Vertex1 = m_vVertices[ triangle[0] ];
      Vertex2 = m_vVertices[ triangle[1] ];
    } 
  else 
    std::cerr<<"ERROR: ComputeAngleWrtSide: improperly formed triangle."<<std::endl;
  
  CovariantVectorType v0 = Vertex2-Vertex0;
  CovariantVectorType v1 = Vertex1-Vertex0;

  if (v0.GetNorm() != 0)
    v0.Normalize();
  if (v1.GetNorm() != 0)
    v1.Normalize();

  return ( acos( v0*v1 ) );
}

void ComputeInflation::ComputeVertexInfo( int iCenterId, SVertexDataType & _vertexData)
{
  long double dArea = 0.0;
  
  // go through all the triangles
  int iNrOfAdjacentTriangles = m_vNeighborTriangles[iCenterId].size();

  for ( int iI=0; iI<iNrOfAdjacentTriangles; iI++)
    {
      // get the coordinates and compute the angles
      
      TriangleType currentTriangle = m_vTriangles[ m_vNeighborTriangles[iCenterId][iI] ];
      
      CovariantVectorType Vertex0Tmp = m_vVertices[ currentTriangle[0] ];
      CovariantVectorType Vertex1Tmp = m_vVertices[ currentTriangle[1] ];
      CovariantVectorType Vertex2Tmp = m_vVertices[ currentTriangle[2] ];

      // cycle them so that Vertex0 is the point with id iCenterId
      
      CovariantVectorType Vertex0;
      CovariantVectorType Vertex1;
      CovariantVectorType Vertex2;
      
      if ( currentTriangle[0]==iCenterId )
	{
	  Vertex0 = Vertex0Tmp;
	  Vertex1 = Vertex1Tmp;
	  Vertex2 = Vertex2Tmp;
	} 
      else if ( currentTriangle[1]==iCenterId ) 
	{
	  Vertex0 = Vertex1Tmp;
	  Vertex1 = Vertex2Tmp;
	  Vertex2 = Vertex0Tmp;
	} 
      else if ( currentTriangle[2]==iCenterId ) 
	{
	  Vertex0 = Vertex2Tmp;
	  Vertex1 = Vertex0Tmp;
	  Vertex2 = Vertex1Tmp;
	} 
      else 
	std::cerr<<"ERROR: center point does not belong to triangle."<<std::endl;
    
      // now compute the angles and determine if it is an obtuse triangle or not
      
      // compute angle at the center point first
      long double dCenterAngle = ComputeAngle( Vertex2,Vertex0,Vertex1 );
      long double dAngle1 = ComputeAngle( Vertex0,Vertex1,Vertex2 );
      long double dAngle2 = ComputeAngle( Vertex1,Vertex2,Vertex0 );

      // Compute triangle area
      _vertexData.vTriangleAreas.push_back(ComputeTriangleArea(Vertex0,Vertex1,Vertex2));
      // Compute triangle circumcenter      
      _vertexData.vTriangleCenters.push_back(ComputeTriangleCenter(Vertex0,Vertex1,Vertex2));

      // let's check if the triangle is obtuse
      if ( dCenterAngle>PI/2 || dAngle1>PI/2 || dAngle2>PI/2 )
	{ // is obtuse
	  if ( dCenterAngle>PI/2 )
	    dArea+=_vertexData.vTriangleAreas[iI]/2.0;
	  else
	    dArea+=_vertexData.vTriangleAreas[iI]/4.0;
	}
      else 
	{ // not obtuse, use the Voronoi formula

	  long double dVoronoiArea = 0.0;
	  
	  CovariantVectorType v1 = Vertex2-Vertex0;
	  CovariantVectorType v2 = Vertex1-Vertex0;
	  
	  if (v1.GetNorm() == 0 && v2.GetNorm() == 0)
	    dVoronoiArea = 0.0;
	  else
	    dVoronoiArea = 1.0/8.0*( pow(v1.GetNorm(),2)/tan(dAngle1) + pow(v2.GetNorm(),2)/tan(dAngle2) );
	  dArea += dVoronoiArea;
	}
    }
  _vertexData.dOneRingArea = dArea;
}

long double ComputeInflation::ComputeAngle( CovariantVectorType Vertex2, CovariantVectorType Vertex0, CovariantVectorType Vertex1 )
{
  CovariantVectorType vec1;
  CovariantVectorType vec2;
  
  vec1 = Vertex2-Vertex0;
  vec2 = Vertex1-Vertex0;
  if (vec1.GetNorm() != 0)
    vec1.Normalize();
  if (vec2.GetNorm() != 0)
    vec2.Normalize();

  long double dAngle = acos( vec1*vec2 ); // IS THE SCALAR PRODUCT A PROBLEM HERE, BECAUSE OF THE COVARIANT VECTORS?
  return dAngle;
}

long double ComputeInflation::ComputeTriangleArea( CovariantVectorType Vertex0, CovariantVectorType Vertex1, CovariantVectorType Vertex2 )
{
  CovariantVectorType a;
  CovariantVectorType b;

  a = Vertex2-Vertex0;
  b = Vertex1-Vertex0;

  // compute the cross product
  VectorType vecCP;

  vecCP[0] = -a[2]*b[1]+a[1]*b[2];
  vecCP[1] = a[2]*b[0]-a[0]*b[2];
  vecCP[2] = -a[1]*b[0]+a[0]*b[1];

  return (vecCP.GetNorm()/2.0);
}

CovariantVectorType ComputeInflation::ComputeTriangleCenter(CovariantVectorType Vertex0, CovariantVectorType Vertex1, CovariantVectorType Vertex2)
{
  CovariantVectorType Center;
  Center = (Vertex0 + Vertex1 + Vertex2) / 3.0;
  return Center;
}

void ComputeInflation::ComputeCurvature(int _VertexId, SVertexDataType & _vertexData)
{
  long double Curvature;
  CovariantVectorType VectorDiff, CurvatureVector;

  int NbNeighbors = m_vNeighborVertices[_VertexId].size();
  
  CurvatureVector[0]=0.0;
  CurvatureVector[1]=0.0;
  CurvatureVector[2]=0.0;
  for ( int iN=0; iN<NbNeighbors; iN++ ) 
    {
      VectorDiff = _vertexData.vWeights[iN]* (m_vVertices[_VertexId] - m_vVertices[m_vNeighborVertices[_VertexId][iN]]);
      CurvatureVector = CurvatureVector + VectorDiff;
    }
  CurvatureVector = CurvatureVector / (2*_vertexData.dOneRingArea);
  Curvature = 0.5* CurvatureVector.GetNorm();

  _vertexData.Curvature = Curvature;
}

void ComputeInflation::ComputeAverageVertex(int _VertexId, SVertexDataType & _vertexData)
{
  int NbNeighborTriangles = m_vNeighborTriangles[_VertexId].size();
  long double TotalArea = 0.0;
  CovariantVectorType TotalVertex;
  TotalVertex[0] = 0.0;
  TotalVertex[1] = 0.0;
  TotalVertex[2] = 0.0;  
  for (int iNT=0; iNT<NbNeighborTriangles; iNT++)
    {
      TotalArea += _vertexData.vTriangleAreas[iNT];
      TotalVertex += _vertexData.vTriangleCenters[iNT] * _vertexData.vTriangleAreas[iNT];
    } 
  _vertexData.AverageVertex =  TotalVertex / TotalArea;
}

