/*=========================================================================
  Program:   MeshInflation
  Module:    $RCSfile: ComputeInflation.h,v $
  Language:  C++
  Date:      $Date: 2010/06/07 15:56:36 $
  Version:   $Revision: 1.3 $
  Author:    Clement Vachet (cvachet@email.unc.edu)

  Copyright (c) Clement Vachet. All rights reserved.
  See NeuroLibCopyright.txt or http://www.ia.unc.edu/dev/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/

#ifndef COMPUTEINFLATION_H
#define COMPUTEINFLATION_H

#include <iostream>
#include <cstring>
#include <fstream>
#include <vector>
#include <math.h>

// ITK includes
#include <itkCovariantVector.h>
#include <itkVector.h>
#include <itkArray.h>

static const long double PI = 3.14159265358979323846;
typedef itk::CovariantVector<long double,3> CovariantVectorType;
typedef itk::Vector<int,3> TriangleType;
typedef itk::Vector<long double,3> VectorType;

class ComputeInflation
{
 public:
  ComputeInflation(std::string _InputMesh, std::string _OutputMesh, float _Smoothing, float _MeanCurvature, int _MaxIterations, int _MaxIterationsBeforeFixing, float _MaxCurvature, bool _IsIntermediateMesh, int _IntermediateMeshIteration);
  ~ComputeInflation();
  void ReadMesh();
  void InflateMesh();
  void WriteMesh(std::string _OutputMesh);
  void WriteCurvatureAttributeFile(std::string _CurvatureVectorFile);  

 private:  
  
  // IO
  void SetInputMesh(std::string _InputMesh){m_InputMesh = _InputMesh;};
  void SetOutputMesh(std::string _OutputMesh){m_OutputMesh = _OutputMesh;};
  void SetSmoothing(float _Smoothing){m_Smoothing = _Smoothing;};
  void SetMeanCurvature(float _MeanCurvature){m_MeanCurvature = _MeanCurvature;};
  void SetMaxIterations(int _MaxIterations){m_MaxIterations = _MaxIterations;};
  void SetMaxIterationsBeforeFixing(int _MaxIterationsBeforeFixing){m_MaxIterationsBeforeFixing = _MaxIterationsBeforeFixing;};
  void SetMaxCurvature(float _MaxCurvature){m_MaxCurvature = _MaxCurvature;};
  void SetIsIntermediateMesh(bool _IsIntermediateMesh){m_IsIntermediateMesh = _IsIntermediateMesh;};
  void SetIntermediateMeshIteration(int _IntermediateMeshIteration){m_IntermediateMeshIteration = _IntermediateMeshIteration;};

  std::string GetInputMesh(){return m_InputMesh;};
  std::string GetOutputMesh(){return m_OutputMesh;};
  float GetSmoothing(){return m_Smoothing;};
  float GetMeanCurvature(){return m_MeanCurvature;};
  int GetMaxIterations(){return m_MaxIterations;};
  int GetMaxIterationsBeforeFixing(){return m_MaxIterationsBeforeFixing;};
  float GetMaxCurvature(){return m_MaxCurvature;};
  bool GetIsIntermediateMesh(){return m_IsIntermediateMesh;};
  int GetIntermediateMeshIteration(){return m_IntermediateMeshIteration;};

  // 
  void SetNbVertices(int _NbVertices){m_NbVertices = _NbVertices;};
  int GetNbVertices(){return m_NbVertices;};
  void SetNbTriangles(int _NbTriangles){m_NbTriangles = _NbTriangles;};
  int GetNbTriangles(){return m_NbTriangles;};

  // Compute general neighborhood
  void ComputeNeighborhood();
  void FillNeighborsVector(int _VertexId, int _Neighbor1, int _Neighbor2);
  
  // Compute mesh information
  void UpdateMesh();
  void ComputeMeshInfo();
  long double ComputeCurrentMaxCurvature();
  void ComputeCurrentMeanCurvature();
  
  void SetCurrentMeanCurvature(long double _CurrentMeanCurvature){m_CurrentMeanCurvature = _CurrentMeanCurvature;};
  long double GetCurrentMeanCurvature(){return m_CurrentMeanCurvature;};
  
  // Mesh Fixing
  void WriteBadVerticesInfoFile();
  
  // Compute vertex information
  struct SVertexDataType 
  {
    std::vector<long double> vWeights;
    std::vector<long double> vTriangleAreas;
    std::vector<CovariantVectorType> vTriangleCenters;
    long double dOneRingArea;
    long double Curvature;
    CovariantVectorType AverageVertex;
  };

  void ComputeWeights(int _VertexId, SVertexDataType & _vertexData);
  void ComputeVertexInfo(int iCenterId, SVertexDataType & _vertexData);
  void ComputeCurvature(int _VertexId, SVertexDataType & _vertexData);
  void ComputeAverageVertex(int _VertexId, SVertexDataType & _vertexData);
  void FindTrianglesSharingEdge(int iI, int iJ, int &iT0, int &iT1 );
  long double ComputeAngleWrtSide(int iT, int iI, int iN );
  long double ComputeAngle(CovariantVectorType Vertex2, CovariantVectorType Vertex0, CovariantVectorType Vertex1);
  long double ComputeTriangleArea(CovariantVectorType Vertex0, CovariantVectorType Vertex1, CovariantVectorType Vertex2);
  CovariantVectorType ComputeTriangleCenter(CovariantVectorType Vertex0, CovariantVectorType Vertex1, CovariantVectorType Vertex2);
  
  // IO
  std::string m_InputMesh;
  std::string m_OutputMesh;
  float m_Smoothing;
  float m_MeanCurvature;
  int m_MaxIterations;
  int m_MaxIterationsBeforeFixing;
  float m_MaxCurvature;
  bool m_IsIntermediateMesh;
  int m_IntermediateMeshIteration;

  int m_NbVertices;
  int m_NbTriangles;

  std::vector<CovariantVectorType> m_vVertices;
  std::vector<TriangleType> m_vTriangles;  
  
  // general neighborhood
  std::vector<int> *m_vNeighborVertices;
  std::vector<int> *m_vNeighborTriangles;
  
  // Mesh information
  std::vector<CovariantVectorType> m_vAverageVertices;
  std::vector<long double> m_vCurvatures;
  std::vector<long double> m_vArea;
  long double m_CurrentMeanCurvature;
};

#endif
