function [Y_band, FX_band, conf_band, band_all, coord_ROIs, img_dim] = ssfmri_band(imgnames, Xtime, confounds, bands, ROIm, brainmask)
%
% [Y_band, FX_band, conf_band, band_all, coord_ROIs, img_dim]  
%      = SSFMRI_BAND(imgnames, Xtime, confounds, bands, ROIm, brainmask)
%
% This function is called in ssfmri_ancova. It takes everything 
% (seed, imag voxel time-series, confounds) into freq_space.
%
%-------------------------------------------------------------------------
% Input parameters
% 
% imgnames - cell array of filenames
% Xtime - vector double of seed timecourse
% counfounds - matrix of confounds [time_pts x num_confounds]
% bands - a cell array of desired frequency bands (eg 2:49)
% ROIm - struct containing information about ROIs: must have either:
%       .Mask - a cell array of masks (num ROIs implied by num masks)
%   -or-
%       .Label - a labeled volume 
%       AND
%       .LabelVal - a vector/array of label values in which you are 
%           interested (from the set available on the labled volume)
% brainmask - a  brainmask that gets AND'd with the ROI's to make sure 
%   everything is in the brain
% 
% -------------------------------------------------------------------------
% Output parameters
% 
% Y_band - cell array (num_ROI) of matricies 
%       [num_vox_in_ROI x num_freqs_selected]
% FX_band - matrix of desired frequecies [num_freqs x size(Xtime,2)]
% conf_band - matrix of desired frequecies [num_freqs x size(confounds,2)]
% band_all - concated array of indicies of all freqs kept
% coord_ROIs - cell array (num_ROI) of matricies: x,y,z coordinates for each 
%       voxel in that ROI [num_vox_in_ROI x 3] 
% img_dim - size of the input for each timepoint 
%       (either 2 for flat slices, or 3 for 3d volumes) 
% 
%--------------------------------------------------------------------------


% imgnames is cell array
imgstrs = spm_vol(imgnames);
num_img = length(imgnames);
img_data = cell(num_img,1);

% get or create set (num-used-labels long) of logical label masks 
if isfield(ROIm,'Mask')
    roi_str = spm_vol(ROIm.Mask);
    num_ROI = length(ROIm.Mask);
    roi_maskimg = cell(num_ROI,1);
    for i = 1:num_ROI
        roi_maskimg{i} = spm_read_vols(roi_str{i});
    end
elseif isfield(ROIm,'Label') && isfield(ROIm,'LabelVal') 
    label_str = spm_vol(ROIm.Label); % label hdr info
    label_img = spm_read_vols(label_str); % actual label vol
    num_ROI = length(ROIm.LabelVal); % this is the set of labels that we want to use
    roi_maskimg = cell(num_ROI,1);
    for i = 1:num_ROI
        roi_maskimg{i} = (label_img == ROIm.LabelVal(i)); % create num_used_label logical maps
    end
else
    error(['In ' mfilename ': Need either .Mask -or- .Label AND .LabelVal!']); 
end

% read all 3d vols 
for i = 1:num_img
    img_data{i} = spm_read_vols(imgstrs{i}); % the 4-D set of images
end

img_dim = size(img_data{1});

if ~isempty(brainmask)
    brain_str = spm_vol(brainmask);
    brain_img = spm_read_vols(brain_str);
    for i = 1:num_ROI
        roi_maskimg{i} = roi_maskimg{i} & brain_img; % this should be sorta redundant... but you never know
    end
end

%% Find coordinates of ROI-specific voxel, for each ROI
coord_ROIs = cell(num_ROI,1);
num_voxROI = zeros(num_ROI,1);
dimROI = size(roi_maskimg{1});
ndim = numel(dimROI);
for clust = 1:num_ROI
    num_voxROI(clust) = length(find(roi_maskimg{clust}>0));
    coord_ROIs{clust} = zeros(num_voxROI(clust),ndim);

    if ndim == 2 
        [tmpR tmpC] = find(roi_maskimg{clust}>0);
        coord_ROIs{clust} = [tmpR tmpC];
    elseif ndim == 3
        [tmpR tmpC tmpS] = ind2sub(size(roi_maskimg{clust}),find(roi_maskimg{clust}>0));
        coord_ROIs{clust} = [tmpR tmpC tmpS];
    else
        error('error ROI dimension'); 
    end
end

%% Fast Fourier Transformation
num_event = size(Xtime,2); % width of X mat
FX = zeros(num_img,num_event); % FX = X in freq domain

for i=1:num_event
    FX(:,i)=(1/sqrt(num_img))*fft(Xtime(:,i)); % take each time-course (columns of X) into frequency domain.
%     FX(:,i)=(1/sqrt(num_img))*fft(tempX(2:end,i));
%     FX(:,i)=fftshift(FX(:,i));
end

% if applicable, take each row of confounds into freq domain
if ~isempty(confounds)
    fx_conf = zeros(size(confounds));
    for i = 1:size(confounds,2)
        % MATLAB does not scale the output of fft by the length of input
        % in MATLAB: F(u) = sum(f(x)exp(-j*2*(pi)*x*u/N), x = 0:N-1
        fx_conf(:,i) = (1/sqrt(num_img))*fft(confounds(:,i)); % from frequency 0:(1-1/128)*fs
%         fx_conf(:,i) = fftshift(fx_conf(:,i)); % from frequency -1/2*fs:(1/2-1/128)*fs
    end
end

ROI_freq = cell(num_ROI,1);
ROI_time = cell(num_ROI,1);
% thred = 0.1*max(img_data{1}(:));
% meancon = mean(img_data{1}(img_data{1}>thred));

% For each ROI (indexed by clust), ROI_time gets the data from that ROI accross all time pts
% (num_voxels tall x time_pts accross)
for clust = 1:num_ROI
    ROI_time{clust} = zeros(num_voxROI(clust),num_img);
    for i = 1:num_img
        ROI_time{clust}(:,i) = img_data{i}(roi_maskimg{clust}>0);
        % ROI_time{clust} is matrix with size of num_voxles_inROI x num_img
    end
%     ROI_time{clust} = ROI_time{clust} - meancon;
end


% For each ROI, take the voxel time-series (across ROI_time rows) into freq domain
for clust = 1:num_ROI
    tempY1 = ROI_time{clust};
    fy1 = zeros(size(ROI_time{clust}));
    for i=1:num_voxROI(clust)
%         tempY1(i,:) = tempY1(i,:) - mean(tempY1(i,:));
        fy1(i,:) = (1/sqrt(num_img))*fft(tempY1(i,:)); % from frequency 0:(1-1/128)*fs
%         fy1(i,:)=fftshift(fy1(i,:)); % from frequency -1/2*fs:(1/2-1/128)*fs
    end

    ROI_freq{clust} = fy1;

end


%% Bands of interest

num_band = length(bands);
% band_loc = cell(num_band,1);
band_all = [];
% 
% all_loc = fftshift(1:num_img);
for i = 1:num_band
    band_all = [band_all bands{i}]; %#ok<AGROW>
% %     bands{i}(bands{i}==1) = []; % the first band is the constant. Not interested in the intercept
% %     band_loc{i} = floor(num_img/2) + bands{i}; % because dft is symetric
%     band_loc1 = find(all_loc==bands{i}(1));
%     band_loc2 = find(all_loc==bands{i}(end));
%     % may use full frequency
%     if bands{i}(end) > all_loc(end) && bands{i}(1) < all_loc(end)      
%         band_locmed = length(all_loc);
%         band_loc{i} = [band_loc1:band_locmed, 1:band_loc2];
%     else
%         band_loc{i} = band_loc1:band_loc2;
%     end
%     band_all = [band_all, band_loc{i}];
end
% band_all_x = band_all;

%% Extracting fourier coefficients corresponding to the bands of interest
Y_band = cell(1,num_ROI);
FX_band = FX(band_all,:); % this is the row direction for FX

if ~isempty(confounds)
    conf_band = fx_conf(band_all,:);
else
    conf_band = [];
end

for clust = 1:num_ROI
    tempY1 = ROI_freq{clust};
    Y_band{clust} = tempY1(:,band_all); % this is the column direction for ROI_freq
end

return
