package edu.jhmi.rad.medic.algorithms;

import edu.jhmi.rad.medic.dialogs.*;
import edu.jhmi.rad.medic.methods.*;
import edu.jhmi.rad.medic.utilities.*;

import gov.nih.mipav.model.algorithms.AlgorithmBase;
import gov.nih.mipav.model.structures.*;
import gov.nih.mipav.model.structures.jama.*;
import gov.nih.mipav.model.file.FileInfoBase;
import gov.nih.mipav.view.*;
import gov.nih.mipav.view.dialogs.*;

import java.io.*;
import java.util.*;
import java.lang.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;


/**
 *	TOADS: TOpology-preserving Anatomy-Driven Segmentation.
 *	<p>
 *	This is the main version of the segmentation. It includes inhomogeneity correction,
 *	template registration and adaptive coefficients.
 *
 *	@version    April 2006
 *	@author     Pierre-Louis Bazin
 *  @see 		JDialogToads
 *	@see 		Toads
 *	@see		RegisterTemplate
 *
*/
public class AlgorithmDOTS extends AlgorithmBase {

    // Fuzzy images require 1 image for each class
    // Hard images 1 image with assigned classes
    private ModelImage  		destImage[];
    private ModelImage  		srcImage[];
    private ModelImage  		lbImage;
    private String      		resultOutput;
    private ViewUserInterface   userInterface;
		
    // image size
	private int 		nx,ny,nz,nc; 		// original image dimensions
	private int 		ntx,nty,ntz; 	// original template dimensions
	private	float		rx,ry,rz;		// image resolutions
	private	float		rtx,rty,rtz;	// template resolutions
	private int			orient;			// image orientation
	private int			orx,ory,orz;	// image axis orientation
	private	String[]	modality;		// image modality
	private	String[]	psfType;		// image psf type
	private	String		algorithmMode;		// variants of the algorithm
	private	String		connectivityType;		// choice of connectivity
	private	String		labelType;		// choice of external labeling maps
	
    // segmentation parameters
    private	String				atlasName;
    private	DotsAtlas		atlas=null;
    private byte[]				objType;

	private float		isoDiffusion;
	private float		gainFactor;
	
	private int 		iterations;
	private float   	maxDistance;
	private	int			nbest;
	
	private	String		alignType = "rigid";
	private	int			levels;
	private	int			initAlignIter = 10;
	private	int			alignIter = 1;
	private	float		alignScale;
	
    private static final	boolean		verbose = true;
	private static final	boolean		debug = true;
	
    /**
    *	Constructor for 3D images in which changes are placed in a predetermined destination image.
    *   @param destImg_      Image model where result image is to stored.
    *   @param srcImg_       Source image model.
    */
	public AlgorithmDOTS(ModelImage[] srcImg_, int nInput_,
											String[] imgModal_,
											String aName_, DotsAtlas atlas_, String segOutput_, 
											ModelImage labeling_, String lbType_,
											float isoDiff_, float gainF_,
											int iterMax_, float distMax_, int nbest_, 
											String alignT_,
											float aScale_,
											int lev_, int initR_, int iterR_) {
        super(null, srcImg_[0]);
		userInterface = ViewUserInterface.getReference();
		srcImage = srcImg_;
        nc = nInput_;
		modality = imgModal_;
		
		atlasName = aName_;
		atlas = atlas_;
		
		resultOutput = segOutput_;
		
		lbImage = labeling_;
		labelType = lbType_;
        
		iterations = iterMax_;
		maxDistance = distMax_;
		
		isoDiffusion = isoDiff_;
		if (isoDiffusion==-1) {
			isoDiffusion = 1.0f/atlas.getNumber();
		}
		
		gainFactor = gainF_;
		if (gainFactor==-1) {
			gainFactor = 1.0f/atlas.getNumber();
		}
		
		nbest = nbest_;
		
		levels = lev_;
		alignIter = iterR_;		
		initAlignIter = initR_;
		
		alignType = alignT_;
		
		alignScale = aScale_;
	}

    /**
    *	Prepares this class for destruction.
    */
	public void finalize(){
	    destImage   = null;
	    srcImage    = null;
        super.finalize();
        System.gc();
	}
	
    /**
    *   Starts the segmentation.
    */
	public void runAlgorithm() {

        if (srcImage  == null) {
            displayError("Source Image is null");
            return;
        }
        
		// start the timer to compute the elapsed time
        setStartTime();

		// assume a single image for now		
		float[] buffer = null;
		short[] lbuffer = null;
		float[] lbimg = null;
		String modal = null;
		for (int c=0;c<nc;c++) {
			if (srcImage[c].getNDims()!=4) {
				displayError("Source Image is not 4D");
				return;
			}
		}
		
        try {
			for (int c=0;c<nc;c++) {
				// image length is length in 3 dims
				int length = srcImage[c].getExtents()[0]*srcImage[c].getExtents()[1]*srcImage[c].getExtents()[2]*srcImage[c].getExtents()[3];
				
				// retrieve data
				if (modality[c].equals("CFARI_directions_(dxN)")) {
					lbuffer = new short[length];
					srcImage[c].exportData(0, length, lbuffer);
				} else {
					buffer = new float[length]; 
					srcImage[c].exportData(0,length, buffer); // locks and releases lock
					modal = modality[c];
				}
			}
			if (!labelType.equals("none")) {
				// image length is length in 3 dims
				int length = lbImage.getExtents()[0]*lbImage.getExtents()[1]*lbImage.getExtents()[2];
				
				// retrieve data
				lbimg = new float[length]; 
				lbImage.exportData(0,length, lbimg); // locks and releases lock
			}
		} catch (IOException error) {
			buffer = null;
			lbimg = null;
			lbuffer = null;
			errorCleanUp("Algorithm: source image locked", true);
			return;
		} catch (OutOfMemoryError e){
			buffer = null;
			lbimg = null;
			lbuffer = null;
			errorCleanUp("Algorithm: Out of memory creating process buffer", true);
			return;
		}
	
		// init dimensions, resolutions, orientations
		nx = srcImage[0].getExtents()[0];
		ny = srcImage[0].getExtents()[1];
		nz = srcImage[0].getExtents()[2];
			
		ntx = atlas.getImageDim()[0];
		nty = atlas.getImageDim()[1];
		ntz = atlas.getImageDim()[2];
		
		rx = srcImage[0].getFileInfo()[0].getResolutions()[0];
		ry = srcImage[0].getFileInfo()[0].getResolutions()[1];
		rz = srcImage[0].getFileInfo()[0].getResolutions()[2];
		
		// check for non-mm units
		if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[0]==FileInfoBase.CENTIMETERS) rx = 10.0f*rx;
		if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[1]==FileInfoBase.CENTIMETERS) ry = 10.0f*ry;
		if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[2]==FileInfoBase.CENTIMETERS) rz = 10.0f*rz;
		
		rtx = atlas.getImageRes()[0];
		rty = atlas.getImageRes()[1];
		rtz = atlas.getImageRes()[2];
		
		orient = srcImage[0].getImageOrientation();
		orx = srcImage[0].getFileInfo()[0].getAxisOrientation(0);
		ory = srcImage[0].getFileInfo()[0].getAxisOrientation(1);
		orz = srcImage[0].getFileInfo()[0].getAxisOrientation(2);
		
 		//---------------------------------------/
		//-          PRE-PROCESSING             -/
		//---------------------------------------/
		
		if (verbose) 	MedicUtilPublic.displayMessage("DTI Tract Segmentation\n");
		if (debug) 		MedicUtilPublic.displayMessage(getParameterText());
		
		// record time
		long start_time = System.currentTimeMillis();

		fireProgressStateChanged(srcImage[0].getImageName(), "Initialization");
		fireProgressStateChanged(10);
		if (verbose) MedicUtilPublic.displayMessage("Initialization\n");
		
		int nw=0;
		if (modality[0].startsWith("CFARI")) {
			nw = srcImage[0].getExtents()[3];
		}
		
		DotsPreprocess preprocess = new DotsPreprocess(buffer, lbuffer, nw, nx, ny, nz, rx, ry, rz,
															orient, orx, ory, orz,
															atlas.getTemplate(), ntx, nty, ntz, rtx, rty, rtz,
															lbimg,
															(byte)1, 0.0f, 5, modal);
		
		buffer = null;
		if (debug) MedicUtilPublic.displayMessage(preprocess.displayTransform());
		
		// find unused regions
		preprocess.findCroppingBoundaries();
		// crop images and atlas
		preprocess.cropImages();
		if (debug) System.out.println("cropped: "+preprocess.displayTransform());
		int[] dim = preprocess.getCurrentImageDimensions();
		nx = dim[0]; ny = dim[1]; nz = dim[2];
		float[] res = preprocess.getSignedImageResolutions();
		rx = res[0]; ry = res[1]; rz = res[2];
		
		fireProgressStateChanged(20);
			
		// align the atlas using FA
		if (verbose) MedicUtilPublic.displayMessage("Atlas alignment\n");
		
		// align both images centers ? not very efficient
		if (debug) System.out.println("image dimensions "+nx+" "+ny+" "+nz);
		atlas.setImageInfo(nx,ny,nz,rx,ry,rz,orient,orx,ory,orz);
		
		
		if (!alignType.equals("none")) atlas.initShapeRegistration(preprocess.getFAmap(), "rigid", initAlignIter, levels);	
		else atlas.initShapeRegistration(preprocess.getFAmap(), "none", initAlignIter, levels);
		atlas.setTransform(preprocess.getTransform());
		
		if (debug) System.out.println("pre-process: "+preprocess.displayTransform());
		
		if (!alignType.equals("none")) atlas.updateShapeRegistration(preprocess.getFAmap(), "rigid", initAlignIter, levels);
		if (debug) System.out.println("atlas: "+atlas.displayTransform(atlas.getTransform()));
		
		fireProgressStateChanged(30);
		
		if (!alignType.equals("none"))
			atlas.registerShapesPyramid();
		
		// add a last step ? not needed, can be nice as an option
		//atlas.updateShapeRegistration(preprocess.getFAmap(), alignType, alignIter, 1);
		if (debug) System.out.println("atlas: "+atlas.displayTransform(atlas.getTransform()));
		
		fireProgressStateChanged(50);
		
		//atlas.registerShapes();
		
		if (debug) System.out.println("aligned atlas: "+atlas.displayTransform(atlas.getTransform()));
		if (debug) System.out.println("aligned pre-process: "+preprocess.displayTransform());
		
		preprocess.uncropImages();
		preprocess.findCroppingBoundaries();
		preprocess.cropImages();

		if (debug) System.out.println("final pre-process: "+preprocess.displayTransform());
		if (debug) System.out.println("final aligned atlas: "+atlas.displayTransform(atlas.getTransform()));
		
		dim = preprocess.getCurrentImageDimensions();
		nx = dim[0]; ny = dim[1]; nz = dim[2];
		atlas.setImageInfo(nx,ny,nz,rx,ry,rz,orient,orx,ory,orz);
		
		atlas.computeTransformedTemplate();
		preprocess.updateTransformedTemplate(atlas.getTemplate());
		
		fireProgressStateChanged(60);
			
		// create the segmentation algorithm
		DotsPropagation segmentation = new DotsPropagation(preprocess.getModality(), preprocess.getDirections(), preprocess.getWeights(),
																	preprocess.getMDmap(),
																	preprocess.getMask(), atlas,
																	preprocess.getLabeling(), labelType,
																	nbest, 
																	nx, ny, nz, preprocess.getDirectionNumber(),
																	rx, ry, rz,
																	isoDiffusion, 
																	gainFactor);

		// get the centroids from initialization
		//segmentation.computeInitialCentroids();
		
		// re-initialize the atlas alignment
		atlas.updateShapeRegistration(segmentation.getGain(), segmentation.getLabels(), nbest, alignType, alignIter, 1, 1.0f, alignScale);
		MedicUtilPublic.displayMessage("aligned atlas: "+atlas.displayTransform(atlas.getTransform()));
			
		//MedicUtilPublic.displayMessage("atlas tuning\n");
		//atlas.changeFiberShapePriors(otherPrior);	
		//atlas.normalizeShapePriors();
		
		fireProgressStateChanged(70);
			
		if (debug) System.out.print("pre-processing time: " + (System.currentTimeMillis()-start_time)+"\n"); 
		
		fireProgressStateChanged(80);
					
		fireProgressStateChanged(90);
			
		// memberships & energy
		//float diff = segmentation.computeTRBPMessages();
		//MedicUtilPublic.displayMessage("initial difference : "+diff+"\n");
		segmentation.computeAtlasPriors();
		//if (algoType.equals("membership-diffusion")) segmentation.gainIntoMemberships();
		if (!alignType.equals("none"))
				atlas.registerShapes();
		
		if (debug) System.out.print("membership initialization time: " + (System.currentTimeMillis()-start_time)+"\n"); 

		// 3. main loop
		//--------------------------------------------------------------------
		MedicUtilPublic.displayMessage("Main computation loop\n");
		boolean stop = false;
		int iter = 0;
		int deg = 0;
		float distance = 1.0f;
		if (iterations==0) stop=true;
		while ((!stop) && (!threadStopped)) {
			long iterationTime=0;
			if (debug) iterationTime = System.currentTimeMillis();
				
			iter++;
			MedicUtilPublic.displayMessage("iteration "+iter+"\n");
			
			// b. update parameter information
			//------------------------------------------------------------
				
			// memberships & energy
			distance = segmentation.computeIterativeGainMaxDiffusion();
			fireProgressStateChanged(30);
			
			if (debug) System.out.print("membership time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			if (debug) iterationTime = System.currentTimeMillis();
			
			// shape alignment
			if (!alignType.equals("none"))
				atlas.registerShapes();
			
			if (debug) System.out.print("alignment time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			if (debug) iterationTime = System.currentTimeMillis();
			fireProgressStateChanged(60);

			// check for convergence
			if (distance < maxDistance) stop = true;
			MedicUtilPublic.displayMessage("distance: "+distance+"\n");
			
			if (iter >= iterations) stop = true;
				
			fireProgressStateChanged(90);
					
			if (debug) System.out.print("update time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			if (debug) iterationTime = System.currentTimeMillis();
		}
		
		//segmentation.cleanUp();
		
		MedicUtilPublic.displayMessage("Preparing output images\n");
		
		// debug
		MedicUtilPublic.displayMessage("total processing time (milliseconds): " + (System.currentTimeMillis()-start_time)); 
		
		fireProgressStateChanged("creating result images...");
		fireProgressStateChanged(30);
				
		try {
			if (resultOutput.equals("hard_segmentation")) {
				// create the output array
				destImage = new ModelImage[2];
			
				// hard segmentation
				destImage[0] = new ModelImage(ModelStorageBase.SHORT, preprocess.getOriginalImageDimensions(),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_class"));				

				short[] shortbuffer = preprocess.uncropAndBuffer(segmentation.exportLabelClassification(),(short)0);
				destImage[0].importData(0, shortbuffer, true);
				shortbuffer = null;
				
				// overlapping segmentations
				destImage[1] = new ModelImage(ModelStorageBase.BOOLEAN, preprocess.getOriginalImageArrayDimensions(atlas.getNumber()),
											JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_seg"));
				
				boolean[] boolbuffer = preprocess.uncropAndBuffer(segmentation.exportSegmentations(),atlas.getNumber(),false);
				destImage[1].importData(0, boolbuffer, true);
				boolbuffer = null;
			} else if  (resultOutput.equals("full_segmentation")) {
				// create the output array
				destImage = new ModelImage[3];
			
				// hard segmentation
				destImage[0] = new ModelImage(ModelStorageBase.SHORT, preprocess.getOriginalImageDimensions(),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_class"));				

				//short[] shortbuffer = preprocess.uncropAndBuffer(segmentation.exportLabelClassification(),(short)0);
				short[] shortbuffer = preprocess.uncropAndBuffer(segmentation.exportCrosshatchLabelClassification(),(short)0);
				destImage[0].importData(0, shortbuffer, true);
				shortbuffer = null;
				
				// overlapping segmentations
				destImage[1] = new ModelImage(ModelStorageBase.BOOLEAN, preprocess.getOriginalImageArrayDimensions(atlas.getNumber()),
											JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_seg"));
				
				boolean[] boolbuffer = preprocess.uncropAndBuffer(segmentation.exportSegmentations(),atlas.getNumber(),false);
				destImage[1].importData(0, boolbuffer, true);
				boolbuffer = null;
				
				// memberships : only inside each structure (=> max membership)
				destImage[2] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_maxmem"));
				buffer = preprocess.uncropAndBuffer(segmentation.exportMaxMemberships(),0.0f);
				destImage[2].importData(0, buffer, true);
				buffer = null;
				
			} else if  (resultOutput.equals("contrasts")) {
				// create the output array
				destImage = new ModelImage[2];
			
				// diffusion weights
				destImage[0] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(preprocess.getDirectionNumber()),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_weights"));
				
				buffer = preprocess.uncropAndBuffer(preprocess.getWeights(),preprocess.getDirectionNumber(),0.0f);
				destImage[0].importData(0, buffer, true);
				
				// diffusion directions
				destImage[1] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(3*preprocess.getDirectionNumber()),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_dirs"));
				
				buffer = preprocess.uncropAndBuffer(preprocess.getDirections(),3*preprocess.getDirectionNumber(),0.0f);
				destImage[1].importData(0, buffer, true);					
				buffer = null;
				
			} else if  (resultOutput.equals("all_images")) {
				// create the output array
				destImage = new ModelImage[7];
			
				// hard segmentation
				destImage[0] = new ModelImage(ModelStorageBase.SHORT, preprocess.getOriginalImageDimensions(),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_class"));				

				short[] shortbuffer = preprocess.uncropAndBuffer(segmentation.exportLabelClassification(),(short)0);
				destImage[0].importData(0, shortbuffer, true);
				shortbuffer = null;
				
				// probabilities
				destImage[1] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(atlas.getNumber()),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_mems"));
				buffer = preprocess.uncropAndBuffer(segmentation.exportMemberships(),atlas.getNumber(),0.0f);
				destImage[1].importData(0, buffer, true);
				buffer = null;
				
				// initial factor
				destImage[2] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(atlas.getNumber()),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_init"));
				buffer = preprocess.uncropAndBuffer(segmentation.exportInitialFactor(),atlas.getNumber(),0.0f);
				destImage[2].importData(0, buffer, true);
				
				// main direction
				destImage[3] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(6),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_diffusion"));
				
				buffer = preprocess.uncropAndBuffer(segmentation.exportDiffusionMaps(),6,0.0f);
				destImage[3].importData(0, buffer, true);		

				// diffusion anisotropy
				destImage[4] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(3),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_ani"));
				
				buffer = preprocess.uncropAndBuffer(segmentation.exportAnisotropyFactor(),3,0.0f);
				destImage[4].importData(0, buffer, true);		
				
				// prior seg
				destImage[5] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
											JDialogBase.makeImageName(srcImage[0].getImageName(), "_DTS_priorseg"));
				
				byte[] bytebuffer = preprocess.uncropAndBuffer(segmentation.generatePriorClassification(),(byte)0);
				destImage[5].importData(0, bytebuffer, true);
				
				// overlapping segmentations
				destImage[6] = new ModelImage(ModelStorageBase.BOOLEAN, preprocess.getOriginalImageArrayDimensions(atlas.getNumber()),
											JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_seg"));
				
				boolean[] boolbuffer = preprocess.uncropAndBuffer(segmentation.exportSegmentations(),atlas.getNumber(),false);
				destImage[6].importData(0, boolbuffer, true);
				boolbuffer = null;
			} else if  (resultOutput.equals("energy_terms")) {
				// create the output array
				destImage = new ModelImage[7];
			
				// hard segmentation
				destImage[0] = new ModelImage(ModelStorageBase.SHORT, preprocess.getOriginalImageDimensions(),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_class"));				

				short[] shortbuffer = preprocess.uncropAndBuffer(segmentation.exportLabelClassification(),(short)0);
				destImage[0].importData(0, shortbuffer, true);
				shortbuffer = null;
				
				// prior factor
				destImage[1] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_maxprior"));				

				buffer = preprocess.uncropAndBuffer(segmentation.exportMaxPriorFactor(),0.0f);
				destImage[1].importData(0, buffer, true);
				buffer = null;
				
				// direction factor
				destImage[2] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_maxdir"));
				buffer = preprocess.uncropAndBuffer(segmentation.exportMaxAngleFactor(),0.0f);
				destImage[2].importData(0, buffer, true);
				buffer = null;
				
				// diffusion coefficients
				destImage[3] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(6),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_diffusion"));
				
				buffer = preprocess.uncropAndBuffer(segmentation.exportDiffusionMaps(),6,0.0f);
				destImage[3].importData(0, buffer, true);		

				// anisotropy factors
				destImage[4] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(3),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_ani"));
				
				buffer = preprocess.uncropAndBuffer(segmentation.exportAnisotropyFactor(),3,0.0f);
				destImage[4].importData(0, buffer, true);		
				
				// prior seg
				destImage[5] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
											JDialogBase.makeImageName(srcImage[0].getImageName(), "_DTS_priorseg"));
				
				byte[] bytebuffer = preprocess.uncropAndBuffer(segmentation.generatePriorClassification(),(byte)0);
				destImage[5].importData(0, bytebuffer, true);
				
				// prior factor
				destImage[6] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
											JDialogBase.makeImageName(srcImage[0].getImageName(), "_DTS_maskprior"));
				
				buffer = preprocess.uncropAndBuffer(segmentation.getMaskPrior(),0.0f);
				destImage[6].importData(0, buffer, true);
				
			} else if  (resultOutput.equals("registration_images")) {
				// create the output array
				//if (atlas.isDeformable()) destImage = new ModelImage[5];
				//else destImage = new ModelImage[4];
				destImage = new ModelImage[6];
				
				// hard segmentation
				destImage[0] = new ModelImage(ModelStorageBase.SHORT, preprocess.getOriginalImageDimensions(),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_class"));				

				short[] shortbuffer = preprocess.uncropAndBuffer(segmentation.exportLabelClassification(),(short)0);
				destImage[0].importData(0, shortbuffer, true);
				shortbuffer = null;
				
				// probabilities
				destImage[1] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_maxmem"));
				buffer = preprocess.uncropAndBuffer(segmentation.exportMaxMappedGain(),0.0f);
				destImage[1].importData(0, buffer, true);
				buffer = null;
				
				// diffusion anisotropy
				destImage[2] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_maxatl"));
				
				buffer = preprocess.uncropAndBuffer(segmentation.exportMaxMappedAtlas(),0.0f);
				destImage[2].importData(0, buffer, true);		
				
				// prior seg
				destImage[3] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
											JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_prior"));
				
				byte[] bytebuffer = preprocess.uncropAndBuffer(segmentation.generatePriorClassification(),(byte)0);
				destImage[3].importData(0, bytebuffer, true);
				
				// prior dir
				destImage[4] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(3),
											JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_dir"));
				
				buffer = preprocess.uncropAndBuffer(segmentation.generatePriorDirectionMap(),3,0.0f);
				destImage[4].importData(0, buffer, true);
				
				if (atlas.isDeformable()) {
					destImage[5] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(3),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_vec"));
					buffer = preprocess.uncropAndBuffer(atlas.exportDeformationField(),3,0.0f);
					destImage[5].importData(0, buffer, true);
				}	
				else {
					destImage[5] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(3),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_vec"));
					buffer = preprocess.uncropAndBuffer(atlas.exportRigidDeformationField(),3,0.0f);
					destImage[5].importData(0, buffer, true);	
				}
			} else if  (resultOutput.equals("raw_segmentation")) {
				// create the output array
				destImage = new ModelImage[3];
			
				// hard segmentation
				destImage[0] = new ModelImage(ModelStorageBase.SHORT, preprocess.getOriginalImageArrayDimensions(nbest),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_labels"));				

				short[] shortbuffer = preprocess.uncropAndBuffer(segmentation.getLabels(),nbest,(short)0);
				destImage[0].importData(0, shortbuffer, true);
				shortbuffer = null;
				
				// gain
				destImage[1] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(nbest),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_gains"));
				buffer = preprocess.uncropAndBuffer(segmentation.getGain(),nbest,0.0f);
				destImage[1].importData(0, buffer, true);
				buffer = null;
				
				// probabilities
				destImage[2] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(nbest),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_mems"));
				buffer = preprocess.uncropAndBuffer(segmentation.exportRawGainMemberships(),nbest,0.0f);
				destImage[2].importData(0, buffer, true);
				buffer = null;
			}  else if  (resultOutput.equals("detail_images")) {
				// create the output array
				destImage = new ModelImage[7];
			
				// hard segmentation
				destImage[0] = new ModelImage(ModelStorageBase.SHORT, preprocess.getOriginalImageDimensions(),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_class"));				

				short[] shortbuffer = preprocess.uncropAndBuffer(segmentation.exportLabelClassification(),(short)0);
				destImage[0].importData(0, shortbuffer, true);
				shortbuffer = null;
				
				// overlapping segmentations
				destImage[1] = new ModelImage(ModelStorageBase.BOOLEAN, preprocess.getOriginalImageArrayDimensions(atlas.getNumber()),
											JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_seg"));
				
				boolean[] boolbuffer = preprocess.uncropAndBuffer(segmentation.exportSegmentations(),atlas.getNumber(),false);
				destImage[1].importData(0, boolbuffer, true);
				boolbuffer = null;
				
				// memberships
				destImage[2] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_maxmem"));
				buffer = preprocess.uncropAndBuffer(segmentation.exportMaxMemberships(),0.0f);
				destImage[2].importData(0, buffer, true);
				buffer = null;
				
				// initial factor
				destImage[3] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(atlas.getNumber()),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_init"));
				buffer = preprocess.uncropAndBuffer(segmentation.exportInitialFactor(),atlas.getNumber(),0.0f);
				destImage[3].importData(0, buffer, true);
				
				// prior seg
				destImage[4] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
											JDialogBase.makeImageName(srcImage[0].getImageName(), "_DTS_priorseg"));
				
				byte[] bytebuffer = preprocess.uncropAndBuffer(segmentation.generatePriorClassification(),(byte)0);
				destImage[4].importData(0, bytebuffer, true);
				
				// prior dir
				destImage[5] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(3),
											JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_dir"));
				
				buffer = preprocess.uncropAndBuffer(segmentation.generatePriorDirectionMap(),3,0.0f);
				destImage[5].importData(0, buffer, true);
				
				// deformation map
				if (atlas.isDeformable()) {
					destImage[6] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(3),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_vec"));
					buffer = preprocess.uncropAndBuffer(atlas.exportDeformationField(),3,0.0f);
					destImage[6].importData(0, buffer, true);
				}	
				else {
					destImage[6] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(3),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_DOTS_vec"));
					buffer = preprocess.uncropAndBuffer(atlas.exportRigidDeformationField(),3,0.0f);
					destImage[6].importData(0, buffer, true);	
				}
			}
		} catch (OutOfMemoryError e) {
			buffer = null;
			errorCleanUp("Segmentation TOADS: Out of memory creating hard classification", true);
			System.err.println(e.getMessage());
			finalize();
			
			setCompleted(false);
			return;
		} catch (IOException error) {
			errorCleanUp("Segmentation TOADS: export problem to destImage[]", true);
			System.err.println(error.getMessage());
			finalize();
			
			setCompleted(false);
			return;
		}
		if (segmentation!=null) segmentation.finalize();
		segmentation = null;

		fireProgressStateChanged(90);
		
		// clean-up the algorithms
		if (atlas!=null) atlas.finalize();
		atlas = null;
		if (preprocess!=null) preprocess.finalize();
		preprocess = null;
		
        if (debug) MedicUtilPublic.displayMessage("output...\n");

		setCompleted(true);
		threadStopped = false;
       // compute the elapsed time
        computeElapsedTime();
    } // end run()

    public ModelImage[] getResultImages(){
    	return destImage;
    }

    /**
     * Construct a readable list of the parameters to this segmentation.
     * @return       the parameter string
     */
	public String getParameterText() {
		String delim = "\n";
        String str = new String();
		for (int c=0;c<nc;c++) {
			str += "image "+(c+1)+": "+  srcImage[c].getImageName() + " (modality: "+modality[c]+")"+ delim;
		}
        str += "atlas name: " + atlasName + delim;
        str += "results: "+ resultOutput + delim;
		
		str += "max difference: "+ maxDistance + delim;
        str += "iterations: "+ iterations + delim;
        str += "gain factor: "+ gainFactor + delim;
		
		str += "iso diffusion: "+ isoDiffusion + delim;
       
		str += "nbest: "+ nbest + delim;
        
		str += "alignment scale: "+ alignScale + delim;
		
		str += "alignment: "+ alignType + delim;
		
		str += "registration nbest (init,iter): " + initAlignIter +", "+ alignIter + delim;
        str += "registration levels: "+ levels + delim;
        
		return str;
    }

	/** access the atlas information */
	public final DotsAtlas getAtlas() { return atlas; }
}
