package edu.jhmi.rad.medic.dialogs;

import edu.jhmi.rad.medic.algorithms.*;
import edu.jhmi.rad.medic.methods.*;
import edu.jhmi.rad.medic.utilities.*;

import gov.nih.mipav.view.*;
import gov.nih.mipav.view.dialogs.*;
import gov.nih.mipav.model.structures.*;
import gov.nih.mipav.model.file.*;
import gov.nih.mipav.model.algorithms.*;
import gov.nih.mipav.model.scripting.*;
import gov.nih.mipav.model.scripting.parameters.*;

import java.awt.event.*;
import java.awt.*;
import java.util.*;
import java.io.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

/** 
*   
*   Dialog box for Toads (atlas-based version)
*
*
*	@version    Aug 2006
*	@author     Pilou Bazin
*   @see        AlgorithmDTIToads
*
*
*/  
public class JDialogDOTS extends JDialogScriptableBase implements AlgorithmInterface {
    
    private     AlgorithmDOTS 	algo = null;
    
	// input
	private     ModelImage              image = null;       // source image(s)
    private     ModelImage[]            images = null;       // source image(s)
    private     int                     imageNumber;
	private     String[]    			imageModality;            
	private		String[] 				imageModalities = {"TensorD_(xx,yy,zz,xy,yz,zx)",
														   "TensorT_(xx,xy,xz,yy,yz,zz)",
														   "TensorLT_(xx,yx,yy,zx,zy,zz)",
															"FA+dir_(FA,x,y,z)"};
															/*
														   "Mixture_({x,y,z,w}xN)",
														   "CFARI_weights_(wxN)",
														   "CFARI_directions_(dxN)"};
														   */
	private		String					defaultModality = "TensorD_(xx,yy,zz,xy,yz,zx)";									   

	private     ModelImage              priorMask = null;       // optional source labels
	private		String[]				priorMaskTypes = {"none","lesions"};
	private		String					priorMaskType = "none";

	// atlas
	private		DotsAtlas				atlas 		= 	null;
	private		String					atlasName	= 	null;
	
	// output
	private     ModelImage[]            resultImage = null; // result image(s)
	/*
	private		String[]				resultOutputs	=		{"hard_segmentation", 
																 "full_segmentation", 
																 "contrasts",
																 "all_images",
																 "raw_segmentation",
																 "registration_images",
																 "energy_terms",
																 "detail_images"};
	private		String					resultOutput	=		"hard_segmentation";
	*/
	
	// parameters
	/*
	private		float		isoDiffusion 	= 	0.1f; 
	private		float		gainFactor			=	0.02f;		// threshold for cropping the background
	*/
	
	// algorithm options
	private 	int 		iterations		=       50;
	private 	float 		maxDistance		=       0.01f;
	private		int			nbest			=		8;			// minimum number of nbest between first and last 
	
	// registration
	private		String[]	alignTypes = {"rigid","single_scale","scaled_rigid","fully_affine",
							"multi_rigid","multi_single_scale","multi_scaled_rigid","multi_fully_affine"};
							/*
							{"none","rigid","single_scale","quadratic_scale","scaled_rigid","fully_affine",
											"deformable_gauss_diffusion","deformable_gauss_fluid","deformable_gauss_mixed"};
											*/
	private		String		alignType 		= 	"rigid";
	private		int			alignIter		= 	5;
	private		int			initAlignIter	= 	50;
	private		int			levels			= 	4;
	//private		float		atlasScale 		= 	3.0f;
	
	// constants
	private static final int MAX_SEED = 50;
		
    // dialog elements
	private		JTabbedPane	tabPane;
	private		JPanel		mainPanel;
	private		JPanel		paramPanel;
    private		JPanel		registrationPanel;
    private		JPanel		optionPanel;
    private		JPanel		atlasPanel;
    private		JPanel		algoPanel;
    private		JPanel			imagePanel;
	
	private 	JTextField  textMaxIter;
	private 	JLabel  	labelMaxIter;
	private 	JTextField  textMaxDist;
	private 	JLabel  	labelMaxDist;
	//private 	JLabel  	labelGainFactor;
    //private 	JTextField  textGainFactor;
	
	private     JLabel      labelLimit;
	private 	JTextField  textFirstLimit;
	private 	JTextField  textLastLimit;
	private 	JLabel  	labelNbest;
	private 	JTextField  textNbest;
	
	//private 	JLabel  	labelIsoDiffusion;
	//private 	JTextField  textIsoDiffusion;
	
	private 	JLabel		labelAlignType;
	private		JComboBox	comboAlignType;
	
	private 	JLabel		labelOptMask;
	private		JComboBox	comboOptMask;
	
	private 	JLabel		labelPriorMaskType;
	private		JComboBox	comboPriorMaskType;
	
	private		JButton			loadButton;
	private		JFileChooser	loadDialog;
	
	//private		JComboBox	comboOutput;
	//private		JLabel		labelOutput;	
	
	private 	JTextField  textRegIter;
	private 	JTextField  textInitIter;
	private 	JLabel  	labelRegIter;
	private 	JTextField  textRegLevels;
	private 	JLabel  	labelRegLevels;
	
	private 	JTextField  textAtlasName;
	private 	JLabel  	labelAtlasName;
	
	private		JLabel			labelModality;
	private		JComboBox		comboModality;
	
	//private 	JTextField  textAtlasScale;
	//private 	JLabel  	labelAtlasScale;
	
	/*
	// label list
	private		JPanel			imageListPanel;
	private		JCheckBox[]		checkImage;
	private		JLabel[]		labelOrder;
	private		JComboBox[]		comboModality;
	private		int				nImage = 0;
	private		JLabel			labelImage;
	private		JPanel			viewPanel;
	private		JScrollPane		scrollImage;
	private		int[]			order;
	private		int				nOrder;
	*/
	
    /**
    *  Creates dialog for plugin.
    *  @param theParentFrame          Parent frame.
    *  @param im              Source image.
    */
    public JDialogDOTS(Frame theParentFrame, ModelImage im) {
		super(theParentFrame, false);
		if (im.getType() == ModelImage.BOOLEAN || im.isColorImage() || im.getNDims()!=4) {
            MipavUtil.displayError("Source Image must be a 4D image"); 
            dispose();
            return;
        }
        image = im;
		
		loadDefaults();
		init();
	}
	
    /**
    *	Used primarily for the script to store variables and run the algorithm.  No
    *	actual dialog will appear but the set up info and result image will be stored here.
    *	@param UI   The user interface, needed to create the image frame.
    *	@param im	Source image.
    */
    public JDialogDOTS(ViewUserInterface UI, ModelImage im) {
        super();
    	if (im.getType() == ModelImage.BOOLEAN || im.isColorImage() || im.getNDims()!=4) {
            MipavUtil.displayError("Source Image must be a 4D image"); 
            dispose();
            return;
        }
    	image = im;
    }
    
    /**
     * Empty constructor needed for dynamic instantiation.
     */
    public JDialogDOTS() {};

    /**
     * Construct a delimited string that contains the parameters to this algorithm.
     * @param delim  the parameter delimiter (defaults to " " if empty)
     * @return       the parameter string
     */
	public String getParameterString( String delim ) {
        if ( delim.equals( "" ) ) {
            delim = " ";
        }

        String str = new String();
        
		str += defaultModality + delim;
		str += atlasName + delim;
		//str += resultOutput + delim;
		str += priorMaskType + delim;
		
		//str += isoDiffusion + delim;
		//str += gainFactor + delim;
		
		str += iterations + delim;
        str += maxDistance + delim;
        str += nbest + delim;
		
		str += alignType + delim;
		//str += atlasScale + delim;
        str += levels + delim;
        str += initAlignIter + delim;
        str += alignIter + delim;
         
		return str;
    }
	
    /**
     * Construct a readable list of the parameters to this algorithm.
     * @return       the parameter string
     */
	public String getParameterText() {
		String delim = "\n";
        String str = new String();
        
		str += "atlas name: " + atlasName + delim;
        //str += "results: "+ resultOutput + delim;
		
		//str += "iso diffusion: "+ isoDiffusion + delim;
		//str += "gain factor: "+ gainFactor + delim;
		
		str += "iterations: "+ iterations + delim;
        str += "max difference: "+ maxDistance + delim;
        str += "nbest: "+ nbest + delim;
        
		str += "alignment type: "+ alignType + delim;
		//str += "atlas scale: "+ atlasScale + delim;
		str += "registration levels: "+ levels + delim;
        str += "registration nbest (init,reg): " + initAlignIter +", "+ alignIter + delim;
           
        return str;
    }
	
 	/**
     *  Loads the default settings from Preferences to set up the dialog
     */
	public void loadDefaults() {
        String defaultsString = Preferences.getDialogDefaults(getDialogName());

        if (defaultsString != null) {

            try {
                StringTokenizer st = new StringTokenizer(defaultsString, ",");
				
				defaultModality = st.nextToken();
				atlasName = st.nextToken();
				//resultOutput = st.nextToken();
				priorMaskType = st.nextToken();
				
				//isoDiffusion = MipavUtil.getFloat(st);
				//gainFactor = MipavUtil.getFloat(st);
				
				iterations = MipavUtil.getInt(st);
				maxDistance = MipavUtil.getFloat(st);
				nbest = MipavUtil.getInt(st);
				
				alignType = st.nextToken();
				//atlasScale = MipavUtil.getFloat(st);
				levels = MipavUtil.getInt(st);
				initAlignIter = MipavUtil.getInt(st);
				alignIter = MipavUtil.getInt(st);
				
				System.out.println(getParameterString(","));
			}
            catch (Exception ex) {
                // since there was a problem parsing the defaults string, start over with the original defaults
                System.out.println( "Resetting defaults for dialog: " + getDialogName() );
				System.out.println( "Error: "+ex.getMessage());
                System.out.println( "String: "+defaultsString);
 				System.out.println(getParameterString(","));
				Preferences.removeProperty( getDialogName() );
            }
        } else {
			System.out.println( "no saved dialogs for "+getDialogName() );
		}
		return;
    }
		
    /**
     * Saves the default settings into the Preferences file
     */
	
    public void saveDefaults() {
        String defaultsString = new String( getParameterString(",") );
        System.out.println(defaultsString);
        saveDefaults(defaultsString);
    }
	 
	/** create a dialog box for choosing a template data file */
	private void buildLoadDialog() {
        loadDialog = new javax.swing.JFileChooser();
        loadDialog.setDialogTitle("Load Atlas data file");
        loadDialog.setDialogType(JFileChooser.OPEN_DIALOG);
        loadDialog.setMaximumSize(new java.awt.Dimension(500, 326));
        loadDialog.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                loadFileActionPerformed(evt);
            }
        });
        loadDialog.setFileSelectionMode(JFileChooser.FILES_ONLY);
        loadDialog.showOpenDialog(this);
	}

    /**
    *	Sets up the GUI (panels, buttons, etc) and displays it on the screen.
    */
	private void init(){
        setForeground(Color.black);
        setTitle("DOTS: DTI Overlapping Tract Segmentation");
				
 		labelAtlasName = new JLabel("Atlas: ");
		labelAtlasName.setForeground(Color.black);
		labelAtlasName.setFont(serif12);

		if (atlasName==null) {
			String defaultAtlasName = "Atlas/dots-atlas-2010/dots-wm-atlas.txt";
			try {
				ClassLoader cl = Thread.currentThread().getContextClassLoader();
				atlasName = cl.getResource(defaultAtlasName).getFile();
			} catch (Exception e) {
				System.err.print("Error: Unable to set default atlas:\n"+e.getMessage());
			}
		}

		textAtlasName = new JTextField(5);
		textAtlasName.setText(String.valueOf(atlasName));
		textAtlasName.setFont(serif12);
	
		loadButton = new JButton("Load");
		loadButton.setForeground(Color.black);
        loadButton.setFont(serif12);
        loadButton.addActionListener(this);
        loadButton.setActionCommand("Load");
		loadButton.setToolTipText("Load the template information from a text file");
        
		labelModality = new JLabel("Image type: ");
        labelModality.setForeground(Color.black);
        labelModality.setFont(serif12);
		
		comboModality = new JComboBox(imageModalities);
		comboModality.setSelectedItem(defaultModality);
		comboModality.setFont(serif12);
        comboModality.setBackground(Color.white);
		
		/*
		labelOutput = new JLabel("Output: ");
        labelOutput.setForeground(Color.black);
        labelOutput.setFont(serif12);
		
		comboOutput = new JComboBox(resultOutputs);
		comboOutput.setSelectedItem(resultOutput);
		comboOutput.setFont(serif12);
        comboOutput.setBackground(Color.white);
		*/
		labelOptMask = new JLabel("optional mask: ");
        labelOptMask.setFont(serif12);
        labelOptMask.setForeground(Color.black);
        		
		comboOptMask = buildImageList();

		labelPriorMaskType = new JLabel("label type: ");
        labelPriorMaskType.setForeground(Color.black);
        labelPriorMaskType.setFont(serif12);
		
		comboPriorMaskType = new JComboBox(priorMaskTypes);
		comboPriorMaskType.setSelectedItem(priorMaskType);
		comboPriorMaskType.setFont(serif12);
        comboPriorMaskType.setBackground(Color.white);
		/*
		labelIsoDiffusion = new JLabel("iso diffusion");
		labelIsoDiffusion.setForeground(Color.black);
		labelIsoDiffusion.setFont(serif12);
	
		textIsoDiffusion = new JTextField(5);
		textIsoDiffusion.setText(String.valueOf(isoDiffusion));
		textIsoDiffusion.setFont(serif12);
		
		labelGainFactor = new JLabel("Gain factor");
        labelGainFactor.setForeground(Color.black);
        labelGainFactor.setFont(serif12);

        textGainFactor = new JTextField(5);
        textGainFactor.setText(String.valueOf(gainFactor));
        textGainFactor.setFont(serif12);
		*/
		labelMaxIter = new JLabel("Maximum number of iterations");
		labelMaxIter.setForeground(Color.black);
		labelMaxIter.setFont(serif12);
	
		textMaxIter = new JTextField(5);
		textMaxIter.setText(String.valueOf(iterations));
		textMaxIter.setFont(serif12);
	
		labelMaxDist = new JLabel("Maximum distance");
		labelMaxDist.setForeground(Color.black);
		labelMaxDist.setFont(serif12);
	
		textMaxDist = new JTextField(5);
		textMaxDist.setText(String.valueOf(maxDistance));
		textMaxDist.setFont(serif12);
	
		labelNbest = new JLabel("number of retained tracts");
        labelNbest.setForeground(Color.black);
        labelNbest.setFont(serif12);

        textNbest = new JTextField(5);
        textNbest.setText(String.valueOf(nbest));
        textNbest.setFont(serif12);
		
 		labelAlignType = new JLabel("Alignment type: ");
        labelAlignType.setForeground(Color.black);
        labelAlignType.setFont(serif12);
		
		comboAlignType = new JComboBox(alignTypes);
		comboAlignType.setSelectedItem(alignType);
		comboAlignType.setFont(serif12);
        comboAlignType.setBackground(Color.white);
		/*
		labelAtlasScale = new JLabel("Atlas alignment scale: ");
		labelAtlasScale.setForeground(Color.black);
		labelAtlasScale.setFont(serif12);
	
		textAtlasScale = new JTextField(5);
		textAtlasScale.setText(String.valueOf(atlasScale));
		textAtlasScale.setFont(serif12);
		*/
        labelRegLevels = new JLabel("Alignment levels: ");
		labelRegLevels.setForeground(Color.black);
		labelRegLevels.setFont(serif12);
	
		textRegLevels = new JTextField(5);
		textRegLevels.setText(String.valueOf(levels));
		textRegLevels.setFont(serif12);
	
		labelRegIter = new JLabel("Alignment iterations (init|main) ");
		labelRegIter.setForeground(Color.black);
		labelRegIter.setFont(serif12);
	
		textInitIter = new JTextField(3);
		textInitIter.setText(String.valueOf(initAlignIter));
		textInitIter.setFont(serif12);
	
		textRegIter = new JTextField(3);
		textRegIter.setText(String.valueOf(alignIter));
		textRegIter.setFont(serif12);
	
		/*
		scrollImage = new JScrollPane();
		scrollImage.setMaximumSize(new Dimension(200,200));
		scrollImage.setPreferredSize(new Dimension(100,100));
		scrollImage.setMinimumSize(new Dimension(30,30));
		displayImageList();
		*/
		//imagePanel = new JPanel(new GridBagLayout());
		registrationPanel = new JPanel(new GridBagLayout());
		atlasPanel = new JPanel(new GridBagLayout());
		paramPanel = new JPanel(new GridBagLayout());
		algoPanel = new JPanel(new GridBagLayout());
		
		GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridwidth = 1;
        gbc.gridheight = 1;
        gbc.anchor = gbc.WEST;
        gbc.insets = new Insets(5, 5, 5, 5);
        /*
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.weightx = 0;
        gbc.gridwidth = 1;
        gbc.fill = gbc.NONE;
        paramPanel.add(labelIsoDiffusion, gbc);
        gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.weightx = 1;
		gbc.gridwidth = 2;
        gbc.fill = gbc.HORIZONTAL;
        paramPanel.add(textIsoDiffusion, gbc);
        gbc.gridx = 0;
        gbc.gridy = 7;
        gbc.weightx = 0;
        gbc.gridwidth = 1;
        gbc.fill = gbc.NONE;
        paramPanel.add(labelGainFactor, gbc);
        gbc.gridx = 1;
        gbc.gridy = 7;
        gbc.weightx = 1;
        gbc.gridwidth = 2;
        gbc.fill = gbc.HORIZONTAL;
        paramPanel.add(textGainFactor, gbc);
        */
        gbc.gridx = 0;
        gbc.gridy = 8;
        gbc.weightx = 0;
        gbc.gridwidth = 1;
        gbc.fill = gbc.NONE;
        paramPanel.add(labelOptMask, gbc);
        gbc.gridx = 1;
        gbc.gridy = 8;
        gbc.weightx = 1;
        gbc.gridwidth = 2;
        gbc.fill = gbc.HORIZONTAL;
        paramPanel.add(comboOptMask, gbc);
        gbc.gridx = 0;
        gbc.gridy = 9;
        gbc.weightx = 0;
        gbc.gridwidth = 1;
        gbc.fill = gbc.NONE;
        paramPanel.add(labelPriorMaskType, gbc);
        gbc.gridx = 1;
        gbc.gridy = 9;
        gbc.weightx = 1;
        gbc.gridwidth = 2;
        gbc.fill = gbc.HORIZONTAL;
        paramPanel.add(comboPriorMaskType, gbc);
		
		gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.weightx = 0;
        gbc.gridwidth = 1;
        gbc.fill = gbc.NONE;
        algoPanel.add(labelMaxIter, gbc);
        gbc.gridx = 1;
        gbc.gridy = 0;
        gbc.weightx = 1;
        gbc.gridwidth = 2;
        gbc.fill = gbc.HORIZONTAL;
        algoPanel.add(textMaxIter, gbc);
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.weightx = 0;
        gbc.gridwidth = 1;
        gbc.fill = gbc.NONE;
        algoPanel.add(labelMaxDist, gbc);
        gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.weightx = 1;
        gbc.gridwidth = 2;
        gbc.fill = gbc.HORIZONTAL;
        algoPanel.add(textMaxDist, gbc);
        gbc.gridx = 0;
        gbc.gridy = 2;
        gbc.weightx = 0;
        gbc.gridwidth = 1;
        gbc.fill = gbc.NONE;
        algoPanel.add(labelNbest, gbc);
        gbc.gridx = 1;
        gbc.gridy = 2;
        gbc.weightx = 1;
        gbc.gridwidth = 2;
        gbc.fill = gbc.HORIZONTAL;
        algoPanel.add(textNbest, gbc);
		
		gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.weightx = 0;
        gbc.gridwidth = 1;
        gbc.fill = gbc.NONE;
        registrationPanel.add(labelAlignType, gbc);
        gbc.gridx = 1;
        gbc.gridy = 0;
        gbc.weightx = 1;
        gbc.gridwidth = 2;
        gbc.fill = gbc.HORIZONTAL;
        registrationPanel.add(comboAlignType, gbc);
		gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        registrationPanel.add(labelRegLevels, gbc);
        gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.gridwidth = 2;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        registrationPanel.add(textRegLevels, gbc);
		gbc.gridx = 0;
        gbc.gridy = 2;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        registrationPanel.add(labelRegIter, gbc);
        gbc.gridx = 1;
        gbc.gridy = 2;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        registrationPanel.add(textInitIter, gbc);
        gbc.gridx = 2;
        gbc.gridy = 2;
        gbc.anchor = gbc.WEST;
        registrationPanel.add(textRegIter, gbc);
        /*
        gbc.gridx = 0;
        gbc.gridy = 3;
        gbc.gridwidth = 1;
        gbc.weightx = 1;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.WEST;
        registrationPanel.add(labelAtlasScale, gbc);
        gbc.gridx = 1;
        gbc.gridy = 3;
        gbc.gridwidth = 2;
        gbc.weightx = 0;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        registrationPanel.add(textAtlasScale, gbc);
		*/
		
		imagePanel = new JPanel(new GridBagLayout());
	
		/*
		// image panel
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 2;
        gbc.weightx = 1;
        gbc.fill = gbc.BOTH;
        imagePanel.add(scrollImage, gbc);
		gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.weightx = 1;
        gbc.gridwidth = 1;
        gbc.fill = gbc.HORIZONTAL;
        imagePanel.add(labelOutput, gbc);
		gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.weightx = 1;
        gbc.gridwidth = 1;
        gbc.fill = gbc.HORIZONTAL;
        imagePanel.add(comboOutput, gbc);
		*/
		gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.weightx = 1;
        gbc.gridwidth = 1;
        gbc.fill = gbc.HORIZONTAL;
        imagePanel.add(labelModality, gbc);
		gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.weightx = 1;
        gbc.gridwidth = 1;
        gbc.fill = gbc.HORIZONTAL;
        imagePanel.add(comboModality, gbc);
		
		atlasPanel = new JPanel(new GridBagLayout());
		
		gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.weightx = 0;
        gbc.gridwidth = 1;
        gbc.fill = gbc.NONE;
        atlasPanel.add(labelAtlasName, gbc);
        gbc.gridx = 1;
        gbc.gridy = 0;
        gbc.weightx = 1;
        gbc.gridwidth = 3;
        gbc.fill = gbc.HORIZONTAL;
        atlasPanel.add(textAtlasName, gbc);
        gbc.gridx = 2;
        gbc.gridy = 1;
        gbc.weightx = 0;
        gbc.gridwidth = 1;
        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.EAST;
        atlasPanel.add(loadButton, gbc);
        
		optionPanel = new JPanel(new GridBagLayout());
		
		gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.weightx = 1;
        gbc.gridwidth = 1;
        gbc.fill = gbc.HORIZONTAL;
        optionPanel.add(algoPanel, gbc);
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.weightx = 1;
        gbc.gridwidth = 1;
        gbc.fill = gbc.HORIZONTAL;
        optionPanel.add(registrationPanel, gbc);
        
		
		mainPanel = new JPanel(new GridBagLayout());

        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.weightx = 1;
        gbc.gridwidth = 1;
        gbc.fill = gbc.HORIZONTAL;
        mainPanel.add(imagePanel, gbc);
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.weightx = 1;
        gbc.gridwidth = 1;
        gbc.fill = gbc.HORIZONTAL;
        mainPanel.add(atlasPanel, gbc);
        gbc.gridx = 0;
        gbc.gridy = 2;
        gbc.weightx = 1;
        gbc.gridwidth = 1;
        gbc.fill = gbc.HORIZONTAL;
        mainPanel.add(optionPanel, gbc);
        gbc.gridx = 0;
        gbc.gridy = 3;
        gbc.weightx = 1;
        gbc.gridwidth = 1;
        gbc.fill = gbc.HORIZONTAL;
        mainPanel.add(paramPanel, gbc);

        getContentPane().add(mainPanel);
        getContentPane().add(buildButtons(), BorderLayout.SOUTH);

        pack();
        setVisible(true); 
		setResizable(false);
    	System.gc();
		
	} // end init()
	
   private JComboBox buildImageList() {
        int j;
		JComboBox combo;
       
        combo = new JComboBox();
        combo.setFont(serif12);
        combo.setBackground(Color.white);
        combo.addItem("-- none --");

        Enumeration names = ViewUserInterface.getReference().getRegisteredImageNames();

        // Add images from user interface that have the same exact dimensionality
        while (names.hasMoreElements()) {
            String name = (String) names.nextElement();
			ModelImage img = ViewUserInterface.getReference().getRegisteredImageByName(name);
			if (ViewUserInterface.getReference().getFrameContainingImage(img) != null) {
				combo.addItem(name);
			}
		}
		return combo;
    }//buildTargetList
    /*
	void displayImageList() {
		int j;
        
        Enumeration names = ViewUserInterface.getReference().getRegisteredImageNames();

        // Add images from user interface that have the same exact dimensionality
        // ! note: if the images have different dimensions, problems occur in Algorithms
        
		// 1. count the images
		nImage = 0;
		while (names.hasMoreElements()) {
            String name = (String) names.nextElement();
            ModelImage img = ViewUserInterface.getReference().getRegisteredImageByName(name);
			if (ViewUserInterface.getReference().getFrameContainingImage(img) != null) {
				if ( (image.getNDims() == img.getNDims()) && (!img.isColorImage()) ) {
					nImage++;
				}
            }
        }
		// 2. create the checkboxes
		checkImage = new JCheckBox[nImage];
		labelOrder = new JLabel[nImage];
		comboModality = new JComboBox[nImage];
		order = new int[nImage];
		nOrder = 0;
		for (int n=0;n<nImage;n++) order[n] = -1;
		int n = 0;
		names = ViewUserInterface.getReference().getRegisteredImageNames();
		while (names.hasMoreElements()) {
            String name = (String) names.nextElement();
            ModelImage img = ViewUserInterface.getReference().getRegisteredImageByName(name);
			if (ViewUserInterface.getReference().getFrameContainingImage(img) != null) {
				if ( (image.getNDims() == img.getNDims()) && (!img.isColorImage()) ) {
					labelOrder[n] = new JLabel(".");
					labelOrder[n].setFont(serif12);
					labelOrder[n].setBackground(Color.white);
					
					checkImage[n] = new JCheckBox(name);
					checkImage[n].setFont(serif12);
					checkImage[n].addActionListener(this);
					checkImage[n].setActionCommand("Image "+n);
					if (image.getImageName().equals(img.getImageName())) {
						checkImage[n].setSelected(true);
						order[n] = nOrder;
						labelOrder[n].setText(""+(nOrder+1));
						nOrder++;
					} else {
						checkImage[n].setSelected(false);
					}
					
					comboModality[n] = new JComboBox(imageModalities);
					comboModality[n].setFont(serif12);
					comboModality[n].setBackground(Color.white);
					comboModality[n].setSelectedItem(defaultModality);
					
					n++;
				}
            }
        }
		// display
		viewPanel = new JPanel(new GridBagLayout());
		GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridwidth = 1;
        
		for (n=0;n<nImage;n++) {
			gbc.gridy = n;
			gbc.gridx = 0;
			gbc.fill = gbc.NONE;
			gbc.weightx = 0;
			viewPanel.add(labelOrder[n], gbc);
			gbc.gridx = 1;
			gbc.fill = gbc.HORIZONTAL;
			gbc.weightx = 1;
			viewPanel.add(checkImage[n], gbc);
			gbc.gridx = 2;
			gbc.fill = gbc.NONE;
			gbc.weightx = 1;
			viewPanel.add(comboModality[n], gbc);
		}
		scrollImage.setViewportView(viewPanel);
		scrollImage.revalidate();
		pack();
    }
	*/
	
    /**
    *  Accessor that returns the image.
    *  @return          The result image.
    */
    public ModelImage[] getResultImage() {
        return resultImage;
    }
	
    /**
    *	Accessor that sets the parameters
    */
    public void setParameters() {}
	
    //************************************************************************
    //************************** Event Processing ****************************
    //************************************************************************

	/**
	*  Closes dialog box when the OK button is pressed and calls the algorithm.
	*  @param event       Event that triggers function.
	*/
	public void actionPerformed(ActionEvent event) {
		String command = event.getActionCommand();
	
		if (command.equals("OK")) {
			if (setVariables()) { 
				callAlgorithm();
			}    
		} else if (command.equals("Cancel")) {
			dispose();
		} else if (command.equals("Help")) {
            //MipavUtil.showHelp("10027");
        } else if (command.equals("Load")) {
			buildLoadDialog();
            loadDialog.setSize(500,326);		
		} 
		/*
		else {
			for (int n=0;n<nImage;n++) {
				if (command.equals("Image "+n)) {
					if (checkImage[n].isSelected()) {
						order[n] = nOrder;
						labelOrder[n].setText(""+(nOrder+1));
						nOrder++;
					} else {
						for (int m=0;m<nImage;m++) {
							if (order[m]>order[n]) {
								order[m]--;
								labelOrder[m].setText(""+(order[m]+1));
							}
						}
						order[n] = -1;
						labelOrder[n].setText(".");
						nOrder--;
					}
				}
			}
		}
		*/
    }

	public void loadFileActionPerformed(ActionEvent evt) {
		if ( JFileChooser.APPROVE_SELECTION.equals(evt.getActionCommand()) ) {
			String filename = loadDialog.getSelectedFile().getAbsolutePath();
			System.out.println("load file: "+filename);
			// load the file here
			atlasName = filename;
			textAtlasName.setText(atlasName);
			System.out.println("get the atlas: "+atlasName);
			// debug: load the atlas
		}
		//loadDialog.setVisible(false);
	}
	
    //************************************************************************
    //************************** Algorithm Events ****************************
    //************************************************************************
    
    /** 
    *	This method is required if the AlgorithmPerformed interface is implemented. 
    *   It is called by the algorithm when it has completed or failed to to complete, 
    *   so that the dialog can be display the result image and/or clean up.
    *   @param algorithm   Algorithm that caused the event.
    */
    public void algorithmPerformed(AlgorithmBase algorithm) {
       
		if (Preferences.isPreference(Preferences.PREF_SAVE_DEFAULTS) && this.getOwner() != null && !isScriptRunning()) {
          saveDefaults();
        }
	    
        if ( algorithm instanceof AlgorithmDOTS) {
            image.clearMask();
			resultImage = ((AlgorithmDOTS)algorithm).getResultImages();
			
			System.out.println("got output");
			
			if (algorithm.isCompleted() == true && resultImage != null) {
                System.out.println("send it out");
				//The algorithm has completed and produced a new image to be displayed.
				
				ViewJFrameImage imageFrame[] = new ViewJFrameImage[resultImage.length];
				for (int i = 0; i < resultImage.length; i++) {
                    MedicUtilPublic.updateFileInfo(image, resultImage[i]);
                    
                    try {
                        imageFrame[i] = new ViewJFrameImage(resultImage[i], null,
                                           new Dimension(610, 200 + i * 20));
                    } catch (OutOfMemoryError error) {
                        System.gc();
                        JOptionPane.showMessageDialog(null, 
                                                "Out of memory: unable to open new frame",
                                                "Error", JOptionPane.ERROR_MESSAGE);
                    }
                }
            } else if (resultImage != null) {
                //algorithm failed but result image still has garbage
                for (int i = 0; i < resultImage.length; i++) {
                    if (resultImage[i] != null) {
                        resultImage[i].disposeLocal(); // Clean up memory of result image
                        resultImage[i] = null;
                    }
                }
                resultImage = null;
                System.gc();
            }
			if (algorithm.isCompleted()) {
				insertScriptLine();
			}
		}
		algorithm.finalize();
		algorithm = null;
		if (atlas!=null) atlas.finalize();
		atlas = null;
		dispose();
    }// end AlgorithmPerformed()
    
  
    /**
    *	Use the GUI results to set up the variables needed to run the algorithm.
    *	@return		<code>true</code> if parameters set successfully, <code>false</code> otherwise.
    */
    /**
    *	Use the GUI results to set up the variables needed to run the algorithm.
    *	@return		<code>true</code> if parameters set successfully, <code>false</code> otherwise.
    */
    private boolean setVariables() {
    	String tmpStr;

		atlasName = textAtlasName.getText();
		
		//resultOutput = (String)comboOutput.getSelectedItem();
		
		priorMaskType = (String)comboPriorMaskType.getSelectedItem();
		if (!priorMaskType.equals("none") && !((String)comboOptMask.getSelectedItem()).equals("-- none --")) {
			priorMask = ViewUserInterface.getReference().getRegisteredImageByName((String)comboOptMask.getSelectedItem());
			if (priorMask==null) {
				comboOptMask.requestFocus();
				return false;
			}
		} else {
			priorMask = null;
		}
		/*
		tmpStr = textIsoDiffusion.getText();
        if (testParameter(tmpStr, 0, 500)) {
          isoDiffusion = Float.valueOf(tmpStr).floatValue();
        } else {
          textIsoDiffusion.requestFocus();
          textIsoDiffusion.selectAll();
          return false;
        }
      	tmpStr = textGainFactor.getText();
        if (testParameter(tmpStr, 0, 10000)) {
          gainFactor = Float.valueOf(tmpStr).floatValue();
        } else {
          textGainFactor.requestFocus();
          textGainFactor.selectAll();
          return false;
        }
		*/
		tmpStr = textMaxIter.getText();
        if ( testParameter(tmpStr, -500, 500) ){
            iterations = Integer.valueOf(tmpStr).intValue();
        } else {
            textMaxIter.requestFocus();
            textMaxIter.selectAll();
            return false;
        }
       	tmpStr = textMaxDist.getText();
        if ( testParameter(tmpStr, 0, 1000) ){
            maxDistance = Float.valueOf(tmpStr).floatValue();
        } else {
            textMaxDist.requestFocus();
            textMaxDist.selectAll();
            return false;
        }
 		tmpStr = textNbest.getText();
        if (testParameter(tmpStr, 0, 1000)) {
          nbest = Integer.valueOf(tmpStr).intValue();
        } else {
          textNbest.requestFocus();
          textNbest.selectAll();
          return false;
        }
		
		alignType = (String)comboAlignType.getSelectedItem();
		/*
		tmpStr = textAtlasScale.getText();
        if (testParameter(tmpStr, 0, 500)) {
          atlasScale = Float.valueOf(tmpStr).floatValue();
        } else {
          textAtlasScale.requestFocus();
          textAtlasScale.selectAll();
          return false;
        }
        */
		tmpStr = textRegLevels.getText();
        if (testParameter(tmpStr, 0, 100)) {
          levels = Integer.valueOf(tmpStr).intValue();
        } else {
          textRegLevels.requestFocus();
          textRegLevels.selectAll();
          return false;
        }
        tmpStr = textInitIter.getText();
        if (testParameter(tmpStr, 0, 500)) {
          initAlignIter = Integer.valueOf(tmpStr).intValue();
        } else {
          textInitIter.requestFocus();
          textInitIter.selectAll();
          return false;
        }
		tmpStr = textRegIter.getText();
        if (testParameter(tmpStr, 0, 500)) {
          alignIter = Integer.valueOf(tmpStr).intValue();
        } else {
          textRegIter.requestFocus();
          textRegIter.selectAll();
          return false;
        }
		/*
		imageNumber = 0;
		for (int i=0;i<nImage;i++) 
			if (checkImage[i].isSelected() ) imageNumber++;
	
		if (imageNumber==0) {
			tabPane.setSelectedComponent(imagePanel);
			imageListPanel.requestFocus();
			return false;
		}
		*/
		/*
		images = new ModelImage[imageNumber];
		imageModality = new String[imageNumber];
		// put images in proper order
		for (int i=0;i<nImage;i++) if (checkImage[i].isSelected() ) {
			images[order[i]] = ViewUserInterface.getReference().getRegisteredImageByName(checkImage[i].getText());
			imageModality[order[i]] = (String)comboModality[i].getSelectedItem();
		}
		defaultModality = imageModality[0];
		
		
		if (imageNumber==0) {
		*/
		imageNumber = 1;
		images = new ModelImage[imageNumber];
		images[0] = image;
		imageModality = new String[imageNumber];
		imageModality[0] = (String)comboModality.getSelectedItem();
		defaultModality = imageModality[0];
		//}

		saveDefaults();
		
    	return true;  	
    }   // end setVariables()
    
    /**
    *	Once all the necessary variables are set, call the Gaussian Blur
    *	algorithm based on what type of image this is and whether or not there
    *	is a separate destination image.
    */
    protected void callAlgorithm() {
        int i;
        
		atlas = new DotsAtlas(atlasName);

		try {           
            // Create algorithm first : get important parameters
			algo = new AlgorithmDOTS(images, imageNumber,
									  imageModality, 
									  atlasName, atlas, "full_segmentation", 
									  priorMask, priorMaskType,
									  -1, -1, 
									  iterations, maxDistance, nbest,
									  alignType,
									  3.0f,
									  levels, initAlignIter, alignIter
									  );
									  
			// This is very important. Adding this object as a listener allows the algorithm to
            // notify this object when it has completed or failed. See algorithm performed event.
            // This is made possible by implementing AlgorithmedPerformed interface
            algo.addListener(this);
                            
            setVisible(false);  // Hide dialog
			
			createProgressBar(image.getImageName(), algo);
                
			// for linux debugs: no threading
			//setSeparateThread(false);

            if (runInSeparateThread) {
                // Start the thread as a low priority because we wish to still have user interface work fast.
                if (algo.startMethod(Thread.MIN_PRIORITY) == false) {
                    MipavUtil.displayError("A thread is already running on this object");
                }
            } else {
                //algo.setActiveImage(isActiveImage);
                algo.run();
            }
        } catch (OutOfMemoryError x) {
            System.gc();
            MipavUtil.displayError( "Dialog TOADS: unable to allocate enough memory");
            return;
        }
    } // end callAlgorithm()
    
	/**
     * Perform any actions required after the running of the algorithm is complete.
     */
    protected void doPostAlgorithmActions() {
		for (int n=0;n<resultImage.length;n++)
			AlgorithmParameters.storeImageInRunner(resultImage[n]);
    }
    /**
     * Set up the dialog GUI based on the parameters before running the algorithm as part of a script.
     */
    protected void setGUIFromParams() {
		//imageNumber = scriptParameters.getParams().getInt("input_images");
		imageNumber = 1;
		images = new ModelImage[imageNumber];
		for (int n=0;n<imageNumber;n++) {
			images[n] = scriptParameters.retrieveInputImage(n+1);
		}
		image = images[0];
        parentFrame = image.getParentFrame();
		
		atlasName = scriptParameters.getParams().getString("atlas_filename");
		//resultOutput = scriptParameters.getParams().getString("result_output");
		
		priorMaskType = scriptParameters.getParams().getString("opt_mask_type");
		if (!priorMaskType.equals("none")) {
			priorMask = scriptParameters.retrieveInputImage(imageNumber);
		}
		
		imageModality = new String[imageNumber];
		/*
		for (int n=0;n<imageNumber;n++) {
			imageModality[n] = scriptParameters.getParams().getString("image_modality_"+(n+1));
		}
		*/
		imageModality[0] = scriptParameters.getParams().getString("image_modality");
		defaultModality = imageModality[0];
		/*
		isoDiffusion = scriptParameters.getParams().getFloat("smoothing_weight");
		gainFactor = scriptParameters.getParams().getFloat("gain_factor");
		*/
		iterations = scriptParameters.getParams().getInt("iterations");
		maxDistance = scriptParameters.getParams().getFloat("max_distance");
        nbest = scriptParameters.getParams().getInt("nbest");
		
		alignType = scriptParameters.getParams().getString("align_type");
		//atlasScale = scriptParameters.getParams().getFloat("atlas_scale");
		levels = scriptParameters.getParams().getInt("levels");
		alignIter = scriptParameters.getParams().getInt("align_init");
		initAlignIter = scriptParameters.getParams().getInt("align_iter");		
    }

	/**
     * Store the parameters from the dialog to record the execution of this algorithm.
     * 
     * @throws  ParserException  If there is a problem creating one of the new parameters.
     */
    protected void storeParamsFromGUI() throws ParserException {
        scriptParameters.getParams().put(ParameterFactory.newParameter("input_images", imageNumber));
		for (int n=0;n<imageNumber;n++) {
			scriptParameters.storeInputImage(images[n]);
		}
		if (!priorMaskType.equals("none")) {
			scriptParameters.storeInputImage(priorMask);
		}
		for (int n=0;n<resultImage.length;n++) {
			scriptParameters.storeImageInRecorder(resultImage[n]);
		}
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("atlas_filename", atlasName));
		//scriptParameters.getParams().put(ParameterFactory.newParameter("result_output", resultOutput));
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("opt_mask_type", priorMaskType));
		/*
		for (int n=0;n<imageNumber;n++) {
			scriptParameters.getParams().put(ParameterFactory.newParameter("image_modality_"+(n+1), imageModality[n]));
		}
		*/
		scriptParameters.getParams().put(ParameterFactory.newParameter("image_modality", defaultModality));
		/*
		scriptParameters.getParams().put(ParameterFactory.newParameter("smoothing_weight", isoDiffusion));
		scriptParameters.getParams().put(ParameterFactory.newParameter("gain_factor", gainFactor));
		*/
		scriptParameters.getParams().put(ParameterFactory.newParameter("iterations", iterations));
		scriptParameters.getParams().put(ParameterFactory.newParameter("max_distance", maxDistance));
		scriptParameters.getParams().put(ParameterFactory.newParameter("nbest", nbest));
		
		scriptParameters.getParams().put(ParameterFactory.newParameter("align_type", alignType));
		//scriptParameters.getParams().put(ParameterFactory.newParameter("atlas_scale", atlasScale));
		scriptParameters.getParams().put(ParameterFactory.newParameter("levels", levels));
		scriptParameters.getParams().put(ParameterFactory.newParameter("align_init", initAlignIter));
		scriptParameters.getParams().put(ParameterFactory.newParameter("align_iter", alignIter));
	}
}
