package edu.jhmi.rad.medic.methods;

import java.io.*;
import java.util.*;
import java.net.URL;

import gov.nih.mipav.view.*;
import gov.nih.mipav.model.structures.jama.*;
import gov.nih.mipav.model.file.FileInfoBase;

import edu.jhmi.rad.medic.libraries.*;
import edu.jhmi.rad.medic.structures.*;
import edu.jhmi.rad.medic.utilities.*;

/**
 *
 *  This algorithm performs a simple alignment of the image with a topology template
 *	<p>
 *	The algorithm handles all the little things needed for image cropping, conversion
 * 	to an image buffer, padding, etc.
 *
 *	@version    March 2007
 *	@author     Pierre-Louis Bazin
 *		
 *
 */
 
public class DotsPreprocess {
		
	// data buffers
	private 	float[][]		imdir;  				// principal direction of original image
	private 	float[][]		imval;  				// weights for each direction of original image
	private 	float[]			famap;  				// FA map of original image
	private 	float[]			mdmap;  				// MD map of original image
	private 	float[]			labeling;  				// prior labeling of original image
	private 	boolean[]		mask;  					// mask of original image
	private 	byte[]			topology;  				// topology template
	private		float[]			transform; 				// rigid transform from atlas space to image space
	private 	int				nix,niy,niz;   			// image dimensions
	private 	int				mix,miy,miz;   			// cropped image dimensions
	private 	float			rix,riy,riz;   			// image resolutions
	private 	int				x0i,y0i,z0i;   			// cropped image origin
	private 	int				xNi,yNi,zNi;   			// cropped image final point
	private 	float			xCi,yCi,zCi;   			// coordinates of the center
	private 	float			xCt,yCt,zCt;   			// coordinates of the center
	private 	int				ntx,nty,ntz;   			// template dimensions
	private 	float			rtx,rty,rtz;   			// template resolutions
	private   	float			orx,ory,orz;			// image axis orientation
	private		int				nw;						// number of image layers
	private		boolean			cropped;				// check whether the images are the originals or the cropped ones
	private		boolean			transformed;			// check whether the topology is the originals or the cropped ones
	private		boolean			normalized;				// check whether the images are normalized in [0,1]
	private		String			modality;
	
	// structure parameters
	private 	int 		nobj;    	// number of structures
	private		byte		bgLabel;    // label of the background (usually 1)
	private		float		bgRatio;	// ratio of image intensity used as background
	private		int			bs;	// the amount of extra borders
	//private		float[]		Ilow,Ihigh;	// image robust normalised min,max
	
	// for debug and display
	private static final boolean		debug				=	true;
	private static final boolean		verbose				=	true;
    
	// constants
	private static final	float   ISQRT2 = (float)(1.0/Math.sqrt(2.0f));
	private static final	float   ZERO = 1E-20f;
	private static final	float   INF = 1E20f;
	
	/**
	 *  constructor
	 */
	public DotsPreprocess(float[] img_, short[] lb_, int nw_,
									int nix_, int niy_, int niz_,
									float rix_, float riy_, float riz_,
									int orient_, int orx_, int ory_, int orz_,
									byte[] topo_, 
									int ntx_, int nty_, int ntz_,
									float rtx_, float rty_, float rtz_,
									float[] lbimg_,
									byte bgLb_, float bgRatio_,
									int bdSz_, String modal_) {
	
		if (modal_.equals("TensorD_(xx,yy,zz,xy,yz,zx)")) 
			initTensorsDiagonal(img_, nix_, niy_, niz_, rix_, riy_, riz_,orient_,orx_,ory_,orz_,
								topo_,ntx_,nty_,ntz_,rtx_,rty_,rtz_,bgLb_,bgRatio_,bdSz_);
		else if (modal_.equals("TensorT_(xx,xy,xz,yy,yz,zz)")) 
			initTensorsTriangular(img_, nix_, niy_, niz_, rix_, riy_, riz_,orient_,orx_,ory_,orz_,
								topo_,ntx_,nty_,ntz_,rtx_,rty_,rtz_,bgLb_,bgRatio_,bdSz_);
		else if (modal_.equals("TensorLT_(xx,yx,yy,zx,zy,zz)")) 
			initTensorsLowerTriangular(img_, nix_, niy_, niz_, rix_, riy_, riz_,orient_,orx_,ory_,orz_,
								topo_,ntx_,nty_,ntz_,rtx_,rty_,rtz_,bgLb_,bgRatio_,bdSz_);
		else if (modal_.equals("FA+dir_(FA,x,y,z)"))
			initFadir(img_, nix_, niy_, niz_, rix_, riy_, riz_,orient_,orx_,ory_,orz_,
						topo_,ntx_,nty_,ntz_,rtx_,rty_,rtz_,bgLb_,bgRatio_,bdSz_);
		else if (modal_.equals("Mixture_({x,y,z,w}xN)"))
			initDirectionMixture(img_, nw_, nix_, niy_, niz_, rix_, riy_, riz_,orient_,orx_,ory_,orz_,
								topo_,ntx_,nty_,ntz_,rtx_,rty_,rtz_,bgLb_,bgRatio_,bdSz_);
		else if (modal_.equals("CFARI_weights_(wxN)"))
			initCFARIModel(img_, lb_, nw_, nix_, niy_, niz_, rix_, riy_, riz_,orient_,orx_,ory_,orz_,
								topo_,ntx_,nty_,ntz_,rtx_,rty_,rtz_,bgLb_,bgRatio_,bdSz_);
		else System.out.println("wrong input type!");
		
		modality = modal_;
		
		labeling = lbimg_;
	}			
																			
	
	/**
	 *  constructor
	 */
	public void initFadir(float[] img_,
									int nix_, int niy_, int niz_,
									float rix_, float riy_, float riz_,
									int orient_, int orx_, int ory_, int orz_,
									byte[] topo_, 
									int ntx_, int nty_, int ntz_,
									float rtx_, float rty_, float rtz_,
									byte bgLb_, float bgRatio_,
									int bdSz_) {
										
		if (debug) MedicUtilPublic.displayMessage("create FA+dir image\n");

		nix = nix_;
		niy = niy_;
		niz = niz_;
		rix = rix_;
		riy = riy_;
		riz = riz_;
		xCi = nix/2.0f;
		yCi = niy/2.0f;
		zCi = niz/2.0f;
		cropped = false;
		normalized = false;
		nw = 1;
		imval = null;
		
		// create the image
		imdir = new float[3][nix*niy*niz];
		famap = new float[nix*niy*niz];
		mdmap = new float[nix*niy*niz];
		mask = new boolean[nix*niy*niz];
		for (int xyz=0;xyz<nix*niy*niz;xyz++) {
			if (img_[xyz]>0.0f) {
				famap[xyz] = img_[xyz];
				mdmap[xyz] = 1.0f;
				for (int d=0;d<3;d++) imdir[d][xyz] = img_[xyz + nix*niy*niz*(d+1)];
				mask[xyz] = true;
			} else {
				famap[xyz] = 0.0f;
				mdmap[xyz] = 0.0f;
				for (int d=0;d<3;d++) imdir[d][xyz] = 0.0f;
				mask[xyz] = false;
			}
		}
		
		topology = topo_;
		ntx = ntx_;
		nty = nty_;
		ntz = ntz_;
		rtx = rtx_;
		rty = rty_;
		rtz = rtz_;
		xCt = ntx/2.0f;
		yCt = nty/2.0f;
		zCt = ntz/2.0f;
		transformed = false;
		
		bgLabel = bgLb_;
		bgRatio = bgRatio_;
		bs = bdSz_;
		
		// orientation: initialize a rotation
		transform = new float[6];
		// note : assumes a rotation around the image center
		if (orient_==FileInfoBase.AXIAL) {
			transform[0] = 0.0f;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			if (orx_==FileInfoBase.ORI_L2R_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_P2A_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_S2I_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else if (orient_==FileInfoBase.CORONAL) {
			transform[0] = -ISQRT2;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			if (orx_==FileInfoBase.ORI_L2R_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_I2S_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_P2A_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else if (orient_==FileInfoBase.SAGITTAL) {
			transform[0] = -0.5f;
			transform[1] = -0.5f;
			transform[2] = 0.5f;
			
			if (orx_==FileInfoBase.ORI_P2A_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_I2S_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_R2L_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else {
			// default is axial
			transform[0] = 0.0f;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			orx = 1.0f;
			ory = 1.0f;
			orz = 1.0f;
		}
		transform[3] = 0.0f;
		transform[4] = 0.0f;
		transform[5] = 0.0f;
		
		// init the image cropping					
		x0i = 0;
		y0i = 0;
		z0i = 0;
		xNi = nix-1;
		yNi = niy-1;
		zNi = niz-1;
		
		// compute min,max ??
		if (debug) MedicUtilPublic.displayMessage("initialisation: done\n");
	}

	/**
	 *  constructor
	 */
	public void initTensorsTriangular(float[] img_, 
									int nix_, int niy_, int niz_,
									float rix_, float riy_, float riz_,
									int orient_, int orx_, int ory_, int orz_,
									byte[] topo_, 
									int ntx_, int nty_, int ntz_,
									float rtx_, float rty_, float rtz_,
									byte bgLb_, float bgRatio_,
									int bdSz_) {
										
		if (debug) MedicUtilPublic.displayMessage("create mixture from tensor (triangular convention)\n");

		nix = nix_;
		niy = niy_;
		niz = niz_;
		rix = rix_;
		riy = riy_;
		riz = riz_;
		xCi = nix/2.0f;
		yCi = niy/2.0f;
		zCi = niz/2.0f;
		cropped = false;
		normalized = false;
		nw = 3;
		
		// create the image
		imdir = new float[3*nw][nix*niy*niz];
		imval = new float[nw][nix*niy*niz];
		famap = new float[nix*niy*niz];
		mdmap = new float[nix*niy*niz];
		mask = new boolean[nix*niy*niz];
		float[][] mat = new float[3][3];
		for (int xyz=0;xyz<nix*niy*niz;xyz++) {
			float[] image = new float[6];
			float sum=0.0f;
			// (xx,yy,zz,xy,yz,zx) <- (xx,xy,xz,yy,yz,zz)
			image[0] = img_[xyz + nix*niy*niz*0];
			image[1] = img_[xyz + nix*niy*niz*3];
			image[2] = img_[xyz + nix*niy*niz*5];
			image[3] = img_[xyz + nix*niy*niz*1];
			image[4] = img_[xyz + nix*niy*niz*4];
			image[5] = img_[xyz + nix*niy*niz*2];
			for (int d=0;d<6;d++) {
				sum+=Numerics.abs(image[d]);
			}
			if (sum>0) {
				double[] eigen = Tensor.eigenDecomposition(Tensor.matrixForm(image));
				famap[xyz] = Tensor.fractionalAnisotropy(eigen);
				mdmap[xyz] = Tensor.meanDiffusivity(eigen);
				float norm = Numerics.max( image[0]+image[1]+image[2], ZERO );
				for (int w=0;w<3;w++) {
					if (eigen[0]*eigen[0]>ZERO) imval[w][xyz] = (float)Numerics.abs(eigen[w]/eigen[0]);
					else imval[w][xyz] = (float)Numerics.abs(eigen[w]);
					imdir[3*w+0][xyz] = (float)eigen[3+3*w+0];
					imdir[3*w+1][xyz] = (float)eigen[3+3*w+1];
					imdir[3*w+2][xyz] = (float)eigen[3+3*w+2];
				}
				mask[xyz] = true;
			} else {
				famap[xyz] = 0.0f;
				mdmap[xyz] = 0.0f;
				for (int w=0;w<3;w++) {
					imval[w][xyz] = 0.0f;
					for (int d=0;d<3;d++) imdir[3*w+d][xyz] = 0.0f;
				}
				mask[xyz] = false;
			}
		}
			
		topology = topo_;
		ntx = ntx_;
		nty = nty_;
		ntz = ntz_;
		rtx = rtx_;
		rty = rty_;
		rtz = rtz_;
		xCt = ntx/2.0f;
		yCt = nty/2.0f;
		zCt = ntz/2.0f;
		transformed = false;
		
		bgLabel = bgLb_;
		bgRatio = bgRatio_;
		bs = bdSz_;
		
		// orientation: initialize a rotation
		transform = new float[6];
		// note : assumes a rotation around the image center
		if (orient_==FileInfoBase.AXIAL) {
			transform[0] = 0.0f;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			if (orx_==FileInfoBase.ORI_L2R_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_P2A_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_S2I_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else if (orient_==FileInfoBase.CORONAL) {
			transform[0] = -ISQRT2;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			if (orx_==FileInfoBase.ORI_L2R_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_I2S_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_P2A_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else if (orient_==FileInfoBase.SAGITTAL) {
			transform[0] = -0.5f;
			transform[1] = -0.5f;
			transform[2] = 0.5f;
			
			if (orx_==FileInfoBase.ORI_P2A_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_I2S_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_R2L_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else {
			// default is axial
			transform[0] = 0.0f;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			orx = 1.0f;
			ory = 1.0f;
			orz = 1.0f;
		}
		transform[3] = 0.0f;
		transform[4] = 0.0f;
		transform[5] = 0.0f;
		
		// init the image cropping					
		x0i = 0;
		y0i = 0;
		z0i = 0;
		xNi = nix-1;
		yNi = niy-1;
		zNi = niz-1;
		
		// compute min,max ??
		if (debug) MedicUtilPublic.displayMessage("initialisation: done\n");
	}

	/**
	 *  constructor
	 */
	public void initTensorsLowerTriangular(float[] img_, 
									int nix_, int niy_, int niz_,
									float rix_, float riy_, float riz_,
									int orient_, int orx_, int ory_, int orz_,
									byte[] topo_, 
									int ntx_, int nty_, int ntz_,
									float rtx_, float rty_, float rtz_,
									byte bgLb_, float bgRatio_,
									int bdSz_) {
										
		if (debug) MedicUtilPublic.displayMessage("create mixture from tensor (triangular convention)\n");

		nix = nix_;
		niy = niy_;
		niz = niz_;
		rix = rix_;
		riy = riy_;
		riz = riz_;
		xCi = nix/2.0f;
		yCi = niy/2.0f;
		zCi = niz/2.0f;
		cropped = false;
		normalized = false;
		nw = 3;
		
		// create the image
		imdir = new float[3*nw][nix*niy*niz];
		imval = new float[nw][nix*niy*niz];
		famap = new float[nix*niy*niz];
		mdmap = new float[nix*niy*niz];
		mask = new boolean[nix*niy*niz];
		float[][] mat = new float[3][3];
		for (int xyz=0;xyz<nix*niy*niz;xyz++) {
			float[] image = new float[6];
			float sum=0.0f;
			// (xx,yy,zz,xy,yz,zx) <- (xx,yx,yy,zx,zy,zz)
			image[0] = img_[xyz + nix*niy*niz*0];
			image[1] = img_[xyz + nix*niy*niz*2];
			image[2] = img_[xyz + nix*niy*niz*5];
			image[3] = img_[xyz + nix*niy*niz*1];
			image[4] = img_[xyz + nix*niy*niz*4];
			image[5] = img_[xyz + nix*niy*niz*3];
			for (int d=0;d<6;d++) {
				sum+=Numerics.abs(image[d]);
			}
			if (sum>0) {
				double[] eigen = Tensor.eigenDecomposition(Tensor.matrixForm(image));
				famap[xyz] = Tensor.fractionalAnisotropy(eigen);
				mdmap[xyz] = Tensor.meanDiffusivity(eigen);
				float norm = Numerics.max( image[0]+image[1]+image[2], ZERO );
				for (int w=0;w<3;w++) {
					if (eigen[0]*eigen[0]>ZERO) imval[w][xyz] = (float)Numerics.abs(eigen[w]/eigen[0]);
					else imval[w][xyz] = (float)Numerics.abs(eigen[w]);
					imdir[3*w+0][xyz] = (float)eigen[3+3*w+0];
					imdir[3*w+1][xyz] = (float)eigen[3+3*w+1];
					imdir[3*w+2][xyz] = (float)eigen[3+3*w+2];
				}
				mask[xyz] = true;
			} else {
				famap[xyz] = 0.0f;
				mdmap[xyz] = 0.0f;
				for (int w=0;w<3;w++) {
					imval[w][xyz] = 0.0f;
					for (int d=0;d<3;d++) imdir[3*w+d][xyz] = 0.0f;
				}
				mask[xyz] = false;
			}
		}
			
		topology = topo_;
		ntx = ntx_;
		nty = nty_;
		ntz = ntz_;
		rtx = rtx_;
		rty = rty_;
		rtz = rtz_;
		xCt = ntx/2.0f;
		yCt = nty/2.0f;
		zCt = ntz/2.0f;
		transformed = false;
		
		bgLabel = bgLb_;
		bgRatio = bgRatio_;
		bs = bdSz_;
		
		// orientation: initialize a rotation
		transform = new float[6];
		// note : assumes a rotation around the image center
		if (orient_==FileInfoBase.AXIAL) {
			transform[0] = 0.0f;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			if (orx_==FileInfoBase.ORI_L2R_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_P2A_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_S2I_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else if (orient_==FileInfoBase.CORONAL) {
			transform[0] = -ISQRT2;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			if (orx_==FileInfoBase.ORI_L2R_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_I2S_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_P2A_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else if (orient_==FileInfoBase.SAGITTAL) {
			transform[0] = -0.5f;
			transform[1] = -0.5f;
			transform[2] = 0.5f;
			
			if (orx_==FileInfoBase.ORI_P2A_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_I2S_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_R2L_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else {
			// default is axial
			transform[0] = 0.0f;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			orx = 1.0f;
			ory = 1.0f;
			orz = 1.0f;
		}
		transform[3] = 0.0f;
		transform[4] = 0.0f;
		transform[5] = 0.0f;
		
		// init the image cropping					
		x0i = 0;
		y0i = 0;
		z0i = 0;
		xNi = nix-1;
		yNi = niy-1;
		zNi = niz-1;
		
		// compute min,max ??
		if (debug) MedicUtilPublic.displayMessage("initialisation: done\n");
	}

	/**
	 *  constructor
	 */
	public void initTensorsDiagonal(float[] img_, 
									int nix_, int niy_, int niz_,
									float rix_, float riy_, float riz_,
									int orient_, int orx_, int ory_, int orz_,
									byte[] topo_, 
									int ntx_, int nty_, int ntz_,
									float rtx_, float rty_, float rtz_,
									byte bgLb_, float bgRatio_,
									int bdSz_) {
										
		if (debug) MedicUtilPublic.displayMessage("create mixture from tensor (diagonal convention)\n");

		nix = nix_;
		niy = niy_;
		niz = niz_;
		rix = rix_;
		riy = riy_;
		riz = riz_;
		xCi = nix/2.0f;
		yCi = niy/2.0f;
		zCi = niz/2.0f;
		cropped = false;
		normalized = false;
		nw = 3;
		
		// create the image
		imdir = new float[3*nw][nix*niy*niz];
		imval = new float[nw][nix*niy*niz];
		famap = new float[nix*niy*niz];
		mdmap = new float[nix*niy*niz];
		mask = new boolean[nix*niy*niz];
		float[][] mat = new float[3][3];
		for (int xyz=0;xyz<nix*niy*niz;xyz++) {
			float[] image = new float[6];
			float sum=0.0f;
			for (int d=0;d<6;d++) {
				image[d] = img_[xyz + nix*niy*niz*d];
				sum+=Numerics.abs(image[d]);
			}
			if (sum>0) {
				double[] eigen = Tensor.eigenDecomposition(Tensor.matrixForm(image));
				famap[xyz] = Tensor.fractionalAnisotropy(eigen);
				mdmap[xyz] = Tensor.meanDiffusivity(eigen);
				float norm = Numerics.max( image[0]+image[1]+image[2], ZERO );
				for (int w=0;w<3;w++) {
					if (eigen[0]*eigen[0]>ZERO) imval[w][xyz] = (float)Numerics.abs(eigen[w]/eigen[0]);
					else imval[w][xyz] = (float)Numerics.abs(eigen[w]);
					imdir[3*w+0][xyz] = (float)eigen[3+3*w+0];
					imdir[3*w+1][xyz] = (float)eigen[3+3*w+1];
					imdir[3*w+2][xyz] = (float)eigen[3+3*w+2];
				}
				mask[xyz] = true;
			} else {
				famap[xyz] = 0.0f;
				mdmap[xyz] = 0.0f;
				for (int w=0;w<3;w++) {
					imval[w][xyz] = 0.0f;
					for (int d=0;d<3;d++) imdir[3*w+d][xyz] = 0.0f;
				}
				mask[xyz] = false;
			}
		}
			
		topology = topo_;
		ntx = ntx_;
		nty = nty_;
		ntz = ntz_;
		rtx = rtx_;
		rty = rty_;
		rtz = rtz_;
		xCt = ntx/2.0f;
		yCt = nty/2.0f;
		zCt = ntz/2.0f;
		transformed = false;
		
		bgLabel = bgLb_;
		bgRatio = bgRatio_;
		bs = bdSz_;
		
		// orientation: initialize a rotation
		transform = new float[6];
		// note : assumes a rotation around the image center
		if (orient_==FileInfoBase.AXIAL) {
			transform[0] = 0.0f;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			if (orx_==FileInfoBase.ORI_L2R_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_P2A_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_S2I_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else if (orient_==FileInfoBase.CORONAL) {
			transform[0] = -ISQRT2;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			if (orx_==FileInfoBase.ORI_L2R_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_I2S_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_P2A_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else if (orient_==FileInfoBase.SAGITTAL) {
			transform[0] = -0.5f;
			transform[1] = -0.5f;
			transform[2] = 0.5f;
			
			if (orx_==FileInfoBase.ORI_P2A_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_I2S_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_R2L_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else {
			// default is axial
			transform[0] = 0.0f;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			orx = 1.0f;
			ory = 1.0f;
			orz = 1.0f;
		}
		transform[3] = 0.0f;
		transform[4] = 0.0f;
		transform[5] = 0.0f;
		
		// init the image cropping					
		x0i = 0;
		y0i = 0;
		z0i = 0;
		xNi = nix-1;
		yNi = niy-1;
		zNi = niz-1;
		
		// compute min,max ??
		if (debug) MedicUtilPublic.displayMessage("initialisation: done\n");
	}

	/**
	 *  constructor
	 */
	public void initDirectionMixture(float[] img_, int nw_,
									int nix_, int niy_, int niz_,
									float rix_, float riy_, float riz_,
									int orient_, int orx_, int ory_, int orz_,
									byte[] topo_, 
									int ntx_, int nty_, int ntz_,
									float rtx_, float rty_, float rtz_,
									byte bgLb_, float bgRatio_,
									int bdSz_) {
										
		if (debug) MedicUtilPublic.displayMessage("create direction mixture\n");
		
		nix = nix_;
		niy = niy_;
		niz = niz_;
		rix = rix_;
		riy = riy_;
		riz = riz_;
		xCi = nix/2.0f;
		yCi = niy/2.0f;
		zCi = niz/2.0f;
		cropped = false;
		normalized = false;
		nw = nw_;
		
		// create the image
		imdir = new float[3*nw][nix*niy*niz];
		imval = new float[nw][nix*niy*niz];
		famap = new float[nix*niy*niz];
		mdmap = new float[nix*niy*niz];
		mask = new boolean[nix*niy*niz];
		double[][] mat = new double[3][3];
		for (int xyz=0;xyz<nix*niy*niz;xyz++) {
			for (int w=0;w<nw;w++) {
				imdir[3*w+0][xyz] = img_[xyz + nix*niy*niz*(4*w+0)];
				imdir[3*w+1][xyz] = img_[xyz + nix*niy*niz*(4*w+1)];
				imdir[3*w+2][xyz] = img_[xyz + nix*niy*niz*(4*w+2)];
				imval[w][xyz] = img_[xyz + nix*niy*niz*(4*w+3)];
			}
			if (imval[0][xyz]>0) {
				for (int n=0;n<3;n++) for (int m=0;m<3;m++) {
					mat[n][m] = 0.0;
					for (int w=0;w<nw;w++) {
						mat[n][m] += imval[w][xyz]*imdir[n+3*w][xyz]*imdir[m+3*w][xyz];
					}
				}
				famap[xyz] = Tensor.fractionalAnisotropy(mat);
				mdmap[xyz] = Tensor.meanDiffusivity(mat);
				mask[xyz] = true;
			} else {
				famap[xyz] = 0.0f;
				mdmap[xyz] = 0.0f;
				mask[xyz] = false;
			}
		}
			
		topology = topo_;
		ntx = ntx_;
		nty = nty_;
		ntz = ntz_;
		rtx = rtx_;
		rty = rty_;
		rtz = rtz_;
		xCt = ntx/2.0f;
		yCt = nty/2.0f;
		zCt = ntz/2.0f;
		transformed = false;
		
		bgLabel = bgLb_;
		bgRatio = bgRatio_;
		bs = bdSz_;
		
		// orientation: initialize a rotation
		transform = new float[6];
		// note : assumes a rotation around the image center
		if (orient_==FileInfoBase.AXIAL) {
			transform[0] = 0.0f;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			if (orx_==FileInfoBase.ORI_L2R_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_P2A_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_S2I_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else if (orient_==FileInfoBase.CORONAL) {
			transform[0] = -ISQRT2;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			if (orx_==FileInfoBase.ORI_L2R_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_I2S_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_P2A_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else if (orient_==FileInfoBase.SAGITTAL) {
			transform[0] = -0.5f;
			transform[1] = -0.5f;
			transform[2] = 0.5f;
			
			if (orx_==FileInfoBase.ORI_P2A_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_I2S_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_R2L_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else {
			// default is axial
			transform[0] = 0.0f;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			orx = 1.0f;
			ory = 1.0f;
			orz = 1.0f;
		}
		transform[3] = 0.0f;
		transform[4] = 0.0f;
		transform[5] = 0.0f;
		
		// init the image cropping					
		x0i = 0;
		y0i = 0;
		z0i = 0;
		xNi = nix-1;
		yNi = niy-1;
		zNi = niz-1;
		
		// compute min,max ??
		if (debug) MedicUtilPublic.displayMessage("initialisation: done\n");
	}

	/**
	 *  constructor
	 */
	public void initCFARIModel(float[] img_, short[] lb_, int nw_,
									int nix_, int niy_, int niz_,
									float rix_, float riy_, float riz_,
									int orient_, int orx_, int ory_, int orz_,
									byte[] topo_, 
									int ntx_, int nty_, int ntz_,
									float rtx_, float rty_, float rtz_,
									byte bgLb_, float bgRatio_,
									int bdSz_) {
										
		if (debug) MedicUtilPublic.displayMessage("create CFARI model\n");
		
		nix = nix_;
		niy = niy_;
		niz = niz_;
		rix = rix_;
		riy = riy_;
		riz = riz_;
		xCi = nix/2.0f;
		yCi = niy/2.0f;
		zCi = niz/2.0f;
		cropped = false;
		normalized = false;
		nw = nw_;
		
		// create the image
		imdir = new float[3*nw][nix*niy*niz];
		imval = new float[nw][nix*niy*niz];
		famap = new float[nix*niy*niz];
		mdmap = new float[nix*niy*niz];
		mask = new boolean[nix*niy*niz];
		double[][] mat = new double[3][3];
		
		// load the vector list
		String filename = "directions241.txt";
		float[][] vec = new float[241][3];
		URL res = DotsPreprocess.class.getResource( filename );
		int nv = 0;
		try {
			if (debug) System.out.println("Opening vector list: "+res.toString());
            File f = new File(res.getFile());
			FileReader fr = new FileReader(f);
            BufferedReader br = new BufferedReader(fr);
            String line = br.readLine();
			StringTokenizer st;
			while (line!=null) {
				st = new StringTokenizer(line, " ");
				for (int i=0;i<3;i++) vec[nv][i] = MipavUtil.getFloat(st);
				nv++;
				line = br.readLine();
			}
		} catch (FileNotFoundException e) {
			System.out.println("File not found:");
			System.out.println(e.getMessage());
		} catch (IOException e ) {
			System.out.println("i/o exception:");
			System.out.println(e.getMessage());
		} catch (Exception e) {
			System.out.println(e.getMessage());
        }

		if (nv!=241) System.err.println("wrong number of vectors: "+nv);
					
		for (int xyz=0;xyz<nix*niy*niz;xyz++) {
			for (int w=0;w<nw;w++) {
				if (lb_[xyz + nix*niy*niz*w]==-1) {
					imdir[3*w+0][xyz] = 0.0f;
					imdir[3*w+1][xyz] = 0.0f;
					imdir[3*w+2][xyz] = 0.0f;
					imval[w][xyz] = 0.0f;
				} else {
					imdir[3*w+0][xyz] = vec[lb_[xyz + nix*niy*niz*w]][0];
					imdir[3*w+1][xyz] = vec[lb_[xyz + nix*niy*niz*w]][1];
					imdir[3*w+2][xyz] = vec[lb_[xyz + nix*niy*niz*w]][2];
					imval[w][xyz] = img_[xyz + nix*niy*niz*w];
				}
			}
			if (imval[0][xyz]>0) {
				for (int n=0;n<3;n++) for (int m=0;m<3;m++) {
					mat[n][m] = 0.0;
					for (int w=0;w<nw;w++) {
						mat[n][m] += imval[w][xyz]*imdir[n+3*w][xyz]*imdir[m+3*w][xyz];
					}
				}
				famap[xyz] = Tensor.fractionalAnisotropy(mat);
				mdmap[xyz] = Tensor.meanDiffusivity(mat);
				mask[xyz] = true;
				// re-normalize
				for (int w=0;w<nw;w++) {
					imval[w][xyz] /= imval[0][xyz];
				}
			} else {
				famap[xyz] = 0.0f;
				mdmap[xyz] = 0.0f;
				mask[xyz] = false;
			}
		}
			
		topology = topo_;
		ntx = ntx_;
		nty = nty_;
		ntz = ntz_;
		rtx = rtx_;
		rty = rty_;
		rtz = rtz_;
		xCt = ntx/2.0f;
		yCt = nty/2.0f;
		zCt = ntz/2.0f;
		transformed = false;
		
		bgLabel = bgLb_;
		bgRatio = bgRatio_;
		bs = bdSz_;
		
		// orientation: initialize a rotation
		transform = new float[6];
		// note : assumes a rotation around the image center
		if (orient_==FileInfoBase.AXIAL) {
			transform[0] = 0.0f;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			if (orx_==FileInfoBase.ORI_L2R_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_P2A_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_S2I_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else if (orient_==FileInfoBase.CORONAL) {
			transform[0] = -ISQRT2;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			if (orx_==FileInfoBase.ORI_L2R_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_I2S_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_P2A_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else if (orient_==FileInfoBase.SAGITTAL) {
			transform[0] = -0.5f;
			transform[1] = -0.5f;
			transform[2] = 0.5f;
			
			if (orx_==FileInfoBase.ORI_P2A_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (ory_==FileInfoBase.ORI_I2S_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (orz_==FileInfoBase.ORI_R2L_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else {
			// default is axial
			transform[0] = 0.0f;
			transform[1] = 0.0f;
			transform[2] = 0.0f;
			
			orx = 1.0f;
			ory = 1.0f;
			orz = 1.0f;
		}
		transform[3] = 0.0f;
		transform[4] = 0.0f;
		transform[5] = 0.0f;
		
		// init the image cropping					
		x0i = 0;
		y0i = 0;
		z0i = 0;
		xNi = nix-1;
		yNi = niy-1;
		zNi = niz-1;
		
		// compute min,max ??
		if (debug) MedicUtilPublic.displayMessage("initialisation: done\n");
	}

	final public void finalize() {
		imdir = null;
		imval = null;
		famap = null;
		mdmap = null;
		transform = null;
		System.gc();
	}
	
	public final float[][] getDirections() { return imdir; }
    
	public final float[][] getWeights() { return imval; }
    
	public final int getDirectionNumber() { return nw; }
	
	public final float[] getFAmap() { return famap; }
    
	public final float[] getMDmap() { return mdmap; }
    
	public final boolean[] getMask() { return mask; }
    
	public final float[] getLabeling() { return labeling; }
    
	public final String getModality() { return modality; }
    
	public final void setTransform(float[] trans) {
        if (trans.length==6) transform = trans;
		else System.err.println("wrong transform type");
    }
    
    public final float[] getTransform() {
        return transform;
    }

	public final float[] exportTransform() {
		float[] trans = new float[6];
		for (int i=0;i<6;i++)
			trans[i] = transform[i];
        return trans;
    }
	
	public final String displayTransform() {
		String info;
		
		info = "transform: ("+transform[0]+", "+transform[1]+", "+transform[2]+", "
						     +transform[3]+", "+transform[4]+", "+transform[5]+")\n";
		
		return info;
	}
	
	/** current dimensions */
	public final int[] getCurrentImageDimensions() {
		int[] dim = new int[3];
        if (cropped) {
			dim[0] = mix; dim[1] = miy; dim[2] = miz;
		} else {
			dim[0] = nix; dim[1] = niy; dim[2] = niz;
		}
		if (debug) MedicUtilPublic.displayMessage("current image dimensions: "+dim[0]+" x "+dim[1]+" x "+dim[2]+"\n");
        
		return dim;
    }
	
	public final int[] getOriginalImageDimensions() {
		int[] dim = new int[3];
        dim[0] = nix; dim[1] = niy; dim[2] = niz;
		
		return dim;
    }
	public final int[] getOriginalImageArrayDimensions(int size) {
		int[] dim = new int[4];
        dim[0] = nix; dim[1] = niy; dim[2] = niz; dim[3] = size;
		
		return dim;
    }
	public final int getOriginalImageSize() {
		return nix*niy*niz;
    }
	public final int[] getCroppedImageDimensions() {
		int[] dim = new int[3];
        dim[0] = mix; dim[1] = miy; dim[2] = miz;
		
		return dim;
    }
	
	/** current resolutions, with the sign corresponding to the axis orientation */
	public final float[] getSignedImageResolutions() {
		float[] res = new float[3];
        res[0] = rix/orx; res[1] = riy/ory; res[2] = riz/orz;
		
		if (debug) MedicUtilPublic.displayMessage("current image resolutions: "+res[0]+" x "+res[1]+" x "+res[2]+"\n");
        
		return res;
    }
	
	public final float[] getImageResolutions() {
		float[] res = new float[3];
        res[0] = rix; res[1] = riy; res[2] = riz;
		
		if (debug) MedicUtilPublic.displayMessage("current image resolutions: "+res[0]+" x "+res[1]+" x "+res[2]+"\n");
        
		return res;
    }
	
	public final float[] getImageOrientations() {
		float[] ori = new float[3];
        ori[0] = orx; ori[1] = ory; ori[2] = orz;
		
		if (debug) MedicUtilPublic.displayMessage("current image orientations: "+ori[0]+" x "+ori[1]+" x "+ori[2]+"\n");
        
		return ori;
    }
	
	/** current dimensions */
	public final void updateTransformedTemplate(byte[] tpl) {	
		topology = tpl;
		transformed = true;
	}	
	
    /** sets the bounding box for cropping from a vector image */
	public void findCroppingBoundaries() {
		x0i = nix-1;
        xNi = 0;
        y0i = niy-1;
        yNi = 0;
        z0i = niz-1;
        zNi = 0;

		RotationMatrix rotation = new RotationMatrix();
		rotation.setParameters(transform[0],transform[1],transform[2]);
		
		// check all images 
		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			if (famap[x+nix*y+nix*niy*z]> bgRatio) {
				if (x < x0i) x0i = x;
				if (x > xNi) xNi = x;
				if (y < y0i) y0i = y;
				if (y > yNi) yNi = y;
				if (z < z0i) z0i = z;
				if (z > zNi) zNi = z;
			}
		}
		/*
		// check for the topology: from topology to image space
		for (int x=0;x<ntx;x++) for (int y=0;y<nty;y++) for (int z=0;z<ntz;z++) {
			if (topology[x+ntx*y+ntx*nty*z]>bgLabel) {
				// get image coordinates
				float[] XI = computeImageCoordinates(x,y,z,transform,rotation.getMatrix());
				
				if (XI[0] < x0i) x0i = Numerics.floor(XI[0]);
				if (XI[0] > xNi) xNi = Numerics.ceil(XI[0]);
				if (XI[1] < y0i) y0i = Numerics.floor(XI[1]);
				if (XI[1] > yNi) yNi = Numerics.ceil(XI[1]);
				if (XI[2] < z0i) z0i = Numerics.floor(XI[2]);
				if (XI[2] > zNi) zNi = Numerics.ceil(XI[2]);
			}
		}
		*/
        // debug
		// hack for 2D
		//zNi = zNi-1;
        if (debug) MedicUtilPublic.displayMessage("boundaries: ["+x0i+","+xNi+"] ["+y0i+","+yNi+"] ["+z0i+","+zNi+"]\n");
        
        return;
    }
	
	/** crop the images */
	public void cropImages() {
		if (cropped) return;
		
		mix = xNi-x0i+1+2*bs;
		miy = yNi-y0i+1+2*bs;
		miz = zNi-z0i+1+2*bs;
		
		// update the transform parameters
		xCi = mix/2.0f;
		yCi = miy/2.0f;
		zCi = miz/2.0f;
		
		RotationMatrix rotation = new RotationMatrix();
		rotation.setParameters(transform[0],transform[1],transform[2]);
		float[][] R = rotation.getMatrix();
		transform[3] += R[0][0]*(mix/2.0f+x0i-bs-nix/2.0f)*rix/orx+R[0][1]*(miy/2.0f+y0i-bs-niy/2.0f)*riy/ory+R[0][2]*(miz/2.0f+z0i-bs-niz/2.0f)*riz/orz;
		transform[4] += R[1][0]*(mix/2.0f+x0i-bs-nix/2.0f)*rix/orx+R[1][1]*(miy/2.0f+y0i-bs-niy/2.0f)*riy/ory+R[1][2]*(miz/2.0f+z0i-bs-niz/2.0f)*riz/orz;
		transform[5] += R[2][0]*(mix/2.0f+x0i-bs-nix/2.0f)*rix/orx+R[2][1]*(miy/2.0f+y0i-bs-niy/2.0f)*riy/ory+R[2][2]*(miz/2.0f+z0i-bs-niz/2.0f)*riz/orz;
		
		float[] smallfa = new float[mix*miy*miz];
		float[] smallmd = new float[mix*miy*miz];
		float[][] smalldir = new float[imdir.length][mix*miy*miz];
		float[][] smallval=null;
		if (imval!=null) smallval= new float[nw][mix*miy*miz];
		boolean[] smallmask =  new boolean[mix*miy*miz];
		float[] smalllb=null;
		if (labeling!=null) smalllb= new float[mix*miy*miz];
		
		for (int xyz=0;xyz<mix*miy*miz;xyz++) {
			smallfa[xyz] = 0.0f;
			smallmd[xyz] = 0.0f;
			for (int d=0;d<imdir.length;d++) {
				smalldir[d][xyz] = 0.0f;
			}
			for (int w=0;w<nw;w++) {
				if (imval!=null) smallval[w][xyz] = 0.0f;
			}
			smallmask[xyz] = false;
			if (labeling!=null) smalllb[xyz] = 0.0f;
		}
		for (int x=x0i;x<=xNi;x++) {
			for (int y=y0i;y<=yNi;y++) {
				for (int z=z0i;z<=zNi;z++) {
					if ( (x<0) || (x>=nix) || (y<0) || (y>=niy) || (z<0) || (z>=niz) ) {
						smallfa[x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)] = 0.0f;
						smallmd[x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)] = 0.0f;
						for (int d=0;d<imdir.length;d++) smalldir[d][x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)] = 0.0f;
						for (int w=0;w<nw;w++) {
							if (imval!=null) smallval[w][x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)] = 0.0f;
						}
						smallmask[x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)] = false;
						if (labeling!=null) smalllb[x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)] = 0.0f;
					} else {
						smallfa[x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)] = famap[x+nix*y+nix*niy*z];
						smallmd[x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)] = mdmap[x+nix*y+nix*niy*z];
						for (int d=0;d<imdir.length;d++) smalldir[d][x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)] = imdir[d][x+nix*y+nix*niy*z];
						for (int w=0;w<nw;w++) {
							if (imval!=null) smallval[w][x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)] = imval[w][x+nix*y+nix*niy*z];
						}
						smallmask[x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)] = mask[x+nix*y+nix*niy*z];
						if (labeling!=null) smalllb[x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)] = labeling[x+nix*y+nix*niy*z];
					}
				}
			}
		}
		// replace the original images
		famap = smallfa;
		mdmap = smallmd;
		imdir = smalldir;
		if (imval!=null) imval = smallval;
		mask = smallmask;
		if (labeling!=null) labeling = smalllb;
			
		cropped = true;
		
		return;
	}
	
	/** uncrop the images */
	public void uncropImages() {
		if (!cropped) return;
		
		// update the transform parameters
		xCi = nix/2.0f;
		yCi = niy/2.0f;
		zCi = niz/2.0f;
		
		RotationMatrix rotation = new RotationMatrix();
		rotation.setParameters(transform[0],transform[1],transform[2]);
		float[][] R = rotation.getMatrix();
		transform[3] -= R[0][0]*(mix/2.0f+x0i-bs-nix/2.0f)*rix/orx+R[0][1]*(miy/2.0f+y0i-bs-niy/2.0f)*riy/ory+R[0][2]*(miz/2.0f+z0i-bs-niz/2.0f)*riz/orz;
		transform[4] -= R[1][0]*(mix/2.0f+x0i-bs-nix/2.0f)*rix/orx+R[1][1]*(miy/2.0f+y0i-bs-niy/2.0f)*riy/ory+R[1][2]*(miz/2.0f+z0i-bs-niz/2.0f)*riz/orz;
		transform[5] -= R[2][0]*(mix/2.0f+x0i-bs-nix/2.0f)*rix/orx+R[2][1]*(miy/2.0f+y0i-bs-niy/2.0f)*riy/ory+R[2][2]*(miz/2.0f+z0i-bs-niz/2.0f)*riz/orz;
		
		float[] largefa = new float[nix*niy*niz];
		float[] largemd = new float[nix*niy*niz];
		float[][] largedir = new float[imdir.length][nix*niy*niz];
		float[][] largeval = null;
		if (imval!=null) largeval = new float[nw][nix*niy*niz];
		boolean[] largemask = new boolean[nix*niy*niz];
		float[] largelb = null;
		if (labeling!=null) largelb = new float[nix*niy*niz];
			
		for (int xyz=0;xyz<nix*niy*niz;xyz++) {
			largefa[xyz] = 0.0f;
			largemd[xyz] = 0.0f;
			for (int d=0;d<imdir.length;d++) largedir[d][xyz] = 0.0f;
			for (int w=0;w<nw;w++) {
				if (imval!=null) largeval[w][xyz] = 0.0f;
			}
			largemask[xyz] = false;
			if (labeling!=null) largelb[xyz] = 0.0f;
		}		
		for (int x=Numerics.max(x0i-bs,0);x<=Numerics.min(xNi+bs,nix-1);x++) {
			for (int y=Numerics.max(y0i-bs,0);y<=Numerics.min(yNi+bs,niy-1);y++) {
				for (int z=Numerics.max(z0i-bs,0);z<=Numerics.min(zNi+bs,niz-1);z++) {
					largefa[x+nix*y+nix*niy*z] = famap[x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)];
					largemd[x+nix*y+nix*niy*z] = mdmap[x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)];
					for (int d=0;d<imdir.length;d++) largedir[d][x+nix*y+nix*niy*z] = imdir[d][x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)];
					for (int w=0;w<nw;w++) {
						if (imval!=null) largeval[w][x+nix*y+nix*niy*z] = imval[w][x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)];
					}
					largemask[x+nix*y+nix*niy*z] = mask[x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)];
					if (labeling!=null) largelb[x+nix*y+nix*niy*z] = labeling[x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)];
				}
			}
		}
		famap = largefa;
		mdmap = largemd;
		imdir = largedir;
		if (imval!=null) imval = largeval;
		mask = largemask;
		if (labeling!=null) labeling = largelb;
		
		cropped = false;
		return;
	}

	/** uncrop the image, send it to a 1D buffer*/
	public float[] uncropAndBuffer(float[] src, float bgVal) {
		float[] larger = new float[nix*niy*niz];

		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			larger[x + nix*y + nix*niy*z] = bgVal;
		}		
		for (int x=Numerics.max(x0i-bs,0);x<=Numerics.min(xNi+bs,nix-1);x++) {
			for (int y=Numerics.max(y0i-bs,0);y<=Numerics.min(yNi+bs,niy-1);y++) {
				for (int z=Numerics.max(z0i-bs,0);z<=Numerics.min(zNi+bs,niz-1);z++) {
					larger[x + nix*y + nix*niy*z] = src[x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)];
				}
			}
		}
		src = null;
		
		return larger;
	}

	/** uncrop the image, send it to a 1D buffer*/
	public float[] uncropAndBuffer(float[][] src, int nc, float bgVal) {
		float[] larger = new float[nix*niy*niz*nc];

		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			for (int c=0;c<nc;c++) {
				larger[x + nix*y + nix*niy*z + nix*niy*niz*c] = bgVal;
			}
		}		
		for (int x=Numerics.max(x0i-bs,0);x<=Numerics.min(xNi+bs,nix-1);x++) {
			for (int y=Numerics.max(y0i-bs,0);y<=Numerics.min(yNi+bs,niy-1);y++) {
				for (int z=Numerics.max(z0i-bs,0);z<=Numerics.min(zNi+bs,niz-1);z++) {
					for (int c=0;c<nc;c++) {
						larger[x + nix*y + nix*niy*z + nix*niy*niz*c] = src[c][x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)];
					}
				}
			}
		}
		src = null;
		
		return larger;
	}

	/** uncrop the image, send it to a 1D buffer*/
	public byte[] uncropAndBuffer(byte[] src, byte bgVal) {
		byte[] larger = new byte[nix*niy*niz];

		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			larger[x + nix*y + nix*niy*z] = bgVal;
		}		
		for (int x=Numerics.max(x0i-bs,0);x<=Numerics.min(xNi+bs,nix-1);x++) {
			for (int y=Numerics.max(y0i-bs,0);y<=Numerics.min(yNi+bs,niy-1);y++) {
				for (int z=Numerics.max(z0i-bs,0);z<=Numerics.min(zNi+bs,niz-1);z++) {
					larger[x + nix*y + nix*niy*z] = src[x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)];
				}
			}
		}
		src = null;
		
		return larger;
	}

	/** uncrop the image, send it to a 1D buffer*/
	public boolean[] uncropAndBuffer(boolean[][] src, int nc, boolean bgVal) {
		boolean[] larger = new boolean[nix*niy*niz*nc];

		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			for (int c=0;c<nc;c++) {
				larger[x + nix*y + nix*niy*z + nix*niy*niz*c] = bgVal;
			}
		}		
		for (int x=Numerics.max(x0i-bs,0);x<=Numerics.min(xNi+bs,nix-1);x++) {
			for (int y=Numerics.max(y0i-bs,0);y<=Numerics.min(yNi+bs,niy-1);y++) {
				for (int z=Numerics.max(z0i-bs,0);z<=Numerics.min(zNi+bs,niz-1);z++) {
					for (int c=0;c<nc;c++) {
						larger[x + nix*y + nix*niy*z + nix*niy*niz*c] = src[c][x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)];
					}
				}
			}
		}
		src = null;
		
		return larger;
	}

	/** uncrop the image, send it to a 1D buffer*/
	public byte[] uncropAndBuffer(byte[][] src, int nc, byte bgVal) {
		byte[] larger = new byte[nix*niy*niz*nc];

		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			for (int c=0;c<nc;c++) {
				larger[x + nix*y + nix*niy*z + nix*niy*niz*c] = bgVal;
			}
		}		
		for (int x=Numerics.max(x0i-bs,0);x<=Numerics.min(xNi+bs,nix-1);x++) {
			for (int y=Numerics.max(y0i-bs,0);y<=Numerics.min(yNi+bs,niy-1);y++) {
				for (int z=Numerics.max(z0i-bs,0);z<=Numerics.min(zNi+bs,niz-1);z++) {
					for (int c=0;c<nc;c++) {
						larger[x + nix*y + nix*niy*z + nix*niy*niz*c] = src[c][x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)];
					}
				}
			}
		}
		src = null;
		
		return larger;
	}

	/** uncrop the image, send it to a 1D buffer*/
	public short[] uncropAndBuffer(short[] src, short bgVal) {
		short[] larger = new short[nix*niy*niz];

		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			larger[x + nix*y + nix*niy*z] = bgVal;
		}		
		for (int x=Numerics.max(x0i-bs,0);x<=Numerics.min(xNi+bs,nix-1);x++) {
			for (int y=Numerics.max(y0i-bs,0);y<=Numerics.min(yNi+bs,niy-1);y++) {
				for (int z=Numerics.max(z0i-bs,0);z<=Numerics.min(zNi+bs,niz-1);z++) {
					larger[x + nix*y + nix*niy*z] = src[x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)];
				}
			}
		}
		src = null;
		
		return larger;
	}

	/** uncrop the image, send it to a 1D buffer*/
	public short[] uncropAndBuffer(short[][] src, int nc, short bgVal) {
		short[] larger = new short[nix*niy*niz*nc];

		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			for (int c=0;c<nc;c++) {
				larger[x + nix*y + nix*niy*z + nix*niy*niz*c] = bgVal;
			}
		}		
		for (int x=Numerics.max(x0i-bs,0);x<=Numerics.min(xNi+bs,nix-1);x++) {
			for (int y=Numerics.max(y0i-bs,0);y<=Numerics.min(yNi+bs,niy-1);y++) {
				for (int z=Numerics.max(z0i-bs,0);z<=Numerics.min(zNi+bs,niz-1);z++) {
					for (int c=0;c<nc;c++) {
						larger[x + nix*y + nix*niy*z + nix*niy*niz*c] = src[c][x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)];
					}
				}
			}
		}
		src = null;
		
		return larger;
	}

	/** crop the images */
	public float[][] generateCroppedDirection() {
		if (cropped) return imdir;
		
		int mx = xNi-x0i+1+2*bs;
		int my = yNi-y0i+1+2*bs;
		int mz = zNi-z0i+1+2*bs;
		
		float[][] smaller = new float[3*nw][mx*my*mz];

		for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
			for (int d=0;d<3*nw;d++) smaller[d][x+mx*y+mx*my*z] = 0.0f;
		}
		for (int x=x0i;x<=xNi;x++) {
			for (int y=y0i;y<=yNi;y++) {
				for (int z=z0i;z<=zNi;z++) {
					for (int d=0;d<3*nw;d++) smaller[d][x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)] = imdir[d][x+mx*y+mx*my*z];
				}
			}
		}
		
		return smaller;
	}
	
	/** crop the images */
	public float[][] generateCroppedTensor() {
		if (cropped) return imdir;
		
		int mx = xNi-x0i+1+2*bs;
		int my = yNi-y0i+1+2*bs;
		int mz = zNi-z0i+1+2*bs;
		
		float[][] smaller = new float[6][mx*my*mz];

		for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
			for (int d=0;d<3*nw;d++) smaller[d][x+mx*y+mx*my*z] = 0.0f;
		}
		for (int x=x0i;x<=xNi;x++) {
			for (int y=y0i;y<=yNi;y++) {
				for (int z=z0i;z<=zNi;z++) {
					for (int d=0;d<6;d++) smaller[d][x-x0i+bs + mix*(y-y0i+bs) + mix*miy*(z-z0i+bs)] = imdir[d][x+mx*y+mx*my*z];
				}
			}
		}
		
		return smaller;
	}
	
    /** 
	 *	returns the transformed topology template.
	 *	the template is cropped and a border is added.
	 *	note: this transformation should only be attempted once!!
	 */
	public final void transformTopology() {
		if (transformed) return;
		
		// new dimensions
		int mx = xNi-x0i+1+2*bs;
		int my = yNi-y0i+1+2*bs;
		int mz = zNi-z0i+1+2*bs;
		
		byte[] smaller = new byte[mx*my*mz];
		
		RotationMatrix rotation = new RotationMatrix();
		rotation.setParameters(transform[0],transform[1],transform[2]);
		
		for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
			smaller[x+mx*y+mx*my*z] = bgLabel;
		}
		for (int x=x0i;x<=xNi;x++) for (int y=y0i;y<=yNi;y++) for (int z=z0i;z<=zNi;z++) {
			// coordinates in topology space
			float[] XT = computeTemplateCoordinates(x,y,z,transform,rotation.getMatrix());
			
			smaller[x-x0i+bs + mx*(y-y0i+bs) + mx*my*(z-z0i+bs)] = ImageFunctions.nearestNeighborInterpolation(topology,bgLabel,XT[0],XT[1],XT[2],ntx,nty,ntz);
		}
		topology = smaller;
		transformed = true;
		
		return;
	} // transformTopology

    /** 
	 *	returns the transformed topology template.
	 *	the template is cropped and a border is added.
	 *	note: this transformation should only be attempted once!!
	 */
	public final byte[] generateTransformedTopology() {
		if (transformed) return topology;
		
		// new dimensions
		int mx = xNi-x0i+1+2*bs;
		int my = yNi-y0i+1+2*bs;
		int mz = zNi-z0i+1+2*bs;
		
		byte[] smaller = new byte[mx*my*mz];
		
		RotationMatrix rotation = new RotationMatrix();
		rotation.setParameters(transform[0],transform[1],transform[2]);
		
		for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
			smaller[x+mx*y+mx*my*z] = bgLabel;
		}
		for (int x=x0i;x<=xNi;x++) for (int y=y0i;y<=yNi;y++) for (int z=z0i;z<=zNi;z++) {
			// coordinates in topology space
			float[] XT = computeTemplateCoordinates(x,y,z,transform,rotation.getMatrix());
			
			smaller[x-x0i+bs + mx*(y-y0i+bs) + mx*my*(z-z0i+bs)] = ImageFunctions.nearestNeighborInterpolation(topology,bgLabel,XT[0],XT[1],XT[2],ntx,nty,ntz);
		}
		return smaller;
	} // transformTopology

    /** 
	 *  transform an image aligned with the topology
	 *	the template is cropped and a border is added.
	 */
	public final float[] transformNewImage(float[] img, float bgVal) {
		
		// new dimensions
		int mx = xNi-x0i+1+2*bs;
		int my = yNi-y0i+1+2*bs;
		int mz = zNi-z0i+1+2*bs;
		
		float[] smaller = new float[mx*my*mz];
		
		RotationMatrix rotation = new RotationMatrix();
		rotation.setParameters(transform[0],transform[1],transform[2]);
		
		for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
			smaller[x+mx*y+mx*my*z] = bgVal;
		}
		for (int x=x0i;x<=xNi;x++) for (int y=y0i;y<=yNi;y++) for (int z=z0i;z<=zNi;z++) {
			// coordinates in topology space
			float[] XT = computeTemplateCoordinates(x,y,z,transform,rotation.getMatrix());
			
			smaller[x-x0i+bs + mx*(y-y0i+bs) + mx*my*(z-z0i+bs)] = ImageFunctions.linearInterpolation(img,bgVal,XT[0],XT[1],XT[2],ntx,nty,ntz);
		}
		return smaller;
	} // transformTopology

	/** 
	 *	computes the transformed coordinates from template to image space
	 */
	private final float[] computeImageCoordinates(int x,int y,int z,float[] trans, float[][] rot) {
		float[] XI = new float[3];
		XI[0] = (rot[0][0]*((x-xCt)*rtx-trans[3])+rot[1][0]*((y-yCt)*rty-trans[4])+rot[2][0]*((z-zCt)*rtz-trans[5]))*orx/rix + xCi;
		XI[1] = (rot[0][1]*((x-xCt)*rtx-trans[3])+rot[1][1]*((y-yCt)*rty-trans[4])+rot[2][1]*((z-zCt)*rtz-trans[5]))*ory/riy + yCi;
		XI[2] = (rot[0][2]*((x-xCt)*rtx-trans[3])+rot[1][2]*((y-yCt)*rty-trans[4])+rot[2][2]*((z-zCt)*rtz-trans[5]))*orz/riz + zCi;
		
		return XI;
	}
	
	/** 
	 *	computes the transformed coordinates from image to template space
	 */
	private final float[] computeTemplateCoordinates(int x,int y,int z,float[] trans, float[][] rot) {
		float[] XT = new float[3];
		
		XT[0] = (rot[0][0]*(x-xCi)*rix/orx+rot[0][1]*(y-yCi)*riy/ory+rot[0][2]*(z-zCi)*riz/orz+trans[3])/rtx + xCt;
		XT[1] = (rot[1][0]*(x-xCi)*rix/orx+rot[1][1]*(y-yCi)*riy/ory+rot[1][2]*(z-zCi)*riz/orz+trans[4])/rty + yCt;
		XT[2] = (rot[2][0]*(x-xCi)*rix/orx+rot[2][1]*(y-yCi)*riy/ory+rot[2][2]*(z-zCi)*riz/orz+trans[5])/rtz + zCt;
		
		return XT;
	}
	

}
