package edu.jhmi.rad.medic.methods;

import java.io.*;
import java.util.*;
import gov.nih.mipav.view.*;

import gov.nih.mipav.model.structures.jama.*;

import edu.jhmi.rad.medic.libraries.*;
import edu.jhmi.rad.medic.structures.*;
import edu.jhmi.rad.medic.utilities.*;

/**
 *
 *  This algorithm segments DTI fiber bundles with message propagation
 *  while preserving their topology using a fast marching technique.
 *	
 *	@version    Jun 2008
 *	@author     Pierre-Louis Bazin 
 *		
 *
 */
 
public class DotsPropagation {
	
	// object types

	private	static	final	byte	EMPTY = -1;
			
	public	static	final	byte	X = 0;
	public	static	final	byte	Y = 1;
	public	static	final	byte	Z = 2;
	public	static 	final 	byte 	XpY = 3;
	public	static 	final 	byte 	YpZ = 4;
	public	static 	final 	byte 	ZpX = 5;
	public	static 	final 	byte 	XmY = 6;
	public	static 	final 	byte 	YmZ = 7;
	public	static 	final 	byte 	ZmX = 8;
	public	static 	final 	byte 	XpYpZ = 9;
	public	static 	final 	byte 	XmYmZ = 10;
	public	static 	final 	byte 	XmYpZ = 11;
	public	static 	final 	byte 	XpYmZ = 12;
	public	static	final	byte	mX = 13;
	public	static	final	byte	mY = 14;
	public	static	final	byte	mZ = 15;
	public	static 	final 	byte 	mXpY = 16;
	public	static 	final 	byte 	mYpZ = 17;
	public	static 	final 	byte 	mZpX = 18;
	public	static 	final 	byte 	mXmY = 19;
	public	static 	final 	byte 	mYmZ = 20;
	public	static 	final 	byte 	mZmX = 21;
	public	static 	final 	byte 	mXpYpZ = 22;
	public	static 	final 	byte 	mXmYmZ = 23;
	public	static 	final 	byte 	mXmYpZ = 24;
	public	static 	final 	byte 	mXpYmZ = 25;

	// data types
	public static final byte	FADIR		=	11;
	public static final byte	MIXTURE		=	13;
	
	// labeling types
	public static final byte	NONE		=	21;
	public static final byte	LESIONS		=	22;
	
	// numerical quantities
	private static final	float   INF=1e15f;
	private static final	float   ZERO=1e-15f;
	private static final	float	PI2 = (float)(Math.PI/2.0);
	private static final	float	SQRT2 = (float)Math.sqrt(2.0);
	private static final	float	SQRT3 = (float)Math.sqrt(3.0);
	private static final	byte	UNKNOWN = -1;
	
	// data and membership buffers
	private 	float[][] 		imdir;  			// original images
	private 	float[][] 		imval;  			// original images
	private 	float[] 		adc;  			// original images
	private 	boolean[] 		mask;  				// original images
	private 	float[] 		labeling;   			// prior labeling
	private 	float[] 		maskprior;   			// gain functions for likely tracts
	private 	float[][] 		gain;   			// gain functions for likely tracts
	private 	float[][] 		prev;   			// gain functions for likely tracts
	private 	short[][] 		best;   			// label functions for likely tracts
	private		float[]			similarityF1;
	private		float[]			similarityF2;
	private		float[]			similarityO1;
	private		float[]			similarityO2;
	private		byte[]			neighborF1;
	private 	byte[]			neighborF2;
	private		byte[]			neighborO1;
	private 	byte[]			neighborO2;
	private static	int 		nx,ny,nz,nw;   		// images dimensions
	private static	float 		rx,ry,rz;   		// images resolutions
	private static	int 	  	nt;					// total number of tracts 
	private static	int 	  	nb;					// number of estimated tracts 
	private static	int 	  	nc = 13;			// half number of neighbors 
	private			DotsAtlas	atlas;
	private static 	byte		dataType;
	private static 	byte		lbType;
	//private		float			adci,adcf;
	
	// parameters
	
	private		float			isoDiffusion;
	private		float			overlapGain = 2.0f;		// the amount of the energy spent on priors 
	
	private		float			gainFactor;
	
	private		float			similarityFactor = 1.0f;
	private		static float	priorMinimum = 0.0f;
		
	private		float			maskdist = 10.0f;
	private		float			nbmask;
	
	// for debug and display
	private static final boolean		debug=false;
	private static final boolean		verbose=true;
	
	
	/**
	 *  constructors for different cases: with/out outliers, with/out selective constraints
	 */
	public DotsPropagation(String modal_, float[][] imdir_, float[][] imval_, float[] adc_,
						boolean[] mask_, DotsAtlas atlas_,
						float[] lb_, String lbType_,
						int nb_,
						int nx_, int ny_, int nz_, int nw_,
						float rx_, float ry_, float rz_,
						float isoDiff_, 
						float gain_) {
		
		if (modal_.equals("Tensor") || modal_.equals("CFARI")) dataType = MIXTURE;
		else dataType = FADIR;
		
		imdir = imdir_;
		imval = imval_;
		adc = adc_;
		System.out.println("data type: "+dataType+", length: "+imdir.length);
		
		mask = mask_;
		atlas = atlas_;
		
		labeling = lb_;
		if (lbType_.equals("lesions")) lbType = LESIONS;
		else lbType = NONE;
		
		nt = atlas.getNumber();
		nb = nb_;
		
		nx = nx_;
		ny = ny_;
		nz = nz_;
		nw = nw_;
		
		rx = rx_;
		ry = ry_;
		rz = rz_;
		
		// coefficient (assumes normalized imagess)
		
		isoDiffusion = isoDiff_;
		
		gainFactor = gain_;
		
		// init all the arrays
		try {
			gain = new float[nb][nx*ny*nz];
			prev = new float[nb][nx*ny*nz];
			best = new short[nb][nx*ny*nz];
			
			similarityF1 = new float[nx*ny*nz];
			similarityF2 = new float[nx*ny*nz];
			similarityO1 = new float[nx*ny*nz];
			similarityO2 = new float[nx*ny*nz];
			neighborF1 = new byte[nx*ny*nz];
			neighborF2 = new byte[nx*ny*nz];
			neighborO1 = new byte[nx*ny*nz];
			neighborO2 = new byte[nx*ny*nz];
			
		} catch (OutOfMemoryError e){
			 finalize();
			System.out.println(e.getMessage());
			return;
		}
		
		// init values
		nbmask = 0.0f;
		for (int xyz=0;xyz<nx*ny*nz;xyz++) {
			if (mask[xyz]) {
				for (byte b=0;b<nb;b++) {
					gain[b][xyz] = 0.0f;
					best[b][xyz] = UNKNOWN;
				}
				nbmask++;
			} else {
				gain[0][xyz] = 0.5f;
				best[0][xyz] = 0;
				for (byte b=1;b<nb;b++) {
					gain[b][xyz] = 0.0f;
					best[b][xyz] = UNKNOWN;
				}
			}
			similarityF1[xyz] = 0.0f;
			similarityF2[xyz] = 0.0f;
			similarityO1[xyz] = 0.0f;
			similarityO2[xyz] = 0.0f;
			neighborF1[xyz] = UNKNOWN;
			neighborF2[xyz] = UNKNOWN;
			neighborO1[xyz] = UNKNOWN;
			neighborO2[xyz] = UNKNOWN;
		}
		estimateDiffusionSimilarity();
		//precomputeADCpriors();
		precomputeMaskPrior();
		
		if (debug) MedicUtilPublic.displayMessage("initialization\n");
	}
		
	public void finalize() {
		imdir = null;
		imval = null;
		gain = null;
		prev = null;
		best = null;
		similarityF1 = null;
		similarityF2 = null;
		similarityO1 = null;
		similarityO2 = null;
		neighborF1 = null;
		neighborF2 = null;
		neighborO1 = null;
		neighborO2 = null;
	}
	
	/**
	 *	clean up the computation arrays
	 */
	public final void cleanUp() {
		imdir = null;
		imval = null;
		System.gc();
	}

	public final float[][] getGain() { return gain; }
	
	public final short[][] getLabels() { return best; }
    
	public final float[] getMaskPrior() { return maskprior; }
	
	public final void computeAtlasPriors() {
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			int id = x + nx*y + nx*ny*z;
			if (mask[id]) {
				priorDistance(x,y,z,id);
			}
		}
	}
	
	public final float computeIterativeGainMaxDiffusion() {
		// record previous gain
		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			int id = x + nx*y + nx*ny*z;
			if (mask[id]) {
				for (int n=0;n<nb;n++) {
					prev[n][id] = gain[n][id];
				}
			}
		}
		
		float diff = 0.0f;
		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			int id = x + nx*y + nx*ny*z;
			if (mask[id]) {
				diff += gainIterativePrecomputedSimilarityDiffusion(x,y,z,id);
			}
		}
		MedicUtilPublic.displayMessage(" changed labels: "+diff/nbmask+"\n");
		
		return diff/nbmask;
	}
	
	/**
	 *  distances and other functions for computing memberships
	 */
	private final float signedAngleFiberSimilarityGain(int dir, int id1, int id2) {
		float prod = (imdir[0][id1]*imdir[0][id2]+imdir[1][id1]*imdir[1][id2]+imdir[2][id1]*imdir[2][id2]);
		float dataAngle = (float)Math.acos(Numerics.min(1.0f,Numerics.abs(prod)))/PI2;
		
		float dv1=0, dv2=0;
			
			 if (dir==X) { dv1 = imdir[0][id1]; dv2 = imdir[0][id2]; }
		else if (dir==Y) { dv1 = imdir[1][id1]; dv2 = imdir[1][id2]; }
		else if (dir==Z) { dv1 = imdir[2][id1]; dv2 = imdir[2][id2]; }
		else if (dir==XpY) { dv1 = imdir[0][id1]/SQRT2+imdir[1][id1]/SQRT2; dv2 = imdir[0][id2]/SQRT2+imdir[1][id2]/SQRT2; }
		else if (dir==YpZ) { dv1 = imdir[1][id1]/SQRT2+imdir[2][id1]/SQRT2; dv2 = imdir[1][id2]/SQRT2+imdir[2][id2]/SQRT2; }
		else if (dir==ZpX) { dv1 = imdir[2][id1]/SQRT2+imdir[0][id1]/SQRT2; dv2 = imdir[2][id2]/SQRT2+imdir[0][id2]/SQRT2; }
		else if (dir==XmY) { dv1 = imdir[0][id1]/SQRT2-imdir[1][id1]/SQRT2; dv2 = imdir[0][id2]/SQRT2-imdir[1][id2]/SQRT2; }
		else if (dir==YmZ) { dv1 = imdir[1][id1]/SQRT2-imdir[2][id1]/SQRT2; dv2 = imdir[1][id2]/SQRT2-imdir[2][id2]/SQRT2; }
		else if (dir==ZmX) { dv1 = imdir[2][id1]/SQRT2-imdir[0][id1]/SQRT2; dv2 = imdir[2][id2]/SQRT2-imdir[0][id2]/SQRT2; }
		else if (dir==XpYpZ) { dv1 = imdir[0][id1]/SQRT3+imdir[1][id1]/SQRT3+imdir[2][id1]/SQRT3; dv2 = imdir[0][id2]/SQRT3+imdir[1][id2]/SQRT3+imdir[2][id2]/SQRT3; }
		else if (dir==XmYpZ) { dv1 = imdir[0][id1]/SQRT3-imdir[1][id1]/SQRT3+imdir[2][id1]/SQRT3; dv2 = imdir[0][id2]/SQRT3-imdir[1][id2]/SQRT3+imdir[2][id2]/SQRT3; }
		else if (dir==XpYmZ) { dv1 = imdir[0][id1]/SQRT3+imdir[1][id1]/SQRT3-imdir[2][id1]/SQRT3; dv2 = imdir[0][id2]/SQRT3+imdir[1][id2]/SQRT3-imdir[2][id2]/SQRT3; }
		else if (dir==XmYmZ) { dv1 = imdir[0][id1]/SQRT3-imdir[1][id1]/SQRT3-imdir[2][id1]/SQRT3; dv2 = imdir[0][id2]/SQRT3-imdir[1][id2]/SQRT3-imdir[2][id2]/SQRT3; }
		else if (dir==mX) { dv1 = -imdir[0][id1]; dv2 = -imdir[0][id2]; }
		else if (dir==mY) { dv1 = -imdir[1][id1]; dv2 = -imdir[1][id2]; }
		else if (dir==mZ) { dv1 = -imdir[2][id1]; dv2 = -imdir[2][id2]; }
		else if (dir==mXpY) { dv1 = -(imdir[0][id1]/SQRT2+imdir[1][id1]/SQRT2); dv2 = -(imdir[0][id2]/SQRT2+imdir[1][id2]/SQRT2); }
		else if (dir==mYpZ) { dv1 = -(imdir[1][id1]/SQRT2+imdir[2][id1]/SQRT2); dv2 = -(imdir[1][id2]/SQRT2+imdir[2][id2]/SQRT2); }
		else if (dir==mZpX) { dv1 = -(imdir[2][id1]/SQRT2+imdir[0][id1]/SQRT2); dv2 = -(imdir[2][id2]/SQRT2+imdir[0][id2]/SQRT2); }
		else if (dir==mXmY) { dv1 = -(imdir[0][id1]/SQRT2-imdir[1][id1]/SQRT2); dv2 = -(imdir[0][id2]/SQRT2-imdir[1][id2]/SQRT2); }
		else if (dir==mYmZ) { dv1 = -(imdir[1][id1]/SQRT2-imdir[2][id1]/SQRT2); dv2 = -(imdir[1][id2]/SQRT2-imdir[2][id2]/SQRT2); }
		else if (dir==mZmX) { dv1 = -(imdir[2][id1]/SQRT2-imdir[0][id1]/SQRT2); dv2 = -(imdir[2][id2]/SQRT2-imdir[0][id2]/SQRT2); }
		else if (dir==mXpYpZ) { dv1 = -(imdir[0][id1]/SQRT3+imdir[1][id1]/SQRT3+imdir[2][id1]/SQRT3); dv2 = -(imdir[0][id2]/SQRT3+imdir[1][id2]/SQRT3+imdir[2][id2]/SQRT3); }
		else if (dir==mXmYpZ) { dv1 = -(imdir[0][id1]/SQRT3-imdir[1][id1]/SQRT3+imdir[2][id1]/SQRT3); dv2 = -(imdir[0][id2]/SQRT3-imdir[1][id2]/SQRT3+imdir[2][id2]/SQRT3); }
		else if (dir==mXpYmZ) { dv1 = -(imdir[0][id1]/SQRT3+imdir[1][id1]/SQRT3-imdir[2][id1]/SQRT3); dv2 = -(imdir[0][id2]/SQRT3+imdir[1][id2]/SQRT3-imdir[2][id2]/SQRT3); }
		else if (dir==mXmYmZ) { dv1 = -(imdir[0][id1]/SQRT3-imdir[1][id1]/SQRT3-imdir[2][id1]/SQRT3); dv2 = -(imdir[0][id2]/SQRT3-imdir[1][id2]/SQRT3-imdir[2][id2]/SQRT3); }

		float dirAngle = (float)Math.acos(Numerics.min(1.0f,Numerics.max(Numerics.abs(dv1),Numerics.abs(dv2))))/PI2;
		
		return (1.0f-dirAngle)*(1.0f-2.0f*dataAngle);
	}	
	
	/**
	 *  distances and other functions for computing memberships
	 */
	private final float signedAngleOverlapSimilarityGain(int dir, int id1, int id2) {
		float prod11 = Numerics.abs((imdir[0][id1]*imdir[0][id2]+imdir[1][id1]*imdir[1][id2]+imdir[2][id1]*imdir[2][id2]));
		float prod12 = Numerics.abs((imdir[0][id1]*imdir[3][id2]+imdir[1][id1]*imdir[4][id2]+imdir[2][id1]*imdir[5][id2])*(imval[1][id2]/imval[0][id2]));
		float prod21 = Numerics.abs((imdir[3][id1]*imdir[0][id2]+imdir[4][id1]*imdir[1][id2]+imdir[5][id1]*imdir[2][id2])*(imval[1][id1]/imval[0][id1]));
		float prod22 = Numerics.abs((imdir[3][id1]*imdir[3][id2]+imdir[4][id1]*imdir[4][id2]+imdir[5][id1]*imdir[5][id2])*(imval[1][id2]/imval[0][id2])*(imval[1][id1]/imval[0][id1]));
		byte off1, off2;
		float prod;
		if (prod11>prod12 && prod11>prod21 && prod11>prod22) {
			off1=0;
			off2=0;
			prod = prod11;
		} else if (prod12>prod11 && prod12>prod21 && prod12>prod22) {
			off1=0;
			off2=3;
			prod = prod12;
		} else if (prod21>prod11 && prod21>prod12 && prod21>prod22) {
			off1=3;
			off2=0;
			prod = prod21;
		} else {
			off1=3;
			off2=3;
			prod = prod22;
		}
		
		float dataAngle = (float)Math.acos(Numerics.min(1.0f,prod))/PI2;
		
		float dv1=0, dv2=0;
			
			 if (dir==X) { dv1 = imdir[off1+0][id1]; dv2 = imdir[off2+0][id2]; }
		else if (dir==Y) { dv1 = imdir[off1+1][id1]; dv2 = imdir[off2+1][id2]; }
		else if (dir==Z) { dv1 = imdir[off1+2][id1]; dv2 = imdir[off2+2][id2]; }
		else if (dir==XpY) { dv1 = imdir[off1+0][id1]/SQRT2+imdir[off1+1][id1]/SQRT2; dv2 = imdir[off2+0][id2]/SQRT2+imdir[off2+1][id2]/SQRT2; }
		else if (dir==YpZ) { dv1 = imdir[off1+1][id1]/SQRT2+imdir[off1+2][id1]/SQRT2; dv2 = imdir[off2+1][id2]/SQRT2+imdir[off2+2][id2]/SQRT2; }
		else if (dir==ZpX) { dv1 = imdir[off1+2][id1]/SQRT2+imdir[off1+0][id1]/SQRT2; dv2 = imdir[off2+2][id2]/SQRT2+imdir[off2+0][id2]/SQRT2; }
		else if (dir==XmY) { dv1 = imdir[off1+0][id1]/SQRT2-imdir[off1+1][id1]/SQRT2; dv2 = imdir[off2+0][id2]/SQRT2-imdir[off2+1][id2]/SQRT2; }
		else if (dir==YmZ) { dv1 = imdir[off1+1][id1]/SQRT2-imdir[off1+2][id1]/SQRT2; dv2 = imdir[off2+1][id2]/SQRT2-imdir[off2+2][id2]/SQRT2; }
		else if (dir==ZmX) { dv1 = imdir[off1+2][id1]/SQRT2-imdir[off1+0][id1]/SQRT2; dv2 = imdir[off2+2][id2]/SQRT2-imdir[off2+0][id2]/SQRT2; }
		else if (dir==XpYpZ) { dv1 = imdir[off1+0][id1]/SQRT3+imdir[off1+1][id1]/SQRT3+imdir[off1+2][id1]/SQRT3; dv2 = imdir[off2+0][id2]/SQRT3+imdir[off2+1][id2]/SQRT3+imdir[off2+2][id2]/SQRT3; }
		else if (dir==XmYpZ) { dv1 = imdir[off1+0][id1]/SQRT3-imdir[off1+1][id1]/SQRT3+imdir[off1+2][id1]/SQRT3; dv2 = imdir[off2+0][id2]/SQRT3-imdir[off2+1][id2]/SQRT3+imdir[off2+2][id2]/SQRT3; }
		else if (dir==XpYmZ) { dv1 = imdir[off1+0][id1]/SQRT3+imdir[off1+1][id1]/SQRT3-imdir[off1+2][id1]/SQRT3; dv2 = imdir[off2+0][id2]/SQRT3+imdir[off2+1][id2]/SQRT3-imdir[off2+2][id2]/SQRT3; }
		else if (dir==XmYmZ) { dv1 = imdir[off1+0][id1]/SQRT3-imdir[off1+1][id1]/SQRT3-imdir[off1+2][id1]/SQRT3; dv2 = imdir[off2+0][id2]/SQRT3-imdir[off2+1][id2]/SQRT3-imdir[off2+2][id2]/SQRT3; }
		else if (dir==mX) { dv1 = -imdir[off1+0][id1]; dv2 = -imdir[off2+0][id2]; }
		else if (dir==mY) { dv1 = -imdir[off1+1][id1]; dv2 = -imdir[off2+1][id2]; }
		else if (dir==mZ) { dv1 = -imdir[off1+2][id1]; dv2 = -imdir[off2+2][id2]; }
		else if (dir==mXpY) { dv1 = -(imdir[off1+0][id1]/SQRT2+imdir[off1+1][id1]/SQRT2); dv2 = -(imdir[off2+0][id2]/SQRT2+imdir[off2+1][id2]/SQRT2); }
		else if (dir==mYpZ) { dv1 = -(imdir[off1+1][id1]/SQRT2+imdir[off1+2][id1]/SQRT2); dv2 = -(imdir[off2+1][id2]/SQRT2+imdir[off2+2][id2]/SQRT2); }
		else if (dir==mZpX) { dv1 = -(imdir[off1+2][id1]/SQRT2+imdir[off1+0][id1]/SQRT2); dv2 = -(imdir[off2+2][id2]/SQRT2+imdir[off2+0][id2]/SQRT2); }
		else if (dir==mXmY) { dv1 = -(imdir[off1+0][id1]/SQRT2-imdir[off1+1][id1]/SQRT2); dv2 = -(imdir[off2+0][id2]/SQRT2-imdir[off2+1][id2]/SQRT2); }
		else if (dir==mYmZ) { dv1 = -(imdir[off1+1][id1]/SQRT2-imdir[off1+2][id1]/SQRT2); dv2 = -(imdir[off2+1][id2]/SQRT2-imdir[off2+2][id2]/SQRT2); }
		else if (dir==mZmX) { dv1 = -(imdir[off1+2][id1]/SQRT2-imdir[off1+0][id1]/SQRT2); dv2 = -(imdir[off2+2][id2]/SQRT2-imdir[off2+0][id2]/SQRT2); }
		else if (dir==mXpYpZ) { dv1 = -(imdir[off1+0][id1]/SQRT3+imdir[off1+1][id1]/SQRT3+imdir[off1+2][id1]/SQRT3); dv2 = -(imdir[off2+0][id2]/SQRT3+imdir[off2+1][id2]/SQRT3+imdir[off2+2][id2]/SQRT3); }
		else if (dir==mXmYpZ) { dv1 = -(imdir[off1+0][id1]/SQRT3-imdir[off1+1][id1]/SQRT3+imdir[off1+2][id1]/SQRT3); dv2 = -(imdir[off2+0][id2]/SQRT3-imdir[off2+1][id2]/SQRT3+imdir[off2+2][id2]/SQRT3); }
		else if (dir==mXpYmZ) { dv1 = -(imdir[off1+0][id1]/SQRT3+imdir[off1+1][id1]/SQRT3-imdir[off1+2][id1]/SQRT3); dv2 = -(imdir[off2+0][id2]/SQRT3+imdir[off2+1][id2]/SQRT3-imdir[off2+2][id2]/SQRT3); }
		else if (dir==mXmYmZ) { dv1 = -(imdir[off1+0][id1]/SQRT3-imdir[off1+1][id1]/SQRT3-imdir[off1+2][id1]/SQRT3); dv2 = -(imdir[off2+0][id2]/SQRT3-imdir[off2+1][id2]/SQRT3-imdir[off2+2][id2]/SQRT3); }

		float dirAngle = (float)Math.acos(Numerics.min(1.0f,Numerics.max(Numerics.abs(dv1),Numerics.abs(dv2))))/PI2;
		
		return (1.0f-dirAngle)*(1.0f-2.0f*dataAngle);
	}	
	
	/**
	 *  distances and other functions for computing memberships
	 */
	private final boolean sameDirection(int dir, int id) {
		float dv=0.0f;
			
			 if (dir==X) dv = imdir[0][id];
		else if (dir==Y) dv = imdir[1][id];
		else if (dir==Z) dv = imdir[2][id];
		else if (dir==XpY) dv = imdir[0][id]/SQRT2+imdir[1][id]/SQRT2;
		else if (dir==YpZ) dv = imdir[1][id]/SQRT2+imdir[2][id]/SQRT2;
		else if (dir==ZpX) dv = imdir[2][id]/SQRT2+imdir[0][id]/SQRT2;
		else if (dir==XmY) dv = imdir[0][id]/SQRT2-imdir[1][id]/SQRT2;
		else if (dir==YmZ) dv = imdir[1][id]/SQRT2-imdir[2][id]/SQRT2;
		else if (dir==ZmX) dv = imdir[2][id]/SQRT2-imdir[0][id]/SQRT2;
		else if (dir==XpYpZ) dv = imdir[0][id]/SQRT3+imdir[1][id]/SQRT3+imdir[2][id]/SQRT3;
		else if (dir==XmYpZ) dv = imdir[0][id]/SQRT3-imdir[1][id]/SQRT3+imdir[2][id]/SQRT3;
		else if (dir==XpYmZ) dv = imdir[0][id]/SQRT3+imdir[1][id]/SQRT3-imdir[2][id]/SQRT3;
		else if (dir==XmYmZ) dv = imdir[0][id]/SQRT3-imdir[1][id]/SQRT3-imdir[2][id]/SQRT3;
		else if (dir==mX) dv = -imdir[0][id];
		else if (dir==mY) dv = -imdir[1][id];
		else if (dir==mZ) dv = -imdir[2][id];
		else if (dir==mXpY) dv = -(imdir[0][id]/SQRT2+imdir[1][id]/SQRT2);
		else if (dir==mYpZ) dv = -(imdir[1][id]/SQRT2+imdir[2][id]/SQRT2);
		else if (dir==mZpX) dv = -(imdir[2][id]/SQRT2+imdir[0][id]/SQRT2);
		else if (dir==mXmY) dv = -(imdir[0][id]/SQRT2-imdir[1][id]/SQRT2);
		else if (dir==mYmZ) dv = -(imdir[1][id]/SQRT2-imdir[2][id]/SQRT2);
		else if (dir==mZmX) dv = -(imdir[2][id]/SQRT2-imdir[0][id]/SQRT2);
		else if (dir==mXpYpZ) dv = -(imdir[0][id]/SQRT3+imdir[1][id]/SQRT3+imdir[2][id]/SQRT3);
		else if (dir==mXmYpZ) dv = -(imdir[0][id]/SQRT3-imdir[1][id]/SQRT3+imdir[2][id]/SQRT3);
		else if (dir==mXpYmZ) dv = -(imdir[0][id]/SQRT3+imdir[1][id]/SQRT3-imdir[2][id]/SQRT3);
		else if (dir==mXmYmZ) dv = -(imdir[0][id]/SQRT3-imdir[1][id]/SQRT3-imdir[2][id]/SQRT3);

		if (dv>0) return true;
		else return false;
	}	
	
	/**
	 *  distances and other functions for computing memberships
	 */
	private final float directionAngleFactor(float[] dir, int id) {
		float norm = Numerics.max(1e-30f,(float)Math.sqrt(dir[0]*dir[0]+dir[1]*dir[1]+dir[2]*dir[2]));
				
		// use best direction only
		float prod = (imdir[0][id]*dir[0]+imdir[1][id]*dir[1]+imdir[2][id]*dir[2])/norm;
		
		float angle = (float)Math.acos(Numerics.min(1.0f,Numerics.abs(prod)))/PI2;
		
		return norm*(1.0f-2.0f*angle);
	}
	
	/**
	 *  distances and other functions for computing memberships
	 */
	private final float directionMixAngleFactor(float[] dir1, float[] dir2, int id) {
		float norm1 = Numerics.max(1e-30f,(float)Math.sqrt(dir1[0]*dir1[0]+dir1[1]*dir1[1]+dir1[2]*dir1[2]));
		float norm2 = Numerics.max(1e-30f,(float)Math.sqrt(dir2[0]*dir2[0]+dir2[1]*dir2[1]+dir2[2]*dir2[2]));
				
		float sgn = Numerics.sign( dir1[0]*dir2[0]+dir1[1]*dir2[1]+dir1[2]*dir2[2] );
		float[] meandir = new float[3];
		meandir[0] = dir1[0]/norm1 + sgn*dir2[0]/norm2;
		meandir[1] = dir1[1]/norm1 + sgn*dir2[1]/norm2;
		meandir[2] = dir1[2]/norm1 + sgn*dir2[2]/norm2;
		float normmean = Numerics.max(1e-30f,(float)Math.sqrt(meandir[0]*meandir[0]+meandir[1]*meandir[1]+meandir[2]*meandir[2]));
		
		meandir[0] /= normmean;
		meandir[1] /= normmean;
		meandir[2] /= normmean;
		
		// use first two directions
		float prod1 = imdir[0][id]*meandir[0]+imdir[1][id]*meandir[1]+imdir[2][id]*meandir[2];
		float prod2 = imdir[3][id]*meandir[0]+imdir[4][id]*meandir[1]+imdir[5][id]*meandir[2];
		
		float angle1 = (float)Math.acos(Numerics.min(1.0f,Numerics.abs(prod1)))/PI2;
		float angle2 = (float)Math.acos(Numerics.min(1.0f,Numerics.abs(prod2)))/PI2;
		
		float val1 = (norm1+norm2)/2.0f*(1.0f-2.0f*angle1);
		float val2 = imval[1][id]/Numerics.max(imval[0][id],1e-30f)*(norm1+norm2)/2.0f*(1.0f-2.0f*angle2);
		
		return Numerics.max(val1,val2);
	}
	
	/**
	 *  product for similarity and gain
	 */
	private final float semiProduct(float s, float g, short lb, int id) {
		if (s>0.0f || g>0.0f) return s*g;
		else if (lb==0) return isoDiffusion*g;
		else return 0.0f;
	}
	
	
	private final float isoGain(int id) {
		return imval[2][id]/Numerics.max(imval[0][id],ZERO);
	}
	
	private final float fiberGain(int id) {
		return (imval[0][id] - imval[1][id])/Numerics.max(imval[0][id],ZERO);
	}
	
	private final float crossingGain(int id) {
		return (imval[0][id] - imval[2][id])/Numerics.max(imval[0][id],ZERO);
	}
	
	/*
	private final float isoADC(int id) {
		return 1.0f;
		//if (adc[id]<0.0f) return 1.0f;
		//else return Numerics.bounded( (adc[id]-adcf)/(adci-adcf), 0.5f, 1.0f);
	}
	
	private final float fiberADC(int id) {
		return 1.0f;
		//if (adc[id]<0.0f) return ZERO;
		//else return Numerics.bounded( (adci-adc[id])/(adci-adcf), ZERO, 0.5f);
	}
	*/
	
	private final float isoMask(int id) {
		return 1.0f;
		//if (adc[id]<0.0f) return 1.0f;
		//else return Numerics.bounded( (adc[id]-adcf)/(adci-adcf), 0.5f, 1.0f);
	}
	
	private final float fiberMask(int id) {
		return maskprior[id];
		//return 1.0f;
		//if (adc[id]<0.0f) return ZERO;
		//else return Numerics.bounded( (adci-adc[id])/(adci-adcf), ZERO, 0.5f);
	}

	private final float isoLabeling(int id) {
		if (lbType==LESIONS) return 1.0f-labeling[id];
		else return 1.0f;
	}
	
	private final float fiberLabeling(int id) {
		return 1.0f;
	}

	/**
	 *  distances and other functions for computing memberships
	 *	distance for id2 = lb2 given id1 = lb1 ( d(id1=lb1 => id2=lb2) ) 
	 */
	private final void priorDistance(int x, int y, int z, int id) {
		// shape prior for all tracts
		float[] prior = new float[nt];
		atlas.getTransformedShape(prior,x,y,z);
		
		// compute the possible candidates
		int np = priorCandidateNumber(prior, x, y, z, id);
		short[] labels = priorCandidateLabels(np, prior, x, y, z, id);
		float[] candidates = priorCandidates(labels, np, prior, x, y, z, id);
		
		// retain only the nb best
		bestCandidates(candidates, labels, id, Numerics.min(nb,np) );
				
		return;
	}

	/**
	 *  distances and other functions for computing memberships
	 *	distance for id2 = lb2 given id1 = lb1 ( d(id1=lb1 => id2=lb2) ) 
	 */
	private final int priorCandidateNumber(float[] prior, int x, int y, int z, int id) {
		// identify all possible tracts and pairs
		int np=0;
		for (int t=0;t<nt;t++) {
			if (prior[t]>priorMinimum) {
				np++;
				for (int t2=t+1;t2<nt;t2++) {
					if (atlas.getOverlap(t,t2)>0) {
						if (prior[t2]>priorMinimum) {
							np++;
						}
					}
				}
			}
		}
		return np;
	}

	/**
	 *  distances and other functions for computing memberships
	 *	distance for id2 = lb2 given id1 = lb1 ( d(id1=lb1 => id2=lb2) ) 
	 */
	private final short[] priorCandidateLabels(int np, float[] prior, int x, int y, int z, int id) {
		short[] labels = new short[np];
		
		int p=0;
		for (int t=0;t<nt;t++) {
			if (prior[t]>priorMinimum) {
				labels[p] = (short)t;
				p++;
				
				for (int t2=t+1;t2<nt;t2++) {
					if (atlas.getOverlap(t,t2)>0) {
						if (prior[t2]>priorMinimum) {
							labels[p] = (short)(100*t+t2);
							p++;
						}
					}
				}
			}
		}
		return labels;
	}

	/**
	 *  distances and other functions for computing memberships
	 *	distance for id2 = lb2 given id1 = lb1 ( d(id1=lb1 => id2=lb2) ) 
	 */
	private final float[] priorCandidates(short[] labels, int np, float[] prior, int x, int y, int z, int id) {
		float[][] dir = new float[nt][3];
		float[] vdir = new float[nt];
		
		// shape prior for all tracts
		atlas.getTransformedDirection(dir,x,y,z);
		
		float priorsum = 0.0f;
		float priormax = 0.0f;
		for (int t=0;t<nt;t++) {
			priorsum += prior[t];
			if (prior[t]>priormax) priormax = prior[t];
		}
		
		// FA value
		float uisobg = isoGain(id);
		float ufiber = fiberGain(id);
		float ucross = crossingGain(id);
		
		// Mask distance
		float miso = isoMask(id);
		float mfiber = fiberMask(id);

		// Labeling value
		float liso = isoLabeling(id);
		float lfiber = fiberLabeling(id);
		
		// other tracts direction prior ?
		for (int t=1;t<nt;t++) {
			vdir[t] = directionAngleFactor(dir[t],id);
		}
		vdir[0] = 0.0f;
		
		float[] candidates = new float[np];
		int p=0;
		for (int t=0;t<nt;t++) {
			if (prior[t]>priorMinimum) {
				// compute the gain
				
				// prior coeff
				float uprior = prior[t]*prior[t]/priorsum;
				//float uprior = prior[t];
			
				// direction coeff
				
				if (t==0) candidates[p] = uisobg*miso*liso*uprior*0.5f;
				else candidates[p] 		= ufiber*mfiber*lfiber*uprior*(0.5f+0.5f*vdir[t]);
				p++;
				
				for (int t2=t+1;t2<nt;t2++) {
					if (atlas.getOverlap(t,t2)>0) {
						if (prior[t2]>priorMinimum) {
							// compute the overlapping prior
							uprior = prior[t]*prior[t2]*(prior[t]+prior[t2])/priorsum;
							//uprior = (float)Math.sqrt(prior[t]*prior[t2]);
							
							// direction coeff : average direction
							float ddir = directionMixAngleFactor(dir[t],dir[t2],id);
							
							candidates[p] = ucross*mfiber*lfiber*uprior*(0.5f+0.5f*ddir);
							p++;
						}
					}
				}
			}
		}
		return candidates;
	}
	private final float[] prevCandidates(short[] labels, int np, int id) {
		float[] candidates = new float[np];
		for (int p=0;p<np;p++) {
			candidates[p] = 0.0f;
			// find non-zero memberships
			for (int b=0;b<nb;b++) {
				if (best[b][id]==labels[p]) {
					candidates[p] = prev[b][id];
				}
			}
		}
		return candidates;
	}
	private final float[] gainCandidates(short[] labels, int np, int id) {
		float[] candidates = new float[np];
		for (int p=0;p<np;p++) {
			candidates[p] = 0.0f;
			// find non-zero memberships
			for (int b=0;b<nb;b++) {
				if (best[b][id]==labels[p]) {
					candidates[p] = gain[b][id];
				}
			}
		}
		return candidates;
	}

	private final float bestCandidates(float[] candidates, short[] labels, int xyz, int num) {
		short pbest = best[0][xyz];
		float pgain = gain[0][xyz];
		for (int b=0;b<nb;b++) {
			gain[b][xyz] = -INF;
			best[b][xyz] = UNKNOWN;
		}
		if (debug) {
			for (short m=1;m<candidates.length;m++) if (Float.isNaN(candidates[m])) {
				System.out.println("NaN: "+m+", "+labels[m]+", "+xyz);
			}
		}
		for (int n=0;n<num;n++) {
			short nmax=0;
			for (short m=1;m<candidates.length;m++) if (candidates[m]>candidates[nmax]) {
				nmax = m;
			}
			gain[n][xyz] = candidates[nmax];
			best[n][xyz] = labels[nmax];
			candidates[nmax] = -INF;
		}
		if (best[0][xyz]!=pbest) return 1.0f;
		else return 0.0f;
	}
	
	private final float bestCandidates(float[] candidates, short[] labels, float[] cprev, int xyz, int num) {
		short pbest = best[0][xyz];
		for (int b=0;b<nb;b++) {
			gain[b][xyz] = -INF;
			prev[b][xyz] = -INF;
			best[b][xyz] = UNKNOWN;
		}
		if (debug) {
			for (short m=1;m<candidates.length;m++) if (Float.isNaN(candidates[m])) {
				System.out.println("NaN: "+m+", "+labels[m]+", "+cprev[m]+", "+xyz);
			}
		}
		for (int n=0;n<num;n++) {
			short nmax=0;
			for (short m=1;m<candidates.length;m++) if (candidates[m]>candidates[nmax]) {
				nmax = m;
			}
			gain[n][xyz] = candidates[nmax];
			prev[n][xyz] = cprev[nmax];
			best[n][xyz] = labels[nmax];
			candidates[nmax] = -INF;
		}
		if (best[0][xyz]!=pbest) return 1.0f;
		else return 0.0f;
	}

	/**
	 *  diffusion, assuming we did record the previous state
	 */
	private final float gainIterativePrecomputedSimilarityDiffusion(int x, int y, int z, int id) {
		float[] prior = new float[nt];
		float[][] dir = new float[nt][3];
		
		// shape prior for all tracts
		atlas.getTransformedShape(prior,x,y,z);
		
		
		// diffusion score: pre-computed
		
		// compute the possible candidates
		int np = priorCandidateNumber(prior, x, y, z, id);
		short[] labels = priorCandidateLabels(np, prior, x, y, z, id);
		float[] candidates = priorCandidates(labels, np, prior, x, y, z, id);
		float[] cprev = prevCandidates(labels, np, id);
				
		int p=0;
		for (int t=0;t<nt;t++) {
			if (prior[t]>priorMinimum) {
				float ngb1, ngb2;
				if (t==0) {
					ngb1 = 0.0f;	
					ngb2 = 0.0f;	
				} else {
					ngb1 = -similarityFactor;
					ngb2 = -similarityFactor;
				}
				// compute the diffusion
				for (byte d=0;d<2*nc;d++) {
					int idn = neighborIndex(d,id);
					
					if (t==0) {
						for (int b=0;b<nb;b++) {
							if (best[b][idn]==labels[p]) {
								ngb1 += isoDiffusion*prev[b][idn]/(float)nc;
							}
						}
					} else if (d==neighborF1[id]) {
						for (int b=0;b<nb;b++) {
							if (best[b][idn]==labels[p]) {
								ngb1 = Numerics.max(ngb1, semiProduct(similarityF1[id], prev[b][idn], labels[p], idn) );
							} else if (best[b][idn]>=100) {
								int b2 = best[b][idn]%100;
								int b1 = (best[b][idn]-b2)/100;
								if (b1==labels[p] || b2==labels[p]) {
									ngb1 = Numerics.max(ngb1, semiProduct(similarityF1[id], prev[b][idn], labels[p], idn) );
								}
							}
						}
					} else if (d==neighborF2[id]) {
						for (int b=0;b<nb;b++) {
							if (best[b][idn]==labels[p]) {
								ngb2 = Numerics.max(ngb2, semiProduct(similarityF2[id], prev[b][idn], labels[p], idn) );
							} else if (best[b][idn]>=100) {
								int b2 = best[b][idn]%100;
								int b1 = (best[b][idn]-b2)/100;
								if (b1==labels[p] || b2==labels[p]) {
									ngb2 = Numerics.max(ngb2, semiProduct(similarityF2[id], prev[b][idn], labels[p], idn) );
								}
							}
						}
					}
				}
				
				// add to prior value
				candidates[p] += 0.5f*(ngb1+ngb2);
				
				// increase counter
				p++;
				
				for (int t2=t+1;t2<nt;t2++) {
					if (atlas.getOverlap(t,t2)>0) {
						if (prior[t2]>priorMinimum) {
							// compute the diffusion
							ngb1 = -similarityFactor;
							ngb2 = -similarityFactor;
							for (byte d=0;d<2*nc;d++) {
								int idn = neighborIndex(d,id);
							
								if (d==neighborO1[id]) {
									for (int b=0;b<nb;b++) {
										if (best[b][idn]==labels[p]) {
											ngb1 = Numerics.max(ngb1, semiProduct(similarityO1[id], prev[b][idn], labels[p], idn) ); 
										} else if (best[b][idn]==t) {
											ngb1 = Numerics.max(ngb1, semiProduct(similarityO1[id], prev[b][idn], labels[p], idn) ); 
										} else if (best[b][idn]==t2) {
											ngb1 = Numerics.max(ngb1, semiProduct(similarityO1[id], prev[b][idn], labels[p], idn) ); 
										}
									}
								} else if (d==neighborO2[id]) {
									for (int b=0;b<nb;b++) {
										if (best[b][idn]==labels[p]) {
											ngb2 = Numerics.max(ngb2, semiProduct(similarityO2[id], prev[b][idn], labels[p], idn)); 
										} else if (best[b][idn]==t) {
											ngb2 = Numerics.max(ngb2, semiProduct(similarityO2[id], prev[b][idn], labels[p], idn)); 
										} else if (best[b][idn]==t2) {
											ngb2 = Numerics.max(ngb2, semiProduct(similarityO2[id], prev[b][idn], labels[p], idn)); 
										}
									}
								}
							}
							
							// add prior value
							candidates[p] += 0.5f*(ngb1+ngb2);
							
							// increase counter
							p++;
						}
					}
				}
			}
		}
		// retain only the nb best
		return bestCandidates(candidates, labels, cprev, id, Numerics.min(nb,np) );
	}

	private final void estimateDiffusionSimilarity() {
		float[]	dsF1 = new float[2*nc];
		float[]	dsF2 = new float[2*nc];
		float[]	dsO1 = new float[2*nc];
		float[]	dsO2 = new float[2*nc];
		
		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			int id = x+nx*y+nx*ny*z;
			if (mask[id]) {
				for (byte d=0;d<2*nc;d++) {
					dsF1[d] = -INF;
					dsF2[d] = -INF;
					dsO1[d] = -INF;
					dsO2[d] = -INF;
					
					int idn = neighborIndex(d,id);
					
					if (sameDirection(d, id)) {
						dsF1[d] = signedAngleFiberSimilarityGain(d,id,idn);
						dsO1[d] = signedAngleOverlapSimilarityGain(d,id,idn);
					} else {
						dsF2[d] = signedAngleFiberSimilarityGain(d,id,idn);
						dsO2[d] = signedAngleOverlapSimilarityGain(d,id,idn);
					}
				}
				// find the best for each side
				neighborF1[id] = Numerics.bestIndex(dsF1);
				neighborF2[id] = Numerics.bestIndex(dsF2);
				neighborO1[id] = Numerics.bestIndex(dsO1);
				neighborO2[id] = Numerics.bestIndex(dsO2);
				similarityF1[id] = dsF1[neighborF1[id]];
				similarityF2[id] = dsF2[neighborF2[id]];
				similarityO1[id] = dsO1[neighborO1[id]];
				similarityO2[id] = dsO2[neighborO2[id]];
			}
		}
		isoDiffusion = isoDiffusion;
		
		MedicUtilPublic.displayMessage("isotropic diffusion score: "+isoDiffusion);
		
		return;
	} // 
	
	private final int neighborIndex(byte d, int id) {
		int idn=id;
		
			 if (d==X) 	idn+=1; 		
		else if (d==mX)	idn-=1;
		else if (d==Y) 	idn+=nx;
		else if (d==mY)	idn-=nx;
		else if (d==Z) 	idn+=nx*ny;
		else if (d==mZ)	idn-=nx*ny;
		else if (d==XpY) 	idn+=1+nx;
		else if (d==mXpY) 	idn-=1+nx;
		else if (d==YpZ) 	idn+=nx+nx*nx;
		else if (d==mYpZ)	idn-=nx+nx*ny;
		else if (d==ZpX) 	idn+=1+nx*ny;	
		else if (d==mZpX)	idn-=1+nx*ny;
		else if (d==XmY) 	idn+=1-nx;	
		else if (d==mXmY)	idn-=1-nx;
		else if (d==YmZ) 	idn+=nx-nx*ny;
		else if (d==mYmZ)	idn-=nx-nx*ny;
		else if (d==ZmX) 	idn+=1-nx*ny;
		else if (d==mZmX)	idn-=1-nx*ny;
		else if (d==XpYpZ) 		idn+=1+nx+nx*ny;
		else if (d==mXpYpZ)		idn-=1+nx+nx*ny;
		else if (d==XmYmZ) 		idn+=1-nx-nx*ny; 
		else if (d==mXmYmZ)		idn-=1-nx-nx*ny;
		else if (d==XmYpZ) 		idn+=1-nx+nx*ny;
		else if (d==mXmYpZ)		idn-=1-nx+nx*ny;
		else if (d==XpYmZ) 		idn+=1+nx-nx*ny; 
		else if (d==mXpYmZ)		idn-=1+nx-nx*ny;

		return idn;
	}
	/*
	private final void precomputeADCpriors() {
		
		adci = 0.0f; float deni = 0.0f;
		adcf = 0.0f; float denf = 0.0f;
		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			int id = x+nx*y+nx*ny*z;
			if (mask[id]) {
				float[] prior = new float[nt];
				atlas.getTransformedShape(prior,x,y,z);
				
				adci += prior[0]*prior[0]*adc[id];
				adcf += (1.0f-prior[0])*(1.0f-prior[0])*adc[id];
				
				deni += prior[0]*prior[0];
				denf += (1.0f-prior[0])*(1.0f-prior[0]);
			}
		}
		if (deni>ZERO && denf>ZERO) {
			adci /= deni;
			adcf /= denf;
			
			adci = 3.0f*adcf;
		}
		MedicUtilPublic.displayMessage("average diffusion values (iso | fiber): "+adci+", "+adcf);
		
		return;
	} // 
	*/
	private final void precomputeMaskPrior() {
		
		MedicUtilPublic.displayMessage("Mask prior\n");
		maskprior = new float[nx*ny*nz];
		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			int id = x+nx*y+nx*ny*z;
			if (mask[id]) {
				maskprior[id] = 0.0f;
				if (!mask[id-1] || !mask[id+1] || !mask[id-nx] || !mask[id+nx] || !mask[id-nx*ny] || !mask[id+nx*ny]) {
					maskprior[id] = 1.0f;
				}
			} else {
				maskprior[id] = -1.0f;
			}
		}
		float nt = maskdist/Numerics.min(rx,ry,rz);
		
		// compute the Manhattan distance
		for (int t=1;t<=nt;t++) {
			for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
				int id = x+nx*y+nx*ny*z;
				if (maskprior[id]==t) {
					if (maskprior[id-1]==0) 	maskprior[id-1] = t+1;
					if (maskprior[id+1]==0) 	maskprior[id+1] = t+1;
					if (maskprior[id-nx]==0) 	maskprior[id-nx] = t+1;
					if (maskprior[id+nx]==0)	maskprior[id+nx] = t+1;
					if (maskprior[id-nx*ny]==0)	maskprior[id-nx*ny] = t+1;
					if (maskprior[id+nx*ny]==0)	maskprior[id+nx*ny] = t+1;
				}
			}
			MedicUtilPublic.displayMessage("+");
		}
		// build the normalized value
		for (int id=0;id<nx*ny*nz;id++) {
			if (maskprior[id]==-1) maskprior[id] = 0.0f;
			else if (maskprior[id]==0) maskprior[id] = 1.0f;
			else maskprior[id] = Numerics.bounded((maskprior[id]/nt)*(maskprior[id]/nt), ZERO, 1.0f);
		}
		MedicUtilPublic.displayMessage("\n");
		
	} // 
	
	public final float[][] exportMemberships() {
		float[][]	Mems = new float[nt][nx*ny*nz];
		//float maxgain = (float)Math.log(nt);
		
		for (int xyz=0;xyz<nx*ny*nz;xyz++) {
			float num,den = 0.0f;
			for (int t=0;t<nt;t++) {
				Mems[t][xyz] = 0.0f;
			}
			for (int b=0;b<nb;b++) {
				if (best[b][xyz]!=UNKNOWN) {
					num = (float)Math.exp((gain[b][xyz]-gain[0][xyz])/gainFactor);
					den += num;
					if (best[b][xyz]<100) {
						Mems[best[b][xyz]][xyz] += num;
					} else {
						int b2 = best[b][xyz]%100;
						int b1 = (best[b][xyz]-b2)/100;
						Mems[b1][xyz] += num;
						Mems[b2][xyz] += num;
					}
				}
			}
			if (den>ZERO) {
				for (int t=0;t<nt;t++) {
					Mems[t][xyz] /= den;
				}
			}
		}
		return Mems;
	} // exportMemberships
	
	public final float[] exportMaxMemberships() {
		float[]	Mems = new float[nx*ny*nz];
		float[] mem = new float[nt];
		//float maxgain = (float)Math.log(nt);
		
		for (int xyz=0;xyz<nx*ny*nz;xyz++) {
			float num,den = 0.0f;
			for (int t=0;t<nt;t++) {
				mem[t] = 0.0f;
			}
			for (int b=0;b<nb;b++) {
				if (best[b][xyz]!=UNKNOWN) {
					num = (float)Math.exp((gain[b][xyz]-gain[0][xyz])/gainFactor);
					den += num;
					if (best[b][xyz]<100) {
						mem[best[b][xyz]] += num;
					} else {
						int b2 = best[b][xyz]%100;
						int b1 = (best[b][xyz]-b2)/100;
						mem[b1] += num;
						mem[b2] += num;
					}
				}
			}
			if (den>ZERO) {
				for (int t=0;t<nt;t++) {
					mem[t] /= den;
				}
			}
			Mems[xyz] = Numerics.max(mem);
		}
		return Mems;
	} // exportMaxMemberships
	
	public final float[] exportMaxMappedGain() {
		float[]	Mems = new float[nx*ny*nz];
		float[] mem = new float[nt];
		float atlasScale = 5.0f;
		
		for (int xyz=0;xyz<nx*ny*nz;xyz++) {
			float sum = 0.0f;
			float num = 0.0f;
			for (int b=0;b<nb;b++) {
				if (gain[b][xyz]>0) {
					if (best[b][xyz]<100) {
						if (atlas.isRegistered(best[b][xyz])) num += gain[b][xyz];
					} else {
						int b2 = best[b][xyz]%100;
						int b1 = (best[b][xyz]-b2)/100;
						if (atlas.isRegistered(b1) || atlas.isRegistered(b2)) num += gain[b][xyz];
					}
					sum += gain[b][xyz];
				}
			}
			sum = Numerics.max(ZERO,sum);
			
			Mems[xyz] = 0.5f + 0.5f*(2.0f*num - sum)/sum;								
		}
		return Mems;
	} // exportMaxMappedGain
	
	public final float[][] exportRawGainMemberships() {
		float[][]	Mems = new float[nb][nx*ny*nz];
		//float maxgain = (float)Math.log(nt);
		
		for (int xyz=0;xyz<nx*ny*nz;xyz++) {
			float num,den = 0.0f;
			for (int b=0;b<nb;b++) {
				if (best[b][xyz]!=UNKNOWN) {
					num = (float)Math.exp((gain[b][xyz]-gain[0][xyz])/gainFactor);
					den += num;
					Mems[b][xyz] = num;
				}
			}
			if (den>ZERO) {
				for (int b=0;b<nb;b++) {
					Mems[b][xyz] /= den;
				}
			}
		}
		return Mems;
	} // exportMemberships
	
	public final boolean[][] exportSegmentations() {
		boolean[][]	Seg = new boolean[nt][nx*ny*nz];
		
		for (int xyz=0;xyz<nx*ny*nz;xyz++) {
			for (int t=0;t<nt;t++) {
				Seg[t][xyz] = false;
			}
			if (best[0][xyz]!=UNKNOWN) {
				if (best[0][xyz]<100) {
					Seg[best[0][xyz]][xyz] = true;
				} else {
					int b2 = best[0][xyz]%100;
					int b1 = (best[0][xyz]-b2)/100;
					Seg[b1][xyz] = true;
					Seg[b2][xyz] = true;
				}
			}
		}
		return Seg;
	} // exportMemberships
	
	/**  generate atlas image from information
	 */
    final public short[] exportLabelClassification() {
		float dist,max,count;
		short[] img = new short[nx*ny*nz];
		
		for (int xyz=0;xyz<nx*ny*nz;xyz++) {
			if (mask[xyz]) {
				if (best[0][xyz]<100) {
					img[xyz] = (short)(100*best[0][xyz]);
				} else {
					img[xyz] = (short)best[0][xyz];
				}
			} else {
				img[xyz] = 0;
			}
		}
		return img;
	}

	/**  generate atlas image from information (all overlapping regions are merged)
	 */
    final public short[] exportFlattenedLabelClassification() {
		float dist,max,count;
		short[] img = new short[nx*ny*nz];
		
		for (int xyz=0;xyz<nx*ny*nz;xyz++) {
			if (mask[xyz]) {
				if (best[0][xyz]<100) {
					img[xyz] = (short)(1+best[0][xyz]);
				} else {
					img[xyz] = (short)(nt+1);
				}
			} else {
				img[xyz] = 0;
			}
		}
		return img;
	}

	/**  generate atlas image from information (all overlapping regions are rendered in crosshatch)
	 */
    final public short[] exportCrosshatchLabelClassification() {
		float dist,max,count;
		short[] img = new short[nx*ny*nz];
		
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			int xyz = x + nx*y + nx*ny*z;
			if (mask[xyz]) {
				if (best[0][xyz]<100) {
					img[xyz] = (short)(best[0][xyz]);
				} else {
					int b2 = best[0][xyz]%100;
					int b1 = (best[0][xyz]-b2)/100;
					if ( (x+y+z)%2==0 ) img[xyz] = (short)(b1);
					else img[xyz] = (short)(b2);
				}
			} else {
				img[xyz] = 0;
			}
		}
		return img;
	}

	public final float[] exportMaxAngleFactor() {
		float[]	Factor = new float[nx*ny*nz];
		float[][] dir = new float[nt][3];
		
		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			int xyz = x+nx*y+nx*ny*z;
			if (best[0][xyz]!=UNKNOWN && best[0][xyz]>0) {
				atlas.getTransformedDirection(dir, x, y, z);
				if (best[0][xyz]<100) {
					Factor[xyz] = directionAngleFactor(dir[best[0][xyz]], xyz);
				} else {
					int b2 = best[0][xyz]%100;
					int b1 = (best[0][xyz]-b2)/100;
					Factor[xyz] = directionMixAngleFactor(dir[b1], dir[b2], xyz);
				}
			}
		}
		return Factor;
	} // exportMaxMemberships
	
	public final float[] exportMaxPriorFactor() {
		float[]	Factor = new float[nx*ny*nz];
		float[] shape = new float[nt];
		
		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			int xyz = x+nx*y+nx*ny*z;
			if (best[0][xyz]!=UNKNOWN && best[0][xyz]>0) {
				atlas.getTransformedShape(shape, x, y, z);
				float sum = 0.0f;
				for (int t=0;t<nt;t++) sum += shape[t];
				sum = Numerics.max(ZERO,sum);
				
				if (best[0][xyz]<100) {
					Factor[xyz] = shape[best[0][xyz]]*shape[best[0][xyz]]/sum;
				} else {
					int b2 = best[0][xyz]%100;
					int b1 = (best[0][xyz]-b2)/100;
					Factor[xyz] = shape[b1]*shape[b2]*(shape[b1]+shape[b2])/sum;
				}
			}
		}
		return Factor;
	} // exportMaxMemberships
	
	public final float[][] exportDiffusionMaps() {
		float[][]	Imdir = new float[6][nx*ny*nz];
		
		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			int xyz = x+nx*y+nx*ny*z;
			for (int d=0;d<3;d++) {
				int xi = x, yj = y, zl = z;
					 if (d==0) xi = x+1;
				else if (d==1) yj = y+1;
				else if (d==2) zl = z+1;
				
				int xyzn = xi+nx*yj+nx*ny*zl;
			
				int dir = -1;
				if (d==0) dir = X;
				else if (d==1) dir = Y;
				else if (d==2) dir = Z;
				
				Imdir[d][xyz] = signedAngleFiberSimilarityGain(dir,xyz,xyzn);
				Imdir[d+3][xyz] = signedAngleOverlapSimilarityGain(dir,xyz,xyzn);
			}
		}
		return Imdir;
	} // exportImageDirection
	
	public final float[][] exportInitialFactor() {
		float[][]	Imdir = new float[nt][nx*ny*nz];
		float[][] dir = new float[nt][3];
		float[] prior = new float[nt];
		
		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			int xyz = x+nx*y+nx*ny*z;
			atlas.getTransformedShape(prior,x,y,z);
			atlas.getTransformedDirection(dir,x,y,z);
			float priorsum = 0.0f;
			for (int t=0;t<nt;t++) priorsum += prior[t];
			float ufiber = fiberGain(xyz);
			for (int t=0;t<nt;t++) {
				float uprior = prior[t]*prior[t]/priorsum;
				float udir = directionAngleFactor(dir[t],xyz);
				Imdir[t][xyz] =  ufiber*uprior*(0.5f+0.5f*udir);
			}
				
		}
		return Imdir;
	} // exportImageDirection
	
	public final float[][] exportAnisotropyFactor() {
		float[][]	Iso = new float[3][nx*ny*nz];
		
		for (int xyz=0;xyz<nx*ny*nz;xyz++) {
			Iso[0][xyz] = fiberGain(xyz);
			Iso[1][xyz] = crossingGain(xyz);
			Iso[2][xyz] = isoGain(xyz);
		}
		return Iso;
	} // exportisoMap
		
    final public byte[] generatePriorClassification() {
		float dist,max,count;
		byte id=0;
		byte[] img = new byte[nx*ny*nz];
		float[] shape = new float[nt];
	   
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			// compute each class gainbility : attribute the highest
			atlas.getTransformedShape(shape,x,y,z);
			// the background is NOT lowered for convenience (high prior masks the others)
			//max = 0.75f*shape[0]; id = 0;
			max = shape[0]; id = 0;
			for (byte k=1;k<nt;k++) {
				if (shape[k]>max) {
					id = k;
					max = shape[k];
				}
			}
			img[x+nx*y+nx*ny*z] = id;
		}
		return img;
	}	
	
    final public float[][] generatePriorDirectionMap() {
		float dist,max,count;
		byte id=0;
		float[][] img = new float[3][nx*ny*nz];
		float[] shape = new float[nt];
		float[][] dir = new float[nt][3];
		float atlasScale = 3.0f;
		
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			// compute each class gainbility : attribute the highest
			atlas.getTransformedShape(shape,x,y,z);
			atlas.getTransformedDirection(dir, x, y, z);
			
			max = shape[1]; id = 1;
			for (byte k=2;k<nt;k++) {
				if (shape[k]>max) {
					id = k;
					max = shape[k];
				}
			}
			max = Numerics.bounded(atlasScale*(max-0.5f)+0.5f,0.0f,1.0f);
			max = (1.0f-shape[0])*max;
			img[0][x+nx*y+nx*ny*z] = max*Numerics.abs(dir[id][0]);
			img[1][x+nx*y+nx*ny*z] = max*Numerics.abs(dir[id][1]);
			img[2][x+nx*y+nx*ny*z] = max*Numerics.abs(dir[id][2]);
		}
		return img;
	}	
	
	final public float[] exportMaxMappedAtlas() {
		float dist,max,count;
		byte id=0;
		float[] img = new float[nx*ny*nz];
		float[] shape = new float[nt];
		float atlasScale = 5.0f;
	   
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			// compute each class gainbility : attribute the highest
			atlas.getTransformedShape(shape,x,y,z);
			
			// the background is NOT lowered for convenience (high prior masks the others)
			max = 0.0f; id = -1;
			for (byte k=0;k<nt;k++) {
				if (shape[k]>max && atlas.isRegistered(k)) {
					id = k;
					max = shape[k];
				}
			}
			img[x+nx*y+nx*ny*z] = Numerics.bounded(atlasScale*(max-0.5f)+0.5f,0.0f,1.0f);
			
		}
		return img;
	}	
}
