package edu.jhu.ece.iacl.plugins.dti.tractography;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.awt.Color;

import javax.vecmath.Point3f;
import javax.vecmath.Point3i;

import edu.jhu.ece.iacl.algorithms.manual_label.LabelImage;
import edu.jhu.ece.iacl.algorithms.manual_label.ROI;
import edu.jhu.ece.iacl.jist.io.*;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamDouble;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamObject;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.fiber.FiberCollection;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataMipav;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.fiber.Fiber;
import edu.jhu.ece.iacl.jist.structures.fiber.FiberCollection;
import edu.jhu.ece.iacl.jist.structures.fiber.XYZ;
import edu.jhu.ece.iacl.jist.structures.geom.GridPt;
import edu.jhu.ece.iacl.jist.structures.image.RGB;
import edu.jhu.ece.iacl.jist.structures.image.USbyte;

import gov.nih.mipav.model.structures.ModelLUT;

public class MedicAlgorithmApplyFiberDOTS extends ProcessingAlgorithm {
	//input param
	private ParamObject<FiberCollection> fiberset;
	private ParamVolume seg;
	private ParamOption operation;
	private ParamInteger label;
	private ParamOption outputtype;
	private ParamDouble threshold;
	private ParamDouble fiberlength;
	
	//output param
	private ParamFileCollection fibersubset;
	
	private FiberCollectionReaderWriter fcrw = FiberCollectionReaderWriter.getInstance();
	private CurveVtkReaderWriter cvrw = CurveVtkReaderWriter.getInstance();

	private static final String cvsversion = "$Revision: 1.2 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Gets a subset of the input fibers based on the input DOTS result.";
	private static final String longDescription = "";


	@Override
	protected void createInputParameters(ParamCollection inputParams) {
		inputParams.add(fiberset = new ParamObject<FiberCollection>("Fiber Set", new FiberCollectionReaderWriter()));
		inputParams.add(seg = new ParamVolume("Segmentation (4D)"));
		inputParams.add(operation = new ParamOption("ROI Operation", new String[]{"Include", "Maximum", "MaximumIncluded"}));
		operation.setDescription("Type of labeling: 'Include' means all fibers of sufficient length and amount within the threshold are included (a fiber may have multiple labels)\n"
									+"'Maximum' means the fibers get the most probable label and 'MaximumIncluded' combines both, i.e the fiber gets the most probable label if it is within the threshold.");
		inputParams.add(label = new ParamInteger("Label",-1));
		label.setDescription("Integer Label defining the ROI.  A value of negative one indicates that all values greater than zero will be included in the region of interest");

		inputParams.add(outputtype=new ParamOption("Output Type",new String[]{"DtiStudio dat","Vtk"}));
		inputParams.add(threshold = new ParamDouble("Threshold",0.5f));
		threshold.setDescription("[0,1] threshold for the 'include' mode: only fibers with at least that much of their length inside the ROI are included.");

		inputParams.add(fiberlength = new ParamDouble("Min. fiber length",50));
		threshold.setDescription("Only fiber larger than the specified length are considered");


		inputParams.setPackage("IACL");
		inputParams.setCategory("DTI.Fiber");
		inputParams.setLabel("Apply Fiber DOTS");
		inputParams.setName("Apply_Fiber_DOTS");


		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("");
		info.add(new AlgorithmAuthor("John Bogovic", "bogovic@jhu.edu", "http://putter.ece.jhu.edu/John"));
		info.add(new AlgorithmAuthor("Pierre-Louis Bazin", "pbazin1@jhmi.edu", "http://medic.rad.jhu.edu/pbazin/"));
		info.setAffiliation("Johns Hopkins University, Department of Radiology and Radiological Science");
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.RC);
	}


	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(fibersubset = new ParamFileCollection("Fiber Sub-Sets",new FileExtensionFilter(new String[]{"dat","vtk"})));
	}


	@Override
	protected void execute(CalculationMonitor monitor)
			throws AlgorithmRuntimeException {
		File dir = new File(this.getOutputDirectory()+File.separator+edu.jhu.ece.iacl.jist.utility.FileUtil.forceSafeFilename(this.getAlgorithmName()));
		try{
			if(!dir.isDirectory()){
				(new File(dir.getCanonicalPath())).mkdir();
			}
		}catch(IOException e){
			e.printStackTrace();
		}
		
		FiberCollection[] selected = fuzzyFiberROIs(fiberset.getObject(), seg.getImageData(), operation.getValue(), threshold.getFloat(), 1, "lb");
		FiberCollection grouped = new FiberCollection();
		grouped.setDimensions(selected[0].getDimensions());
		grouped.setResolutions(selected[0].getResolutions());
		for (int n=0;n<selected.length;n++) {
			File f = writeFibers(selected[n],dir);
			fibersubset.add(f);
			if (n>1) grouped.addAll(selected[n]);
		}
		
		String filename = fiberset.getObject().getName()+"_"+operation.getValue()+"_"+"all";
		grouped.setName(filename);

		File f = writeFibers(grouped,dir);
		fibersubset.add(f);
			

	}

	private File writeFibers(FiberCollection fibers, File dir){
		File out = null;
		if(outputtype.getIndex()==0){
			out = fcrw.write(fibers, dir);
		}else{
			out = cvrw.write(fibers.toCurveCollection(), dir);
		}
		return out;
	}
	
		/**
	 * 
	 * A method to parcellate fibers based to DTIStudio like regions of interest ROIs
	 * 
	 * @param tractin	The fibercollection containing the fibers to be chosen
	 * @param op		The operation to be applied
	 * @param param		The optional parameter for the operation of interest
	 * @param label		The label of interest.  (-1 indicates all labels >0 
	 * @return			A FiberCollection containing the chosen fibers
	 * @author 			John Bogovic, bogovic@jhu.edu	
	 * @author 			Pierre-Louis Bazin, pbazin1@jhmi.edu
	 */
	private FiberCollection[] fuzzyFiberROIs(FiberCollection tractin, ImageData vol, String op, float param, int label, String name){
		
		System.out.println("Fuzzy Fiber-ROI overlap, mode: "+op+" ("+param+", "+label+")");
		
		String filename ="";
		filename = tractin.getName()+"_"+op+"_"+name;

		FiberCollection[] tractout = new FiberCollection[vol.getComponents()];
		for(int l=0; l<vol.getComponents(); l++) {
			tractout[l] = new FiberCollection();
			tractout[l].setDimensions(new Point3i(vol.getRows(), vol.getCols(), vol.getSlices()));
			tractout[l].setResolutions(new Point3f(vol.getHeader().getDimResolutions()[0],
													vol.getHeader().getDimResolutions()[1],
													vol.getHeader().getDimResolutions()[2]));
			tractout[l].setName(filename+(l+1));
		}
		
		ModelLUT lut = new ModelLUT(ModelLUT.STRIPED,256,new int[]{4,256});
		
		RGB[] color = new RGB[vol.getComponents()];
		for(int l=0; l<vol.getComponents(); l++) {
			Color c = lut.getColor(l);
			System.out.println("color scheme : "+l+" -> ("+c.getRed()+", "+c.getGreen()+", "+c.getBlue()+")");
			USbyte usb = new USbyte();
			color[l] = new RGB(usb.unsigned2signed(c.getRed()),usb.unsigned2signed(c.getGreen()),usb.unsigned2signed(c.getBlue()));
			
			// debug
			//color[l] = new RGB((byte)255,(byte)0,(byte)0);
			System.out.println("color scheme : "+l+" -> ("+color[l].r+", "+color[l].g+", "+color[l].b+")");
		}
		
		// select the fibers and components of interest
		for(Fiber f : tractin){
			int[] included = new int[vol.getComponents()];
			for (int i : included) included[i] = 0;
			int length = 0;
			
			for(XYZ pt : f.getXYZChain()){
				
				GridPt p = new GridPt((int)Math.floor(pt.x),(int)Math.floor(pt.y),(int)Math.floor(pt.z), 0);
				
				// gather statistics first, then perform operation specific selection
				length++;
				for(int l=0; l<vol.getComponents(); l++) {
					if(vol.getBoolean(p.x, p.y, p.z, l)) {
						included[l]++;
					}
				}
			}
			//System.out.println("Fiber length: "+length);
			
			// results
			if (length >= fiberlength.getDouble()) {			
				if (op.equals("Include")) {
					for(int l=0; l<vol.getComponents(); l++) {
						if (included[l] >= param*length) {
							f.setColor(color[l]);
							tractout[l].add(f);
							//System.out.println("added to label "+(l+1));
						}
					}
				} else if (op.equals("Maximum")) {
					int max = 0;
					for(int l=0; l<vol.getComponents(); l++) {
						if (included[l] > included[max]) max = l;
					}
					f.setColor(color[max]);
					tractout[max].add(f);
					//System.out.println("added to label "+(max+1));
				} else if (op.equals("MaximumIncluded")) {
					int max = 0;
					for(int l=0; l<vol.getComponents(); l++) {
						if (included[l] > included[max]) max = l;
					}
					if (included[max] >= param*length) {
						f.setColor(color[max]);
						tractout[max].add(f);
						//System.out.println("added to label "+(max+1));
					}
				} else {
					//System.out.println("Unsupported option!");
				}
			}
		}
		return tractout;
	}
	

}
