#ifndef TRIMESH_H
#define TRIMESH_H
/*
Szymon Rusinkiewicz
Princeton University

TriMesh.h
Class for triangle meshes.
*/

#define  LARGENUM  10000000.0
#define  ONE       1 
#define  CURVATURE 2 
#define  NOISE     3
//#define SPEEDTYPE ONE

#include "Vec.h"
#include "Color.h"
#include "KDtree.h"
#include "math.h"
#include <vector>
#include <list>
#include <map>
#include <limits>
#include <iostream>
#include <fstream>

using std::vector;
using std::map;

#define  PI 3.141592653589793

#ifndef MIN
  #define MIN(a,b) ((a)<(b))?(a):(b) 
#endif

#ifndef MAX
  #define MAX(a,b) ((a)>(b))?(a):(b)
#endif


class TriMesh {
protected:
	static bool read_helper(const char *filename, TriMesh *mesh);

public:
	// Types
	struct Face {
		int v[3];
		float speedInv;
		float T[3];
		vec3 edgeLens;  // edge length for 01, 12, 20

		Face() {}
		Face(const int &v0, const int &v1, const int &v2)
		{
			v[0] = v0; v[1] = v1; v[2] = v2;
		}
		Face(const int *v_)
		{
			v[0] = v_[0]; v[1] = v_[1]; v[2] = v_[2];
		}
		int &operator[] (int i) { return v[i]; }
		const int &operator[] (int i) const { return v[i]; }
		operator const int * () const { return &(v[0]); }
		operator const int * () { return &(v[0]); }
		operator int * () { return &(v[0]); }
		int indexof(int v_) const
		{
			return (v[0] == v_) ? 0 :
			       (v[1] == v_) ? 1 :
			       (v[2] == v_) ? 2 : -1;
		}
	};

	class BBox {
  public:
    point min, max;
    bool valid;
    
    // Construct as empty
	BBox() : min(point(std::numeric_limits<float>::max(),
										 std::numeric_limits<float>::max(),
										 std::numeric_limits<float>::max())),
      max(point(std::numeric_limits<float>::min(),
								std::numeric_limits<float>::min(),
								std::numeric_limits<float>::min())),
      valid(false)
				{}
    
    // Initialize to one point or two points
  BBox(const point &p) : min(p), max(p), valid(true)
      {}
  BBox(const point &min_, const point &max_) :
    min(min_), max(max_), valid(true)
      {}
    
    // Mark invalid
    void clear()
    {
      min = point(std::numeric_limits<float>::max(),
									std::numeric_limits<float>::max(),
		  std::numeric_limits<float>::max());
      max = point(std::numeric_limits<float>::min(),
									std::numeric_limits<float>::min(),
									std::numeric_limits<float>::min());
      valid = false;
    }
    
    // Return center point and (vector) diagonal
    point center() const { return 0.5f * (min+max); }
    vec size() const { return max - min; }
    
    // Grow a bounding box to encompass a point
    BBox &operator += (const point &p)
      { min.min(p); max.max(p); return *this; }
    BBox &operator += (const BBox &b)
      { min.min(b.min); max.max(b.max); return *this; }
    
    // The following appear to be necessary for Visual Studio,
    // despite the fact that the operators shouldn't need
    // to be friends...
    friend const TriMesh::BBox operator + (const TriMesh::BBox &b, const point &p);
    friend const TriMesh::BBox operator + (const point &p, const TriMesh::BBox &b);
    friend const TriMesh::BBox operator + (const TriMesh::BBox &b1, const TriMesh::BBox &b2);
  };
	/*
		struct BBox {
		point min, max;
		point center() const { return 0.5f * (min+max); }
		vec size() const { return max - min; }
		bool valid;
		BBox() : valid(false)
			{}
	};
	*/

	struct BSphere {
		point center;
		float r;
		bool valid;
		BSphere() : valid(false)
			{}
	};

	// Enums
	enum tstrip_rep { TSTRIP_LENGTH, TSTRIP_TERM };
	enum { GRID_INVALID = -1 };
	//enum speed_type { ONE = 0, CURVATURE, NOISE };

	// The basics: vertices and faces
	vector< point > vertices;
	vector<Face> faces;

	int speedType;



 
	// Triangle strips
	vector<int> tstrips;

	// Grid, if present
	vector<int> grid;
	int grid_width, grid_height;

	// Other per-vertex properties
	vector<Color> colors;
	vector<float> confidences;
	vector<unsigned> flags;
	unsigned flag_curr;

	// Computed per-vertex properties
	vector<vec> normals;
	vector<vec> pdir1, pdir2;
	vector<float> curv1, curv2;
	vector<float> abs_curv;
	vector< Vec<4,float> > dcurv;
	vector<vec> cornerareas;
	vector<float> pointareas;

  KDtree *kd;
  double maxEdgeLength;
  vector< map<unsigned int, float> > geodesicMap;
	float *geodesic;

  vector< vector<float> > features;

	// Bounding structures
	BBox bbox;
	BSphere bsphere;

	// Connectivity structures:
	//  For each vertex, all neighboring vertices
	vector< vector<int> > neighbors;
	//  For each vertex, all neighboring faces
	vector< vector<int> > adjacentfaces;

	vector<double> radiusInscribe;

	vector<int>  getTwoNeighbors(int v){
		vector<int> twoNeighbors;
		
		// for each neighbor
		for(int i=0; i < this->neighbors[v].size(); i++){
			// add self
			int n = this->neighbors[v][i];
			twoNeighbors.push_back(n);
			
			// add neighbors
			for(int j=0; j < this->neighbors[n].size(); j++)
				twoNeighbors.push_back( this->neighbors[n][j] );
		}


		return twoNeighbors;
	}


	vector< vector<Face> > vertOneringFaces;
	//  For each face, the three faces attached to its edges
	//  (for example, across_edge[3][2] is the number of the face
	//   that's touching the edge opposite vertex 2 of face 3)
	vector<Face> across_edge;

	vector<float> noiseOnVert;
	
	 int getSpeedType(){
		 return speedType;
	 }
	//int SPEEDTYPE;
	// Compute all this stuff...
	void setSpeedType(int st)
	{
		//ST = st;
		speedType = st;
		if(st == ONE){
			//iMap = &geoIndex;
			//dMap = &geoMap;
		}
		else if(st == CURVATURE){
			//iMap = &adaptIndex;
			//dMap = &adaptMap;
		}
		else{
			std::cout  << "Impossible SpeedType set" << std::endl;
			exit(1337);
		}
	}
	void need_tstrips();
	void convert_strips(tstrip_rep rep);
	void unpack_tstrips();
	void triangulate_grid();
	void need_faces()
	{
		if (!faces.empty())
			return;
		if (!tstrips.empty())
			unpack_tstrips();
		else if (!grid.empty())
			triangulate_grid();
	}

	void need_faceedges();
	void need_speed();
	void need_noise(int nNoiseIter);
	void need_oneringfaces();
  void need_kdtree();
  void need_maxedgelength();
	void need_normals();
	void need_pointareas();
	void need_curvatures();
	void need_abs_curvatures();
	void need_dcurv();
	void need_bbox();
	void need_bsphere();
	void need_neighbors();
	void need_adjacentfaces();
	void need_across_edge();
	void need_meshinfo();
	void need_Rinscribe();

	// Input and output
	static TriMesh *read(const char *filename);
	bool write(const char *filename);

	// Statistics
	// XXX - Add stuff here
	float feature_size();

	// Useful queries
	// XXX - Add stuff here
	bool is_bdy(int v)
	{
		if (neighbors.empty()) need_neighbors();
		if (adjacentfaces.empty()) need_adjacentfaces();
		return neighbors[v].size() != adjacentfaces[v].size();
	}


vec trinorm(int f)
{
	if (faces.empty()) need_faces();
	return ::trinorm(vertices[faces[f][0]], vertices[faces[f][1]],
		vertices[faces[f][2]]);
}

// FIM: check angle for at a given vertex, for a given face
bool IsNonObtuse(int v, Face f)
{		
	int iV = f.indexof(v);

	point A = this->vertices[v];
	point B = this->vertices[f[(iV+1)%3]];
	point C = this->vertices[f[(iV+2)%3]];

	float a = dist(B,C);
	float b = dist(A,C);
	float c = dist(A,B);

	float angA = 0.0; /* = acos( (b*b + c*c - a*a) / (2*b*c) )*/

	if ((a > 0) && (b > 0) && (c > 0))//  Manasi stack overflow
	{//  Manasi stack overflow
		angA = acos( (b*b + c*c - a*a) / (2*b*c) );//  Manasi stack overflow
	}//  Manasi stack overflow

	//return ( angA - PI/2.0f < -0.00001 );
	return ( angA < M_PI/2.0f );
}

// FIM: given a vertex, find an all-acute neighborhood of faces
void SplitFace(vector<Face> &acFaces, int v, Face cf, int nfAdj/*, int currentVert*/)
{		
	// get all the four vertices in order
	/* v1         v4
	+-------+
	\     . \
	\   .   \
	\ .     \
	+-------+
	v2         v3 */

	int iV = cf.indexof(v);											// get index of v in terms of cf
	int v1 = v;
	int v2 = cf[(iV+1)%3];
	int v4 = cf[(iV+2)%3];
	iV = this->faces[nfAdj].indexof(v2);				// get index of v in terms of adjacent face
	int v3 = this->faces[nfAdj][(iV+1)%3];

	// create faces (v1,v3,v4) and (v1,v2,v3), check angle at v1
	Face f1(v1,v3,v4);													
	//f1.T[f1.indexof(v1)] = this->vertT[currentVert][v1];
	//f1.T[f1.indexof(v3)] = this->vertT[currentVert][v3];
	//f1.T[f1.indexof(v4)] = this->vertT[currentVert][v4];
	

	Face f2(v1,v2,v3);
	//f2.T[f2.indexof(v1)] = this->vertT[currentVert][v1];
	//f2.T[f2.indexof(v2)] = this->vertT[currentVert][v2];
	//f2.T[f2.indexof(v3)] = this->vertT[currentVert][v3];
	
	
	if (IsNonObtuse(v,f1))
	{
		//switch (SPEEDTYPE)
		switch(speedType)
		{
		case CURVATURE:
			/*
			f1.speedInv = ( abs(curv1[f1[0]] + curv2[f1[0]]) + 
											abs(curv1[f1[1]] + curv2[f1[1]]) + 
											abs(curv1[f1[2]] + curv2[f1[2]]) ) / 6;
			*/
			f1.speedInv = ( abs_curv[f1[0]] + 
               abs_curv[f1[1]] + 
               abs_curv[f1[2]] ) / 3.0;
               
			break;
		case ONE:
			f1.speedInv = 1.0;
			break;
		case NOISE:
			f1.speedInv = (noiseOnVert[f1[0]] + 
										 noiseOnVert[f1[1]] + 
										 noiseOnVert[f1[2]]) / 3;
		default:
			f1.speedInv = 1.0;
			break;
		}
		
		vec3 edge01 = (vec3)(vertices[f1[1]] - vertices[f1[0]]);
		vec3 edge12 = (vec3)(vertices[f1[2]] - vertices[f1[1]]);
		vec3 edge20 = (vec3)(vertices[f1[0]] - vertices[f1[2]]);
		f1.edgeLens[0] =sqrt(edge01[0]*edge01[0] + edge01[1]*edge01[1] + edge01[2]*edge01[2]);
		f1.edgeLens[1] =sqrt(edge12[0]*edge12[0] + edge12[1]*edge12[1] + edge12[2]*edge12[2]);
		f1.edgeLens[2] =sqrt(edge20[0]*edge20[0] + edge20[1]*edge20[1] + edge20[2]*edge20[2]);
		acFaces.push_back(f1);
	}
	else
	{
		int nfAdj_new = this->across_edge[nfAdj][this->faces[nfAdj].indexof(v2)];
		if (nfAdj_new > -1)
		{
			SplitFace(acFaces,v,f1,nfAdj_new/*, currentVert*/);
		}
		else
		{
			//printf("NO cross edge!!! Maybe a hole!!\n");
		}
		//SplitFace(acFaces,v,f1,nfAdj_new, currentVert);
	}

	if (IsNonObtuse(v,f2))
	{
		//switch (SPEEDTYPE)
		switch(speedType)
		{
		case CURVATURE:
			/*
			f2.speedInv = ( abs(curv1[f2[0]] + curv2[f2[0]]) + 
												abs( curv1[f2[1]] + curv2[f2[1]]) + 
												abs(curv1[f2[2]] + curv2[f2[2]]) ) / 6;
			*/
			f2.speedInv = ( abs_curv[f2[0]] + 
               abs_curv[f2[1]] + 
               abs_curv[f2[2]] ) / 3.0;

			break;
		case ONE:
			f2.speedInv = 1.0;
			break;
		case NOISE:
			f2.speedInv = (noiseOnVert[f2[0]] + 
										 noiseOnVert[f2[1]] + 

										 noiseOnVert[f2[2]]) / 3;

			break;
		default:
			f2.speedInv = 1.0;
			break;
		}


		vec3 edge01 = (vec3)(vertices[f2[1]] - vertices[f2[0]]);
		vec3 edge12 = (vec3)(vertices[f2[2]] - vertices[f2[1]]);
		vec3 edge20 = (vec3)(vertices[f2[0]] - vertices[f2[2]]);
		f2.edgeLens[0] =sqrt(edge01[0]*edge01[0] + edge01[1]*edge01[1] + edge01[2]*edge01[2]);
		f2.edgeLens[1] =sqrt(edge12[0]*edge12[0] + edge12[1]*edge12[1] + edge12[2]*edge12[2]);
		f2.edgeLens[2] =sqrt(edge20[0]*edge20[0] + edge20[1]*edge20[1] + edge20[2]*edge20[2]);
		acFaces.push_back(f2);
	}
	else
	{
		int nfAdj_new = this->across_edge[nfAdj][this->faces[nfAdj].indexof(v4)];
		if (nfAdj_new > -1)
		{
			SplitFace(acFaces,v,f2,nfAdj_new/*,currentVert*/);
		}
		else
		{
			//printf("NO cross edge!!! Maybe a hole!!\n");
		}
		//SplitFace(acFaces,v,f2,nfAdj_new,currentVert);
	}
}


// FIM: one ring function
vector<Face> GetOneRing(int v/*, int currentVert*/)
{
  // make sure we have the across-edge map
	if (this->across_edge.empty())
		this->need_across_edge();

	// variables required
	vector<Face> oneRingFaces;
	vector<Face> t_faces;

	// get adjacent faces
	int naf = this->adjacentfaces[v].size();

	if (!naf)
	{
		std::cout << "vertex " << v << " has 0 adjacent faces..." << std::endl;
	}
	else
	{
		for (int af = 0; af < naf; af++)
		{
			Face cf = this->faces[adjacentfaces[v][af]];

			t_faces.clear();
			if(IsNonObtuse(v,cf))// check angle: if non-obtuse, return existing face
			{
				//this->colors[cf[0]] = Color::red();
				//this->colors[cf[1]] = Color::red();
				//this->colors[cf[2]] = Color::red();
				t_faces.push_back(cf);					
			}
			else
			{
				//t_faces.push_back(cf);
				int nfae = this->across_edge[this->adjacentfaces[v][af]][cf.indexof(v)];
				if (nfae > -1)
				{
					SplitFace(t_faces,v,cf,nfae/*,currentVert*/);// if obtuse, split face till we get all acute angles
				}
				else
				{
					//printf("NO cross edge!!! Maybe a hole!!\n");
				}
				//SplitFace(t_faces,v,cf,nfae,currentVert);// if obtuse, split face till we get all acute angles
			}

			for (int tf = 0; tf < t_faces.size(); tf++)
			{					
				//this->colors[t_faces[tf][0]] = Color::red();
				//this->colors[t_faces[tf][1]] = Color::red();
				//this->colors[t_faces[tf][2]] = Color::red();
				oneRingFaces.push_back(t_faces[tf]);
			}
		}				
	}
	//this->colors[v] = Color::green();
	return oneRingFaces;
}


	// FIM: initialize attributes
	//typedef std::<int> ListType;
void InitializeAttributes(int currentVert , std::vector<int> seeds = vector<int>() )
{
	int nv = this->vertices.size();
	this->geodesic = new float[nv];
	
	for(int v= 0; v < nv; v++){
		geodesic[v] = LARGENUM;
	}

	// initialize seed points if present...
	if (!seeds.empty()){
		int ns = seeds.size();
		for (int s = 0; s < ns; s++){
			//this->vertMap[currentVert][seeds[s]].d = 0;
			geodesic[seeds[s]] = 0;
		}
	}


  // pre-compute faces, normals, and other per-vertex properties that may be needed
	this->need_neighbors();
	this->need_normals();
	this->need_adjacentfaces();
	this->need_across_edge();		
	this->need_faces();

	/* HOW DO WE DO THIS USING NEW GEODESIC DATA STRUCTURE?
	// for all faces: initialize per-vertex travel time and face-speed		
	int nf = this->faces.size();
	for (int f = 0; f < nf; f++)
	{
		Face cf = this->faces[f];
		
		// travel time
		faces[f].T[0] = this->vertT[currentVert][cf[0]];
		faces[f].T[1] = this->vertT[currentVert][cf[1]];
		faces[f].T[2] = this->vertT[currentVert][cf[2]];
	}
	*/
}

// FIM:  Remove data lingering from computation
void CleanupAttributes(int currentVert)
{
	delete [] this->geodesic;
}

float GetEuclideanDistance(int v1,int v2)
{
	float d = 0.000001f;
	point p1, p2;
	
	p1 = this->vertices[v1];
	p2 = this->vertices[v2];

	d = dist(p1,p2);

	return d;
}


float GetGeodesicDistance(int v1,int v2)
{
  float gDist = 0.000001f;

  if (v1 == v2) return gDist;

  int vert = v1;
  int key = v2;

  if (v2 > v1)
  {
    vert = v2;
    key = v1;
  }

  std::map<unsigned int,float>::iterator geoIter = this->geodesicMap[vert].find(key);
  if (geoIter != this->geodesicMap[vert].end())
  {
    gDist = geoIter->second;
  }
  else
  {
    gDist = LARGENUM;
  }

  return gDist;
}

double GetGeodesicDistance(point x, point y)
{
  // get vertex closest to first point - x
  int vertX = this->FindNearestVertex(x);

  // scan all adjacent faces to see which face (f) includes point x
  float alphaX, betaX, gammaX;
  Face triangleX = this->faces[ this->adjacentfaces[vertX][0] ];
  for (unsigned int fNumber = 0; fNumber < this->adjacentfaces[vertX].size(); fNumber++)
  {
    // check if face contains x and store barycentric coordinates for x in face f
    triangleX = this->faces[ this->adjacentfaces[vertX][fNumber] ];
    vec barycentric = this->ComputeBarycentricCoordinates(x,triangleX);
    alphaX = barycentric[0];
    betaX = barycentric[1];
    gammaX = barycentric[2];

    if ( ( ( barycentric[0] >= 0 ) && ( barycentric[0] <= 1 ) ) &&
         ( ( barycentric[1] >= 0 ) && ( barycentric[1] <= 1 ) ) &&
         ( ( barycentric[2] >= 0 ) && ( barycentric[2] <= 1 ) ) )
    {
      fNumber = this->adjacentfaces[vertX].size();
    }
  }

  // repeat this process for point y
  int vertY = this->FindNearestVertex(y);

  // scan all adjacent faces to see which face (f) includes point y
  float alphaY, betaY, gammaY;
  Face triangleY = this->faces[ this->adjacentfaces[vertY][0] ];
  for (unsigned int fNumber = 0; fNumber < this->adjacentfaces[vertY].size(); fNumber++)
  {
    // check if face contains y and store barycentric coordinates for y in face f
    triangleY = this->faces[ this->adjacentfaces[vertY][fNumber] ];
    vec barycentric = this->ComputeBarycentricCoordinates(y,triangleY);
    alphaY = barycentric[0];
    betaY = barycentric[1];
    gammaY = barycentric[2];

    if ( ( barycentric[0] >= 0 ) && ( barycentric[0] <= 1 ) &&
         ( barycentric[1] >= 0 ) && ( barycentric[1] <= 1 ) &&
         ( barycentric[2] >= 0 ) && ( barycentric[2] <= 1 ) )
    {
      fNumber = this->adjacentfaces[vertY].size();
    }
  }

  // compute geodesic distance by interpolation
  // level one, interpolate distance from source triangle to distination point (i.e. D(triangleX, y))
  float dx0y = ( alphaY * this->GetGeodesicDistance( triangleX.v[0], triangleY.v[0] ) ) +
               ( betaY  * this->GetGeodesicDistance( triangleX.v[0], triangleY.v[1] ) ) +
               ( gammaY * this->GetGeodesicDistance( triangleX.v[0], triangleY.v[2] ) );

  float dx1y = ( alphaY * this->GetGeodesicDistance( triangleX.v[1], triangleY.v[0] ) ) +
               ( betaY  * this->GetGeodesicDistance( triangleX.v[1], triangleY.v[1] ) ) +
               ( gammaY * this->GetGeodesicDistance( triangleX.v[1], triangleY.v[2] ) );

  float dx2y = ( alphaY * this->GetGeodesicDistance( triangleX.v[2], triangleY.v[0] ) ) +
               ( betaY  * this->GetGeodesicDistance( triangleX.v[2], triangleY.v[1] ) ) +
               ( gammaY * this->GetGeodesicDistance( triangleX.v[2], triangleY.v[2] ) );

  // level 2, interpolate distance between x & y
  float dxy = (alphaX * dx0y) + (betaX * dx1y) + (gammaX * dx2y);

  return dxy;
}

int FindNearestVertex(point pt)
{
	if ( !kd )
	{
    kd = new KDtree(this->vertices);
  }	

  if (maxEdgeLength == 0.0)
  {
    need_maxedgelength();
  }

  const float *match = kd->closest_to_pt(pt,sqr(maxEdgeLength));
  int imatch = 0;
  if (!match)
  {
    std::cout << "failed to find vertex within " << maxEdgeLength << " for point " << pt << ". using vertex 0" << std::endl;
    return imatch;
  }

  imatch = (match - (const float *) &(vertices[0][0])) / 3;
  return imatch;
}

vec3 ComputeBarycentricCoordinates(point p, Face f)
{
  vec3 bCoords; bCoords.clear();
  point a,b,c;
  a = this->vertices[ f.v[0] ];
  b = this->vertices[ f.v[1] ];
  c = this->vertices[ f.v[2] ];

  point n = (b - a) CROSS (c - a);
  normalize(n);

  float denominator = ( (b - a) CROSS (c - a) ) DOT n;

  bCoords[0] = ( ( (c - b) CROSS (p - b) ) DOT n ) / denominator;
  bCoords[1] = ( ( (a - c) CROSS (p - c) ) DOT n ) / denominator;
  bCoords[2] = ( ( (b - a) CROSS (p - a) ) DOT n ) / denominator;

  return bCoords;
}

void ReadFeatureFromFile(const char *infilename)
{
  std::ifstream infile(infilename, std::ios::binary);
	if (!infile.is_open())
  {
    std::cout << "File Not Found" << std::endl;
	}
  else
  {
    // read # vertices
    unsigned int numVert;
    infile.read(reinterpret_cast<char *>(&numVert), sizeof(unsigned int));

    if ( numVert != this->vertices.size() )
    {
      std::cout << "size of feature vector does not match # vertices in mesh" << std::endl;      
    }
    else
    {
      std::cout << "reading feature from file " << infilename << std::endl;

      vector< float > tmpFeatureVec;
      // loop over vertices
      for (int i = 0; i < numVert; i++)
      {
        // read feature value
        float value;
        infile.read( reinterpret_cast<char *>(&value), sizeof(float) );
        tmpFeatureVec.push_back(value);
      }

      this->features.push_back( tmpFeatureVec );
    }

		infile.close();
  }

}


void ReadFeatureFromList(const char *infilename)
{
  std::ifstream infile(infilename);
	if (!infile.is_open())
  {
    std::cout << "File Not Found" << std::endl;
	}
  else
  {
    std::cout << "reading feature from file " << infilename << std::endl;

    vector< float > tmpFeatureVec;
    float value;
    // loop over vertices
    while (infile)
    {
      // read feature value
      infile >> value;
      tmpFeatureVec.push_back(value);
    }
    tmpFeatureVec.pop_back();

    if ( tmpFeatureVec.size() == this->vertices.size() )
    {
      this->features.push_back( tmpFeatureVec );
    }
    else
    {
      std::cout << "size of feature vector does not match # vertices in the mesh ! Aborting..." << std::endl;
      exit(1);
    }

		infile.close();
  }

}



void WriteFeatureToFile(int featureIndex, const char *outfilename)
{
  //std::ofstream outfile(outfilename, std::ios::binary);

  //// write numVertices to facilitate reading later
  //int numVert = this->vertices.size();
  //outfile.write( reinterpret_cast<char *>(&numVert), sizeof(int) );

  //// loop over each vertex
  //for (int i = 0; i < numVert; i++)
  //{
  //  // write distance to curve
  //  unsigned short value = this->(features[featureIndex])[i];
  //  outfile.write( reinterpret_cast<char *>(&value), sizeof(unsigned short) );
  //}

  //outfile.close();
}

float GetFeatureValue(point x, int featureIndex)
{
  // find vertex nearest to x
  int vertX = this->FindNearestVertex(x);

  // check one-ring to see which triangle contains x
  float alphaX, betaX, gammaX;
  Face triangleX = this->faces[ this->adjacentfaces[vertX][0] ];
  for (unsigned int fNumber = 0; fNumber < this->adjacentfaces[vertX].size(); fNumber++)
  {
    // check if face contains x and store barycentric coordinates for x in face f
    triangleX = this->faces[ this->adjacentfaces[vertX][fNumber] ];
    vec barycentric = this->ComputeBarycentricCoordinates(x,triangleX);
    alphaX = barycentric[0];
    betaX = barycentric[1];
    gammaX = barycentric[2];

    if ( ( ( barycentric[0] >= 0 ) && ( barycentric[0] <= 1 ) ) &&
         ( ( barycentric[1] >= 0 ) && ( barycentric[1] <= 1 ) ) &&
         ( ( barycentric[2] >= 0 ) && ( barycentric[2] <= 1 ) ) )
    {
      fNumber = this->adjacentfaces[vertX].size() + 1;
    }
  }

  // interpolate feature values on triangle face
  float f0 = this->features[featureIndex][ triangleX.v[0] ];
  float f1 = this->features[featureIndex][ triangleX.v[1] ];
  float f2 = this->features[featureIndex][ triangleX.v[2] ];
  float featureValue = (alphaX * f0) + (betaX * f1) + (gammaX * f2);

  return featureValue;
}

point GetFeatureDerivative(point p, int fIndex = 0)
{
  point dP; dP.clear();
  dP[0] = 0.0f; dP[1] = 0.0f; dP[2] = 0.0f;

  // find nearest vertex
  int vertP = this->FindNearestVertex(p);

  // find triangle where point belongs
  float alphaP, betaP, gammaP;
  Face triangleP = this->faces[ this->adjacentfaces[vertP][0] ];
  for (unsigned int fNumber = 0; fNumber < this->adjacentfaces[vertP].size(); fNumber++)
  {
    // check if face contains x and store barycentric coordinates for x in face f
    triangleP = this->faces[ this->adjacentfaces[vertP][fNumber] ];
    vec barycentric = this->ComputeBarycentricCoordinates(p,triangleP);
    alphaP = barycentric[0];
    betaP = barycentric[1];
    gammaP = barycentric[2];

    if ( ( ( barycentric[0] >= 0 ) && ( barycentric[0] <= 1 ) ) &&
         ( ( barycentric[1] >= 0 ) && ( barycentric[1] <= 1 ) ) &&
         ( ( barycentric[2] >= 0 ) && ( barycentric[2] <= 1 ) ) )
    {
      fNumber = this->adjacentfaces[vertP].size() + 1;
    }
  }

  // compute derivative at 3 vertices (A,B,C)
  int A = triangleP.v[0];
  int B = triangleP.v[1];
  int C = triangleP.v[2];

  point dA = ComputeFeatureDerivative(A,fIndex);
  point dB = ComputeFeatureDerivative(B,fIndex);
  point dC = ComputeFeatureDerivative(C,fIndex);


  // interpolate
   dP[0] = ( alphaP * dA[0] ) + ( betaP * dB[0] ) + ( gammaP * dC[0] );
   dP[1] = ( alphaP * dA[1] ) + ( betaP * dB[1] ) + ( gammaP * dC[1] );
   dP[2] = ( alphaP * dA[2] ) + ( betaP * dB[2] ) + ( gammaP * dC[2] );

  return dP;
}

point ComputeFeatureDerivative(int v,int nFeature = 0)
{
  point df; df.clear();
  df[0] = 0.0f; df[1] = 0.0f; df[2] = 0.0f;

  // feature value at v
  float valueV = this->features[nFeature][v];
  point ptV = this->vertices[v];

  // iterate over neighbors of v to compute derivative as central difference
  for (unsigned int n = 0; n < this->neighbors[v].size(); n++)
  {
    int indexN = this->neighbors[v][n];
    float valueN = this->features[nFeature][indexN];
    point ptN = this->vertices[indexN];

    float valueDiff = valueN - valueV;
    point ptDiff = ptN - ptV;

    df[0] = df[0] + (valueDiff / ptDiff[0]) + 0.0001f;
    df[1] = df[1] + (valueDiff / ptDiff[1]) + 0.0001f;
    df[2] = df[2] + (valueDiff / ptDiff[2]) + 0.0001f;
  }

  df[0] = df[0] / (float) ( this->neighbors[v].size() );
  df[1] = df[1] / (float) ( this->neighbors[v].size() );
  df[2] = df[2] / (float) ( this->neighbors[v].size() );
  
  return df;
}

int GetNumberOfFeatures()
{
  return this->features.size();  
}



// Debugging printout, controllable by a "verbose"ness parameter

static int verbose;
static void set_verbose(int);
static void (*dprintf_hook)(const char *);
static void set_dprintf_hook(void (*hook)(const char *));
static void dprintf(const char *format, ...);

// Same as above, but fatal-error printout
static void (*eprintf_hook)(const char *);
static void set_eprintf_hook(void (*hook)(const char *));
static void eprintf(const char *format, ...);

// Constructor
 TriMesh() : grid_width(-1), grid_height(-1), flag_curr(0), speedType(ONE), maxEdgeLength(0.0)
{
	//iMap = &geoIndex;
	//dMap = &geoMap;
  kd = NULL;
}
};


inline const TriMesh::BBox operator + (const TriMesh::BBox &b, const point &p)
{
  return TriMesh::BBox(b) += p;
}


inline const TriMesh::BBox operator + (const point &p, const TriMesh::BBox &b)
{
  return TriMesh::BBox(b) += p;
}


inline const TriMesh::BBox operator + (const TriMesh::BBox &b1, const TriMesh::BBox &b2)
{
  return TriMesh::BBox(b1) += b2;
}

#endif

