/*******************************************************

This program performs subset feature selection and builds classifiers.

last modified : Oct 6, 2008

yong.fan@ieee.org
kanterae@uphs.upenn.edu

*******************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "nrutil_compare.h"
#include "general.h"
#include "StandardSVM.h"
#include "Kernel.h"
#include "UserKernel.h"
#include "IOTorch.h"
#include "COMPARE_SVM_rfe.h"
#define SCALE 100
#include <getopt.h>

#define EXEC_NAME "COMPARE_build"

#ifndef RELEASE_ID
#define RELEASE_ID "0.0_super_alpha"
#endif

#define SVNFILEVER "$Id: COMPARE_build.cc 18 2008-10-08 21:39:27Z kanterae $"

#ifndef SVN_REV
#define SVN_REV "NO_SVN" 
#endif


typedef struct
{
    string regression_mode;
    bool multi_mode;
    int first_class;
    int last_class;
    string c;
    string eps_regression;
    string unshrink_mode;
    string n_max_unshrink;
    int  kernel_type;
    int  d;
    real std;
    real s;
    real r;
    string u;
    string cache_size;
    string eps_fin;
    string h;
    string eps_shrink;
    string eps_bornes;
    string sparse_mode;
    bool bin_mode;
    int  load;
} parametres;

void version(){
  printf("\n\n%s Version information\n",EXEC_NAME);
  printf("  Release          : %s\n", RELEASE_ID);
  printf("  Svn Revision     : %s\n", SVN_REV);
  printf("  Svn File Version : %s\n", SVNFILEVER);
}

void usage(){
  printf("\n This program performs subset feature selection and builds classifiers\n");
  
  printf("\n Usage: %s (data_directory) (num_subject) (training_filename_base) (testing_filename_base) [options]\n");

  printf("\n Required arguments:");
  printf("\n   data_directory          : absolute path to the training and testing files");
  printf("\n   num_subject             : number of files to look for");
  printf("\n   training_filename_base  : training file prefix (ex. r_roi_train)");
  printf("\n   testing_filename_base   : testing file prefix (ex. r_roi_test)\n");
  
  printf("\n Optional arguments:\n");
  printf("\n   [-num_feature   value]    the number of candidate features to be used for feature selection and classifcation (default:150)"); 
  printf("\n   [-c             value]    SVM parameter:trade-off between training error and the margin (default:10)");
  printf("\n   [-std           value]    SVM parameter:kernel size in rbf kernel (default:100)");
  printf("\n   [-start_point   value]    the start point of a range of features for final classification (default:0)");
  printf("\n   [-end_point     value]    the end point of a range of features for final classification (default:150)");
  printf("\n   [-u      --usage]       Display this message");
  printf("\n   [-h       --help]       Display this message");
  printf("\n   [-v    --version]       Display version information\n");
  
  printf("\n More information:");
  printf("\n   https://www.rad.upenn.edu/sbia/software/");
  printf("\n   sbia-software@uphs.upenn.edu");
  printf("\n\n");
}

void roc(float *t, float *y, float *tp, float *fp, int n)
  //  generate a receiver operating characteristic curve
  
  //  gives the true-positive rate (TP) and false positive rate (FP), 
  //  where y is a vector giving the score assigned to each pattern 
  //  and t indicates the true class (a value above zero represents
  //  the positive class and anything else represents the negative class). 
  //  Fawcett, T., "ROC graphs : Notes and practical considerations for researchers", Technical report, HP
  //  Laboratories, MS 1143, 1501 Page Mill Road, Palo Alto CA 94304, USA, April 2004.
{
  // process targets
  float *tt = new float [n];
  for(int i=0; i<n; i++)
    if(t[i]>0)
      tt[i] = 1;
    else
      tt[i] = 0;
  
  // sort by classifier output
  unsigned long *idx = new unsigned long [n];
  float *yy = new float [n];
  for(int i=0; i<n; i++)
    yy[i] = -y[i];
  
  indexx(n, yy, idx);

  for(int i=0; i<n; i++)
    yy[i] =tt[idx[i]];

  //compute true positive and false positive rates
  tp[0] = yy[0];
  fp[0] = 1.0 - yy[0];
  
  for(int i=1; i<n; i++)
    {
      tp[i] = tp[i-1] + yy[i];
      fp[i] = fp[i-1] + 1.0 - yy[i];
    }
  
  float sum = 0;
  for(int i=0; i<n; i++)
    sum+=yy[i];
  sum += 1.0e-8;
  for(int i=0; i<n; i++)
    {
      tp[i] /= sum;
      fp[i] /= (n-sum)+1.0e-8;
    }

  delete [] tt;
  delete [] yy;
  delete [] idx;
}

void svm_model(real **data, real *y, int l, int c, parametres *params, string file_out)
{
 
  StandardSVM svmclass;
  
  svmclass.setOption("regression_mode", params->regression_mode);
  svmclass.setOption("C", params->c);
  svmclass.setOption("eps_regression", params->eps_regression);
  svmclass.setOption("unshrink_mode", params->unshrink_mode);
  svmclass.setOption("n_max_unshrink", params->n_max_unshrink);
  
  svmclass.setOption("cache_size_meg", params->cache_size);
  svmclass.setOption("eps_fin", params->eps_fin);
  svmclass.setOption("n_iter_min_to_shrink", params->h);
  svmclass.setOption("eps_shrink", params->eps_shrink);
  svmclass.setOption("eps_bornes", params->eps_bornes);
  svmclass.setOption("sparse_mode", params->sparse_mode);
  
  // creat a kernel for the svm classifier
  Kernel *he;
  switch(params->kernel_type)
  {
    case 0:
      he = new DotKernel(svmclass);
       break;
    case 1:
      he = new PolynomialKernel(svmclass);
      ((PolynomialKernel *)he)->setParametres(params->d, params->s, params->r);
      break;
    case 2:
      he = new GaussianKernel(svmclass);
       ((GaussianKernel *)he)->setParametres(params->std);
       break;
    case 3:
      he = new SigmoidKernel(svmclass);
      ((SigmoidKernel *)he)->setParametres(params->s, params->r);
      break;
    case 4:
      he = new UserKernel(svmclass);
      ((UserKernel *)he)->setParametres(params->u);
      break;
    default:
      cout << "$ What's this kernel ?!\n\n";
      exit(1);
  }
 
  sreal **sdata = NULL;
  svmclass.train(data, sdata, y, l, c);
  svmclass.save(file_out, "");
}

void svm_model1(real **data, real *y, int l, int c, real **tdata, real *py, int tl, parametres *params, string file_out)
{
 
  StandardSVM svmclass;
  
  svmclass.setOption("regression_mode", params->regression_mode);
  svmclass.setOption("C", params->c);
  svmclass.setOption("eps_regression", params->eps_regression);
  svmclass.setOption("unshrink_mode", params->unshrink_mode);
  svmclass.setOption("n_max_unshrink", params->n_max_unshrink);
  
  svmclass.setOption("cache_size_meg", params->cache_size);
  svmclass.setOption("eps_fin", params->eps_fin);
  svmclass.setOption("n_iter_min_to_shrink", params->h);
  svmclass.setOption("eps_shrink", params->eps_shrink);
  svmclass.setOption("eps_bornes", params->eps_bornes);
  svmclass.setOption("sparse_mode", params->sparse_mode);
  
  // creat a kernel
  Kernel *he;
  switch(params->kernel_type)
  {
    case 0:
      he = new DotKernel(svmclass);
       break;
    case 1:
      he = new PolynomialKernel(svmclass);
      ((PolynomialKernel *)he)->setParametres(params->d, params->s, params->r);
      break;
    case 2:
      he = new GaussianKernel(svmclass);
       ((GaussianKernel *)he)->setParametres(params->std);
       break;
    case 3:
      he = new SigmoidKernel(svmclass);
      ((SigmoidKernel *)he)->setParametres(params->s, params->r);
      break;
    case 4:
      he = new UserKernel(svmclass);
      ((UserKernel *)he)->setParametres(params->u);
      break;
    default:
      cout << "$ What's this kernel ?!\n\n";
      exit(1);
  }
 
  sreal **sdata = NULL;
  svmclass.train(data, sdata, y, l, c);
  svmclass.save(file_out, "");

  for(int i=0; i<tl; i++)
    py[i] = svmclass.use(tdata[i]);

}

void svmmodel_train_test(real **data, real **tdata, real *y, real *ty,  int l, int tl, int c, parametres *params, int *correction)
{
  StandardSVM svmclass;
  
  svmclass.setOption("regression_mode", params->regression_mode);
  svmclass.setOption("C", params->c);
  svmclass.setOption("eps_regression", params->eps_regression);
  svmclass.setOption("unshrink_mode", params->unshrink_mode);
  svmclass.setOption("n_max_unshrink", params->n_max_unshrink);
  
  svmclass.setOption("cache_size_meg", params->cache_size);
  svmclass.setOption("eps_fin", params->eps_fin);
  svmclass.setOption("n_iter_min_to_shrink", params->h);
  svmclass.setOption("eps_shrink", params->eps_shrink);
  svmclass.setOption("eps_bornes", params->eps_bornes);
  svmclass.setOption("sparse_mode", params->sparse_mode);
  
  Kernel *he;
  switch(params->kernel_type)
  {
    case 0:
      he = new DotKernel(svmclass);
       break;
    case 1:
      he = new PolynomialKernel(svmclass);
      ((PolynomialKernel *)he)->setParametres(params->d, params->s, params->r);
      break;
    case 2:
      he = new GaussianKernel(svmclass);
       ((GaussianKernel *)he)->setParametres(params->std);
       break;
    case 3:
      he = new SigmoidKernel(svmclass);
      ((SigmoidKernel *)he)->setParametres(params->s, params->r);
      break;
    case 4:
      he = new UserKernel(svmclass);
      ((UserKernel *)he)->setParametres(params->u);
      break;
    default:
      cout << "$ What's this kernel ?!\n\n";
      exit(1);
  }
  
  sreal **sdata = NULL;
  svmclass.train(data, sdata, y, l, c);
  
  real *y_pred = new real [tl];
  for(int i=0; i<tl; i++)
    {
      y_pred[i] = svmclass.use(tdata[i]);
      
      if(y_pred[i]*ty[i]>0)
	correction[i] = 1;
      else
	correction[i] = 0;
    }

  delete [] y_pred;
}

void featurescore(real **data, real *y, real* featurescore, int l, int c, parametres *params)
{
  if(params->regression_mode != "1")
  {
    // cout << "# Checking class...";
    cout.flush();
    for(int i = 0; i < l; i++)
    {
      if( (y[i] != 1) && (y[i] != -1) )
      {
        cout << "\n$ Error: class label must be -1 or +1\n" << endl;
        exit(1);
      }
    }
    //cout << "OK" << endl;
  }

  SVMrfe svmclass;
  
  svmclass.setOption("regression_mode", params->regression_mode);
  svmclass.setOption("C", params->c);
  svmclass.setOption("eps_regression", params->eps_regression);
  svmclass.setOption("unshrink_mode", params->unshrink_mode);
  svmclass.setOption("n_max_unshrink", params->n_max_unshrink);
  
  svmclass.setOption("cache_size_meg", params->cache_size);
  svmclass.setOption("eps_fin", params->eps_fin);
  svmclass.setOption("n_iter_min_to_shrink", params->h);
  svmclass.setOption("eps_shrink", params->eps_shrink);
  svmclass.setOption("eps_bornes", params->eps_bornes);
  svmclass.setOption("sparse_mode", params->sparse_mode);
  
  // creat a kernel
  Kernel *he;
  switch(params->kernel_type)
  {
    case 0:
      he = new DotKernel(svmclass);
       break;
    case 1:
      he = new PolynomialKernel(svmclass);
      ((PolynomialKernel *)he)->setParametres(params->d, params->s, params->r);
      break;
    case 2:
      he = new GaussianKernel(svmclass);
       ((GaussianKernel *)he)->setParametres(params->std);
       break;
    case 3:
      he = new SigmoidKernel(svmclass);
      ((SigmoidKernel *)he)->setParametres(params->s, params->r);
      break;
    case 4:
      he = new UserKernel(svmclass);
      ((UserKernel *)he)->setParametres(params->u);
      break;
    default:
      cout << "$ What's this kernel ?!\n\n";
      exit(1);
  }
  
  sreal **sdata=NULL;
  svmclass.train(data, sdata, y, l, c);
  svmclass.scorecomputation_diff(featurescore);
  //  for(int j=0; j< n_input_dim; j++)
  //  printf("\n * %f\n",feature_score[i]);

}

void featurescore_forward(real **data, real **add_data, real *y, real *featurescore, int l, int c, int add_c, parametres *params)
{
  if(params->regression_mode != "1")
    {
      //cout << "# Checking class...";
      cout.flush();
      for(int i = 0; i < l; i++)
	{
	  if( (y[i] != 1) && (y[i] != -1) )
	    {
	      cout << "\n$ Error: class label must be -1 or +1\n" << endl;
	      exit(1);
	    }
	}
      //cout << "OK" << endl;
    }
  
  SVMrfe svmclass;
  
  svmclass.setOption("regression_mode", params->regression_mode);
  svmclass.setOption("C", params->c);
  svmclass.setOption("eps_regression", params->eps_regression);
  svmclass.setOption("unshrink_mode", params->unshrink_mode);
  svmclass.setOption("n_max_unshrink", params->n_max_unshrink);
  
  svmclass.setOption("cache_size_meg", params->cache_size);
  svmclass.setOption("eps_fin", params->eps_fin);
  svmclass.setOption("n_iter_min_to_shrink", params->h);
  svmclass.setOption("eps_shrink", params->eps_shrink);
  svmclass.setOption("eps_bornes", params->eps_bornes);
  svmclass.setOption("sparse_mode", params->sparse_mode);
  
  // creat a kernel
  Kernel *he;
  switch(params->kernel_type)
    {
    case 0:
      he = new DotKernel(svmclass);
      break;
    case 1:
      he = new PolynomialKernel(svmclass);
      ((PolynomialKernel *)he)->setParametres(params->d, params->s, params->r);
      break;
    case 2:
      he = new GaussianKernel(svmclass);
      ((GaussianKernel *)he)->setParametres(params->std);
      break;
    case 3:
      he = new SigmoidKernel(svmclass);
      ((SigmoidKernel *)he)->setParametres(params->s, params->r);
      break;
    case 4:
      he = new UserKernel(svmclass);
      ((UserKernel *)he)->setParametres(params->u);
      break;
    default:
      cout << "$ What's this kernel ?!\n\n";
      exit(1);
  }
  
  sreal **sdata = NULL;
  svmclass.train(data, sdata, y, l, c);
  svmclass.scorecomputation_forward(featurescore, add_data, add_c);
  //for(int j=0; j< add_c; j++)
  //  printf("\n * %f\n",featurescore[j]);

}
void featureselection(real **data, real *y, int l, int c, int feature_num,  parametres *params, int *order)
{  
  if (c< feature_num)
    {
      printf("\n The feature num must be smaller than feature dimension.\n");
      feature_num = c;
    }

  real **data_ = new real *[l];
  for(int i=0; i<l; i++)
    data_[i] = new real [feature_num];

  real tmp =0;
  for(int i=0; i<l; i++)
    {
      tmp = 0;
      for(int j=0; j<feature_num; j++)
	tmp +=data[i][j]*data[i][j];

      tmp =sqrt(tmp)+1.0e-6;
      for(int j=0; j<feature_num; j++)
	data_[i][j] = data[i][j] /tmp *100;
    }
  
  int *SurvivingList = new int [feature_num];
  int *RankedList    = new int [feature_num];
  for(int i=0; i<feature_num; i++)
    SurvivingList[i]=i;
  
 
  
  real *feature_score = new real [feature_num];
  
  if(data != NULL)
    {
      real **tmpdata = new real *[l];
      for(int i=0; i< l; i++)
	tmpdata[i]= new real[feature_num];
      
      for(int i=feature_num; i>= 1; i--)
	{
	  for(int ii=0; ii< l; ii++)
	    for(int jj=0; jj<i; jj++)
	      tmpdata[ii][jj]=data_[ii][SurvivingList[jj]];
	  
	  for(int ii=0; ii< l; ii++)
	    {
	      tmp = 0;
	      for(int jj=0; jj<i; jj++)
		tmp += tmpdata[ii][jj]*tmpdata[ii][jj];

	      tmp = sqrt(tmp) +1.0e-6;
	      for(int jj=0; jj<i; jj++)
		tmpdata[ii][jj] = tmpdata[ii][jj]/tmp*100;
	    }

	  featurescore(tmpdata, y, feature_score, l, i, params);
	  
	  //for(int j=0; j< i; j++)
	  //  printf("\n * %f\n",feature_score[j]);
	  
	  real min_score = feature_score[i-1];
	  RankedList[i-1] = SurvivingList[i-1];
	  
	  for(int j=0; j< i-1; j++)
	    {
	      if(feature_score[j]<min_score)
		{
		  //printf("\n feature_score[%d]=%f,minscore=%f",j,feature_score[j],min_score);
                  min_score = feature_score[j];
		  RankedList[i-1]=SurvivingList[j];
		}
	    }
	  
	  int data_index=0;
	  for(int jj=0; jj<i; jj++)
	    {
	      if(SurvivingList[jj]!=RankedList[i-1])
		SurvivingList[data_index++]=SurvivingList[jj];
	    }
	}
      RankedList[0]=SurvivingList[0];
      
      for(int i=0; i<l; i++)
	delete [] tmpdata[i];
      delete tmpdata;
    }
  
  for(int i=0; i<feature_num; i++)
    order[i] = RankedList[i];
  
  for(int i=0; i<l; i++)
    delete [] data_[i];
  delete data_;
  delete [] feature_score;
  delete [] SurvivingList;
  delete [] RankedList;
}


void featureselection_forward(real **data, real *y, int l, int c, int feature_num,  parametres *params, int start_point, int *order)
{  
  if (c< feature_num)
    {
      printf("\n The feature num must be smaller than feature dimension.\n");
      feature_num = c;
    }
  real **data_ = new real *[l];
  for(int i=0; i<l; i++)
    data_[i] = new real [feature_num];

  real tmp =0;
  for(int i=0; i<l; i++)
    {
      tmp = 0;
      for(int j=0; j<feature_num; j++)
	tmp +=data[i][j]*data[i][j];

      tmp =sqrt(tmp)+1.0e-6;
      for(int j=0; j<feature_num; j++)
	data_[i][j] = data[i][j] /tmp *100;
    }
  int *SurvivingList = new int [feature_num-start_point];
  int *RankedList    = new int [feature_num-start_point];
  for(int i=0; i<feature_num-start_point; i++)
    SurvivingList[i] = RankedList[i] = i+start_point;
  
  real **add_data = new real *[l];
  for(int i=0; i< l; i++)
    add_data[i]= new real[feature_num - start_point];
  
  real *feature_score = new real [feature_num-start_point];
  
  if(data != NULL)
    {
      real **tmpdata = new real *[l];
      for(int i=0; i< l; i++)
	tmpdata[i]= new real[feature_num];
      for(int ii=0; ii< l; ii++)
	for(int jj=0; jj<feature_num; jj++)
	  tmpdata[ii][jj] = data[ii][jj];
      
      for(int i= 0; i<feature_num-start_point-1; i++)
	{
	  for(int ii=0; ii< l; ii++)
	    {
	      for(int jj=0; jj<i; jj++)
		tmpdata[ii][jj+start_point-1]=data_[ii][RankedList[jj]];
	      
	      for(int jj=0; jj<feature_num-start_point-i; jj++)
		add_data[ii][jj] = data_[ii][SurvivingList[jj]];
	    }
	  
	   for(int ii=0; ii< l; ii++)
	    {
	      tmp = 0;
	      for(int jj=0; jj<i+start_point; jj++)
		tmp += tmpdata[ii][jj]*tmpdata[ii][jj];

	      tmp = sqrt(tmp)+1.0e-6;
	      for(int jj=0; jj<i+start_point; jj++)
		tmpdata[ii][jj] = tmpdata[ii][jj]/tmp*100;
	    }
	      
	  featurescore_forward(tmpdata, add_data, y, feature_score, l, c, feature_num-start_point-i, params);
	  
	  //for(int j=0; j< i; j++)
	  //  printf("\n * %f\n",feature_score[j]);
	  
	  real min_score = feature_score[0];
	  RankedList[i] = SurvivingList[0];
	  
	  for(int j=0; j< feature_num-start_point-i; j++)
	    {
	      if(feature_score[j]<min_score)
		{
		  //printf("\n feature_score[%d]=%f,minscore=%f",j,feature_score[j],min_score);
                  min_score = feature_score[j];
		  RankedList[i]=SurvivingList[j];
		}
	    }
	  
	  int data_index=0;
	  for(int jj=0; jj<feature_num-start_point-i; jj++)
	    {
	      if(SurvivingList[jj]!=RankedList[i])
		SurvivingList[data_index++]=SurvivingList[jj];
	    }
	}
      
      RankedList[feature_num-start_point-1]=SurvivingList[0];
      
      for(int i=0; i<l; i++)
	delete [] tmpdata[i];
      delete tmpdata;
    }

  for(int i=0; i<l; i++)
    delete [] add_data[i];
  delete add_data;


  for(int i=0; i<feature_num-start_point; i++)
    order[i] = RankedList[i];
  
  delete [] feature_score;
  delete [] SurvivingList;
  delete [] RankedList;
}



int main(int argc, char *argv[])
{

  static struct option long_options[] =
  {
    {"usage",       no_argument,            0, 'u'},
    {"help",        no_argument,            0, 'h'},
    {"version",     no_argument,            0, 'v'},
  };

  int b, option_index = 0;

  if(argc<5){
    while ( (b = getopt_long (argc, argv, "uhv",
                long_options,&option_index)) != -1){
      switch (b)
      {
        case 'v':
          version();
          return 0;

        case 'u':
          usage();
          return 0;

        case 'h':
          usage();
          return 0;

        case '?':
          /* getopt_long already printed an error message. */
          break;

        default:
          return 1;
      }
    }
    usage();
    exit(1);
  }
  
 
  char *data_directory= argv[1];
  argc--;
  argv++;
  int  num_subject    = atoi(argv[1]);
  argc--;
  argv++;
  char *train_name_base = argv[1];
  argc--;
  argv++;
  char *test_name_base = argv[1];
  argc--;
  argv++;

  int  num_feature    = 150;
  string c            = "10";
  float std           = 100;
  int f_start_point   = 0;
  int f_end_point     = 0;

  while(argc>1){
    if(strcmp(argv[1],"-num_feature")==0){
	    argc--;
	    argv++;
	    num_feature = atoi(argv[1]);
	    argc--;
	    argv++;
	  }
    else if(strcmp(argv[1],"-c")==0){
	    argc--;
	    argv++;
	    c = argv[1];
	    argc--;
	    argv++;
	  }
    else if(strcmp(argv[1],"-std")==0){
	    argc--;
	    argv++;
	    std = atof(argv[1]);
	    argc--;
	    argv++;
	  }
    else if(strcmp(argv[1],"-start_point")==0){
	    argc--;
	    argv++;
	    f_start_point = atoi(argv[1]);
	    argc--;
	    argv++;
	  }
    else if(strcmp(argv[1],"-end_point")==0){
	    argc--;
	    argv++;
	    f_end_point = atoi(argv[1]);
	    argc--;
	    argv++;
	  }
    else if((strcmp(argv[1],"-v")==0)||(strcmp(argv[1],"-version")==0)){
	    argc--;
	    argv++;
      version();
      exit(0);
	  }
    else{
	    usage();
      exit(1);
    }
  }
      
  printf("\n The input options are below:"); 
  printf("\n ===============================================\n");
  printf("\n data_directory:          %s",data_directory);
  printf("\n num_subject :            %d",num_subject);
  printf("\n training_file_name:      %s",train_name_base);
  printf("\n testing_file_name:       %s",test_name_base);
  printf("\n num_feature:             %d\n",num_feature);
  printf("\n std:                     %f",std);
  printf("\n start_point:             %d",f_start_point);
  printf("\n end_point:               %d",f_end_point);
  printf("\n ==============================================\n");
  
  parametres params;

  params.regression_mode = "0";
  params.multi_mode = false;
  params.first_class = -1;
  params.last_class = -1;
  params.c = c;
  params.eps_regression = "0.5";
  params.unshrink_mode = "0";
  params.n_max_unshrink = "2";
  params.kernel_type = 2;
  params.d = 2;
  params.std = std;
  params.s = 1;
  params.r = 1;
  params.u = "10";
  params.cache_size = "50";
  params.eps_fin = "0.01";
  params.h = "100";
#ifdef USEDOUBLE
  params.eps_shrink = "1E-9";
  params.eps_bornes = "1E-12";
#else
  params.eps_shrink = "1E-4";
  params.eps_bornes = "1E-4";
#endif
  params.sparse_mode = "0";
  params.bin_mode = false;    
  params.load = -1;
  
  // read data
  printf("\n reading data ...");
  FILE *fp;
  char filename[1024];
  sprintf(filename, "%s/%s_%d.txt",data_directory, train_name_base, 0);
  if((fp=fopen(filename,"r"))==NULL)
    {
      printf("\n cannot open training file %s",filename);
      exit(0);
    }
  int num_train_sample = 0;
  int num_train_feature = 0;
  fscanf(fp,"%d %d",&num_train_sample, &num_train_feature);
  fclose(fp);
  num_train_feature -= 1;
  float ***training = f3tensor(num_subject, num_train_sample, num_feature);
  float **train_y   = matrix(num_subject, num_train_sample);

  sprintf(filename, "%s/%s_%d.txt",data_directory, test_name_base,0);
  if((fp=fopen(filename,"r"))==NULL)
    {
      printf("\n cannot open testing file %s",filename);
      exit(0);
    }
  int num_test_sample = 0;
  int num_test_feature = 0;
  fscanf(fp,"%d %d",&num_test_sample, &num_test_feature);
  fclose(fp);
  num_test_feature -= 1;
  float ***testing = f3tensor(num_subject, num_test_sample, num_feature);
  float **test_y   = matrix(num_subject, num_test_sample);
  
  if(num_feature>num_train_feature)
    {
      printf("\n");
      for(int i=0; i<num_subject; i++)
	{
	  //printf(".");
	  sprintf(filename, "%s/%s_%d.txt",data_directory, train_name_base, i);
	  if((fp=fopen(filename,"r"))==NULL)
	    {
	      printf("\n cannot open training file %s",filename);
	      exit(0);
	    }
	  fscanf(fp,"%d %d",&num_train_sample, &num_train_feature);
	  num_train_feature -= 1;
	  for(int j=0; j<num_train_sample; j++)
	    {
	      for(int k=0; k<num_train_feature; k++)
		fscanf(fp,"%f", &training[i][j][k]);
	      fscanf(fp,"%f", &train_y[i][j]);
	      
	      for(int k=num_train_feature; k<num_feature;  k++)
		training[i][j][k] = 0;
	    }
	  fclose(fp);
	  
	  sprintf(filename, "%s/%s_%d.txt",data_directory, test_name_base, i);
	  if((fp=fopen(filename,"r"))==NULL)
	    {
	      printf("\n cannot open testing file %s",filename);
	      exit(0);
	    }
	  fscanf(fp,"%d %d",&num_test_sample, &num_test_feature);
	  num_test_feature -= 1;
	  for(int j=0; j<num_test_sample; j++)
	    {
	      for(int k=0; k<num_test_feature; k++)
		fscanf(fp,"%f", &testing[i][j][k]);
	      fscanf(fp,"%f", &test_y[i][j]);

	      for(int k=num_test_feature; k<num_feature;  k++)
		testing[i][j][k] = 0;
	    }
	  fclose(fp);
	}
      printf("\n");
    }
  else
    {
      printf("\n");
      float tmp;
      for(int i=0; i<num_subject; i++)
	{
	  //printf(".");
	  sprintf(filename, "%s/%s_%d.txt",data_directory, train_name_base,i);
	  if((fp=fopen(filename,"r"))==NULL)
	    {
	      printf("\n cannot open training file %s",filename);
	      exit(0);
	    }
	  fscanf(fp,"%d %d",&num_train_sample, &num_train_feature);
	  num_train_feature -= 1;
	  for(int j=0; j<num_train_sample; j++)
	    {
	      for(int k=0; k<num_train_feature; k++)
		{
		  if(k<num_feature)
		    fscanf(fp,"%f", &training[i][j][k]);
		  else
		    fscanf(fp,"%f",&tmp);
		}
	      fscanf(fp,"%f", &train_y[i][j]);
	    }
	  fclose(fp);
	  
	  sprintf(filename, "%s/%s_%d.txt",data_directory,test_name_base,i);
	  if((fp=fopen(filename,"r"))==NULL)
	    {
	      printf("\n cannot open testing file %s",filename);
	      exit(0);
	    }
	  fscanf(fp,"%d %d",&num_test_sample, &num_test_feature);
	  num_test_feature -= 1;
	  for(int j=0; j<num_test_sample; j++)
	    {
	      for(int k=0; k<num_test_feature; k++)
		{
		  if(k<num_feature)
		    fscanf(fp,"%f", &testing[i][j][k]);
		  else
		    fscanf(fp,"%f", &tmp);
		}
	      fscanf(fp,"%f", &test_y[i][j]);
	    }
	  fclose(fp);
	}
      printf("\n");
    }
  
  // ranking based search
  printf("\n ranking feature selction ...");
  int **rank_correction = imatrix(num_subject*num_test_sample, num_feature);
  
  double **data = dmatrix(num_train_sample, num_feature);
  double **tdata = dmatrix(num_test_sample, num_feature);
  double *y      = dvector(num_train_sample);
  double *ty     = dvector(num_test_sample);
  int *correction = ivector(num_test_sample);

  for(int i=0; i<num_subject; i++)
    {
      for(int k=0; k<num_train_sample; k++)
	{
	  data[k][0] = training[i][k][0];
	  y[k] = train_y[i][k];
	}
      
      for(int k=0; k<num_test_sample; k++)
	{
	  tdata[k][0] = testing[i][k][0];
	  ty[k] = test_y[i][k];
	}
      
      svmmodel_train_test(data, tdata, y, ty,  num_train_sample, num_test_sample, 1, &params, correction); 

      for(int k=0; k<num_test_sample; k++)
	rank_correction[i*num_test_sample+k][0] = correction[k];


      for(int j=1;j<num_feature;j++)
	{
	  for(int k=0; k<num_train_sample; k++)
	    {
	      for(int l=0; l<=j; l++)
		data[k][l] = training[i][k][l];
	      float tmp = 0;
	      for(int l=0; l<=j; l++)
		tmp += data[k][l]*data[k][l];
	      tmp = SCALE/(sqrt(tmp)+1.0e-6);
	      for(int l=0; l<=j; l++)
		data[k][l] *= tmp;
	      
	      y[k] = train_y[i][k];
	    }
	  
	  for(int k=0; k<num_test_sample; k++)
	    {
	      for(int l=0; l<=j; l++)
		tdata[k][l] = testing[i][k][l];
	      float tmp = 0;
	      for(int l=0; l<=j; l++)
		tmp += tdata[k][l] * tdata[k][l];
	      tmp = SCALE/(sqrt(tmp)+1.0e-6);
	      for(int l=0; l<=j; l++)
		tdata[k][l] *= tmp;

	      ty[k] = test_y[i][k];
	    }
      
	  svmmodel_train_test(data, tdata, y, ty,  num_train_sample, num_test_sample, j+1, &params, correction); 

	  /*
	  FILE *fp;
	  char fname[1024];
	  sprintf(fname,"%s_%d_%d.txt","train",i,j);
	  fp=fopen(fname,"w");
	  for(int k=0; k<num_train_sample; k++)
	    {
	      for(int l=0; l<j+1; l++)
		fprintf(fp,"%f\t",data[k][l]);
	      fprintf(fp,"\n");
	    }
	  fclose(fp);
	  */    
	  for(int k=0; k<num_test_sample; k++)
	    rank_correction[i*num_test_sample+k][j] = correction[k];
	}
    }

  int *rank_rate = ivector(num_feature);
  for(int i=0; i<num_feature; i++)
    {
      rank_rate[i] = 0;
      for(int j=0; j<num_subject*num_test_sample; j++)
	rank_rate[i] += rank_correction[j][i];
    }

  sprintf(filename, "ranking_rate_%d.bin", (int)floor(std));
  if((fp=fopen(filename,"w"))==NULL)
    {
      printf("\n cannot open file %s for writing out classification rates.",filename);
      exit(0);
    }
  for(int j=0; j<num_feature; j++)
    fprintf(fp, "%f\t",(float)rank_rate[j]/(float)(num_subject*num_test_sample));
  fclose(fp);

  int peak_rate = rank_rate[0];
  int peak_loca = 0;
  for(int i=1; i<num_feature; i++)
    {
      if(peak_rate<=rank_rate[i])
	{
	  peak_rate = rank_rate[i];
	  peak_loca = i;
	}
    }


  // subset feature selection   
  printf("\n subset feature selction ...");
  int start_point;
  if(peak_loca <=0 )
    start_point = 1;
  else
    start_point = peak_loca+1;
  
  printf("\n start_point = %d",start_point);
  int **backward_order = imatrix(num_subject, start_point);
  int **forward_order = NULL;
  if(start_point<num_feature)
    forward_order  = imatrix(num_subject, num_feature - start_point);
    
  int **order          = imatrix(num_subject, num_feature); 
  for(int i=0; i<num_subject; i++)
    {
      for(int j=0; j<start_point; j++)
	backward_order[i][j] = j;
     
      if(start_point<num_feature)
         for(int j=0; j<num_feature-start_point; j++)
	   forward_order[i][j] = j+start_point;
    
      for(int k=0; k<num_train_sample; k++)
	{
	  for(int l=0; l<num_feature; l++)
	    data[k][l] = training[i][k][l];

	  float tmp = 0;
	  for(int l=0; l<num_feature; l++)
	    tmp += data[k][l]*data[k][l];
	  tmp = SCALE/(sqrt(tmp)+1.0e-6);
	  for(int l=0; l<num_feature; l++)
	    data[k][l] *= tmp;
	  
	  y[k] = train_y[i][k];
	}
      if(start_point >= 1)
	featureselection(data, y, num_train_sample, num_feature, start_point, &params, backward_order[i]);
      
      if(start_point<num_feature)
	featureselection_forward(data, y, num_train_sample, num_feature, num_feature,  &params, start_point, forward_order[i]);
      else
	start_point = num_feature;

      for(int j=0; j<start_point; j++)
	order[i][j] = backward_order[i][j];
      if(start_point<num_feature)
         for(int j=0; j<num_feature-start_point; j++)
	   order[i][start_point+j] = forward_order[i][j];
	
      //printf("\n %d ;;;",i);
    }

  //classification result based on subset feature selection;
  printf("\n model selction ...");
  int **feat_correction = imatrix(num_subject*num_test_sample, num_feature);
  
  for(int i=0; i<num_subject; i++)
    {
      for(int k=0; k<num_train_sample; k++)
	{
	  data[k][0] = training[i][k][order[i][0]];
	  y[k] = train_y[i][k];
	}
      
      for(int k=0; k<num_test_sample; k++)
	{
	  tdata[k][0] = testing[i][k][order[i][0]];
	  ty[k] = test_y[i][k];
	}
      
      svmmodel_train_test(data, tdata, y, ty,  num_train_sample, num_test_sample, 1, &params, correction); 
      
      for(int k=0; k<num_test_sample; k++)
	feat_correction[i*num_test_sample+k][0] = correction[k];


      for(int j=1;j<num_feature;j++)
	{
	  for(int k=0; k<num_train_sample; k++)
	    {
	      for(int l=0; l<=j; l++)
		data[k][l] = training[i][k][order[i][l]];
	      float tmp = 0;
	      for(int l=0; l<=j; l++)
		tmp += data[k][l]*data[k][l];
	      tmp = SCALE/(sqrt(tmp)+1.0e-6);
	      for(int l=0; l<=j; l++)
		data[k][l] *= tmp;

	      y[k] = train_y[i][k];
	    }
	  
	  for(int k=0; k<num_test_sample; k++)
	    {
	      for(int l=0; l<=j; l++)
		tdata[k][l] = testing[i][k][order[i][l]];
	      float tmp = 0;
	      for(int l=0; l<=j; l++)
		tmp += tdata[k][l] * tdata[k][l];
	      tmp = SCALE/(sqrt(tmp)+1.0e-6);
	      for(int l=0; l<=j; l++)
		tdata[k][l] *= tmp;

	      ty[k] = test_y[i][k];
	    }
	  
	  svmmodel_train_test(data, tdata, y, ty,  num_train_sample, num_test_sample, j+1, &params, correction); 
	  
	  for(int k=0; k<num_test_sample; k++)
	    feat_correction[i*num_test_sample+k][j] = correction[k];
	}
    }

  
  int *feat_rate = ivector(num_feature);
  for(int i=0; i<num_feature; i++)
    {
      feat_rate[i] = 0;
      for(int j=0; j<num_subject*num_test_sample; j++)
	feat_rate[i] += feat_correction[j][i];
    }
  
  int best_rate = feat_rate[num_feature-1];
  int best_loca = num_feature-1;
  for(int i=num_feature-1; i>=0; i--)
    {
      if(best_rate<feat_rate[i])
	{
	  best_rate = feat_rate[i];
	  best_loca = i;
	}
    }
    

  printf("\n the best classification rate is %f with %d features.",(float)best_rate/(float)(num_subject*num_test_sample), best_loca+1);

  for(int i=0; i<num_subject; i++)
    {
      sprintf(filename, "order_%d.bin", i);
      if((fp=fopen(filename,"w"))==NULL)
	{
	  printf("\n cannot open file %s for writing out order file.",filename);
	  exit(0);
	}
      for(int j=0; j<num_feature; j++)
	fprintf(fp, "%d\t",order[i][j]);
      fclose(fp);
    }
  
  sprintf(filename, "classification_rate_%d.bin", (int)floor(std));
  if((fp=fopen(filename,"w"))==NULL)
    {
      printf("\n cannot open file %s for writing out classification rates.",filename);
      exit(0);
    }
  
  fprintf(fp,"\n***the best classification rate is %f with %d features.\n\n",(float)best_rate/(float)(num_subject*num_test_sample), best_loca+1);
  fprintf(fp,"\n***the classification rates with respect to the number of features used in classification are:\n");
  for(int j=0; j<num_feature; j++)
    fprintf(fp, "%f\t",(float)feat_rate[j]/(float)(num_subject*num_test_sample));
  

  float *score = new float [num_subject];
  if((f_start_point == 0) && (f_end_point == 0))
    {
      fprintf(fp,"\n\n***the number of features used in final classification is %d\n\n",best_loca+1);
      
      for(int i=0; i<num_subject; i++)
	{
	  for(int k=0; k<num_train_sample; k++)
	    {
	      for(int l=0; l<=best_loca; l++)
		data[k][l] = training[i][k][order[i][l]];
	      float tmp = 0;
	      for(int l=0; l<=best_loca; l++)
		tmp += data[k][l]*data[k][l];
	      tmp = SCALE/(sqrt(tmp)+1.0e-6);
	      for(int l=0; l<=best_loca; l++)
		data[k][l] *= tmp;
	      
	      y[k] = train_y[i][k];
	    }
	  
	  for(int k=0; k<num_test_sample; k++)
	    {
	      for(int l=0; l<=best_loca; l++)
		tdata[k][l] = testing[i][k][order[i][l]];
	      float tmp = 0;
	      for(int l=0; l<=best_loca; l++)
		tmp += tdata[k][l] * tdata[k][l];
	      tmp = SCALE/(sqrt(tmp)+1.0e-6);
	      for(int l=0; l<=best_loca; l++)
		tdata[k][l] *= tmp;
	      
	      ty[k] = test_y[i][k];
	    }
	  
	  sprintf(filename, "svm_model_%d_0.bin", i);
	  
	  double tmp;
	  // svm_model(data, y, num_train_sample, best_loca+1, &params, filename);
	  svm_model1(data, y, num_train_sample, best_loca+1, tdata, &tmp, 1, &params, filename);
	  score[i] = tmp;
	}
    }
  else
    {
      if(f_start_point == 0)
	f_start_point = best_loca+1;
      if(f_end_point == 0)
	f_end_point = best_loca+1;
      
      if(f_end_point < f_start_point)
	f_end_point = f_start_point;
      
      fprintf(fp,"\n***the features used in final classification range between %d and %d\n\n",f_start_point, f_end_point);
      
      for(int i=0; i<num_subject; i++)
	{
	  score[i] = 0;
	  for(int n_feat=f_start_point; n_feat<=f_end_point; n_feat++)
	    {
	      for(int k=0; k<num_train_sample; k++)
		{
		  for(int l=0; l<n_feat; l++)
		    data[k][l] = training[i][k][order[i][l]];
		  float tmp = 0;
		  for(int l=0; l<n_feat; l++)
		    tmp += data[k][l]*data[k][l];
		  tmp = SCALE/(sqrt(tmp)+1.0e-6);
		  for(int l=0; l<n_feat; l++)
		    data[k][l] *= tmp;
		  
		  y[k] = train_y[i][k];
		}
	      
	      for(int k=0; k<num_test_sample; k++)
		{
		  for(int l=0; l<n_feat; l++)
		    tdata[k][l] = testing[i][k][order[i][l]];
		  float tmp = 0;
		  for(int l=0; l<n_feat; l++)
		    tmp += tdata[k][l] * tdata[k][l];
		  tmp = SCALE/(sqrt(tmp)+1.0e-6);
		  for(int l=0; l<n_feat; l++)
		    tdata[k][l] *= tmp;
		  
		  ty[k] = test_y[i][k];
		}
	      
	      sprintf(filename, "svm_model_%d_%d.bin", i,n_feat-f_start_point);
	      
	      double tmp;
	      // svm_model(data, y, num_train_sample, best_loca+1, &params, filename);
	      svm_model1(data, y, num_train_sample, n_feat, tdata, &tmp, 1, &params, filename);
	      score[i] += tmp;
	    }
	  score[i] /= (f_end_point-f_start_point+1);
	}
    }
  
  float *fp1 = new float [num_subject];
  float *tp1 = new float [num_subject];
  float *t  = new float [num_subject];
  for(int i=0; i<num_subject; i++)
    t[i] = test_y[i][0];
  
  roc(t, score, tp1, fp1, num_subject);
  
  fprintf(fp,"\n***the SVM scores corresponding to the subjects are:\n");
  for(int i=0; i<num_subject; i++)
    fprintf(fp,"%f\t",score[i]);
  
  fprintf(fp,"\n\n***the data points for ploting ROC curve:\n");
  fprintf(fp,"\n fp: 0.0\t");
  for(int i=0; i<num_subject; i++)
    fprintf(fp, "%f\t",fp1[i]);
  fprintf(fp, "%f\n",1.0);
  
  fprintf(fp,"\n tp: 0.0\t");
  for(int i=0; i<num_subject; i++)
    fprintf(fp, "%f\t",tp1[i]);
  fprintf(fp, "%f\n",1.0);
  
  fclose(fp);
  
  delete [] score;
  delete [] t;
  delete [] fp1;
  delete [] tp1;
  free_f3tensor(training, num_subject, num_train_sample, num_feature);
  free_matrix(train_y,num_subject, num_train_sample);
  free_f3tensor(testing, num_subject, num_test_sample, num_feature);
  free_matrix(test_y, num_subject, num_test_sample);

  free_imatrix(rank_correction, num_subject*num_test_sample, num_feature);
  
  free_dmatrix(data, num_train_sample, num_feature);
  free_dmatrix(tdata, num_test_sample, num_feature);
  free_dvector(y, num_train_sample);
  free_dvector(ty, num_test_sample);
  free_ivector(correction, num_test_sample);

  return 0;
}
