/*******************************************************

This program calculates sum of spatial discrinination maps computed in leave-out-out experiments.

last modified : Oct 6, 2008

yong.fan@ieee.org
kanterae@uphs.upenn.edu

*******************************************************/

#include <stdio.h>
#include <stdlib.h>
#include "nrutil_compare.h"
#include <getopt.h>

#define EXEC_NAME "COMPARE_float_sum"

#ifndef RELEASE_ID
#define RELEASE_ID "0.2.1"
#endif

#define SVNFILEVER "$Id: COMPARE_float_sum.cc 17 2008-10-08 17:27:42Z kanterae $"

#ifndef SVN_REV
#define SVN_REV "NO_SVN" 
#endif


void version(){
  printf("\n\n%s Version information\n",EXEC_NAME);
  printf("  Release          : %s\n", RELEASE_ID);
  printf("  Svn Revision     : %s\n", SVN_REV);
  printf("  Svn File Version : %s\n", SVNFILEVER);
}

void usage(){
  printf("\n This program calculates sum of spatial discrinination maps computed in leave-out-out experiments.\n");

  printf("\n Usage: %s (output_file) (feature_file_base1) (feature_file_base2) (file_start_num) (file_end_num) (xdim) (ydim) (zdim)",EXEC_NAME);
  printf("\n Usage: %s [-u/h/v]\n",EXEC_NAME);
  
  printf("\n Required arguments:");
  printf("\n   output_file          : output image filename");
  printf("\n   feature_file_base1   : prefix of feature files to look for (ex. COMPARE_spatial_map)");
  printf("\n   feature_file_base2   : suffix of feature files to look for (ex. diff_0.img)");
  printf("\n   file_start_num       : index of first file to look for (starting at 0)");
  printf("\n   file_end_num         : index of last file");
  printf("\n        --> The program will look for a range of N files, N in {file_start_num..file_end_num},");
  printf("\n        --> following the pattern \"feature_file_base1_N_feature_file_base2\" (ex. COMPARE_spatial_map_0_diff_0.img");
  printf("\n   xdim                 : x dimention of map");
  printf("\n   ydim                 : y dimention of map");
  printf("\n   zdim                 : z dimention of map\n");
  
  printf("\n Optional arguments:\n");
  printf("\n   [-u      --usage]     Display this message");
  printf("\n   [-h       --help]     Display this message");
  printf("\n   [-v    --version]     Display version information\n");
  
  printf("\n More information:");
  printf("\n   https://www.rad.upenn.edu/sbia/software/");
  printf("\n   sbia-software@uphs.upenn.edu");
  printf("\n\n");
}

int main(int argc, char *argv[])
{

  static struct option long_options[] =
  {
    {"usage",       no_argument,            0, 'u'},
    {"help",        no_argument,            0, 'h'},
    {"version",     no_argument,            0, 'v'},
  };

  int c, option_index = 0;

  if(argc!=9){
    while ( (c = getopt_long (argc, argv, "uhv",
                long_options,&option_index)) != -1){
      switch (c)
      {
        case 'v':
          version();
          return 0;

        case 'u':
          usage();
          return 0;

        case 'h':
          usage();
          return 0;

        case '?':
          /* getopt_long already printed an error message. */
          break;

        default:
          return 1;
      }
    }
    usage();
    exit(1);
  }
  
  char *output_file         = argv[1];
  char *feature_file_base1  = argv[2];
  char *feature_file_base2  = argv[3];
  int  file_start_num     = atoi(argv[4]);
  int  file_end_num       = atoi(argv[5]);
  int  xdim               = atoi(argv[6]);
  int  ydim               = atoi(argv[7]);
  int  zdim               = atoi(argv[8]);
  
  printf("\n The input options are below:");
  printf("\n output_file:         %s",output_file);
  printf("\n feature_file_base1:  %s",feature_file_base1);
  printf("\n feature_file_base2:  %s",feature_file_base2);
  printf("\n file_start_num:      %d",file_start_num);
  printf("\n file_end_num:        %d",file_end_num);
  printf("\n xsize:               %d",xdim);
  printf("\n ysize:               %d",ydim);
  printf("\n zsize:               %d",zdim);
  printf("\n ======================================\n");

  float ***im = f3tensor(xdim,ydim,zdim);
  for(int k=0; k<zdim; k++)
    for(int j=0; j<ydim; j++)
      for(int i=0; i<xdim; i++)
	im[i][j][k] = 0;
  
  char file_name[1024];
  FILE *fp; 
  
  float tmp_float;
  for(int i=file_start_num; i<file_end_num; i++)
    {
      sprintf(file_name,"%s_%d_%s",feature_file_base1,i,feature_file_base2);
        
      if((fp=fopen(file_name,"r"))==NULL)
	{ 
	  printf("\n Cannot open file %s\n",file_name);
	  exit(1);
	}
      for(int kk=0; kk<zdim; kk++)
	for(int jj=0; jj<ydim; jj++)
	  for(int ii=0; ii<xdim; ii++)
	    {
	      fread(&tmp_float,sizeof(float),1,fp);
	      im[ii][jj][kk]+=tmp_float;
	    }
      fclose(fp);
    }

  float num=file_end_num-file_start_num +1;
   for(int kk=0; kk<zdim; kk++)
     for(int jj=0; jj<ydim; jj++)
       for(int ii=0; ii<xdim; ii++)
	 im[ii][jj][kk] /= num;
	 

  if((fp = fopen(output_file, "w")) == NULL)
    {
      printf("\n Could not open file %s",output_file);
      exit(1);
    }
  
  for(int k=0; k<zdim; k++)
    for(int j=0; j<ydim; j++)
      for(int i=0; i<xdim; i++)
	fwrite(&im[i][j][k],sizeof(float),1,fp);
  fclose(fp);
  
  free_f3tensor(im,xdim,ydim,zdim);
  return 0;
}
