/*******************************************************

This program projects group difference values to brain regions.

last modified : Oct 6, 2008

yong.fan@ieee.org
kanterae@uphs.upenn.edu

*******************************************************/

#include <stdio.h>
#include <stdlib.h>
#include "nrutil_compare.h"
#include <getopt.h>

#define EXEC_NAME "COMPARE_map_svm_diff_to_region"

#ifndef RELEASE_ID
#define RELEASE_ID "0.0_super_alpha"
#endif

#define SVNFILEVER "$Id: COMPARE_map_svm_diff_to_region.cc 35 2010-09-22 14:41:15Z batmangn@UPHS.PENNHEALTH.PRV $"

#ifndef SVN_REV
#define SVN_REV "NO_SVN" 
#endif


void version(){
  printf("\n\n%s Version information\n",EXEC_NAME);
  printf("  Release          : %s\n", RELEASE_ID);
  printf("  Svn Revision     : %s\n", SVN_REV);
  printf("  Svn File Version : %s\n", SVNFILEVER);
}

void usage(){
  printf("\n This program projects group difference values to brain regions.\n");

  printf("\n Usage: %s (output_file_base) (feature_file) (order_file) (weighting_file) (feature_start_id) (feature_end_id) (region_file_base) (xdim) (ydim) (zdim)",EXEC_NAME);
  printf("\n Usage: %s [-u/h/v]\n",EXEC_NAME);
  
  printf("\n Required arguments:");
  printf("\n   output_file_base  : name of the outputs (each kind of feature map has one regional map)");
  printf("\n   feature_file      : name of file storing the information of feature extraction");
  printf("\n   order_file        : name of file storing the information of feature selection");
  printf("\n   weighting_file    : name of file storing the group difference vector");
  printf("\n   feature_start_id  : start of number of features used for classification (starting at 0)");
  printf("\n   feature_end_id    : end of number of features used for classification");
  printf("\n   region_file_base  : prefix of files storing the brain regions");
  printf("\n   xdim              : x dimention of feature map");
  printf("\n   ydim              : y dimention of feature map");
  printf("\n   zdim              : z dimention of feature map\n");
  
  printf("\n Optional arguments:\n");
  printf("\n   [-u      --usage]     Display this message");
  printf("\n   [-h       --help]     Display this message");
  printf("\n   [-v    --version]     Display version information\n");
  
  printf("\n More information:");
  printf("\n   https://www.rad.upenn.edu/sbia/software/");
  printf("\n   sbia-software@uphs.upenn.edu");
  printf("\n\n");
}

int main(int argc, char *argv[])
{

  static struct option long_options[] =
  {
    {"usage",       no_argument,            0, 'u'},
    {"help",        no_argument,            0, 'h'},
    {"version",     no_argument,            0, 'v'},
  };

  int c, option_index = 0;

  if(argc!=11){
    while ( (c = getopt_long (argc, argv, "uhv",
                long_options,&option_index)) != -1){
      switch (c)
      {
        case 'v':
          version();
          return 0;

        case 'u':
          usage();
          return 0;

        case 'h':
          usage();
          return 0;

        case '?':
          /* getopt_long already printed an error message. */
          break;

        default:
          return 1;
      }
    }
    usage();
    exit(1);
  }
  
  char *output_file_base  = argv[1];
  char *feature_file      = argv[2];
  char *order_file        = argv[3];
  char *weighting_file    = argv[4];
  int  feature_start_id   = atoi(argv[5]);
  int  feature_end_id     = atoi(argv[6]);
  char *region_file_base  = argv[7];
  int  xdim               = atoi(argv[8]);
  int  ydim               = atoi(argv[9]);
  int  zdim               = atoi(argv[10]);
  
  printf("\n The input options are below:");
  printf("\n output_file_base:    %s",output_file_base);
  printf("\n feature_file:        %s",feature_file);
  printf("\n order_file:          %s",order_file);
  printf("\n weighting_file:      %s",weighting_file);
  printf("\n feature_start_id:    %d",feature_start_id);
  printf("\n feature_end_id:      %d",feature_end_id);
  printf("\n region_file_base:    %s",region_file_base);
  printf("\n xsize:               %d",xdim);
  printf("\n ysize:               %d",ydim);
  printf("\n zsize:               %d",zdim);
  printf("\n ======================================\n");

  int **features = imatrix(1800, 2);
  float *feature_score = vector(1800);
  float *weight = vector(1800);

  int num_feature = 0;
  char fname[1024];
  FILE *fp;
  
  //read weighting file
  sprintf(fname, "%s", weighting_file);
  if((fp=fopen(fname,"r"))==NULL)
    {
      printf("\n Could not open file %s.\n",fname);
      exit(1);
    }
  
  while(fscanf(fp,"%f",weight+num_feature)>0)
    num_feature += 1;
  fclose(fp);
  
  
  int *order     = ivector(num_feature);
  
  //read feature file
  sprintf(fname, "%s",feature_file);
  if((fp=fopen(fname,"r"))==NULL)
    {
      printf("\n Could not open file %s.\n",fname);
      exit(1);
    }
  for(int i=0; i<6; i++)
    fscanf(fp,"%s", fname);
  int tmp_int;
  float tmp_float;
  for(int i=0; i<1800; i++)
    {
      fscanf(fp,"%d",&tmp_int);
      fscanf(fp,"%f",feature_score+i);
      for(int j=0; j<2; j++)
	fscanf(fp,"%d",&features[i][j]);
      fscanf(fp,"%f",&tmp_float);
      fscanf(fp,"%f",&tmp_float);
    }
  fclose(fp);

  //read order file
  sprintf(fname, "%s", order_file);
   if((fp=fopen(fname,"r"))==NULL)
    {
      printf("\n Could not open file %s.\n",fname);
      exit(1);
    }
   
   for(int i=0; i<num_feature; i++)
     fscanf(fp,"%d",order+i);
   fclose(fp);

   
   //read region files
   short ****region_file = s4tensor(feature_end_id-feature_start_id+1,xdim,ydim,zdim);
   for(int i=feature_start_id; i<=feature_end_id; i++)
     {
       sprintf(fname, "%s_%d.bin", region_file_base,i);
       if((fp=fopen(fname,"r"))==NULL)
	 {
	   printf("\n Could not open file %s.\n",fname);
	   exit(1);
	 }
       for(int kk=0; kk<zdim; kk++)
	 for(int jj=0; jj<ydim; jj++)
	   for(int ii=0; ii<xdim; ii++)
	     fread(&region_file[i-feature_start_id][ii][jj][kk], sizeof(short), 1, fp);
       fclose(fp);
     }
     
   //output region file in which regions are selected with impact
   //========================================================================================================
   float ***region_selected_impact = f3tensor(xdim,ydim,zdim);
   for(int id=feature_start_id; id<=feature_end_id; id++)
     {
       for(int kk=0; kk<zdim; kk++)
	 for(int jj=0; jj<ydim; jj++)
	   for(int ii=0; ii<xdim; ii++)
	     region_selected_impact[ii][jj][kk] = 0;
       
       for(int kk=0; kk<zdim; kk++)
	 for(int jj=0; jj<ydim; jj++)
	   for(int ii=0; ii<xdim; ii++)
	     { 
	       for(int ll=0; ll<num_feature; ll++)
		 {
		   if((features[order[ll]][0] ==id)&&(region_file[id][ii][jj][kk]-1==features[order[ll]][1]))
		     region_selected_impact[ii][jj][kk] = weight[ll];
		 }
	     }
       
       sprintf(fname, "%s_diff_%d.img",output_file_base,id);
       if((fp=fopen(fname,"w"))==NULL)
	 {
	   printf("\n Could not open file %s.\n",fname);
	   exit(1);
	 }
       
       for(int kk=0; kk<zdim; kk++)
	 for(int jj=0; jj<ydim; jj++)
	   for(int ii=0; ii<xdim; ii++)
	     fwrite(&region_selected_impact[ii][jj][kk], sizeof(float),1,fp);
       fclose(fp);
     }
   free_f3tensor(region_selected_impact,xdim,ydim,zdim);   
   
   
   free_imatrix(features,1800, 2);
   free_ivector(order,num_feature);
   free_vector(weight,1800);
   free_vector(feature_score, 1800);
   free_s4tensor(region_file,feature_end_id-feature_start_id+1,xdim,ydim,zdim);
   
   return 0;
}
