/*******************************************************

Feature Extraction Class.

last modified : Oct 6, 2008

yong.fan@ieee.org
kanterae@uphs.upenn.edu

*******************************************************/

#include <stdlib.h>
#include <stdio.h>
#include "FeatureExtraction_model.h"
#include "general.h"
#include "StandardSVM.h"
#include "Kernel.h"
#include "UserKernel.h"
#include "IOTorch.h"
#define SCALE 100

FeatureExtraction_model::FeatureExtraction_model()
{
  // Default parameters for debugging
  _DebugFlag = False;
}

FeatureExtraction_model::~FeatureExtraction_model()
{
}

void FeatureExtraction_model :: SetInput (char *sublist, char *region_file_base, char *feature_location_file, char *feature_order_file, char *svm_model_file)
{
  this->Debug("SetInput");
  
  _file_sublist = sublist;
  
  
  _data_root_dir      = new char [10240];
  
  _region_file_base  = region_file_base;
  _feature_location_file = feature_location_file;
  _feature_order_file    = feature_order_file;
  _svm_model_file        = svm_model_file;
  
  
  _total_num_feature = 1800;
}

void FeatureExtraction_model::GetInput()
{
  this->Debug("GetInput");
  FILE *fp;
  if((fp=fopen(_file_sublist,"r"))==NULL)
    {
      printf("\n cannot open file %s.\n",_file_sublist);
      exit(1);
    }

  if(fscanf(fp,"%d %d",&_num_subject,&_num_feature)!=2)
    {
      printf("\n please check file %s (num_subject).\n",_file_sublist);
      exit(1);
    }
    
  if(fscanf(fp,"%d %d %d",&_xdim,&_ydim,&_zdim)!=3)
    {
      printf("\n please check file %s (dimension).\n",_file_sublist);
      exit(1);
    }
  if(fscanf(fp,"%s",_data_root_dir)!=1)
    {
      printf("\n please check file %s.(data root)\n",_file_sublist);
      exit(1);
    }

  _file_name = cc3tensor(_num_subject,_num_feature,1024);
  _label  = vector(_num_subject);
  int tmp;
  for(int i=0; i<_num_subject; i++)
    {
      for(int j=0; j<_num_feature; j++)
        {
          if(fscanf(fp,"%s", _file_name[i][j])!=1)
             {
               printf("\n please check file (file_name) %s.\n",_file_sublist);
               exit(1);
             }
        }
      if(fscanf(fp,"%d", &tmp)!=1)
        {
          printf("\n please check file %s.(label)\n",_file_sublist);
          exit(1);
        }
     _label[i] = tmp;
    }
  fclose(fp);          
}

void FeatureExtraction_model::Initialize()
{
  
  this->Debug("Initialize");
  
  _im_size = _xdim * _ydim * _zdim;

  _feature_data = f4tensor(_num_subject, _xdim, _ydim, _zdim);
  
  _r_region = s4tensor(_num_feature, _xdim, _ydim, _zdim);

  _r_roi_volume = f3tensor(_num_feature, _num_subject,  _total_num_feature);
  _r_roi_size   = matrix(_num_feature, _total_num_feature);
  
  for(int i=0; i<_num_feature; i++)
    for(int k=0; k<_num_subject; k++)
      for(int l=0; l<_total_num_feature; l++)
	_r_roi_volume[i][k][l] = 0;
  
  for(int i=0; i<_num_feature; i++)
    for(int k=0; k<_total_num_feature; k++)
      _r_roi_size[i][k] = 0;
 
}

void FeatureExtraction_model::Finalize()
{
  
  this->Debug("Finalize");
  
  free_s4tensor(_r_region,_num_feature,  _xdim, _ydim, _zdim);
  free_f4tensor(_feature_data, _num_subject, _xdim, _ydim, _zdim);
  
  free_imatrix(_r_roi_feature_location, _num_classification_feature, 2);
  
  free_matrix(_r_roi_feature,_num_subject, _num_classification_feature);
  
  free_f3tensor(_r_roi_volume,_num_feature, _num_subject,  _total_num_feature);
  free_matrix(_r_roi_size, _num_feature,_total_num_feature);
 
  free_matrix(_whitten_para, _num_classification_feature, 2);
  free_cc3tensor(_file_name, _num_subject,_num_feature,1024);
  delete [] _data_root_dir;
}


void FeatureExtraction_model::ReadData(int feat)
{
  this->Debug("ReadData");
  
  FILE *fp;
  short tmp;
  
  char filename[1024];
  
  for(int i=0; i< _num_subject; i++)
    {
      sprintf(filename, "%s/%s",_data_root_dir,_file_name[i][feat]);
      if((fp = fopen(filename, "r"))== NULL)
        {
          printf("\n could not open file: %s\n",filename);
          exit(1);
        }
      printf(".");
      
      for(int kk=0; kk<_zdim; kk++)
        for(int jj=0; jj<_ydim; jj++)
          for(int ii=0; ii<_xdim; ii++)
            {
              fread(&tmp,sizeof(short),1,fp);
              _feature_data[i][ii][jj][kk] = tmp;
            }
      fclose(fp);
    }
    
}

void FeatureExtraction_model::ComputeRegionFeat(int feat)
{
  this->Debug("Read Region File");
  // get the region information using watershed segmentation method
  
  char filename[1024];
  sprintf(filename,"%s_%d.bin",_region_file_base, feat);
  FILE *fp;
  if((fp=fopen(filename,"r"))==NULL)
    {
      printf("\n Cannot open file: %s.\n",filename);
      exit(1);
    }
  for(int k=0; k<_zdim; k++)
    for(int j=0; j<_ydim; j++)
      for(int i=0; i<_xdim; i++)
	fread(&_r_region[feat][i][j][k],sizeof(short),1,fp);
  fclose(fp);
}

void FeatureExtraction_model::ReadOrderFile()
{
  this->Debug("Read Order File");
  FILE *fp;
  if((fp=fopen(_feature_order_file,"r"))==NULL)
    {
      printf("\n Cannot open feature order file %s.\n",_feature_order_file);
      exit(1);
    }
  _num_classification_feature = 0;
  int tmp;
  while(fscanf(fp,"%d",&tmp)>0)
    {
      _num_classification_feature += 1;
    }
  fclose(fp);
  _order = ivector(_num_classification_feature);
  if((fp=fopen(_feature_order_file,"r"))==NULL)
    {
      printf("\n Cannot open feature order file %s.\n",_feature_order_file);
      exit(1);
    }
  for(int i=0; i<_num_classification_feature; i++)
    fscanf(fp,"%d",&_order[i]);
  fclose(fp);  

  _r_roi_feature_location    = imatrix(_num_classification_feature,2);
  _r_roi_feature = matrix(_num_subject, _num_classification_feature);
  _whitten_para = matrix(_num_classification_feature, 2);
  
  
}

void FeatureExtraction_model::ReadFeatureLocation()
{
  this->Debug("Read Feature Locations");
  FILE *fp;

  if((fp=fopen(_feature_location_file,"r"))==NULL)
    {
      printf("\n cannot open file %s\n",_feature_location_file);
      exit(1);
    }
  
  char tmpchar[1024];

  for(int i=0; i<6; i++)
    fscanf(fp, "%s", tmpchar);
  
  int tmpint;
  float tmpfloat;
  for(int i=0; i<_num_classification_feature; i++)
    {
      fscanf(fp, "%d", &tmpint);
      fscanf(fp, "%f", &tmpfloat);
      fscanf(fp, "%d", &_r_roi_feature_location[i][0]);
      fscanf(fp, "%d", &_r_roi_feature_location[i][1]);
      fscanf(fp, "%f", &_whitten_para[i][0]);
      fscanf(fp, "%f", &_whitten_para[i][1]);
    }
  fclose(fp);
}

void FeatureExtraction_model::GetFeature(int feat)
{
  this->Debug("GetFeature");
  
  // roi feature from region growing
  for(int k=0; k<_zdim; k++)
    for(int j=0; j<_ydim; j++)
      for(int i=0; i<_xdim; i++)
	{
	  int id = (int)_r_region[feat][i][j][k] -1;
	  if((id>=0)&&(id<_total_num_feature))
	    {
	      _r_roi_size[feat][id] += 1;
	      
	      for(int ii=0; ii<_num_subject; ii++)
		_r_roi_volume[feat][ii][id] += _feature_data[ii][i][j][k];
	    }
	}
  
  for(int ii=0; ii<_total_num_feature; ii++)
    for(int jj=0; jj<_num_subject; jj++)
      _r_roi_volume[feat][jj][ii] /= (_r_roi_size[feat][ii]+1);
}

void FeatureExtraction_model::RankROIFeature()
{
  this->Debug("ROI Feature");
  
  for(int index=0; index < _num_classification_feature; index++)
    {	
      int level  = _r_roi_feature_location[_order[index]][0];
      int ii     = _r_roi_feature_location[_order[index]][1];
      float mean = _whitten_para[_order[index]][0];
      float var  = _whitten_para[_order[index]][1];
      //    printf("\n mean = %f, var =%f",mean,var);
      for(int ll=0; ll<_num_subject; ll++)
	_r_roi_feature[ll][index]= (_r_roi_volume[level][ll][ii] - mean)/var;
    }  
}  

void FeatureExtraction_model::WriteFiles()
{
  this->Debug("WriteClassificationData");

  FILE *fp;
  char test_file[100];
  
   
  // Write roi feature from region_grow
  sprintf(test_file, "model_test.bin");
  if((fp=fopen(test_file, "w")) == NULL)
    {
      printf("\n Could not open file: %s\n", test_file);
      exit(1);
    }
  
  fprintf(fp,"%d    %d\n", _num_subject, _num_classification_feature+1);
  
  for(int i=0; i<_num_subject; i++)
    {
      for(int j=0; j<_num_classification_feature; j++)
	fprintf(fp,"%f\t",_r_roi_feature[i][j]);
      fprintf(fp,"%f\n",_label[i]);
    }
  fclose(fp);
}

void FeatureExtraction_model::Run( )
{
  GetInput();	
  Initialize();
  for(int feat=0; feat<_num_feature; feat++)
    {
      ReadData(feat);
      ComputeRegionFeat(feat);
      GetFeature(feat); 
    }
  ReadOrderFile();
  ReadFeatureLocation();
  RankROIFeature();
  // WriteFiles();
  SVMClassification();
  Finalize();
}

void FeatureExtraction_model::SVMClassification( )
{
  this->Debug("SVMClassification");
  
  real **data = new real *[_num_subject];
  for(int i=0; i<_num_subject; i++)
    data[i] = new real [_num_classification_feature];
  
  for(int i=0; i<_num_subject; i++)
    for(int j=0; j<_num_classification_feature; j++)
      data[i][j] = _r_roi_feature[i][j];
  
  FILE *fp;
  
  StandardSVM emilie;
  emilie.load(_svm_model_file);
  int num_feature = emilie.n_input_dim;
  
  for(int i=0; i<_num_subject; i++)
    {
      float tmp = 0;
      for(int j=0; j<num_feature; j++)
	tmp += data[i][j] *data[i][j];
      tmp = SCALE/(sqrt(tmp)+1.0e-6);
      
      for(int j=0; j<num_feature; j++)
	data[i][j] *= tmp;
    }
  
  real *y_pred = new real[_num_subject];
  
  for(int i = 0; i < _num_subject; i++)
    y_pred[i] = emilie.use(data[i]);
  
  if((fp=fopen("svm_score.bin","w"))==NULL)
    {
      printf("\n cannot open svm_score.bin for outputing scores.\n");
      exit(1);
    }
  for(int i=0; i<_num_subject; i++)
    fprintf(fp,"%f\t",y_pred[i]);
  fclose(fp);
  
  delete [] y_pred;
  for(int i=0; i<_num_subject; i++)
    delete [] data[i];
  delete [] data;
}
