#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#define FREE_ARG char*

void nrerror(char error_text[])
/*Numerical Recipes standard handler */
{
  fprintf(stderr, "Numerical Recipes run-time error...\n");
  fprintf(stderr, "%s\n", error_text);
  fprintf(stderr, "... now exiting to system ...\n");
  exit(1);
}

float *vector(long nh)
/* allocate a float vector with subscript range v[nl..nh]  */
{
  float *v;
  
  v=(float *)malloc((size_t)((nh)*sizeof(float)));
  if(!v) nrerror("allocation failure in vector()");
  return v;
}

int *ivector(long nh)
/* allocate an int vector with subscript range v[nl..nh]  */
{
  int  *v;
  
  v=(int *)malloc((size_t)((nh)*sizeof(int)));
  if(!v) nrerror("allocation failure in ivector()");
  return v;
}

unsigned char *cvector(long nh)
/* allocate an unsigned char vector with subscript range v[nl..nh]  */
{
  unsigned char *v;
  
  v=(unsigned char *)malloc((size_t)((nh)*sizeof(unsigned char)));
  if(!v) nrerror("allocation failure in cvector()");
  return v;
} 
  
unsigned long  *lvector(long nh)
/* allocate an unsigned long vector with subscript range v[nl..nh]  */
{
  unsigned long *v;
  
  v=(unsigned long *)malloc((size_t)((nh)*sizeof(unsigned long)));
  if(!v) nrerror("allocation failure in lvector()");
  return v;
}

double *dvector(long nh)
/* allocate a double vector with subscript range v[nl..nh]  */
{
  double *v;
  
  v=(double *)malloc((size_t)((nh)*sizeof(double)));
  if(!v) nrerror("allocation failure in dvector()");
  return v;
}

float **matrix(long nrh, long nch)
    /* allocate a float matrix wiht subscript range m[0..nrh-1,0..nch-1]  */
{
  long i, nrow=nrh, ncol=nch;
  float **m;
  
  /* allocate pointers to rows */
  m=(float **) malloc((size_t)((nrow)*sizeof(float *)));
  if (!m) nrerror("allocate failure 1 in matrix()");



  /* allocate rows and set  pointers to them  */
  for(i=0; i<nrh; i++)
    m[i]=(float *) malloc((size_t)((ncol)*sizeof(float)));
  if(!m[nrh-1]) nrerror("allocate failure 2 in matrix()");
 
  /* return pointers to array of pointers to rows  */
  return m;
}


double **dmatrix(long nrh, long nch)
    /* allocate a double matrix wiht subscript range m[0..nrh-1,0..nch-1]  */
{
  long i, nrow=nrh, ncol=nch;
  double **m;
  
  /* allocate pointers to rows */
  m=(double **) malloc((size_t)((nrow)*sizeof(double *)));
  if (!m) nrerror("allocate failure 1 in dmatrix()");
 
  /* allocate rows and set  pointers to them  */
   for(i=0; i<nrh; i++)
     m[i]=(double *) malloc((size_t)((ncol)*sizeof(double)));
  if(!m[nrh-1]) nrerror("allocate failure 2 in dmatrix()");
   
  /* return pointers to array of pointers to rows  */
  return m;
}

int **imatrix(long nrh, long nch)
    /* allocate an int matrix wiht subscript range m[0..nrh-1,0..nch-1]  */
{
  long i, nrow=nrh, ncol=nch;
  int **m;
  
  /* allocate pointers to rows */
  m=(int **) malloc((size_t)((nrow)*sizeof(int *)));
  if (!m) nrerror("allocate failure 1 in imatrix()");
  
  /* allocate rows and set  pointers to them  */
   for(i=0; i<nrh; i++)
     m[i]=(int *) malloc((size_t)((ncol)*sizeof(int)));
   if(!m[nrh-1]) nrerror("allocate failure 2 in imatrix()");
 
  
  /* return pointers to array of pointers to rows  */
  return m;
}

char **cmatrix(long nrh, long nch)
    /* allocate a char matrix wiht subscript range m[0..nrh-1,0..nch-1]  */
{
  long i, nrow=nrh, ncol=nch;
  char **m;
  
  /* allocate pointers to rows */
  m=(char **) malloc((size_t)((nrow)*sizeof(char *)));
  if (!m) nrerror("allocate failure 1 in cmatrix()");
 
  /* allocate rows and set  pointers to them  */
   for(i=0; i<nrh; i++)
     m[i]=(char *) malloc((size_t)((ncol)*sizeof(char)));
  if(!m[nrh-1]) nrerror("allocate failure 2 in cmatrix()");
   
  /* return pointers to array of pointers to rows  */
  return m;
}




float **submatrix(float **a, long oldrl, long oldrh, long oldcl, long oldch)
    /* point a submatrix [0..][0..] to a[oldrl..oldrh,oldcl..oldch]  */
{
  long i,j, nrow=oldrh-oldrl+1, ncol=oldch-oldcl+1;
  float **m;
  
  /* allocate array of pointers to rows */
  m=(float **) malloc((size_t)((nrow)*sizeof(float *)));
  if (!m) nrerror("allocate failure in submatrix()");
 
  /* set pointers to rows  */
  for(i=oldrl,j=0; i<=oldrh; i++,j++) m[j]=a[i];
  
  /* return pointers to array of pointers to rows  */
  return m;
}

unsigned char ***c3tensor(long nrh, long nch, long ndh)
/* allocate an unsigned  char 3tensor with range t[0..nrh-1][0..nch-1][0..ndh-1] */
{
  long i,j,nrow=nrh, ncol=nch, ndep=ndh;
  unsigned  char  ***t;
  
  /* allocate pointers to pointers to  rows */
  t=(unsigned char  ***) malloc((size_t)((nrow)*sizeof(unsigned char **)));
  if (!t) nrerror("allocate failure 1 in c3tensor()");
  

  /* allocate pointers to rows and set pointers to them */
  for(i=0;i<nrow;i++)  
    t[i]=(unsigned char **) malloc((size_t)((ncol)*sizeof(unsigned char*)));
  if(!t[nrh-1]) nrerror("allocation failure 2 in c3tensor()");
  

  /* allocate rows and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
        t[j][i]=(unsigned char *) malloc((size_t)((ndep)*sizeof(unsigned char)));
  if(!t[nrh-1][nch-1]) nrerror(" allocation failure 3 in c3tensor()");
    
  /* return pointers to array of pointers to rows  */
  return t;
}

char ***cc3tensor(long nrh, long nch, long ndh)
/* allocate an unsigned  char 3tensor with range t[0..nrh-1][0..nch-1][0..ndh-1] */
{
  long i,j,nrow=nrh, ncol=nch, ndep=ndh;
  char  ***t;
  
  /* allocate pointers to pointers to  rows */
  t=(char  ***) malloc((size_t)((nrow)*sizeof(char **)));
  if (!t) nrerror("allocate failure 1 in cc3tensor()");
  

  /* allocate pointers to rows and set pointers to them */
  for(i=0;i<nrow;i++)  
    t[i]=(char **) malloc((size_t)((ncol)*sizeof(char*)));
  if(!t[nrh-1]) nrerror("allocation failure 2 in cc3tensor()");
  

  /* allocate rows and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
        t[j][i]=(char *) malloc((size_t)((ndep)*sizeof(char)));
  if(!t[nrh-1][nch-1]) nrerror(" allocation failure 3 in cc3tensor()");
    
  /* return pointers to array of pointers to rows  */
  return t;
}

int ***i3tensor(long nrh, long nch, long ndh)
/* allocate an int 3tensor with range t[0..nrh-1][0..nch-1][0..ndh-1] */
{
  long i,j,nrow=nrh, ncol=nch, ndep=ndh;
  int  ***t;
  
  /* allocate pointers to pointers to  rows */
  t=(int ***) malloc((size_t)((nrow)*sizeof(int **)));
  if (!t) nrerror("allocate failure 1 in i3tensor()");
  

  /* allocate pointers to rows and set pointers to them */
  for(i=0;i<nrow;i++)  
    t[i]=(int **) malloc((size_t)((ncol)*sizeof(int*)));
  if(!t[nrh-1]) nrerror("allocation failure 2 in i3tensor()");
  

  /* allocate rows and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
        t[j][i]=(int *) malloc((size_t)((ndep)*sizeof(int)));
  if(!t[nrh-1][nch-1]) nrerror(" allocation failure 3 in i3tensor()");
    
  /* return pointers to array of pointers to rows  */
  return t;
}

short ***s3tensor(long nrh, long nch, long ndh)
/* allocate an short 3tensor with range t[0..nrh-1][0..nch-1][0..ndh-1] */
{
  long i,j,nrow=nrh, ncol=nch, ndep=ndh;
  short  ***t;
  
  /* allocate pointers to pointers to  rows */
  t=(short ***) malloc((size_t)((nrow)*sizeof(short **)));
  if (!t) nrerror("allocate failure 1 in s3tensor()");
  

  /* allocate pointers to rows and set pointers to them */
  for(i=0;i<nrow;i++)  
    t[i]=(short **) malloc((size_t)((ncol)*sizeof(short *)));
  if(!t[nrh-1]) nrerror("allocation failure 2 in s3tensor()");
  

  /* allocate rows and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
        t[j][i]=(short *) malloc((size_t)((ndep)*sizeof(short)));
  if(!t[nrh-1][nch-1]) nrerror(" allocation failure 3 in s3tensor()");
    
  /* return pointers to array of pointers to rows  */
  return t;
}
float ***f3tensor(long nrh, long nch, long ndh)
/* allocate a float 3tensor with range t[0..nrh-1][0..nch-1][0..ndh-1] */
{
  long i,j,nrow=nrh, ncol=nch, ndep=ndh;
  float ***t;
  
  /* allocate pointers to pointers to  rows */
  t=(float ***) malloc((size_t)((nrow)*sizeof(float **)));
  if (!t) nrerror("allocate failure 1 in f3tensor()");
  

  /* allocate pointers to rows and set pointers to them */
  for(i=0;i<nrow;i++)  
    t[i]=(float **) malloc((size_t)((ncol)*sizeof(float*)));
  if(!t[nrh-1]) nrerror("allocation failure 2 in f3tensor()");
  

  /* allocate rows and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
        t[j][i]=(float *) malloc((size_t)((ndep)*sizeof(float)));
  if(!t[nrh-1][nch-1]) nrerror(" allocation failure 3 in f3tensor()");
    
  /* return pointers to array of pointers to rows  */
  return t;
}

double ***d3tensor(long nrh, long nch, long ndh)
/* allocate a double 3tensor with range t[0..nrh-1][0..nch-1][0..ndh-1] */
{
  long i,j,nrow=nrh, ncol=nch, ndep=ndh;
  double  ***t;
  
  /* allocate pointers to pointers to  rows */
  t=(double ***) malloc((size_t)((nrow)*sizeof(double **)));
  if (!t) nrerror("allocate failure 1 in d3tensor()");
  

  /* allocate pointers to rows and set pointers to them */
  for(i=0;i<nrow;i++)  
    t[i]=(double **) malloc((size_t)((ncol)*sizeof(double*)));
  if(!t[nrh-1]) nrerror("allocation failure 2 in d3tensor()");
  

  /* allocate rows and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
        t[j][i]=(double *) malloc((size_t)((ndep)*sizeof(double)));
  if(!t[nrh-1][nch-1]) nrerror(" allocation failure 3 in d3tensor()");
    
  /* return pointers to array of pointers to rows  */
  return t;
}

float ****f4tensor(long nrh, long nch, long ndh, long nth)
/* allocate a float 4tensor with range t[0..nrh-1][0..nch-1][0..ndh-1][0..nth-1] */
{
  long i,j,k,nrow=nrh, ncol=nch, ndep=ndh, ntim=nth;
  float ****t;
  
  /* allocate pointers to pointers to  rows */
  t=(float ****) malloc((size_t)((nrow)*sizeof(float ***)));
  if (!t) nrerror("allocate failure 1 in f4tensor()");
  

  /* allocate pointers to rows and set pointers to them */
  for(i=0;i<nrow;i++)  
    t[i]=(float ***) malloc((size_t)((ncol)*sizeof(float**)));
  if(!t[nrh-1]) nrerror("allocation failure 2 in f4tensor()");
  

  /* allocate rows and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
        t[j][i]=(float **) malloc((size_t)((ndep)*sizeof(float *)));
  if(!t[nrh-1][nch-1]) nrerror(" allocation failure 3 in f4tensor()");

   /* allocate cols  and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
      for(k=0;k<ndep;k++)
        t[j][i][k]=(float *) malloc((size_t)((ntim)*sizeof(float)));
  if(!t[nrh-1][nch-1][ndh-1]) nrerror(" allocation failure 4 in f4tensor()");
    
  /* return pointers to array of pointers to rows  */
  return t;
}

short ****i4tensor(long nrh, long nch, long ndh, long nth)
/* allocate a float 4tensor with range t[0..nrh-1][0..nch-1][0..ndh-1][0..nth-1] */
{
  long i,j,k,nrow=nrh, ncol=nch, ndep=ndh, ntim=nth;
  short ****t;
  
  /* allocate pointers to pointers to  rows */
  t=(short ****) malloc((size_t)((nrow)*sizeof(short ***)));
  if (!t) nrerror("allocate failure 1 in i4tensor()");
  

  /* allocate pointers to rows and set pointers to them */
  for(i=0;i<nrow;i++)  
    t[i]=(short ***) malloc((size_t)((ncol)*sizeof(short**)));
  if(!t[nrh-1]) nrerror("allocation failure 2 in i4tensor()");
  

  /* allocate rows and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
        t[j][i]=(short **) malloc((size_t)((ndep)*sizeof(short *)));
  if(!t[nrh-1][nch-1]) nrerror(" allocation failure 3 in i4tensor()");

   /* allocate cols  and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
      for(k=0;k<ndep;k++)
        t[j][i][k]=(short *) malloc((size_t)((ntim)*sizeof(short)));
  if(!t[nrh-1][nch-1][ndh-1]) nrerror(" allocation failure 4 in i4tensor()");
    
  /* return pointers to array of pointers to rows  */
  return t;
}
short ****s4tensor(long nrh, long nch, long ndh, long nth)
/* allocate a float 4tensor with range t[0..nrh-1][0..nch-1][0..ndh-1][0..nth-1] */
{
  long i,j,k,nrow=nrh, ncol=nch, ndep=ndh, ntim=nth;
  short ****t;
  
  /* allocate pointers to pointers to  rows */
  t=(short ****) malloc((size_t)((nrow)*sizeof(short ***)));
  if (!t) nrerror("allocate failure 1 in i4tensor()");
  

  /* allocate pointers to rows and set pointers to them */
  for(i=0;i<nrow;i++)  
    t[i]=(short ***) malloc((size_t)((ncol)*sizeof(short**)));
  if(!t[nrh-1]) nrerror("allocation failure 2 in i4tensor()");
  

  /* allocate rows and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
        t[j][i]=(short **) malloc((size_t)((ndep)*sizeof(short *)));
  if(!t[nrh-1][nch-1]) nrerror(" allocation failure 3 in i4tensor()");

   /* allocate cols  and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
      for(k=0;k<ndep;k++)
        t[j][i][k]=(short *) malloc((size_t)((ntim)*sizeof(short)));
  if(!t[nrh-1][nch-1][ndh-1]) nrerror(" allocation failure 4 in i4tensor()");
    
  /* return pointers to array of pointers to rows  */
  return t;
}

float *****f5tensor(long nrh,long nch,long ndh,long nth,long nsh)
/* allocate a float 5 dimension tensor with range t[0..nrh-1][0..nch-1][0..ndh-1][0..nth-1][0..nsh-1] */
{
  long i,j,k,l,nrow=nrh, ncol=nch, ndep=ndh, ntim=nth, ntyp=nsh;
  float *****t;
  
  /* allocate pointers to pointers to  rows */
  t=(float *****) malloc((size_t)((nrow)*sizeof(float ****)));
  if (!t) nrerror("allocate failure 1 in f5tensor()");
  

  /* allocate pointers to rows and set pointers to them */
  for(i=0;i<nrow;i++)  
    t[i]=(float ****) malloc((size_t)((ncol)*sizeof(float***)));
  if(!t[nrh-1]) nrerror("allocation failure 2 in f5tensor()");
  

  /* allocate rows and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
        t[j][i]=(float ***) malloc((size_t)((ndep)*sizeof(float **)));
  if(!t[nrh-1][nch-1]) nrerror(" allocation failure 3 in f5tensor()");

   /* allocate cols  and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
      for(k=0;k<ndep;k++)
        t[j][i][k]=(float **) malloc((size_t)((ntim)*sizeof(float *)));
  if(!t[nrh-1][nch-1][ndh-1]) nrerror(" allocation failure 4 in f5tensor()");

  /* allocate cols  and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
      for(k=0;k<ndep;k++)
	for(l=0;l<ntim;l++)
	  t[j][i][k][l]=(float *) malloc((size_t)((ntyp)*sizeof(float )));
  if(!t[nrh-1][nch-1][ndh-1][ntim-1]) nrerror(" allocation failure 5 in f5tensor()");

    
  /* return pointers to array of pointers to rows  */
  return t;
}

short *****s5tensor(long nrh,long nch,long ndh,long nth,long nsh)
/* allocate a short 5 dimension tensor with range t[0..nrh-1][0..nch-1][0..ndh-1][0..nth-1][0..nsh-1] */
{
  long i,j,k,l,nrow=nrh, ncol=nch, ndep=ndh, ntim=nth, ntyp=nsh;
  short *****t;
  
  /* allocate pointers to pointers to  rows */
  t=(short *****) malloc((size_t)((nrow)*sizeof(short ****)));
  if (!t) nrerror("allocate failure 1 in f5tensor()");
  

  /* allocate pointers to rows and set pointers to them */
  for(i=0;i<nrow;i++)  
    t[i]=(short ****) malloc((size_t)((ncol)*sizeof(short***)));
  if(!t[nrh-1]) nrerror("allocation failure 2 in f5tensor()");
  

  /* allocate rows and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
        t[j][i]=(short ***) malloc((size_t)((ndep)*sizeof(short **)));
  if(!t[nrh-1][nch-1]) nrerror(" allocation failure 3 in f5tensor()");

   /* allocate cols  and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
      for(k=0;k<ndep;k++)
        t[j][i][k]=(short **) malloc((size_t)((ntim)*sizeof(short *)));
  if(!t[nrh-1][nch-1][ndh-1]) nrerror(" allocation failure 4 in f5tensor()");

  /* allocate cols  and set  pointers to them  */
  for(j=0;j<nrow;j++)
    for(i=0;i<ncol;i++)
      for(k=0;k<ndep;k++)
	for(l=0;l<ntim;l++)
	  t[j][i][k][l]=(short *) malloc((size_t)((ntyp)*sizeof(short )));
  if(!t[nrh-1][nch-1][ndh-1][ntim-1]) nrerror(" allocation failure 5 in f5tensor()");

    
  /* return pointers to array of pointers to rows  */
  return t;
}

void free_vector(float *v, long nh)
     /* free a float vector allocated with vector()  */
{
  free((FREE_ARG)(v));
}

void free_ivector(int *v, long nh)
     /* free an int vector allocated with ivector()  */
{
  free((FREE_ARG)(v));
}

void free_cvector(unsigned char *v, long nh)
     /* free an unsigned char vector allocated with cvector()  */
{
  free((FREE_ARG)(v));
}

void free_lvector(unsigned long *v, long nh)
     /* free an unsigned long vector allocated with lvector()  */
{
  free((FREE_ARG)(v));
}

void free_dvector(double *v, long nh)
     /* free a double vector allocated with dvector()  */
{
  free((FREE_ARG)(v));
}

void free_matrix(float **m, long nrh, long nch)
     /* free a float matrix allocated by matrix()  */
{
  long i;
  for(i=0;i<nrh;i++)
    free((FREE_ARG)(m[i])); 
  free((FREE_ARG)(m)); 
}

void free_dmatrix(double **m, long nrh, long nch)
     /* free a double matrix allocated by dmatrix()  */
{
  long i;
  for(i=0;i<nrh;i++)
    free((FREE_ARG)(m[i])); 
  free((FREE_ARG)(m));  
}

void free_imatrix(int **m, long nrh, long nch)
     /* free an int matrix allocated by imatrix()  */
{
  long i;
  for(i=0;i<nrh;i++)
    free((FREE_ARG)(m[i])); 
  free((FREE_ARG)(m));  
}

void free_cmatrix(char **m, long nrh, long nch)
     /* free an int matrix allocated by imatrix()  */
{
  long i;
  for(i=0;i<nrh;i++)
    free((FREE_ARG)(m[i])); 
  free((FREE_ARG)(m));  
}

void free_submatrix(float **b, long nrl, long nrh, long ncl, long nch)
     /* free a submatrix allocated by submatrix()  */
{
  long i;
  for(i=nrl;i<=nrh;i++)
    free((FREE_ARG)(b[i])); 
  free((FREE_ARG)(b));  
}

void free_c3tensor(unsigned char  ***t, long nrh, long nch, long ndh)
     /* free an unsigned char  c3tensor allocated by c3tensor()  */
{
  long i,j;
  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
       free((FREE_ARG)(t[i][j])); 
  
  for(i=0;i<nrh;i++)
    free((FREE_ARG)(t[i])); 
  free((FREE_ARG)(t));   
}

void free_cc3tensor(char  ***t, long nrh, long nch, long ndh)
     /* free an unsigned char  c3tensor allocated by c3tensor()  */
{
  long i,j;
  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
       free((FREE_ARG)(t[i][j])); 
  
  for(i=0;i<nrh;i++)
    free((FREE_ARG)(t[i])); 
  free((FREE_ARG)(t));   
}


void free_i3tensor(int  ***t, long nrh, long nch, long ndh)
     /* free a int  i3tensor allocated by i3tensor()  */
{
  long i,j;
  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
       free((FREE_ARG)(t[i][j])); 
  
  for(i=0;i<nrh;i++)
    free((FREE_ARG)(t[i])); 
  free((FREE_ARG)(t));   
}

void free_s3tensor(short  ***t, long nrh, long nch, long ndh)
     /* free a int  s3tensor allocated by s3tensor()  */
{
  long i,j;
  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
       free((FREE_ARG)(t[i][j])); 
  
  for(i=0;i<nrh;i++)
    free((FREE_ARG)(t[i])); 
  free((FREE_ARG)(t));   
}

void free_f3tensor(float ***t, long nrh, long nch, long ndh)
     /* free a float f3tensor allocated by f3tensor()  */
{
  long i,j;
  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
       free((FREE_ARG)(t[i][j])); 
  
  for(i=0;i<nrh;i++)
    free((FREE_ARG)(t[i])); 
  free((FREE_ARG)(t));   
}

void free_d3tensor(double ***t, long nrh, long nch, long ndh)
     /* free a float f3tensor allocated by f3tensor()  */
{
  long i,j;
  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
       free((FREE_ARG)(t[i][j])); 
  
  for(i=0;i<nrh;i++)
    free((FREE_ARG)(t[i])); 
  free((FREE_ARG)(t));   
}

void free_f4tensor(float ****t, long nrh, long nch, long ndh, long nth)
     /* free a float f4tensor allocated by f4tensor()  */
{
  long i,j,k;
  
  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
      for(k=0;k<ndh;k++)
       free((FREE_ARG)(t[i][j][k]));


  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
       free((FREE_ARG)(t[i][j])); 
  
  for(i=0;i<nrh;i++)
    free((FREE_ARG)(t[i])); 
  free((FREE_ARG)(t));   
}

void free_i4tensor(short ****t, long nrh, long nch, long ndh, long nth)
     /* free a short i4tensor allocated by i4tensor()  */
{
  long i,j,k;
  
  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
      for(k=0;k<ndh;k++)
       free((FREE_ARG)(t[i][j][k]));


  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
       free((FREE_ARG)(t[i][j])); 
  
  for(i=0;i<nrh;i++)
    free((FREE_ARG)(t[i])); 
  free((FREE_ARG)(t));   
}

void free_s4tensor(short ****t, long nrh, long nch, long ndh, long nth)
     /* free a short i4tensor allocated by i4tensor()  */
{
  long i,j,k;
  
  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
      for(k=0;k<ndh;k++)
       free((FREE_ARG)(t[i][j][k]));


  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
       free((FREE_ARG)(t[i][j])); 
  
  for(i=0;i<nrh;i++)
    free((FREE_ARG)(t[i])); 
  free((FREE_ARG)(t));   
}

void free_f5tensor(float *****t, long nrh, long nch, long ndh, long nth, long nsh)
     /* free a float f4tensor allocated by f4tensor()  */
{
  long i,j,k,l;
  
  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
      for(k=0;k<ndh;k++)
	for(l=0;l<nth;l++)
	  free((FREE_ARG)(t[i][j][k][l]));

  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
      for(k=0;k<ndh;k++)
	free((FREE_ARG)(t[i][j][k]));
  
  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
      free((FREE_ARG)(t[i][j])); 
  
  for(i=0;i<nrh;i++)
    free((FREE_ARG)(t[i])); 
  free((FREE_ARG)(t));   
}

void free_s5tensor(short *****t, long nrh, long nch, long ndh, long nth, long nsh)
     /* free a short s5tensor allocated by s5tensor()  */
{
  long i,j,k,l;
  
  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
      for(k=0;k<ndh;k++)
	for(l=0;l<nth;l++)
	  free((FREE_ARG)(t[i][j][k][l]));

  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
      for(k=0;k<ndh;k++)
	free((FREE_ARG)(t[i][j][k]));
  
  for(i=0;i<nrh;i++)
    for(j=0;j<nch;j++)
      free((FREE_ARG)(t[i][j])); 
  
  for(i=0;i<nrh;i++)
    free((FREE_ARG)(t[i])); 
  free((FREE_ARG)(t));   
}

#define NRANSI
#define SWAP(a,b) itemp=(a);(a)=(b);(b)=itemp;
#define M 7
#define NSTACK 50

void indexx(long n, float arr[], unsigned long indx[])
{
	long i,indxt,ir=n-1,itemp,j,k,l=0;
	int jstack=0,*istack;
	float a;

	istack= ivector(NSTACK);

	for (j=0;j<n;j++) indx[j]=j;
	for (;;) {
		if (ir-l < M) {
			for (j=l+1;j<=ir;j++) {
				indxt=indx[j];
				a=arr[indxt];
				for (i=j-1;i>=0;i--) {
					if (arr[indx[i]] <= a) break;
					indx[i+1]=indx[i];
				}
				indx[i+1]=indxt;
			}
			if (jstack == 0) break;
			ir=istack[jstack--];
			l=istack[jstack--];
		} else {
			k=(l+ir) >> 1;
			SWAP(indx[k],indx[l+1]);
			if (arr[indx[l]] > arr[indx[ir]]) {
				SWAP(indx[l],indx[ir])
			}
			if (arr[indx[l+1]] > arr[indx[ir]]) {
				SWAP(indx[l+1],indx[ir])
			}
			if (arr[indx[l]] > arr[indx[l+1]]) {
				SWAP(indx[l+1],indx[l])
			}
			i=l+1;
			j=ir;
			indxt=indx[l+1];
			a=arr[indxt];
			for (;;) {
				do i++; while (arr[indx[i]] < a);
				do j--; while (arr[indx[j]] > a);
				if (j < i) break;
				SWAP(indx[i],indx[j])
			}
			indx[l+1]=indx[j];
			indx[j]=indxt;
			jstack += 2;
			if (jstack > NSTACK) printf("NSTACK too small in iindexx.");
			if (ir-i+1 >= j-l) {
				istack[jstack]=ir;
				istack[jstack-1]=i;
				ir=j-1;
			} else {
				istack[jstack]=j-1;
				istack[jstack-1]=l;
				l=i;
			}
		}
	}
	
	

	
	free_ivector(istack,NSTACK);
}
#undef M
#undef NSTACK
#undef SWAP
#undef NRANSI
