#!/usr/bin/perl -w
# Test the program
# Compare the output to precomputed values

use diagnostics;
use strict;
use strict;
use Digest::MD5 qw(md5_hex);
use Getopt::Long qw(:config no_ignore_case bundling pass_through);

my $SVNVersion='$Id: test.pl 35 2010-09-22 14:41:15Z batmangn@UPHS.PENNHEALTH.PRV $';
my $NAME;
($NAME=$0)=~s#^.*/##;


sub usage
{
	print STDERR "$NAME -p filename  [-n number] [-m] [-i filename] [-e env_module] -- [command to execute]\n";
	print STDERR "$NAME [--help|-h]\n";


	print STDERR << "E-O-USAGE";
	Options:
    [--precomputed|-p] file	name of file with precomputed results (required)
    [--number|-n] number	number of lines of output file to use in comparison (optional)
    [--md5sum|-m]		flag that precomputed file is already an MD5 checksum (optional)
    [--input|-i] filename	name of input data file to compare to precomputed file (optional)
    [--environment|-e] module	load the named module to configure the testing environment
    [--help|-h			display extended help
    -- command			command to execute (preceeded by "--", required if "-i" is not used)

E-O-USAGE
        print STDERR "\n\n@_\n";
        exit 1;
}

# This program uses POD (Perl's internal documentation). This makes it easy to
# maintain the documentation within the executable itself, yet produce nicely
# formatted output


=head1 $NAME

=head1 SYNOPSIS

B<$NAME> -p filename  [-n number] [-m] [-i filename] [-e env_module] -- [command to execute]

B<$NAME> [--help|-h]

=head1 DESCRIPTION

This script is a testing tool, used to compare the output of a program
to a pre-computed ouput. The data being compared can be textual or binary
(including images).

=head1 RETURN VALUES
	
The test program returns 0 if the comparison was successful, and 1 if it was not.

=head1 ARGUMENTS

Arguments:
    -n|--number	#
	    lines from the beginning of the output to use in comparison
	    (optional, default = ALL)

    -m|--md5sum	
   	flag to signify that the precomputed data is in the form of an md5 checksum

    -p|--precomputed file
    	name of file containing precomputed output for comparison (required)

    -i|--input file
	    name of file containing data produced by the command to compare
	    to the precomputed data. If this is specified, then the command
	    can be eliminated. This is typically used when the command
	    produces multiple output files.  (optional)

    -e|--environment modulename
	    name of module file to load to configure the testing environment. This 
	    option can be given multiple times to load additional modules. This has
	    the same affect as running "module load modulename" from the command line.
	    (optional)
   
	    The system modules that are available to be loaded can be listed with:

	      module avail
	     Local module files can also be specified with a path.


    command
	   the command to run to produce output. The command (and all
	   it's options) must be separated from the options to the testing
	   program with two dashes (--).
	   (required if "-i" is not specified)

    
Paths to the precomputed expected output are relative to the current
directory if not specified absolutely.


=head1 CREATING MD5 CHECKSUMS

An md5 checksum can be precomputed with the command "md5sum", as in:

  md5sum reference_stripped.img > precomputed_reference_stripped

=head1 EXAMPLES

    test.pl --precomputed reference_stripped.img -- ../newstripimg -i reference_input.dcm
	    compare the output produced by the program "newstripimg" to the
	    file "reference_stripped.img"

    test.pl -p precomputed_reference_stripped -- ../newstripimg -i reference_input.dcm
	    compare the output produced by the program "newstripimg"
	    to the precomputed MD5 checksum stored in the file
	    "precomputed_reference_stripped"


    test.pl -n 1 -p expectedoutput -- ../example -g 3 input.img
    	compare the first line produced by the command
	    "../example -g 3 input.img" to the data in the file
	    "expectedoutput"
	
    test.pl -p expected -e python2.5 -e ~/testing/myenv -- alterimg -g 3 input.img
	    load the system environment module "python2.5" and the personal module
		definitions from the file "~/testing/myenv", then run the command
		    alterimg -g 3 input.img
		compare the results to the file "expected"


    test.pl -p expectedoutput -m -i /tmp/output -- ../example -x 255,255,128 -o /tmp/output
	    compare the file "/tmp/output" to the file "expectedoutput",
	    which is already an md5 checksum. The file "/tmp/output" will be
	    produced by the command "../example -x 255,255,128 -o /tmp/output"

    test.pl -p out1.md5 -m -i out1 -- example -x 255,255,128 -o out1 -o2 out2 -o3 out3
	    run the command
	    "example -x 255,255,128 -o out1 -o2 out2 -o3 out3"
	    to produce 3 output files. Compare the file "out1" to the
	    md5 checksum found in the file "out1.md5"
    
    test.pl -p expectedoutput2 --md5sum -i /tmp/output2
	    Compare the file "/tmp/output2" (which must already exist)
	    to the md5 checksum found in the file "expectedoutput2"

    test.pl -p expectedoutput3 --input /tmp/output3 
	    Compare the file "/tmp/output3" (which must already exist)
	    to the complete data found in the file "expectedoutput2"

=head1 AUTHOR

Mark Bergman <mark.bergman@uphs.upenn.edu>

=head1 CONTACT

For questions, contact <sbia-software@uphs.upenn.edu>

=head1 RIGHTS

(c) 2008, University of Pennsylvania

May be distributed under the terms of the GNU GENERAL PUBLIC LICENSE.

=cut


my $linecount=0;
my $precompfile;
my $prechecked=0;
my $inputfile;
my $results;
my @precomputed;
my @tmparray;
my $expected;
my @envmodules;
my $moduletoload;

GetOptions('number|n=i'	=> \$linecount,
			'input|i=s' => \$inputfile,
			'md5sum|m' => \$prechecked,
			'environment|e=s' => \@envmodules,
			'help|h|?' => sub { exec("perldoc $0")},
			'precomputed|p=s'	=> \$precompfile);



if ( ! defined($precompfile))
{
	usage("The file containing the precomputed results must be specified with the \"--precomputed\" option.");
	exit 1;
}

if (! -f $precompfile )
{
	print STDERR "No such file ($precompfile)\n";
	exit 1;
}

# manually test for the presence of the "modulecmd" executable somewhere in $PATH. This is
# because there's no [easy?] way to trap failures in the
#		eval `modulecmd`
# call later on, and we want to avoid ugly errors if the modulecmd is not available
my $nomodulecmd = 1;
my $searchpath;
foreach $searchpath ( split(/:/,$ENV{'PATH'}))
{
	if ( -x "$searchpath/modulecmd" )
	{
		$nomodulecmd = 0;
		last;
	}
}

if ( $nomodulecmd )
{
	print STDERR "No modulecmd found...cannot load any modules.\n";
}
else
{
	# Load any specified environment modules
	foreach $moduletoload ( @envmodules )
	{
		eval `modulecmd perl load $moduletoload` or print STDERR "Error loading module $moduletoload: $!\n";
	}
}

if ( $linecount != 0 )
{
	# $linecount was defined...do some basic sanity checks
	if ( $linecount < 0 ) 
	{
		print STDERR "The number of lines of output to compare to the precomputed results must be a positive number\n";
		exit 1;
	}

	my $precompsize = -s $precompfile;

	if ( $linecount > $precompsize ) 
	{
		print STDERR "The number of lines of output to compare to the precomputed results ($linecount) must be less than or equal to the number of lines ($precompsize) in the precomputed results\n";
		exit 1;
	}
}
			


shift(@ARGV);	# remove "--"
if ( $#ARGV  == -1 )
{
	# There was no command specified...that's OK if $inputfile is defined
	if ( ! defined($inputfile))
	{
		# Error!
		usage("Must specify a command to run or the name of an existing file to use as input with the \"-i\" option.");
	}
}
else
{
	$results=`@ARGV`;
}

if (defined($inputfile))
{
	if (! -f $inputfile )
	{
		print STDERR "Specified input file ($inputfile) does not exist\n";
		exit 1;
	}
	open(OUTPUT,$inputfile) or die "Could not open input file ($inputfile): $!";
	@tmparray=<OUTPUT>;
	$results=join("",@tmparray);
	close(OUTPUT) or die "Could not close input file ($inputfile): $!";
}

if ($linecount != 0 )
{
	# We only want the first $linecount lines in $results
	# assumes that lines end in \n

	@tmparray=split(/\n/,$results);
	$results=join("\n",splice(@tmparray,0,$linecount));
	$results.="\n";
}

# replace $results with it's own checksum
$results=md5_hex("$results");

open(PRECOMPUTED,$precompfile) or die "Could not open file with expected results ($precompfile): $!";
if ( $prechecked == 0 )
{
	# must generate the checksum of the precomputed results file
	@precomputed=<PRECOMPUTED>;
	close(PRECOMPUTED) or die "Could not close file with expected results ($precompfile): $!";
	
	$expected=md5_hex(@precomputed);
}
else
{
	$expected=<PRECOMPUTED>;
	chomp($expected);
	$expected=~s/\s.*//;	# strip off any whitespace, filename, or "-"
}

if ( $expected !~ $results ) 
{
	print STDERR "Error in test. Results differ from expected values.\n";
	exit 1;
}


print "test successful\n";
exit 0;
