/*******************************************************

This program calculates the gradient directional difference between support vectors belonging to differnt classes.

last modified : Oct 6, 2008

yong.fan@ieee.org
kanterae@uphs.upenn.edu

Ref: P. Golland. Discriminative Direction for Kernel Classifiers. 
Proceedings of NIPS: Advances in Neural Information Processing Systems 14, 745-752, 2002.
*******************************************************/

#include "general.h"
#include "StandardSVM.h"
#include "Kernel.h"
#include "UserKernel.h"
#include "IOTorch.h"
/* this program requires the SVMTorch II package */
#include <getopt.h>

#define EXEC_NAME "COMPARE_SVM_discriminate"

#ifndef RELEASE_ID
#define RELEASE_ID "0.3.1"
#endif

#define SVNFILEVER "$Id: COMPARE_SVM_discriminate.cc 109 2012-03-26 18:44:41Z schuha@UPHS.PENNHEALTH.PRV $"

#ifndef SVN_REV
#define SVN_REV "NO_SVN" 
#endif


void version(){
  printf("\n\n%s Version information\n",EXEC_NAME);
  printf("  Release          : %s\n", RELEASE_ID);
  printf("  Svn Revision     : %s\n", SVN_REV);
  printf("  Svn File Version : %s\n", SVNFILEVER);
}

void usage(){
  printf("\n This program calculates the gradient directional difference between support vectors belonging to differnt classes.\n");

  printf("\n Usage: %s (model_file) (difference_file)",EXEC_NAME);
  printf("\n Usage: %s [-u/h/v]\n",EXEC_NAME);
  
  printf("\n Required arguments:");
  printf("\n   model_file        : SVM model file (output from SVMTorch)");
  printf("\n   difference_file   : output file (ex. COMPARE_diff.txt)\n");
  
  printf("\n Optional arguments:\n");
  printf("\n   [-u      --usage]     Display this message");
  printf("\n   [-h       --help]     Display this message");
  printf("\n   [-v    --version]     Display version information\n");
  
  printf("\n More information:");
  printf("\n   https://www.rad.upenn.edu/sbia/software/");
  printf("\n   sbia-software@uphs.upenn.edu");
  printf("\n\n");
}


void impact(string file_svm, real *diff, int *n_input_dim)
{
  StandardSVM svmmodel;
  
  svmmodel.load(file_svm);

  
  *n_input_dim = svmmodel.n_input_dim;
  
  real *agent = new real [svmmodel.n_input_dim];
  real *direction = new real[svmmodel.n_input_dim];

  for(int i=0; i<svmmodel.n_input_dim; i++)
    diff[i] = 0;
  
  real std;

  ((GaussianKernel *)svmmodel.kernel)->getParametres(std);
  //getParametres(string &u)
  real g = 1./(2*std*std);  
  
  for(int i=0; i<svmmodel.n_support_vectors; i++)
    {
      real step=0.005;
      real magnitude = 0;
      for(int t=0; t<svmmodel.n_input_dim; t++)
	{ 
	
	  agent[t] = svmmodel.data[svmmodel.support_vectors[i]][t];
	  magnitude += agent[t]*agent[t];
	}
      
      magnitude = sqrt(magnitude)+1.0e-6;
      step *= magnitude;
      for(int iter=0; (iter<10000)&&((svmmodel.sv_alpha[svmmodel.support_vectors[i]])*(svmmodel.use(agent))>0); iter++)
	{ 
           for(int t=0; t<svmmodel.n_input_dim; t++)
	      direction[t] = 0;
	    
	   real *tmp_diff = new real [svmmodel.n_input_dim];
           for(int k=0; k<svmmodel.n_support_vectors; k++)
	     {  
	       real xnorm=0;
	       for(int t=0; t<svmmodel.n_input_dim; t++)
                {
	           tmp_diff[t]=agent[t]-svmmodel.data[svmmodel.support_vectors[k]][t];
		   xnorm += tmp_diff[t]*tmp_diff[t];
	        }     
	     	      
	       real expdiff=exp(g*xnorm)*svmmodel.sv_alpha[k]; 
		
	       for(int t=0; t<svmmodel.n_input_dim; t++)
	         direction[t] -= expdiff*tmp_diff[t];
	     }
	   
	   delete [] tmp_diff;
	   
	   real tmpmagnitude=0;  	  
	   for(int t=0; t<svmmodel.n_input_dim; t++)
	    tmpmagnitude += direction[t]*direction[t];
	   tmpmagnitude = sqrt(tmpmagnitude)+1.0e-6; 
	   for(int t=0; t<svmmodel.n_input_dim; t++)
	    agent[t] += step * direction[t]/tmpmagnitude;
	    
	    if(iter==10000-1)
	      printf("\n exceed maximum iteration number\n");
	}
      
     
      for(int t=0; t<svmmodel.n_input_dim; t++)
	diff[t] += fabs(agent[t]-svmmodel.data[svmmodel.support_vectors[i]][t])/magnitude;
      //      cout << "support vector "<< i <<  endl;
    }
  
   
  real maxvalue = diff[0];
  for(int t=1; t<svmmodel.n_input_dim; t++)
     if(maxvalue<diff[t])
       maxvalue=diff[t];
       
  for(int t=0; t<svmmodel.n_input_dim; t++)
    diff[t] /= maxvalue;
    
  delete [] agent;
  delete [] direction;
  return ;
}


int main(int argc, char **argv)
{

  static struct option long_options[] =
  {
    {"usage",       no_argument,            0, 'u'},
    {"help",        no_argument,            0, 'h'},
    {"version",     no_argument,            0, 'v'},
  };

  int c, option_index = 0;

  if(argc!=3){
    while ( (c = getopt_long (argc, argv, "uhv",
                long_options,&option_index)) != -1){
      switch (c)
      {
        case 'v':
          version();
          return 0;

        case 'u':
          usage();
          return 0;

        case 'h':
          usage();
          return 0;

        case '?':
          /* getopt_long already printed an error message. */
          break;

        default:
          return 1;
      }
    }
    usage();
    exit(1);
  }

  
  int ndim;
  real *diff = new real [5000];
  
  impact(argv[argc-2], diff, &ndim);
  
  FILE *fp;
  if((fp=fopen(argv[argc-1],"w"))==NULL)
    {
      printf("\n could not open file %s.\n",argv[argc-1]);
      exit(1);
    }
  for(int i=0; i<ndim; i++)
    fprintf(fp,"%f\t",diff[i]);
  
  delete [] diff;  

  return 0;
}
