/*!
 * \file  COMPARE_check_input.cc
 * \brief This program checks if the inputs are in correct format.
 *
 * This program is used to check if the subject list input files are
 * in correct format for COMPARE package. If the input files are good,
 * the program will generate text files containing information about
 * the training and testing sets.
 *
 * The program call class "ImagesReader" for all function.
 *
 * For copyright information please see Copyright.txt in the root
 * directory of the project.
 *
 * Contact: SBIA Group <sbia-software@uphs.upenn.edu>
 */


#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "nrutil_compare.h"
#include "ImagesReader.h"
#include <getopt.h>

#include "SbiaCOMPAREConfig.h"

#define EXEC_NAME "COMPARE_check_input"


void version(){
  printf("\n\n%s Version information\n",EXEC_NAME);
  printf("  Version:  %s\n", SBIA_COMPARE_VERSION);
#if SBIA_COMPARE_REVISION
  printf("  Revision: %s\n", SBIA_COMPARE_REVISION);
#endif
}


void usage(){
  printf("\n This program is used to check if the subject list input files are in correct format for COMPARE package. If the input files are good, the program will generate text files containing information about the training and testing sets\n");

  printf("\n Usage: %s <train_sub_list> <test_sub_list>\n",EXEC_NAME);
  printf("\n Usage: %s [-u/h/v]\n",EXEC_NAME);
  
  printf("\n Required arguments:");
  printf("\n   <train_sub_list>     Input file for training set. Set to '0' to skip check.");
  printf("\n   <test_sub_list>      Input file for test set. Set to '0' to skip check.\n");
  
  printf("\n Optional arguments:\n");
  printf("\n   [-u      --usage]       Display this message");
  printf("\n   [-h       --help]       Display this message");
  printf("\n   [-v    --version]       Display version information\n");
  
  printf("\n More information:");
  printf("\n   https://www.rad.upenn.edu/sbia/software/");
  printf("\n   sbia-software@uphs.upenn.edu");
  printf("\n\n");
}

int main(int argc, char *argv[])
{
  static struct option long_options[] =
  {
    {"usage",       no_argument,            0, 'u'},
    {"help",        no_argument,            0, 'h'},
    {"version",     no_argument,            0, 'v'},
    {0, 0, 0, 0}
  };

  int c, option_index = 0;

  if(argc!=3){
    while ( (c = getopt_long (argc, argv, "uhv",
                long_options,&option_index)) != -1){
      switch (c)
      {
        case 'v':
          version();
          return 0;

        case 'u':
          usage();
          return 0;

        case 'h':
          usage();
          return 0;

        case '?':
          /* getopt_long already printed an error message. */
          break;

        default:
          return 1;
      }
    }
    usage();
    exit(1);
  }
  
  char* train_file = argv[1];
  char* test_file  = argv[2];

  ImagesReader reader;

  FILE *fp;
  int num_subj, num_feat;
  int xdim, ydim, zdim;
  std::string root;
  if ( strcmp(train_file, "0")!=0 )
  {
	  reader.SetSubjectList(train_file);
	  if (! reader.ParseSubjectList(0))
	  {
		  printf("please check file %s.\n", train_file);
		  exit(1);
	  }
	  if (! reader.CheckInput())
	  {
		  printf("please check images in %s.\n", train_file);
		  exit(1);
	  }
	  
	  num_subj = reader.GetNumberOfSubjects();
	  num_feat = reader.GetNumberOfFeatures();
	  xdim = reader.GetImageDimensions()[0];
	  ydim = reader.GetImageDimensions()[1];
	  zdim = reader.GetImageDimensions()[2];
	  root = reader.GetDataRoot();
  
	  if((fp=fopen("train_para.txt","w"))==NULL)
	  {
		  printf("Check storage space!!!\n");
		  exit(1);
	  }
	  fprintf(fp,"%s\n%d\n%d\n%d\n%d\n%d",root.c_str(),num_subj,num_feat,xdim,ydim,zdim);
	  fclose(fp);
  
  }
  if ( strcmp(test_file, "0")!=0 )
  {
	  reader.SetSubjectList(test_file);
	  if (! reader.ParseSubjectList(1))
	  {
		  printf("please check file %s.\n",test_file);
		  exit(1);
	  }
	  if (! reader.CheckInput())
	  {
		  printf("please check images in %s.\n", test_file);
		  exit(1);
	  }
	  int test_num_subj, test_num_feat;
	  int test_xdim, test_ydim, test_zdim;
	  std::string test_root;
	  test_num_subj = reader.GetNumberOfSubjects();
	  test_num_feat = reader.GetNumberOfFeatures();
	  test_xdim = reader.GetImageDimensions()[0];
	  test_ydim = reader.GetImageDimensions()[1];
	  test_zdim = reader.GetImageDimensions()[2];
	  test_root = reader.GetDataRoot();
	  
	  if ( strcmp(train_file,"0")!=0 && (test_num_feat!=num_feat || test_xdim!=xdim || test_ydim!=ydim || test_zdim!=zdim))
	  {
		  printf("\n train list and test list do not match.\n");
		  exit(1);
	  }

	  if((fp=fopen("test_para.txt","w"))==NULL)
  	  {
		  printf("Check storage space!!!\n");
		  exit(1);
  	  }
    	  fprintf(fp,"%s\n%d\n%d\n%d\n%d\n%d",test_root.c_str(),test_num_subj,num_feat,xdim,ydim,zdim);
    	  fclose(fp);
  }
  
  return 0;
}




/*  FILE *fp;
  if((fp=fopen(train_file,"r"))==NULL)
  {
      printf("\n cannot open file %s.\n",train_file);
      exit(1);
  }

  int num_subj, num_feat;
  int xdim, ydim, zdim;
  char root[10240];
  
  if(fscanf(fp,"%d %d",&num_subj,&num_feat)!=2)
    {
      printf("\n please check file %s.\n",train_file);
      exit(1);
    }
  if(fscanf(fp,"%d %d %d",&xdim,&ydim,&zdim)!=3)
    {
      printf("\n please check file %s.\n",train_file);
      exit(1);
    }
  if(fscanf(fp,"%s",root)!=1)
    {
      printf("\n please check file %s.\n",train_file);
      exit(1);
    }

  char ***file_name = cc3tensor(num_subj,num_feat,1024);
  int  *label  = ivector(num_subj);
  for(int i=0; i<num_subj; i++)
    {
      for(int j=0; j<num_feat; j++)
	{
	  if(fscanf(fp,"%s", file_name[i][j])!=1)
	     {
	       printf("\n please check file %s.\n",train_file);
	       exit(1);
	     }
	}
      if(fscanf(fp,"%d", &label[i])!=1)
	{
	  printf("\n please check file %s.\n",train_file);
	  exit(1);
	}
    }
  fclose(fp);

  char fname[1024];
  for(int i=0; i<num_feat; i++)
    {
      sprintf(fname,"train_sub_%d.lst",i);
      if((fp=fopen(fname,"w"))==NULL)
	{
	  printf("Check storage space!!!\n");
	  exit(1);
	}
      for(int j=0; j<num_subj; j++)
	fprintf(fp,"%s\t%d\n",file_name[j][i],label[j]);
      fclose(fp);
    }
  free_cc3tensor(file_name,num_subj,num_feat,1024);
  free_ivector(label,num_subj);

  int test_num_subj, test_num_feat;
  int test_xdim, test_ydim, test_zdim;	       
  char test_root[1024];

  if(strcmp(test_file,"0")!=0)
    {
      if((fp=fopen(test_file,"r"))==NULL)
	{
	  printf("\n cannot open file %s\n.",test_file);
	  exit(1);
	}
      
      if(fscanf(fp,"%d %d",&test_num_subj,&test_num_feat)!=2)
	{
	  printf("\n please check file %s.\n",test_file);
	  exit(1);
	}
      if(test_num_feat!=num_feat)
	{
	  printf("\n please check train file and test file\n");
	  exit(1);
	}
      
      if(fscanf(fp,"%d %d %d",&test_xdim,&test_ydim,&test_zdim)!=3)
	{
	  printf("\n please check file %s.\n",test_file);
	  exit(1);
	}
      
      if((test_xdim!=xdim)||(test_ydim!=ydim)||(test_zdim!=zdim))
	{
	  printf("\n please check train file and test file\n");
	  exit(1);
	}

      if(fscanf(fp,"%s",test_root)!=1)
	{
	  printf("\n please check file %s.\n",test_file);
	  exit(1);
	}
      
      file_name = cc3tensor(test_num_subj,test_num_feat,1024);
      label  = ivector(test_num_subj);
      for(int i=0; i<test_num_subj; i++)
	{
	  for(int j=0; j<num_feat; j++)
	    {
	      if(fscanf(fp,"%s", file_name[i][j])!=1)
		{
		  printf("\n please check file %s.\n",test_file);
		  exit(1);
		}
	    }
	  if(fscanf(fp,"%d", &label[i])!=1)
	    {
	      printf("\n please check file %s.\n",test_file);
	      exit(1);
	    }
	}
      fclose(fp);    
      for(int i=0; i<num_feat; i++)
	{
	  sprintf(fname,"test_sub_%d.lst",i);
	  if((fp=fopen(fname,"w"))==NULL)
	    {
	      printf("Check storage space!!!\n");
	      exit(1);
	    }
	  for(int j=0; j<test_num_subj; j++)
	    fprintf(fp,"%s\t%d\n",file_name[j][i],label[j]);
	  fclose(fp);
	}
    }

  if((fp=fopen("train_para.txt","w"))==NULL)
  {
  	  printf("Check storage space!!!\n");
    	  exit(1);
  }
  fprintf(fp,"%s\n%d\n%d\n%d\n%d\n%d",root,num_subj,num_feat,xdim,ydim,zdim);
  fclose(fp);
  
  if(strcmp(test_file,"0")!=0)
  {
    	  if((fp=fopen("test_para.txt","w"))==NULL)
  	  {
		  printf("Check storage space!!!\n");
		  exit(1);
  	  }
    	  fprintf(fp,"%s\n%d\n%d\n%d\n%d\n%d",test_root,test_num_subj,num_feat,xdim,ydim,zdim);
    	  fclose(fp);
  }
  
  return 0;
}*/
