/*!
 * \file  COMPARE_extract.cc
 * \brief This program extracts discriminative regional features by learning from training samples.
 *
 * For copyright information please see Copyright.txt in the root
 * directory of the project.
 *
 * Contact: SBIA Group <sbia-software@uphs.upenn.edu>
 */


#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "FeatureExtraction.h"
#include <getopt.h>

#include "SbiaCOMPAREConfig.h"

#define EXEC_NAME "COMPARE_extract"


void version(){
  printf("\n\n%s Version information\n",EXEC_NAME);
  printf("  Version:  %s\n", SBIA_COMPARE_VERSION);
#if SBIA_COMPARE_REVISION
  printf("  Revision: %s\n", SBIA_COMPARE_REVISION);
#endif
}

void usage(){
  printf("\n This program extracts discriminative regional features by learning from training samples.\n");

  printf("\n Usage: %s (subject_list) (loo_id) [options]\n",EXEC_NAME);
  
  printf("\n Required arguments:");
  printf("\n   subject_list    : input file for training");
  printf("\n   loo_id          : index of individual to use from the training set (starting at 0)\n");
  
  printf("\n Optional arguments:\n");
  printf("\n   [-sigma       value]    Gaussian smoothing kernel size (default:3.0)");
  printf("\n   [-u      --usage]     Display this message");
  printf("\n   [-h       --help]     Display this message");
  printf("\n   [-v    --version]     Display version information\n");
  
  printf("\n More information:");
  printf("\n   https://www.rad.upenn.edu/sbia/software/");
  printf("\n   sbia-software@uphs.upenn.edu");
  printf("\n\n");
}

int main(int argc, char *argv[])
{

  static struct option long_options[] =
  {
    {"usage",       no_argument,            0, 'u'},
    {"help",        no_argument,            0, 'h'},
    {"version",     no_argument,            0, 'v'},
    {0, 0, 0, 0}
  };

  int c, option_index = 0;

  if(argc<3){
    while ( (c = getopt_long (argc, argv, "uhv",
                long_options,&option_index)) != -1){
      switch (c)
      {
        case 'v':
          version();
          return 0;

        case 'u':
          usage();
          return 0;

        case 'h':
          usage();
          return 0;

        case '?':
          /* getopt_long already printed an error message. */
          break;

        default:
          return 1;
      }
    }
    usage();
    exit(1);
  }
 
  char *sublist  = argv[1];
  argc--;
  argv++;
  int subject_id      = atoi(argv[1]);
  argc--;
  argv++;

  long num_feature    = 1800;
  float smooth_size   = 3.0;
  float p             = 1.0;
  
  while(argc>1){
    if(strcmp(argv[1],"-sigma")==0){
	    argc--;
	    argv++;
	    smooth_size = atof(argv[1]);
	    argc--;
	    argv++;
	  }
    else if((strcmp(argv[1],"-v")==0)||(strcmp(argv[1],"-version")==0)){
	    argc--;
	    argv++;
      version();
      exit(0);
	  }
    else{
	    usage();
      exit(1);
    }
  }
  
  printf("\n The input options are below:"); 
  printf("\n ===============================================\n");
  printf("\n sublist:      %s",sublist);
  printf("\n subject_id:   %d",subject_id);
  printf("\n smooth_size:  %f",smooth_size);
  
  printf("\n ==============================================\n");
  
  FeatureExtraction *featExt = new FeatureExtraction;  
  
  featExt->SetInput (sublist, subject_id, num_feature, smooth_size, p);
  featExt->Run();
  
  return 0;
}
  

