#!/bin/bash

##############################################################################
# \file  Compare.sh
# \brief This is the script to test a classifier of COMPARE software.
#
# For copyright information please see Copyright.txt in the root
# directory of the project.
#
# Contact: SBIA Group <sbia-software@uphs.upenn.edu>
##############################################################################

EXEC_NAME="Compare_test";
VERSION="@PROJECT_VERSION@";
REVISION="@PROJECT_REVISION@";

# Set the PATH to include the libexec directory for the package...this should be relative to the installation

# function replacing relative path with absolute path
makefullpath(){
	if [ "`echo $1 | grep '^/.*'`" == "" ]; then
		echo "$2/$1"
	else
		echo $1
	fi
}

# record current directory
CurrentDir=`pwd`

libexec=`dirname $0`
libexec=$(makefullpath $libexec $CurrentDir)
if [ ! -d $libexec ] ; then
	echo "Could not find libexec directory at \"$libexec\"" 1>&2 
	exit 1
fi

# check for required executables
for exe in COMPARE_extract COMPARE_check_input COMPARE_average_score COMPARE_build COMPARE_build_loo COMPARE_build_check COMPARE_SVM_discriminate COMPARE_map_svm_diff_to_region COMPARE_float_sum COMPARE_model
do
	if [ ! -e ${libexec}/$exe ] ; then
		echo "Could not find required executable \"$exe\" in \"$libexec\"" 1>&2
		exit 1
	fi
done

# We're OK to go on...
export PATH="${libexec}:${PATH}"

version(){
  echo " $EXEC_NAME Version information";
  echo "  Version:  $VERSION";
  if [ ! -z "$REVISION" ]; then
  echo "  Revision: $REVISION";
  fi
}

usage() {
  echo
  echo "  This program classifies the samples in data_list_file using model_file, and outputs SVM scores to result_file";
  echo
  echo "  Usage: $EXEC_NAME -M <model_file> -I <data_list_file> -O <result_file>";
  echo "  Usage: $EXEC_NAME [-v/h/u]";
	echo
  echo "  Required parameters:";
  echo "    [-M   model_file     ]  the classifier constructed by Compare";
  echo "    [-I   data_list_file ]  input file containing the test set";
  echo "    [-O   result_file    ]  classification scores for each input subject";
  echo
  echo "  Optional parameters:";
  echo "    [-v]   version information";
  echo "    [-h]   this message";
  echo "    [-u]   this message";
  echo
  echo "  More information:";
  echo "    https://www.rad.upenn.edu/sbia/software/";
  echo "    sbia-software@uphs.upenn.edu";
  echo
}
checkandexit(){
  if [ $1 != 0 ] ; then
    # There was an error, bail out!
    echo "$2" 1>&2
    if [ "$tmpdir" != "" ]; then
	    /bin/cp $tmpdir/a.log $CurrentDir/COMPARE_test_crash.log
	    echo " See the whole log file COMPARE_test_crash.log at $CurrentDir."
    fi
    exit $1
  fi
}
# clean up temporary files on exit
trap cleanup EXIT
function cleanup(){
	if [ "$tmpdir" != "" ]; then
		/bin/rm -rf $tmpdir
		echo " cleaned."
	fi
}

model_file="";
test_file="";
result_file="";
tmpdir="";

while getopts  M:I:O:vhu opt
do
 case "$opt" in
    M) model_file="$OPTARG";;
    I) test_file="$OPTARG";;
    O) result_file="$OPTARG";;
    v) version; exit 0;;
    h) usage; exit 0;;
    u) usage; exit 0;;
    *) usage; exit 1;;
  esac
done

if [ "$model_file" == "" ]; then
	usage
	echo "Need model file!" 1>&2
	exit 1
fi
if [ "$test_file" == "" ]; then
	usage
	echo "Need test file!" 1>&2
	exit 1
fi
if [ "$result_file" == "" ]; then
	usage
	echo "Result file is not defined!" 1>&2
	exit 1
fi



# make temporary directory
if [ -d "${SBIA_TMPDIR}" ]; then
	tmpdir=`mktemp -d ${SBIA_TMPDIR}/COMPARE_XXXXXX`;
	checkandexit $? "Failed to make a temporary directory in ${SBIA_TMPDIR}!"
else
	tmpdir=`mktemp -d -p "/tmp/" COMPARE_XXXXXX`;
	checkandexit $? "Failed to make a temporary directory in /tmp !"
fi

# COMPARE_test will work in the temporary directory
cd $tmpdir
checkandexit $? "Cannot work in $tmpdir!"
echo " Using temporary directory $tmpdir"

# function checking output path and creating it if not exist
checkpath(){
	name_path=${1%/*}
	if [ ! -d "$name_path" ]; then
		/bin/mkdir -p $name_path
		checkandexit $? "Cannot write files in $name_path!"

	fi
}
model_file=$(makefullpath $model_file $CurrentDir)
test_file=$(makefullpath $test_file $CurrentDir)
result_file=$(makefullpath $result_file $CurrentDir)
checkpath $result_file



echo
echo " checking model file ...";

#decompress the model files
if [ -e $model_file ]
 then
  tar -zxvf $model_file > a.log
 else
  echo "Could not open model file $model_file.";
  exit 1;
fi

train_para="train_para.txt";
if [ -e $train_para ]
 then
    for i in 1 2 3 4 5 6
    do
     if [ $i -eq 1 ]; then 
	    read train_root_dir;
     fi
   
     if [ $i -eq 2 ]; then 
	    read train_num_subj 
     fi
    
     if [ $i -eq 3 ]; then
	    read train_num_feat
     fi

     if [ $i -eq 4 ]; then
	    read xdim  
     fi
    
     if [ $i -eq 5 ]; then
	    read ydim  
     fi

     if [ $i -eq 6 ]; then
	    read zdim 
     fi
    done < $train_para
 else
   echo "could not open input of model";
   exit 1;
fi   

rm $train_para

echo " checking input ...";
COMPARE_check_input 0 $test_file > a.log
checkandexit $? "Error checking input!"

base_dir=`pwd`;
train_file_base="r_roi_train";
test_file_base="r_roi_test";

echo " classifying testing subjects ...";
#test models on test data set

test_para="test_para.txt";
    if [ -e $test_para ]
	then
	    for i in 1 2 3 4 5 6
	    do
	    if [ $i -eq 1 ]; then 
		    read test_root_dir;
	    fi
	    
	    if [ $i -eq 2 ]; then 
		    read test_num_subj 
	    fi
	    
	    if [ $i -eq 3 ]; then
		    read test_num_feat
	    fi
	    
	    if [ $i -eq 4 ]; then
		    read xdim  
		  fi
		
	    if [ $i -eq 5 ]; then
		    read ydim  
	    fi
	    
	    if [ $i -eq 6 ]; then
		    read zdim 
	    fi
	    done < $test_para
	 else
	    echo "could not open input for testing";
	    exit 1;
    fi   

start_id=0;
end_id=$(( $test_num_feat - 1 ));
num_subject=$test_num_subj;
data_root_dir=$test_root_dir;

i=0

while [ $i -lt $train_num_subj ]
do
 	base_dir=`pwd`;
 	feature_location_file="${base_dir}/r_roi_feature_location_${i}.txt";
 	region_file="${base_dir}/r_roi_region_${i}";
 	feature_order="${base_dir}/order_${i}.bin";
 	svm_model_file="${base_dir}/svm_model_${i}.bin";
	
 	COMPARE_model $test_file $region_file $feature_location_file $feature_order $svm_model_file > a.log
 	checkandexit $? "Error classifying test subjects: `cat a.log | tail -1`!"
 	
 	test_score_file="test_score_file_${i}.txt";
 	mv svm_score.bin $test_score_file
    	
 	i=$(( $i + 1 ));
done
    
test_result_file=${result_file##*/};
COMPARE_average_score test_score_file $train_num_subj $num_subject $test_result_file > a.log
checkandexit $? "Error averaging test scores: `cat a.log | tail -1`!"
     
mv $test_result_file $result_file

/bin/rm -rf $tmpdir

echo " finished.";

exit 0;
