/*!
 * \file  FeatureExtraction.cc
 * \brief Feature Extraction Class.
 *
 * For copyright information please see Copyright.txt in the root
 * directory of the project.
 *
 * Contact: SBIA Group <sbia-software@uphs.upenn.edu>
 */

#include "FeatureExtraction.h"

FeatureExtraction::FeatureExtraction()
{
  // Default parameters for debugging
  _DebugFlag = True;

}

FeatureExtraction::~FeatureExtraction()
{
}

void FeatureExtraction :: SetInput (char *sublist, int subject_id, long num_feature, float smooth_size, float p)
{
  this->Debug("SetInput");
  
  _file_sublist = sublist;
  
  _subject_id       = subject_id;
  _num_classification_feature = num_feature;

  _smooth_size   = smooth_size;
  
  _p    = p;
}

void FeatureExtraction::GetInput()
{
  this->Debug("GetInput");

  ImagesReader reader;
  
  reader.SetSubjectList(_file_sublist);
  if (!reader.ParseSubjectList(0))
  {
	  printf("Subject list file parsing fails.\n");
	  exit(1);
  }
  _num_subject = reader.GetNumberOfSubjects();
  _num_feature = reader.GetNumberOfFeatures();
  _xdim = reader.GetImageDimensions()[0];
  _ydim = reader.GetImageDimensions()[1];
  _zdim = reader.GetImageDimensions()[2];

  _label = vector(_num_subject);
  reader.GetLabels(_label);
  
  _num_po_subject = 0;
  _num_ne_subject = 0;
  for(int i=0; i<_num_subject; i++)
  {
    	  if(_label[i]>0)
	  	  _num_po_subject += 1;
    	  else
	  	  _num_ne_subject += 1;
  }   
}

void FeatureExtraction::Initialize()
{

  this->Debug("Initialize");
  
  _im_size = _xdim * _ydim * _zdim;

  _feature_data = f4tensor(_num_subject, _xdim, _ydim, _zdim);
  _score  = f4tensor(_num_feature, _xdim, _ydim, _zdim);
  _loo_score  = f4tensor(_num_feature, _xdim, _ydim, _zdim);
  _spatial_score = f4tensor(_num_feature, _xdim, _ydim, _zdim);
  
  _max_region_size = 3000;
  _w_region = s4tensor(_num_feature, _xdim, _ydim, _zdim);
  _r_region = s4tensor(_num_feature, _xdim, _ydim, _zdim);
  
  _w_mean  = f4tensor(_num_feature, _xdim, _ydim, _zdim);
  _r_mean  = f4tensor(_num_feature, _xdim, _ydim, _zdim);
  
  _region_seed = s3tensor(_num_feature, _max_region_size, 3);
  
  _w_roi_feature_location    = imatrix(_num_classification_feature,2);
  _r_roi_feature_location    = imatrix(_num_classification_feature,2);
  
  _w_roi_feature = matrix(_num_subject, _num_classification_feature);
  _r_roi_feature = matrix(_num_subject, _num_classification_feature);

  
  _w_roi_volume = f3tensor(_num_feature, _num_subject,  _num_classification_feature);
  _w_roi_size   = matrix(_num_feature, _num_classification_feature);
  _w_roi_score  = matrix(_num_feature, _num_classification_feature);

  _r_roi_volume = f3tensor(_num_feature, _num_subject,  _num_classification_feature);
  _r_roi_size   = matrix(_num_feature, _num_classification_feature);
  _r_roi_score  = matrix(_num_feature, _num_classification_feature);

  for(int i=0; i<_num_feature; i++)
    for(int k=0; k<_num_subject; k++)
      for(int l=0; l<_num_classification_feature; l++)
	_w_roi_volume[i][k][l]= _r_roi_volume[i][k][l] = 0;
  
  
  for(int i=0; i<_num_feature; i++)
    for(int k=0; k<_num_classification_feature; k++)
      _w_roi_size[i][k] = _r_roi_size[i][k] = 0;
  
  _region_size = ivector(_num_feature);

  _background = s4tensor(_num_feature, _xdim, _ydim, _zdim);
  for(int l=0; l<_num_feature; l++)
    for(int k=0; k<_zdim; k++)
      for(int j=0; j<_ydim; j++)
	for(int i=0; i<_xdim; i++)
	  _background[l][i][j][k] = 0;

  _whitten_para = f3tensor(4,_num_classification_feature, 2);
}

void FeatureExtraction::Finalize()
{

  this->Debug("Finalize");

  free_f4tensor(_score, _num_feature, _xdim, _ydim, _zdim);
  free_f4tensor(_loo_score, _num_feature, _xdim, _ydim, _zdim);
  free_f4tensor(_spatial_score, _num_feature, _xdim, _ydim, _zdim);
  
  free_s4tensor(_w_region,_num_feature,  _xdim, _ydim, _zdim);
  free_f4tensor(_w_mean, _num_feature, _xdim, _ydim, _zdim);
  free_s4tensor(_r_region,_num_feature,  _xdim, _ydim, _zdim);
  free_f4tensor(_r_mean, _num_feature, _xdim, _ydim, _zdim);
  free_s3tensor(_region_seed,_num_feature, _max_region_size, 3);
  
  free_f4tensor(_feature_data, _num_subject, _xdim, _ydim, _zdim);
  
  free_imatrix(_w_roi_feature_location, _num_classification_feature, 2);
  free_imatrix(_r_roi_feature_location, _num_classification_feature, 2);
  
  free_matrix(_w_roi_feature,_num_subject, _num_classification_feature);
  free_matrix(_r_roi_feature,_num_subject, _num_classification_feature);
  
  free_f3tensor(_w_roi_volume,_num_feature, _num_subject,  _num_classification_feature);
  free_matrix(_w_roi_size,_num_feature,_num_classification_feature);
  free_matrix(_w_roi_score, _num_feature, _num_classification_feature);
  
  free_f3tensor(_r_roi_volume,_num_feature, _num_subject,  _num_classification_feature);
  free_matrix(_r_roi_size, _num_feature,_num_classification_feature);
  free_matrix(_r_roi_score, _num_feature, _num_classification_feature);
  
  free_ivector(_region_size, _num_feature);
  free_s4tensor(_background, _num_feature, _xdim, _ydim, _zdim);
  free_f3tensor(_whitten_para, 4, _num_classification_feature, 2);
  
}

void FeatureExtraction::ReadData (int feat)
{
  this->Debug("ReadData");
  
  ImagesReader reader;
  reader.SetSubjectList(_file_sublist);
  reader.ParseSubjectList(0);
  if (!reader.ReadImages(_feature_data, feat))
  {
	  printf("Image reading process failed.\n");
	  exit(1);
  }

}

void FeatureExtraction::FeatureWhitten0()
{
  this->Debug("FeatureWhitten");
  int len = _num_subject -1;
  
  float *x = vector(len);
  float *y = vector(len);
  
  float mean,var;
  for(int kk=0; kk<_zdim; kk++)
    for(int jj=0; jj<_ydim; jj++)
      for(int ii=0; ii<_xdim; ii++)
	{
	  int indx=0;
	  for(int ll=0; ll<_num_subject; ll++)
            {
              if( ll != _subject_id)
                {
                  x[indx] = _feature_data[ll][ii][jj][kk];
                  
                  y[indx] = _label[ll];
                  
                  indx+=1;
                }
            }
	  
	  if( IsValidVector(x,y,len) > 0 )
	    {
	      mean  = 0; var = 0;
	      for(int i=0; i<len; i++)
		mean += x[i];
	      mean /= (float)(len);
	      
	      for(int i=0; i<len; i++)
		{
		  float tmp = x[i] - mean;
		  var += tmp*tmp;
		}
	      var = sqrt(var/(len-1))+1.0e-6;
	      
	      for(int ll=0; ll<_num_subject; ll++)
		_feature_data[ll][ii][jj][kk]= (_feature_data[ll][ii][jj][kk]-mean)/var;
	      
	    }
	}
  free_vector(x,len);
  free_vector(y,len);
}

void FeatureExtraction::ComputeLOOScore(int feat)
{
  this->Debug("Compute LOO Score");
  
  int len = _num_subject -1;
  
  float *x = vector(len);
  float *y = vector(len);
  
  for(int kk=0; kk<_zdim; kk++)
    for(int jj=0; jj<_ydim; jj++)
      for(int ii=0; ii<_xdim; ii++)
	{
	  int indx=0;
	  for(int ll=0; ll<_num_subject; ll++)
	    {
	      if( ll != _subject_id)
		{
		  x[indx] = _feature_data[ll][ii][jj][kk];
		  
		  y[indx] = _label[ll];
		  
		  indx+=1;
		}
	    }
	  
	  if( IsValidVector(x,y,len) > 0 )
	    _loo_score[feat][ii][jj][kk]  = LOOPearsonCorrelation(x,y,len);
	  else
	    _loo_score[feat][ii][jj][kk]  = 0;
	}  

/*  FILE *fp;
  char fn[1024];
  sprintf(fn,"loo_score_%d_%d.bin",_subject_id,feat);
  if((fp=fopen(fn,"w"))==NULL)
    {
      printf("\n cannot open file %s.\n",fn);
      exit(1);
    }
  fprintf(fp, "%d\t%d\t%d\n", _xdim, _ydim, _zdim);
  for(int kk=0; kk<_zdim; kk++)
    for(int jj=0; jj<_ydim; jj++)
      for(int ii=0; ii<_xdim; ii++)
	fprintf(fp, "%f\t", _loo_score[feat][ii][jj][kk]);
//	fwrite(&_score[feat][i][j][k],sizeof(float),1,fp);
  fclose(fp);
*/  
}

void FeatureExtraction::ComputeSpatialScore(int feat)
{
  this->Debug("Compute Spatial Score");
  
  int len = _num_subject -1;
  
  float *x = vector(len);
  float *y = vector(len);
  int nhsize = 1;
  int datasize = (2*nhsize+1)*(2*nhsize+1)*(2*nhsize+1);
  float **nh = matrix(len, datasize);
  
  for(int kk=nhsize; kk<_zdim-nhsize; kk++)
    for(int jj=nhsize; jj<_ydim-nhsize; jj++)
      for(int ii=nhsize; ii<_xdim-nhsize; ii++)
	{
	  int indx=0;
	  for(int ll=0; ll<_num_subject; ll++)
	    {
	      if( ll != _subject_id)
		{
		  x[indx] = _feature_data[ll][ii][jj][kk];
		  
		  y[indx] = _label[ll];
		  
		  indx+=1;
		}
	    }
	  
	  int isvalid = IsValidVector(x,y,len);
	  if( isvalid > 0 )
	    {
	      int indx = 0;
	      for(int ll=0; ll<_num_subject; ll++)
		{
		  if( ll!= _subject_id)
		    {
		      int indx0 = 0;
		      for(int k=-nhsize; k<=nhsize; k++)
			for(int j=-nhsize; j<=nhsize; j++)
			  for(int i=-nhsize; i<=nhsize; i++)
			    nh[indx][indx0++] = _feature_data[ll][ii+i][jj+j][kk+k];                            
		      
		      indx+=1;
		      //printf("\n index0 = %d",indx0);
		    }
		}         
	      
	      int num_valid = 0;
	      for(int i=0; i<datasize; i++)
		{
		  for(int j=0; j<len; j++)
		    x[j] = nh[j][i];
		  
		  num_valid += IsValidVector(x,y,len);
		}
	      
	      if(num_valid == datasize)
		_spatial_score[feat][ii][jj][kk] = IntraClass(nh,len,datasize);
	      else
		_spatial_score[feat][ii][jj][kk] = 0;
	    }
	  else
	    _spatial_score[feat][ii][jj][kk] = 0;    
	  
	  if(_spatial_score[feat][ii][jj][kk]<0)
	    _spatial_score[feat][ii][jj][kk] = 0;
	}  

/*  FILE *fp;
  char fn[1024];
  sprintf(fn,"spatial_score_%d_%d.bin",_subject_id,feat);
  if((fp=fopen(fn,"w"))==NULL)
    {
      printf("\n cannot open file %s.\n",fn);
      exit(1);
    }
  fprintf(fp, "%d\t%d\t%d\n", _xdim, _ydim, _zdim);
  for(int kk=0; kk<_zdim; kk++)
    for(int jj=0; jj<_ydim; jj++)
      for(int ii=0; ii<_xdim; ii++)
	fprintf(fp, "%f\t", _spatial_score[feat][ii][jj][kk]);
//	fwrite(&_score[feat][i][j][k],sizeof(float),1,fp);
  fclose(fp);
*/ 
  
  free_matrix(nh,len,datasize);
  free_vector(x,len);
  free_vector(y,len);
}

void FeatureExtraction::CombineScore(int feat)
{  
  for(int kk=0; kk<_zdim; kk++)
    for(int jj=0; jj<_ydim; jj++) 
      for(int ii=0; ii<_xdim; ii++)
	_score[feat][ii][jj][kk] = _loo_score[feat][ii][jj][kk]*pow(_spatial_score[feat][ii][jj][kk],_p);  

/*
  FILE *fp;
  char fn[1024];
  sprintf(fn,"Score_%d_%d.bin",_subject_id,feat);
  if((fp=fopen(fn,"w"))==NULL)
    {
      printf("\n cannot open file %s.\n",fn);
      exit(1);
    }
  fprintf(fp, "%d\t%d\t%d\n", _xdim, _ydim, _zdim);
  for(int k=0; k<_zdim; k++)
    for(int j=0; j<_ydim; j++)
      for(int i=0; i<_xdim; i++)
	fprintf(fp, "%f\t", _score[feat][i][j][k]);
//	fwrite(&_score[feat][i][j][k],sizeof(float),1,fp);
  fclose(fp);
*/
}



void FeatureExtraction::ComputeRegionFeat(int feat)
{
  this->Debug("Compute Region Feature");
  // get the region information using watershed segmentation method
  
  for(int k=0; k<_max_region_size; k++)
    for(int l=0; l<3; l++)
      _region_seed[feat][k][l] = 0;
  
  for(int k=0; k<_zdim; k++)
    for(int j=0; j<_ydim; j++)
      for(int i=0; i<_xdim; i++)
	_w_region[feat][i][j][k] = 0; 
  
  watershed(_score[feat], _xdim, _ydim, _zdim, _w_region[feat],_smooth_size);
  
  float background = _w_region[feat][0][0][0];
  for(int k=0; k<_zdim; k++)
    for(int j=0; j<_ydim; j++)
      for(int i=0; i<_xdim; i++)
	{
	  if(_w_region[feat][i][j][k] == background)
	    {
	      _w_region[feat][i][j][k] = 0;
	      _background[feat][i][j][k] = 1;
	    }
	  else if((_w_region[feat][i][j][k] > background)&&(background >0))
	    _w_region[feat][i][j][k] -= 1;
	}
  
//  fill the regions with their means
//  watershed_post(_score[feat], _w_region[feat], _w_mean[feat], _xdim, _ydim, _zdim);
  int num_region = 0;
  for(int k=0; k<_zdim; k++)
    for(int j=0; j<_ydim; j++)
      for(int i=0; i<_xdim; i++)
	if(num_region<_w_region[feat][i][j][k])
	  num_region = (int)_w_region[feat][i][j][k];


/*  FILE *fp;
  char fn[1024];
  sprintf(fn,"Watershed_%d_%d.bin",_subject_id,feat);
  if((fp=fopen(fn,"w"))==NULL)
    {
      printf("\n cannot open file %s.\n",fn);
      exit(1);
    }
  fprintf(fp, "%d\n", num_region);
  for(int k=0; k<_zdim; k++)
    for(int j=0; j<_ydim; j++)
      for(int i=0; i<_xdim; i++)
	      fprintf(fp, "%d\t", (int)_w_region[feat][i][j][k]);
//	fwrite(&_score[feat][i][j][k],sizeof(float),1,fp);
  fclose(fp);

*/




  num_region = (num_region > _max_region_size ? _max_region_size : num_region);	
  _region_size[feat] = num_region;
  
  float *max       = vector(num_region);
  int   *region_id = ivector(num_region);
//  int ** tmp_region_feat_loca = imatrix(num_region, 3);
  
  for(int i=0; i<num_region; i++)
    {
      max[i] = 0;
      region_id[i] = i+1;
//      tmp_region_feat_loca[i][0] = 0;
//      tmp_region_feat_loca[i][1] = 0;
//      tmp_region_feat_loca[i][2] = 0; 
    }

  for(int k=0; k<_zdim; k++)
    for(int j=0; j<_ydim; j++)
      for(int i=0; i<_xdim; i++)
	{
	  for(int rr=0; rr<num_region; rr++)
	    {
	      if(((int)_w_region[feat][i][j][k] == region_id[rr]) && (fabsf(_score[feat][i][j][k]) > max[rr]))
		{
		  max[rr] = fabsf(_score[feat][i][j][k]);
//		  tmp_region_feat_loca[rr][0] = i;
//		  tmp_region_feat_loca[rr][1] = j;
//		  tmp_region_feat_loca[rr][2] = k;

		  _region_seed[feat][rr][0] = i;
		  _region_seed[feat][rr][1] = j;
		  _region_seed[feat][rr][2] = k;
		}
	    }
	}
  //unsigned long *index_max = lvector(num_region);
  //indexx(num_region, max, index_max);
  
  free_ivector(region_id, num_region);
  free_vector(max, num_region);
  //free_imatrix(tmp_region_feat_loca, num_region,3);
  //free_lvector(index_max, num_region);
}


void FeatureExtraction::RegionGrow(int feat)
{
  this->Debug("RegionGrow");
  
  int ***xc; // voxels that have already been allocated to a region have flag 1.
  int ***br; // buffer containing border information. Border voxels are labelled by their region number.
  
  xc = i3tensor(_xdim, _ydim, _zdim);
  br = i3tensor(_xdim, _ydim, _zdim);
  
  int n = _region_size[feat];
  
  //initialize arrays seed, br, xc, and arrays mp, m, and s
  for(int k=0; k<_zdim; k++)
    for(int j=0; j<_ydim; j++)
      for(int i=0; i<_xdim; i++)
	_r_region[feat][i][j][k] = 0;
  for(int i=0; i<n ; i++)
    {
      int ii = _region_seed[feat][i][0];
      int jj = _region_seed[feat][i][1];
      int kk = _region_seed[feat][i][2];
      
      _r_region[feat][ii][jj][kk] = i+1;
    }
  
  for(int k=0; k<_zdim; k++)
    for(int j=0; j<_ydim; j++)
      for(int i=0; i<_xdim; i++)
	{
	  br[i][j][k] = 0;
	  xc[i][j][k] = 0;
	}
  
  float *xx = vector(_num_subject-1);
  float *yy = vector(_num_subject-1);
  int idx = 0;
  for(int i=0; i<_num_subject; i++)
    {
      if(i!=_subject_id)
	yy[idx++] = _label[i];
    }
  
  // store sum of voxel before and after region grow
  float **sum1 = matrix(_num_subject, n);
  float **sum2 = matrix(_num_subject, n); 
  // store region size before and after region grow
  int *np = ivector(n);
  // store region score before and after region grow
  float *sc1 = vector(n);
  float *sc2 = vector(n);
  
  for(int i=0; i<n; i++)
    {
      int ii = _region_seed[feat][i][0];
      int jj = _region_seed[feat][i][1];
      int kk = _region_seed[feat][i][2];
      idx = 0;
      for(int j=0; j<_num_subject; j++)
	{
	  if(j!=_subject_id)
	    {
	      sum1[idx][i] = sum2[idx][i]= xx[idx] = _feature_data[j][ii][jj][kk];
	      idx += 1;
	    } 
	}
      sc1[i] = sc2[i] = LOOPearsonCorrelation(xx,yy,_num_subject -1);
      np[i] =1;
    }
  
  // use seeds from image seed to initialize br, np, m, and s
  for(int k =0; k<_zdim; k++)
    for(int j=0; j<_ydim; j++)
      for(int i=0; i<_xdim; i++)
	{
	  if(_r_region[feat][i][j][k] != 0)
	    {
	      br[i][j][k] = (int)_r_region[feat][i][j][k];
	      xc[i][j][k] = 1;
	    }
	}
  
  // variable allc containing the number of voxels that have not been labelled yet.
  
  int m = 0; 
  int c = 0;
  long ap, allc = 0;
  int times = 0;
  int stop  = 0;
  
  // main segment loop
  // printf("\n *here is ok\n");
  while(stop!=1)
    {
      times++;
      
      // scan the entire image.
      for(int z=1; z<_zdim-1; z++)
	for(int y=1; y<_ydim-1; y++)
	  for(int x=1; x<_xdim-1; x++)
	    {
	      //printf("\n (%d, %d, %d)", x,y,z);
	      m = br[x][y][z];
	      
	      // for any current border point check if there is any pixel in its neighbourhood
	      // that can be merged to the region to which the border belongs.
	      if(m!= 0)
		{
		  // printf("\n m = %d",m);
		  for(int dz=-1; dz<=1; dz++)
		    for(int dy=-1; dy<=1; dy++)
		      for(int dx=-1; dx<=1; dx++)
			{
			  // candidate voxel must not be assigned to a region and its intensity
			  // must be close to the mean value of the region under consideration
			  // (their difference must be less than threshold t). update np, s, xc,br.
			  
			  c = xc[x+dx][y+dy][z+dz];
			  if( (c!=1) && ((int)_w_region[feat][x+dx][y+dy][z+dz] == m) && (_background[feat][x+dx][y+dy][z+dz]==0) ) 
			  // if( (c!=1) && (_background[feat][x+dx][y+dy][z+dz]==0) )
			    {
			      idx = 0;
			      for(int j=0; j<_num_subject; j++)
				{
				  if(j!=_subject_id)
				    {
				      sum2[idx][m-1] += _feature_data[j][x+dx][y+dy][z+dz];
				      xx[idx] = sum2[idx][m-1];
				      idx+=1;
				    }
				}
			      
			      sc2[m-1] = LOOPearsonCorrelation(xx,yy,_num_subject-1);
			      
			      if(fabsf(sc2[m-1])>=fabsf(sc1[m-1]))
				{
				  _r_region[feat][x+dx][y+dy][z+dz] = m;
				  np[m-1] += 1;
				  sc1[m-1] = sc2[m-1];
				  for(int i=0; i<_num_subject-1; i++)
				    sum1[i][m-1] = sum2[i][m-1];
				  
				  xc[x+dx][y+dy][z+dz] = 1;
				  xc[x][y][z] = 1;
				  br[x+dx][y+dy][z+dz] = m;
				  br[x][y][z] = 0;
				}
			      else
				{
				  for(int i=0; i<_num_subject-1; i++)
				    sum2[i][m-1] = sum1[i][m-1];
				}
			    }
			}
		}
	    }
      
      
      // count voxels that have not been assigned to any region.
      //printf("\n ***here is ok\n");
      ap = allc; allc = 0;
      for(int z=0; z<_zdim; z++)
	for(int y=0; y<_ydim; y++)
	  for(int x=0; x<_xdim; x++)
	    allc+=abs(xc[x][y][z]-1);
      
      if(abs(ap-allc)==0)
	stop = 1;
    }
  
  for(int k=0; k<_zdim; k++)
    for(int j=0; j<_ydim; j++)
      for(int i=0; i<_xdim; i++)
	{
	  if(_r_region[feat][i][j][k] == 0)
	    _r_mean[feat][i][j][k] = 0;
	  else
	    _r_mean[feat][i][j][k] = fabs(sc1[(int)_r_region[feat][i][j][k]-1]);
	}
/* 
  FILE *fp;
  char fn[1024];
  sprintf(fn,"Regions_%d_%d.bin",_subject_id,feat);
  if((fp=fopen(fn,"w"))==NULL)
    {
      printf("\n cannot open file %s.\n",fn);
      exit(1);
    }
//  fprintf(fp, "%d\t%d\t%d\n", _xdim, _ydim, _zdim);
  for(int k=0; k<_zdim; k++)
    for(int j=0; j<_ydim; j++)
      for(int i=0; i<_xdim; i++)
	fprintf(fp, "%f\t", _r_mean[feat][i][j][k]);
//	fwrite(&_score[feat][i][j][k],sizeof(float),1,fp);
  fclose(fp);
*/

  free_i3tensor(xc,_xdim, _ydim, _zdim);
  free_i3tensor(br,_xdim, _ydim, _zdim);
  
  free_vector(xx,_num_subject-1);
  free_vector(yy,_num_subject-1);
  
  free_matrix(sum1,_num_subject-1,n);
  free_matrix(sum2,_num_subject-1,n);
  
  free_ivector(np,n);
  
  free_vector(sc1,n);
  free_vector(sc2,n);
}

void FeatureExtraction::GetFeature(int feat)
{
  this->Debug("GetFeature");
  
  // roi feature from watershed segmentation
  for(int k=0; k<_zdim; k++)
    for(int j=0; j<_ydim; j++)
      for(int i=0; i<_xdim; i++)
	{
	  int id = (int)_w_region[feat][i][j][k] -1;
	  if((id>=0)&&(id<_num_classification_feature))
	    {
	      //  printf("\n region id is %d\n",id);
	      _w_roi_size[feat][id] += 1;
	      
	      for(int ii=0; ii<_num_subject; ii++)
		_w_roi_volume[feat][ii][id] += _feature_data[ii][i][j][k];
	    }
	}
  
  // roi feature from region growing
  for(int k=0; k<_zdim; k++)
    for(int j=0; j<_ydim; j++)
      for(int i=0; i<_xdim; i++)
	{
	  int id = (int)_r_region[feat][i][j][k] -1;
	  if((id>=0)&&(id<_num_classification_feature))
	    {
	      //  printf("\n region id is %d\n",id);
	      _r_roi_size[feat][id] += 1;
	      
	      for(int ii=0; ii<_num_subject; ii++)
		_r_roi_volume[feat][ii][id] += _feature_data[ii][i][j][k];
	    }
	}
  
  for(int ii=0; ii<_num_classification_feature; ii++)
    for(int jj=0; jj<_num_subject; jj++)
      {
	_w_roi_volume[feat][jj][ii] /= (_w_roi_size[feat][ii]+1);
	_r_roi_volume[feat][jj][ii] /= (_r_roi_size[feat][ii]+1);
	//printf("\n roi volume is %f",_roi_volume[ll][jj][ii]);
      }
}

void FeatureExtraction::RankROIFeature()
{
  this->Debug("ROI Feature");
  
  int len = _num_subject-1;
  float *x,*y;
  x = vector(len);
  y = vector(len);

  this->Debug("ROI score");
  for(int ll=0; ll<_num_feature; ll++)
    for(int ii=0; ii<_num_classification_feature; ii++)
      {
	int indx = 0;
	for(int jj=0; jj<_num_subject; jj++)
	  {
	    if(jj!=_subject_id)
	      {
		x[indx] = _w_roi_volume[ll][jj][ii];
		
		y[indx] = _label[jj];
		
		indx+=1;
	      }
	  }
	
	if( IsValidVector1(x,y,len) > 0 )
	  _w_roi_score[ll][ii]  = LOOPearsonCorrelation(x,y,len);
	else
	  _w_roi_score[ll][ii] = 0;
	}
  
  for(int ll=0; ll<_num_feature; ll++)
    for(int ii=0; ii<_num_classification_feature; ii++)
      {
	int indx = 0;
	for(int jj=0; jj<_num_subject; jj++)
	  {
	    if(jj!=_subject_id)
	      {
		x[indx] = _r_roi_volume[ll][jj][ii];
		
		y[indx] = _label[jj];
		
		indx+=1;
	      }
	  }
	
	if( IsValidVector1(x,y,len) > 0 )
	  _r_roi_score[ll][ii]  = LOOPearsonCorrelation(x,y,len);
	else
	  _r_roi_score[ll][ii] = 0;
      }
  
  free_vector(x,len);
  free_vector(y,len);
  
  this->Debug("ROI ranking");
  long num = _num_feature * _num_classification_feature;
  float *score = vector(num);
  int **tmp = imatrix(num, 2);
  
  long index = 0;
  for(int ll=0; ll<_num_feature; ll++)
    for(int ii=0; ii<_num_classification_feature; ii++)
      {
	score[index] = fabsf(_w_roi_score[ll][ii]);
	tmp[index][0] = ll;
	tmp[index][1] = ii;
	index+=1;
      }
  
  unsigned long *indx = lvector(num);
  indexx(num, score, indx);
  for(long i=0;i<_num_classification_feature;i++)
    for(int j=0;j<2;j++)
      _w_roi_feature_location[i][j]=tmp[indx[num-1-i]][j];
  
  index = 0;
  for(int ll=0; ll<_num_feature; ll++)
    for(int ii=0; ii<_num_classification_feature; ii++)
      {
	score[index] = fabsf(_r_roi_score[ll][ii]);
	tmp[index][0] = ll;
	tmp[index][1] = ii;
	index+=1;
      }
  
  indexx(num, score, indx);
  for(long i=0;i<_num_classification_feature;i++)
    for(int j=0;j<2;j++)
      _r_roi_feature_location[i][j]=tmp[indx[num-1-i]][j];

  
  for(int index=0; index < _num_classification_feature; index++)
    {	
      int level  = _w_roi_feature_location[index][0];
      int ii     = _w_roi_feature_location[index][1];
      for(int ll=0; ll<_num_subject; ll++)
	_w_roi_feature[ll][index]=_w_roi_volume[level][ll][ii];
      

      level  = _r_roi_feature_location[index][0];
      ii     = _r_roi_feature_location[index][1];
      for(int ll=0; ll<_num_subject; ll++)
	_r_roi_feature[ll][index]=_r_roi_volume[level][ll][ii];
    }  
  
  free_vector(score, num);
  free_lvector(indx,num);
  free_imatrix(tmp,num,2);
}

void FeatureExtraction::FeatureWhitten( )
{
  this->Debug("FeatureWhitten");
  int len = _num_subject -1;
  float *x = vector(len);
  
  float mean,var;
  // roi feature from wathreshed 
  for(int ii=0; ii<_num_classification_feature; ii++)
    {
      int indx=0;
      for(int ll=0; ll<_num_subject; ll++)
	{
	  if( ll != _subject_id)
	    x[indx++] = _w_roi_feature[ll][ii];
	}
      
      mean  = 0; var = 0;
      for(int i=0; i<len; i++)
	mean += x[i];
      mean /= (float)(len);
      
      for(int i=0; i<len; i++)
	{
	  float tmp = x[i] - mean;
	  var += tmp*tmp;
	}
      var = sqrt(var/(len-1))+1.0e-6;
      
      _whitten_para[0][ii][0] = mean;
      _whitten_para[0][ii][1] = var;
      for(int ll=0; ll<_num_subject; ll++)
	_w_roi_feature[ll][ii]= (_w_roi_feature[ll][ii]-mean)/var;
    }
  
  // roi feature from region growing
  for(int ii=0; ii<_num_classification_feature; ii++)
    {
      int indx=0;
      for(int ll=0; ll<_num_subject; ll++)
	{
	  if( ll != _subject_id)
	    x[indx++] = _r_roi_feature[ll][ii];
	}
      
      mean  = 0; var = 0;
      for(int i=0; i<len; i++)
	mean += x[i];
      mean /= (float)(len);
      
      for(int i=0; i<len; i++)
	{
	  float tmp = x[i] - mean;
	  var += tmp*tmp;
	}
      var = sqrt(var/(len-1))+1.0e-6;
      
      _whitten_para[1][ii][0] = mean;
      _whitten_para[1][ii][1] = var;
      for(int ll=0; ll<_num_subject; ll++)
	_r_roi_feature[ll][ii]= (_r_roi_feature[ll][ii]-mean)/var;
    }

  free_vector(x,len);
}


void FeatureExtraction::WriteFiles()
{
  this->Debug("WriteClassificationData");

  FILE *fp1,*fp2;
  char train_file[100];
  char test_file[100];
   
  FeatureWhitten();
 
  // Write roi feature from region_grow
  sprintf(train_file, "r_roi_train_%d.txt",_subject_id);
  sprintf(test_file, "r_roi_test_%d.txt", _subject_id);
  
  if((fp1=fopen(train_file,"w")) == NULL)
    {
      printf("\n Could not open file: %s\n",train_file);
      exit(1);
    }
  if((fp2=fopen(test_file, "w")) == NULL)
    {
      printf("\n Could not open file: %s\n", test_file);
      exit(1);
    }
  
  fprintf(fp1,"%d    %ld\n", _num_subject-1, _num_classification_feature+1);
  fprintf(fp2,"%d    %ld\n", 1,              _num_classification_feature+1);
  
  for(int i=0; i<_num_subject; i++)
    {
      if(i == _subject_id)
	{
	  for(int j=0; j<_num_classification_feature; j++)
	    fprintf(fp2,"%f\t",_r_roi_feature[i][j]);
	  fprintf(fp2,"%f\n",_label[i]);
	}
      else
	{
	  for(int j=0; j<_num_classification_feature; j++)
	    fprintf(fp1,"%f\t",_r_roi_feature[i][j]);
	  fprintf(fp1,"%f\n",_label[i]);
	}
    }
  fclose(fp1);
  fclose(fp2);
  
  
  // Write roi feature from watershed segmentation
  sprintf(train_file, "w_roi_train_%d.txt",_subject_id);
  sprintf(test_file, "w_roi_test_%d.txt", _subject_id);
  
  if((fp1=fopen(train_file,"w")) == NULL)
    {
      printf("\n Could not open file: %s\n",train_file);
      exit(1);
    }
  if((fp2=fopen(test_file, "w")) == NULL)
    {
      printf("\n Could not open file: %s\n", test_file);
      exit(1);
    }
  
  fprintf(fp1,"%d    %ld\n", _num_subject-1, _num_classification_feature+1);
  fprintf(fp2,"%d    %ld\n", 1,              _num_classification_feature+1);
  
  for(int i=0; i<_num_subject; i++)
    {
      if(i == _subject_id)
	{
	  for(int j=0; j<_num_classification_feature; j++)
	    fprintf(fp2,"%f\t",_w_roi_feature[i][j]);
	  fprintf(fp2,"%f\n",_label[i]);
	}
      else
	{
	  for(int j=0; j<_num_classification_feature; j++)
	    fprintf(fp1,"%f\t",_w_roi_feature[i][j]);
	  fprintf(fp1,"%f\n",_label[i]);
	}
    }
  fclose(fp1);
  fclose(fp2);
  
}

void FeatureExtraction::WriteFeatureLocations( )
{
  this->Debug("WriteFeatureLocation");
  
  FILE *fp; 
  char file_name[1024];
  
  
  // roi feature from watershed segmentation
  sprintf(file_name, "w_roi_feature_location_%d.txt", _subject_id);
  if((fp=fopen(file_name,"w"))==NULL)
    {
      printf("\n Could not open file: %s\n",file_name);
      exit(1);
    }
  
  fprintf(fp, "id\t score\t  feature_id\t region_id\t mean\t var\n");
  for(long i=0; i<_num_classification_feature; i++)
    {
      //printf(".");
      int feat   = _w_roi_feature_location[i][0];
      int roi    = _w_roi_feature_location[i][1];
      float score= _w_roi_score[feat][roi];
      float mean = _whitten_para[0][i][0];
      float var  = _whitten_para[0][i][1];
      fprintf(fp, "%ld\t %f\t %d\t %d\t %f\t %f\n", i, score, feat, roi, mean, var);
    }
  fclose(fp);
  
  // roi feature from region growing
  sprintf(file_name, "r_roi_feature_location_%d.txt", _subject_id);
  if((fp=fopen(file_name,"w"))==NULL)
    {
      printf("\n Could not open file: %s\n",file_name);
      exit(1);
    }
  
  fprintf(fp, "id\t score\t  feature_id\t region_id\t mean\t var\n");
  for(long i=0; i<_num_classification_feature; i++)
    {
      //printf(".");
      int feat   = _r_roi_feature_location[i][0];
      int roi    = _r_roi_feature_location[i][1];
      float score= _r_roi_score[feat][roi];
      float mean = _whitten_para[1][i][0];
      float var  = _whitten_para[1][i][1];
      fprintf(fp, "%ld\t %f\t %d\t %d\t %f\t %f\n", i, score, feat, roi, mean, var);
    }
  fclose(fp);  
}

void FeatureExtraction::WriteRegionFiles( )
{
  this->Debug("Write Region Files");
  
  FILE *fp; 
  char file_name[1024];

  short stmp;
  for(int feat = 0; feat<_num_feature; feat++)
    {
      
      // watershed roi
      sprintf(file_name, "w_roi_region_%d_%d.bin", _subject_id,feat);
      if((fp=fopen(file_name,"w"))==NULL)
	{
	  printf("\n Could not open file: %s\n",file_name);
	  exit(1);
	}
      for(int k=0; k<_zdim; k++)
	for(int j=0; j<_ydim; j++)
	  for(int i=0; i<_xdim; i++)
	    {
	      stmp = (short)_w_region[feat][i][j][k];
	      fwrite(&stmp, sizeof(short), 1,fp);
	    }
      fclose(fp);
      
      /*
      sprintf(file_name, "w_roi_mean_%d_%d.bin", _subject_id,feat);
      if((fp=fopen(file_name,"w"))==NULL)
	{
	  printf("\n Could not open file: %s\n",file_name);
	  exit(1);
	}
      for(int k=0; k<_zdim; k++)
	for(int j=0; j<_ydim; j++)
	  for(int i=0; i<_xdim; i++)
	    {
	      ftmp = (float)_w_mean[feat][i][j][k];
	      fwrite(&ftmp, sizeof(float), 1,fp);
	    }
      fclose(fp);
      */

      // regiongrowing roi
      sprintf(file_name, "r_roi_region_%d_%d.bin", _subject_id,feat);
      if((fp=fopen(file_name,"w"))==NULL)
	{
	  printf("\n Could not open file: %s\n",file_name);
	  exit(1);
	}
      for(int k=0; k<_zdim; k++)
	for(int j=0; j<_ydim; j++)
	  for(int i=0; i<_xdim; i++)
	    {
	      stmp = (short)_r_region[feat][i][j][k];
	      fwrite(&stmp, sizeof(short), 1,fp);
	    }
      fclose(fp);
      
      /*
      sprintf(file_name, "r_roi_mean_%d_%d.bin", _subject_id,feat);
      if((fp=fopen(file_name,"w"))==NULL)
	{
	  printf("\n Could not open file: %s\n",file_name);
	  exit(1);
	}
      for(int k=0; k<_zdim; k++)
	for(int j=0; j<_ydim; j++)
	  for(int i=0; i<_xdim; i++)
	    {
	      ftmp = (float)_r_mean[feat][i][j][k];
	      fwrite(&ftmp, sizeof(float), 1,fp);
	    }
      fclose(fp);
      */
    }   
}

void FeatureExtraction::Run( )
{
  GetInput();
  Initialize();
  
  if((_subject_id>=0)&&(_subject_id<_num_subject))
    {
      for(int feat=0; feat<_num_feature; feat++)
	{
	  ReadData(feat);
	  FeatureWhitten0();
	  
	  ComputeLOOScore(feat);
	  ComputeSpatialScore(feat);
	  CombineScore(feat);

	  ReadData(feat);
	  ComputeRegionFeat(feat);
	  RegionGrow(feat);
	  GetFeature(feat); 
	}
      RankROIFeature();
      WriteFiles();
      WriteFeatureLocations();
      WriteRegionFiles();
    }
  else
    {
      printf("\n subject_id should be in the range of 0 to num_subject-1.\n");
      exit(1);
    }
     
  Finalize();
}
