/*!
 * \file  FeatureExtraction.h
 * \brief Feature Extraction Class.
 *
 * For copyright information please see Copyright.txt in the root
 * directory of the project.
 *
 * Contact: SBIA Group <sbia-software@uphs.upenn.edu>
 */


#ifndef _FEATUREEXTRACTION_H_
#define _FEATUREEXTRACTION_H_
#define True  1
#define False 0
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>

#include "watershed.h"
#include "nrutil_compare.h"
#include "ImagesReader.h"


class FeatureExtraction {
  
 public:
 
  int  _DebugFlag;

  /// Constructor
  FeatureExtraction();

  /// Destructor
  ~FeatureExtraction();

  /// Returns the name of the class
  char *NameOfClass();

  /// Prints debugging messages if debugging is enabled
  void Debug(char *);


 public:
  
  char *_file_sublist;
  char ***_file_name;  
  float *_label;
  
  int _num_po_subject;
  int _num_ne_subject;
  int _num_subject;
  
  int  _xdim;
  int  _ydim;
  int  _zdim;
  long _im_size;
  long _num_feature;

  int  _subject_id; // for leave one out test
 
  
  long _num_classification_feature; 
  
  float ****_feature_data;  // store the features of all the subjects [sub][x][y][z];
  
  float ****_score;    // store the score data  [x][y][z]
  float ****_loo_score;     
  float ****_spatial_score; 

  short ****_w_region;       // store the watershed segmentation result
  float ****_w_mean;         // store the watershed segmentation region filled with mean value
  
  short ****_r_region;
  float ****_r_mean;
  short ***_region_seed; 
  int _max_region_size;
  int *_region_size;
  short ****_background;
  
  void RegionGrow(int feat);
  void GetROIFeature();
  void ComputeRegionFeat(int feat);

  int **_ranked_feature_location; // store the ranked feature location [feature_index][feature + feature coordination];
  int **_region_feature_location; // store the region feature location
  int **_w_roi_feature_location;    // store the roi feature location from wathershed segmentation
  int **_r_roi_feature_location;    // store the roi feature location from region growing

  float **_w_roi_feature;
  float **_r_roi_feature;

  float ***_w_roi_volume;
  float **_w_roi_size;
  float **_w_roi_score;
  float ***_r_roi_volume;
  float **_r_roi_size;
  float **_r_roi_score;

  float ***_whitten_para;

  float _smooth_size;  // smooth the enhanced score map
  float _p;
  /// Comput the feature score without subject_id 
  void ComputeLOOScore(int feat);
  void ComputeSpatialScore(int feat);
  void FeatureWhitten0();
  void CombineScore(int feat);
  
  void GetFeature(int feat);
  void RankROIFeature();
  void WriteFiles();
  
  /// Write Feature locations and their corresponding score
  void WriteFeatureLocations();
  void WriteRegionFiles();

 
  
  void ReadData(int feat);
  void FeatureWhitten( );

  /// Determing the data point is not noise
  int IsValidVector(float *x, float *y, int len);
  int IsValidVector(int *x, int *y, int len);
  int IsValidVector1(float *x, float *y, int len);
  int IsValidVector1(int *x, int *y, int len);
  /// Compute the Pearson Correlation
  float PearsonCorrelation(float *x, float *y, int len); 
  float PearsonCorrelation(int *x, int *y, int len);
  float LOOPearsonCorrelation(float *x, float *y, int len);
  float LOOPearsonCorrelation(int *x, int *y, int len);
  float IntraClass(float **data, int nfeat, int nhsize);
  float IntraClass(int **data, int nfeat, int nhsize);
  /// Sets input for the feature selection routine
  void SetInput (char *sublist,int subject_id, long num_feature, float smooth_size, float p);
  void GetInput();  
  /// Initial set up for the feature selection
  void Initialize();
  /// Final set up for the feature selection
  void Finalize();
 

  void Run(); 

};

inline int FeatureExtraction::IsValidVector(float *x, float *y, int len)
{
  int tmp1 = 0;
  int tmp2 = 0;
  
  for(int i=0; i<len; i++)
    {
      if(fabsf(x[i])>1.0e-4)
	{
	  if(y[i]>0.0)
	    tmp1++;
	  else
	    tmp2++;
	}
    }
  
  if((tmp1>_num_po_subject/4)&&(tmp2>_num_ne_subject/4))
    return 1;
  else
    return 0;  
}

inline int FeatureExtraction::IsValidVector(int *x, int *y, int len)
{

  int tmp1 = 0;
  int tmp2 = 0;
  
  for(int i=0; i<len; i++)
    {
      if(x[i]>1)
	{
	  if(y[i]>0)
	    tmp1+=1;
	  else
	    tmp2+=1;
	}
    }
	    
  if((tmp1>_num_po_subject/4)&&(tmp2>_num_ne_subject/4))
    return 1;
  else
    return 0;  
}

inline int FeatureExtraction::IsValidVector1(float *x, float *y, int len)
{
  int tmp1 = 0;
  int tmp2 = 0;
  
  for(int i=0; i<len; i++)
    {
      if(fabsf(x[i])>1.0e-3)
	{
	  if(y[i]>0.0)
	    tmp1++;
	  else
	    tmp2++;
	}
    }
  
  if((tmp1>_num_po_subject-2)&&(tmp2>_num_ne_subject-2))
    return 1;
  else
    return 0;  
}

inline int FeatureExtraction::IsValidVector1(int *x, int *y, int len)
{

  int tmp1 = 0;
  int tmp2 = 0;
  
  for(int i=0; i<len; i++)
    {
      if(x[i]>1)
	{
	  if(y[i]>0)
	    tmp1+=1;
	  else
	    tmp2+=1;
	}
    }
	    
  if((tmp1>_num_po_subject-2)&&(tmp2>_num_ne_subject-2))
    return 1;
  else
    return 0;  
}

inline float FeatureExtraction::PearsonCorrelation(float *x, float *y, int len)
{

  double sumx=0.0,sumy=0.0,sumx2=0.0,sumy2=0.0;
  
  double  sumxy=0.0;
  for(int i=0; i<len; i++)
    {
      sumx+=x[i];
      sumy+=y[i];
      sumx2+=x[i]*x[i];
      sumy2+=y[i]*y[i];
      sumxy+=x[i]*y[i];
    }
  
  double coefficient = (sumxy-sumx*sumy/(float)len)/(sqrt(fabs((1.0e-6+sumx2-sumx*sumx/(float)len)*(1.0e-6+sumy2-sumy*sumy/(float)len)))+1.0e-6 );

  //printf("\n * * %f",coefficient);

  if (coefficient<-1.0)
    coefficient =-1.0;
  if (coefficient>1.0)
    coefficient = 1.0;
  /*
  if(isnanf(coefficient))
    {
      printf("\n ********* non a number *****\n");
      printf("\n sumx^2/2 = %f",sumx*sumx/(float)(len));
      printf("\n sumy^2/2 = %f",sumy*sumy/(float)(len));
      printf("\n sumx2 = %f",sumx2);
      printf("\n sumy2 = %f",sumy2);
      for(int i=0; i<len; i++)
	printf("\n *%f",x[i]);
      exit(0);
    }
  */

  return (float)coefficient;
}

inline void FeatureExtraction::Debug(char *message)
{
  if (_DebugFlag == True) printf("\n %s \n", message);
}

inline float FeatureExtraction::PearsonCorrelation(int *x, int *y, int len)
{

  double sumx=0.0,sumy=0.0,sumx2=0.0,sumy2=0.0;
  
  double  sumxy=0.0;
  for(int i=0; i<len; i++)
    {
      sumx+=(double)x[i];
      sumy+=(double)y[i];
      sumx2+=(double)x[i]*(double)x[i];
      sumy2+=(double)y[i]*(double)y[i];
      sumxy+=(double)x[i]*(double)y[i];
    }
  
  double coefficient = (sumxy-sumx*sumy/(float)len)/(sqrt(fabs((1.0e-6+sumx2-sumx*sumx/(float)len)*(1.0e-6+sumy2-sumy*sumy/(float)len)))+1.0e-6 );

  //printf("\n * * %f",coefficient);

  if (coefficient<-1.0)
    coefficient =-1.0;
  if (coefficient>1.0)
    coefficient = 1.0;
  /*
  if(isnanf(coefficient))
    {
      printf("\n ********* non a number *****\n");
      printf("\n sumx^2/2 = %f",sumx*sumx/(float)(len));
      printf("\n sumy^2/2 = %f",sumy*sumy/(float)(len));
      printf("\n sumx2 = %f",sumx2);
      printf("\n sumy2 = %f",sumy2);
      for(int i=0; i<len; i++)
	printf("\n *%f",x[i]);
      exit(0);
    }
  */

  return (float)coefficient;
}


inline float FeatureExtraction::LOOPearsonCorrelation(float *x, float *y, int len)
{

  int loo_len = len - 1;
  
  float *loo  = vector(len);
  float *loox = vector(loo_len);
  float *looy = vector(loo_len);

  for(int i=0; i<len; i++)
    {
      int index = 0;
      for(int j=0; j<len; j++)
	{
	  if(j!=i)
	    {
	      loox[index] = x[j];
	      looy[index] = y[j];
	      index+=1;
	    }
	}
      loo[i] = PearsonCorrelation(loox,looy,loo_len);
    }
  
  
  float min = fabsf(loo[0]);
  float score = loo[0];
  for(int i=1; i<len; i++)
    {
      float tmp = fabsf(loo[i]);
      if(tmp < min)
	{
	  min = tmp;
	  score = loo[i];
	}
    }
  
  free_vector(loo,  len);
  free_vector(loox, loo_len);
  free_vector(looy, loo_len);
  
  return score;
} 

inline float FeatureExtraction::LOOPearsonCorrelation(int *x, int *y, int len)
{

  int loo_len = len - 1;

  float *loo  = vector(len);
  float *loox = vector(loo_len);
  float *looy = vector(loo_len);

  for(int i=0; i<len; i++)
    {
      int index = 0;
      for(int j=0; j<len; j++)
	{
	  if(j!=i)
	    {
	      loox[index] = x[j];
	      looy[index] = y[j];
	      index+=1;
	    }
	}
      loo[i] = PearsonCorrelation(loox,looy,loo_len);
    }
  
  float min = fabsf(loo[0]);
  float score = loo[0];
  for(int i=1; i<len; i++)
    {
      float tmp = fabsf(loo[i]);
      if(tmp < min)
	{
	  min = tmp;
	  score = loo[i];
	}
    }
  
  free_vector(loo,  len);
  free_vector(loox, loo_len);
  free_vector(looy, loo_len);
  
  return score;
}

inline float FeatureExtraction::IntraClass(int **data, int nfeat, int nhsize)
{
  //printf("\n IntraClass computing ...");

  float *colmean, *rowmean, gm;
  colmean = vector(nhsize);
  rowmean = vector(nfeat);

  float df_col = nhsize - 1;
  float  df_row = nfeat  - 1;
  float  df_err = df_col * df_row;

  for(int i=0; i<nfeat; i++)
    rowmean[i] = 0;
  for(int i=0; i<nhsize; i++)
    colmean[i] = 0;
  
  gm = 0;
  
  for(int i=0; i<nfeat; i++)
    {
      for(int j=0; j<nhsize; j++)
        rowmean[i] += data[i][j];
      
      rowmean[i] /= (float)nhsize;
    }
  
  for(int j=0; j<nhsize; j++)
    {
      for(int i=0; i<nfeat; i++)
        colmean[j] += data[i][j];
      
      gm += colmean[j];
      colmean[j] /= (float)nfeat;
    }
  gm /= (float)(nhsize * nfeat);

  float css = 0;
  float rss = 0;
  float correction = nfeat*nhsize*gm*gm;

  for(int i=0; i<nfeat; i++)
    {
      float tmp = rowmean[i] - gm;
      rss += tmp*tmp;
    }
  rss *= nhsize;

  for(int j=0; j<nhsize; j++)
    {
      float tmp = colmean[j] - gm;
      css += tmp*tmp;
    }
  css *= nfeat;

  float ess = 0;
  for(int i=0; i<nfeat; i++)
    for(int j=0; j<nhsize; j++)
      ess += data[i][j]*data[i][j];

  ess -= correction + css + rss;

  float cms = css / df_col;
  float rms = rss / df_row;
  float ems = ess / df_err;

  free_vector(rowmean, nfeat);
  free_vector(colmean, nhsize);

  // icc(C,1)
  //  return (rms - ems)/(rms + (nhsize-1)*ems + 1.0e-6);
  
  // icc(A,1)
  return (rms - ems)/(rms + (nhsize-1)*ems + (float)(nhsize)/(float)(nfeat)*(cms - ems) +1.0e-6);
}


inline float FeatureExtraction::IntraClass(float **data, int nfeat, int nhsize)
{
  //printf("\n IntraClass computing ...");

  float *colmean, *rowmean, gm;
  colmean = vector(nhsize);
  rowmean = vector(nfeat);

  float df_col = nhsize - 1;
  float df_row = nfeat  - 1;
  float df_err = df_col * df_row;

  for(int i=0; i<nfeat; i++)
    rowmean[i] = 0;
  for(int i=0; i<nhsize; i++)
    colmean[i] = 0;
  
  gm = 0;
  
  for(int i=0; i<nfeat; i++)
    {
      for(int j=0; j<nhsize; j++)
        rowmean[i] += data[i][j];
      
      rowmean[i] /= (float)nhsize;
    }
  
  for(int j=0; j<nhsize; j++)
    {
      for(int i=0; i<nfeat; i++)
        colmean[j] += data[i][j];
      
      gm += colmean[j];
      colmean[j] /= (float)nfeat;
    }
  gm /= (float)(nhsize * nfeat);

  float css = 0;
  float rss = 0;
  float correction = nfeat*nhsize*gm*gm;

  for(int i=0; i<nfeat; i++)
    {
      float tmp = rowmean[i] - gm;
      rss += tmp*tmp;
    }
  rss *= nhsize;

  for(int j=0; j<nhsize; j++)
    {
      float tmp = colmean[j] - gm;
      css += tmp*tmp;
    }
  css *= nfeat;

  float ess = 0;
  for(int i=0; i<nfeat; i++)
    for(int j=0; j<nhsize; j++)
      ess += data[i][j]*data[i][j];

  ess -= correction + css + rss;

  float cms = css / df_col;
  float rms = rss / df_row;
  float ems = ess / df_err;

  free_vector(rowmean, nfeat);
  free_vector(colmean, nhsize);

  // icc(C,1)
  //  return (rms - ems)/(rms + (nhsize-1)*ems + 1.0e-6);
  
  // icc(A,1)
  return (rms - ems)/(rms + (nhsize-1)*ems + (float)(nhsize)/(float)(nfeat)*(cms - ems) +1.0e-6);
}


#endif
