/*!
 * \file  ImagesReader.cc
 * \brief Implementation of subject list parser and image reader module.
 *
 * Copyright (c) University of Pennsylvania. All rights reserved.
 * See LICENSE or Copyright file in the project root directory.
 *
 * Contact: SBIA Group <sbia-software -at- uphs.upenn.edu>
 */

#include "ImagesReader.h"


using namespace std;


//////////////////////////////////////////////////////////////////////////////
// construction / destruction
//////////////////////////////////////////////////////////////////////////////

/*****************************************************************************/
ImagesReader
::ImagesReader ()
:
    _hdr (NULL)
{
}

/*****************************************************************************/
ImagesReader
::~ImagesReader ()
{
        if (_hdr) delete _hdr;
}

//////////////////////////////////////////////////////////////////////////////
// public interface
//////////////////////////////////////////////////////////////////////////////

/*****************************************************************************/
void 
ImagesReader
::SetSubjectList (const std::string &filename)
{
        _listfile = filename;
}

/*****************************************************************************/
bool
ImagesReader
::ParseSubjectList (int type)
{
        string          line;
        vector <string> tokens;
        
        // open subjects list
        ifstream fin;
        fin.open(_listfile.c_str(), ios_base::in);
        if (fin.fail())
        {
                cerr << "Failed to open file " << _listfile << endl;
                return false;
        }
        
        // read number of subjects and features
        if (!getline (fin, line) || parseline (line, tokens) != 2)
        {
                cerr << "Syntax error on line 1 of file " << _listfile << '.' << endl;
                return false;
        }
        
        int num_subject = atoi (tokens [0].c_str ());
        int num_feat    = atoi (tokens [1].c_str ());
        
        if (num_subject <= 0 || num_feat <= 0)
        {
                cerr << "Number of subjects and features must be positive." << endl;
                return false;
        }
        
        tokens.clear();
        
        // read image dimensions
        if (!getline (fin, line) || parseline (line, tokens) != 3)
        {
                cerr << "Syntax error on line 2 of file " << _listfile << '.' << endl;
                return false;
        }
        
        int dim [3];
        
        dim [0] = atoi (tokens [0].c_str ());
        dim [1] = atoi (tokens [1].c_str ());
        dim [2] = atoi (tokens [2].c_str ());
        
        tokens.clear();
        
        // read data root directory used for relative paths
        if (!getline (fin, line) || parseline (line, tokens) != 1)
        {
                cerr << "Syntax error on line 3 of file " << _listfile << '.' << endl;
                return false;
        }
        
        std::string root = tokens [0];
        tokens.clear();

        // If root is a relative path, use directory of subject list file to make it absolute.
        if ( root [0] != '/' )
        {
                int found = _listfile.find_last_of("/");
                if (found != string::npos)
                {
                        string folder = _listfile.substr(0, found+1);
                        root = folder + root;
                }
        }
        
        // read filenames of images
        vector <vector <string> > filenames (num_subject);
        vector <float>            label     (num_subject);
        
        for (int i = 0; i < num_subject; ++i) filenames [i].resize (num_feat);
        
        int count   = 0;
        int columns = 0;
        
        while (getline (fin, line))
        {
                columns = parseline (line, tokens);
               
                // skip empty lines
                if (columns == 0) continue;

                if (type == 0 && columns != num_feat + 1)
                {
                        cerr << "Syntax error on line " << (count + 4) << " of file " << _listfile << '.' << endl;
                        cerr << "Number of features and number of images specified do not match";
                        if ( columns == num_feat ) cerr << ", or missing label";
                        cerr << '.' << endl;
                        return false;
                }
                if (type != 0 && columns != num_feat + 1 && columns != num_feat)
                {
                        cerr << "Syntax error on line " << (count + 4) << " of file " << _listfile << '.' << endl;
                        cerr << "Number of features and number of images specified do not match." << endl;
                        return false; 
                }
                
                if (count > (num_subject - 1))
                {
                        cerr << "More subjects given than specified." << endl;
                        return false;
                }
                
                for (int i = 0; i < num_feat; ++i) filenames [count][i] = tokens [i];
                
                if (type == 0)
                {
                        if (strcmp (tokens [num_feat].c_str (),  "1") != 0
                            && strcmp (tokens [num_feat].c_str (), "-1") != 0)
                        {
                                cerr << "Syntax error on line " << (count + 4) << " of file " << _listfile << '.' << endl;
                                cerr << "Label must be either 1 or -1." << endl;
                                return false;
                        }
                        label [count] = atof (tokens [num_feat].c_str ());
                }
                 
                tokens.clear();
                ++ count;
        }
 
        if (count != num_subject)
        {
                cerr << "Less subjects given than specified." << endl;
                return false;
        }
        
        // store parsed values
        _num_subject = num_subject;
        _num_feat    = num_feat;
        _dim [0]     = dim [0];
        _dim [1]     = dim [1];
        _dim [2]     = dim [2];
        _root        = root;
        _filenames   = filenames;
        _label       = label;
        
        return true;
}

/*****************************************************************************/
bool
ImagesReader
::CheckInput ()
{
        for (int i = 0; i < _num_subject; ++i)
        {
                for (int j = 0; j < _num_feat; ++j)
                {
                        // prepend root directory if file path is relative
                        string fname = _filenames [i][j];
                        if (fname [0] != '/') fname = _root + '/' + fname;
                        
                        // read image header
                        nifti_image *nim = nifti_image_read (fname.c_str (), 0);
                        
                        if (!nim)
                        {
                                cerr << "Failed to open image file " << fname << endl;
                                cerr << "Is the given image a NIfTI image?" << endl;
                                return false;
                        }
                        
                        // check whether image type is supported
                        if (!IsSupportedImage (nim))
                        {
                                cerr << "Image file " << fname << " not supported." << endl;
                                cerr << "Image must be a three-dimensional scalar image of type signed short." << endl;
                                return false;
                        }
                        
                        // check whether image attributes match subject list header
                        int error_code = IsHeaderMatch (nim);
                        
                        if (error_code == 1)
                        {
                                cerr << "Attributes of image " << fname << " do not match information of subject list header." << endl;
                                return false;
                        }
                        else if (error_code == 2)
                        {
                                cerr << "WARNING Image " << fname << " might not be co-registered." << endl;
                                // just a warning, keep going
                        }
                        
                        // free NIfTI image structure
                        nifti_image_free (nim);
                }
        }
        
        return true;
}

/*****************************************************************************/
bool 
ImagesReader
::ReadImages (float ****image, int feat)
{
        for (int i = 0; i < _num_subject; ++i)
        {
                // prepend root directory if file path is relative
                string fname = _filenames [i][feat];
                if (fname [0] != '/') fname = _root + '/' + fname;
                
                // read image including image data
                nifti_image *nim = nifti_image_read (fname.c_str (), 1);
                
                if (!nim)
                {
                        cerr << "Failed to open image file " << fname << endl;
                        cerr << "Is the given image a NIfTI image?" << endl;
                        return false;
                }
                
                // copy image data into own memory
                int count = 0;
                for (int kk = 0; kk < _dim[2]; ++kk)
                {
                        for (int jj = 0; jj < _dim[1]; ++jj)
                        {
                                for (int ii = 0; ii < _dim[0]; ++ii)
                                {
                                        image [i][ii][jj][kk] = static_cast <float> (
                                                        reinterpret_cast <short*> (nim->data) [count]
                                                        );
                                        count ++;
                                }
                        }
                }
                
                // free NIfTI image structure
                nifti_image_free (nim);
        }
        
        return true;
}

/*****************************************************************************/
void
ImagesReader
::GetLabels (float *label) const
{
        for (int i=0; i<_num_subject; i++)
        {
                label[i] = _label[i];
        }
}

///////////////////////////////////////////////////////////////////////////////
// helper
///////////////////////////////////////////////////////////////////////////////

/*****************************************************************************/
int
ImagesReader
::IsHeaderMatch (nifti_image *nim)
{
        int error_code;
        // if first image, initialize header information
        if (_hdr == NULL)
        {
                if (nim->nx != _dim [0] || nim->ny != _dim [1] || nim->nz != _dim[2])
                {
                        cerr << "Dimension of first image does not match the dimensions specified in the subject list." << endl;
                        error_code = 1;
                        return error_code;
                }
                _hdr = new nifti_1_header;
                *_hdr = nifti_convert_nim2nhdr (nim);
        }
        // otherwise, compare image to first image
        else
        {
                // check image dimensions
                for (int i = 0; i < 8; ++i)
                {
                        if (nim->dim [i] != _hdr->dim [i]) 
                        {
                                error_code = 1;
                                cerr << "Image dimension do not match." << endl;
                                return error_code;
                        }
                }
                // check image orientation and location
                if (nim->qform_code != _hdr->qform_code || nim->sform_code != _hdr->sform_code) 
                {
                        error_code = 2;
                        return error_code;
                }
                if (nim->sform_code > 0)
                {
                        for (int c = 0; c < 4; ++c)
                        {
                                if (    fabs (nim->qto_xyz.m [0][c] - _hdr->srow_x [c]) > 1.0e-6
                                                || fabs (nim->qto_xyz.m [1][c] - _hdr->srow_y [c]) > 1.0e-6 
                                                || fabs (nim->qto_xyz.m [2][c] - _hdr->srow_z [c]) > 1.0e-6)
                                {
                                        error_code = 2;
                                        return error_code;
                                }
                        }
                }
                if (nim->qform_code > 0)
                {
                        if (    fabs (nim->quatern_b - _hdr->quatern_b) > 1.0e-6
                                        || fabs (nim->quatern_c - _hdr->quatern_c) > 1.0e-6
                                        || fabs (nim->quatern_d - _hdr->quatern_d) > 1.0e-6
                                        || fabs (nim->qoffset_x - _hdr->qoffset_x) > 1.0e-6
                                        || fabs (nim->qoffset_y - _hdr->qoffset_y) > 1.0e-6
                                        || fabs (nim->qoffset_z - _hdr->qoffset_z) > 1.0e-6 )
                        {
                                error_code = 2;
                                return error_code;
                        }
                }
        }
        
        error_code = 0;
        return error_code;
}

/*****************************************************************************/
bool
ImagesReader
::IsSupportedImage (nifti_image *nim)
{
        if (nim->datatype != DT_SIGNED_SHORT) return false;
        for (int i=1; i<=3; i++)
        {
                if (nim->dim[i] <= 1) return false;
        }
        for (int i=4; i<=7; i++)
        {
                if (nim->dim[i] != 0 && nim->dim[i] != 1) return false;
        }
        return true;
}

/*****************************************************************************/
int
ImagesReader
::parseline (string str, vector <string> &tokens)
{
        istringstream iss (str);
        copy (istream_iterator <string> (iss),
              istream_iterator <string> (),
              back_inserter <vector <string> > (tokens));
        return tokens.size ();
}

