/*!
 * \file  ImagesReader.h
 * \brief Implementation of subject list parser and image reader module.
 *
 * Copyright (c) 2011 University of Pennsylvania. All rights reserved.
 * See LICENSE file in project root or 'doc' directory for details.
 *
 * Contact: SBIA Group <sbia-software -at- uphs.upenn.edu>
 */

#pragma once
#ifndef __IMAGESREADER_H
#define __IMAGESREADER_H


#include <iostream>
#include <fstream>
#include <sstream>
#include <iterator>
#include <vector>
#include <cmath>
#include <nifti1_io.h>


/*!
 * \class ImagesReader
 * \brief Parses subject list file and reads input images.
 *
 * Example:
 * \code
 * ImagesReader reader;
 *
 * // set subjects list file
 * reader.SetSubjectList ("subjects.txt");
 * // parse subjects list file
 * reader.ParseSubjectList (list type);
 * // validate input images and read header information
 * if (!reader.CheckInput ()) {
 *     std::cerr << "Invalid subjects list subjects.txt" << std::endl;
 * }
 * // allocate memory for images
 * const int N = reader.GetNumberOfImages ();
 * const int X = reader.GetImageDimensions ()[1];
 * const int Y = reader.GetImageDimensions ()[2];
 * const int Z = reader.GetImageDimensions ()[3];
 * float ****images = NULL;
 * try {
 *     images = new float [N][X][Y][Z];
 * } catch (...) {
 *     std::cerr << "Failed to allocate memory for input images" << std::endl;
 *     exit (EXIT_FAILURE);
 * }
 * // read images
 * if (!reader.ReadImages (images, feat)) {
 *     std::cerr << Failed to read images listed in subjects.txt" << std::endl;
 *     delete [] images;
 *     exit (EXIT_FAILURE);
 * }
 * \endcode
 */
class ImagesReader
{
    //////////////////////////////////////////////////////////////////////////
    // construction / destruction
    //////////////////////////////////////////////////////////////////////////

public:

    /*!
     * \brief Constructor.
     */
    ImagesReader ();

    /*!
     * \brief Destructor.
     */
    ~ImagesReader ();

    //////////////////////////////////////////////////////////////////////////
    // public interface
    //////////////////////////////////////////////////////////////////////////

public:

    /*!
     * \brief Set path to subject list file.
     *
     * \param [in] filename Path to subject list file.
     */
    void SetSubjectList (const std::string &filename);

    /*!
     * \brief Parse the subject list file.
     *
     * This method checkes the format and parses the information of 
     * subject list file including header, image names and labels. 
     * The parsed information can be obtained from this class after 
     * this method was called using the corresponding getter methods.
     *
     * \see GetNumberOfSubjects ()
     * \see GetNumberOfFeatures ()
     * \see GetNumberOfDimensions ()
     * \see GetDataRoot ()
     *
     * \param [in] type Type of subject list. Can be either 0 for
     *                  a training list or non-0 for a testing list.
     *
     * \return Whether the list could be parsed successfully.
     */
    bool ParseSubjectList (int type);

    /*!
     * \brief Validate the input images and read header information.
     *
     * \return Whether the image list is valid, the input images are
     *         supported and co-registered.
     */
    bool CheckInput ();

    /*!
     * \return Number of subjects.
     */
    int GetNumberOfSubjects () const;

    /*!
     * \return Number of feature images.
     */
    int GetNumberOfFeatures () const;

    /*!
     * \brief Get image dimensions.
     *
     * This function returns a pointer to the dim array of the common
     * NIfTI image, i.e., an array of length 8, where the first entry
     * specifies the number of dimension, the second entry is the image
     * size along the x axis, the third entry the image size along the
     * y axis, and the fourth entry the image size along the z axis.
     *
     * \return Pointer to dim array of NIfTI header.
     */
    const int *GetImageDimensions () const;

    /*!
     * \return Root directory of image files.
     */
    const char *GetDataRoot () const;

    /*!
     * \brief Read images into a 4D array.
     *
     * This function parses the subject list file and reads the images
     * into a 4D array. The memory for the image data has to be allocated
     * by the caller. Therefore, the number of images and the size of the
     * images can be retrieved using the getter methods of this reader.
     *
     * \see GetNumberOfImages ()
     * \see GetImageDimensions ()
     *
     * \param [out] images The read image data.
     *
     * \return Whether the images were read successfully.
     */
    bool ReadImages (float ****image, int feat);

    /*!
     * \brief Get parsed classification labels.
     *
     * \param [out] label The read labels.
     */
    void GetLabels (float *label) const;
    
    //////////////////////////////////////////////////////////////////////
    // helpers
    //////////////////////////////////////////////////////////////////////

protected:

    /*!
     * \brief Check the header information of images.
     *
     * This function checks whether the dimension information of image 
     * header and information provided by input list file match. 
     *
     * \retval 0 Image header information matches the one of the subject
     #           list header.
     * \retval 1 Images don't match each other and this will lead to fatal
     *           calculation error.
     * \retval 2 Images don't match each other but this might only
     *           cause an unreliable result.
     */
    int IsHeaderMatch (nifti_image *nim); 

    /*!
     * \brief Check if the images' datatype is short or not.
     *
     * \return Whether the given NIfTI image is supported.
     */
    bool IsSupportedImage (nifti_image *nim);

    /*!
     * \brief Parse one line of subject list to a vector of strings.
     *
     * \param [in] str    The input string with tokens separated by
     *                    whitespace characters.
     * \param [in] tokens The parsed tokens.
     * 
     * \return Number of parsed tokens.
     */
    int parseline (std::string str, std::vector <std::string> &tokens);

    //////////////////////////////////////////////////////////////////////
    // members
    //////////////////////////////////////////////////////////////////////

private:

    std::string          _listfile;    ///< Path of subject list file.
    int                  _num_subject; ///< Number of subjects.
    int                  _num_feat;    ///< Number of feature types.
    int	                 _dim[3];	     ///< Image dimensions in list file.
    std::string          _root;        ///< Root directory of image files.
    nifti_1_header      *_hdr;         ///< Common header information.
    std::vector <float>  _label;       ///< Classification labels.
    std::vector< std::vector <std::string> > _filenames; ///< Image file names.
}; // class ImagesReader


//////////////////////////////////////////////////////////////////////////
// inline methods
//////////////////////////////////////////////////////////////////////////

inline int
ImagesReader
::GetNumberOfSubjects () const
{
    return _num_subject;
}

inline int
ImagesReader
::GetNumberOfFeatures () const
{
    return _num_feat;
}

inline const int *
ImagesReader
::GetImageDimensions () const
{
    return _dim;
}

inline const char *
ImagesReader
::GetDataRoot () const
{
    return _root.c_str ();
}


#endif // __IMAGESREADER_H

