##############################################################################
# \file  SbiaMacros.cmake
# \brief Definition of CMake macros and functions commonly used at SBIA.
#
# This file contains the definition of CMake macros and functions which are
# used by projects developed at SBIA for convenience.
#
# DO NOT edit this file. Instead, create a new file named "Macros.cmake",
# for example, and include it in the root CMake file of the project.
#
# For copyright information please see Copyright.txt in the root
# directory of the project.
#
# Contact: SBIA Group <sbia-software@uphs.upenn.edu>
##############################################################################

include (CMakeParseArguments)

# ============================================================================
# global constants
# ============================================================================

# Default component used when no component is specified. Especially scripts
# which are added by globbing the directory are added by default to this
# component.
set (SBIA_DEFAULT_COMPONENT "Runtime")

# Base name with extension of the (optional) CMake file with additional
# arguments to build command, e.g., containing list of MEX-files which were
# written to this file by the overwritten target_link_libraries () command.
set (SBIA_DEPENDS_FILE "SbiaDependInfo.cmake")

# ============================================================================
# project
# ============================================================================

# ****************************************************************************
# \macro sbia_project
# \brief SBIA equivalent to CMake's project () command.
#
# The project specific attributes such as project name and project version,
# among others, need to be defined in the Settings.cmake file which can be
# found in the PROJECT_CONFIG_DIR. The PROJECT_CONFIG_DIR is by default set
# by SbiaSettings.cmake to the 'Config' subfolder of the project's source tree.
# The project's Settings.cmake file is included by this macro and the project
# name is passed unchanged to the project () command of CMake.
#
# Dependencies to external packages should be resolved via find_package ()
# commands in the file Depends.cmake which as well has to be located in
# PROJECT_CONFIG_DIR (note that this variable may be modified within
# Settings.cmake. The Depends.cmake file is included by this macro.
#
# Each SBIA project further has to have a ReadMe.txt file in the top directory
# of the source tree which is the root documentation file. This file may only
# consist of a reference to the project's actual documentation files that
# are located in the 'Doc' subfolder of the source tree.
#
# Moreover, a Copyright.txt file with the copyright and license notices must
# be found in the top directory of the source tree.

macro (sbia_project)
  # include project settings
  include ("${PROJECT_CONFIG_DIR}/Settings.cmake")

  # check required project information
  if (NOT PROJECT_NAME)
    message (FATAL_ERROR "PROJECT_NAME not defined.")
  endif ()

  if (NOT PROJECT_VERSION)
    message (FATAL_ERROR "PROJECT_VERSION not defined.")
  endif ()

  # default project information
  if (NOT PROJECT_README_FILE)
    set (PROJECT_README_FILE "${CMAKE_CURRENT_SOURCE_DIR}/ReadMe")
  endif ()
  # note that PROJECT_README_FILE may not exist yet, but get
  # generated during the configure step of CMake!

  if (NOT PROJECT_LICENSE_FILE)
    set (PROJECT_LICENSE_FILE "${CMAKE_CURRENT_SOURCE_DIR}/Copyright")
  endif ()
  if (NOT EXISTS "${PROJECT_LICENSE_FILE}")
    message (FATAL_ERROR "Project Copyright file not found")
  endif ()

  # start CMake project
  project ("${PROJECT_NAME}")

  set (CMAKE_PROJECT_NAME "${PROJECT_NAME}") # variable used by CPack

  # reset cached CMake variables
  set (SBIA_INCLUDE_DIRECTORIES "" CACHE INTERNAL "Include directories." FORCE)

  # convert project name to upper and lower case only, respectively
  string (TOUPPER ${PROJECT_NAME} PROJECT_NAME_UPPER)
  string (TOLOWER ${PROJECT_NAME} PROJECT_NAME_LOWER)

  # extract version numbers from version string
  string (REGEX MATCHALL "[0-9]+" PROJECT_VERSION_PARTS "${PROJECT_VERSION}")
  list (LENGTH PROJECT_VERSION_PARTS PROJECT_VERSION_COUNT)

  if (PROJECT_VERSION_COUNT LESS 2 OR PROJECT_VERSION_COUNT GREATER 3)
    message (FATAL_ERROR "Invalid version string specified for PROJECT_VERSION in ${PROJECT_INFO_FILE}!")
  endif ()

  list (GET PROJECT_VERSION_PARTS 0 PROJECT_VERSION_MAJOR)
  list (GET PROJECT_VERSION_PARTS 1 PROJECT_VERSION_MINOR)

  if (PROJECT_VERSION_COUNT GREATER 2)
    list (GET PROJECT_VERSION_PARTS 2 PROJECT_VERSION_PATCH)
  else ()
    set (PROJECT_VERSION_PATCH 0)
  endif ()

  set (PROJECT_VERSION_COUNT "3")

  # combine version numbers to version strings (also ensures consistency)
  set (PROJECT_VERSION   "${PROJECT_VERSION_MAJOR}.${PROJECT_VERSION_MINOR}.${PROJECT_VERSION_PATCH}")
  set (PROJECT_SOVERSION "${PROJECT_VERSION_MAJOR}.${PROJECT_VERSION_MINOR}")

  # get project revision
  sbia_svn_get_revision ("${PROJECT_SOURCE_DIR}" PROJECT_REVISION)

  # print project information
  message (STATUS "Project ${PROJECT_NAME}")
  message (STATUS "  Version:   ${PROJECT_VERSION}")
  message (STATUS "  SoVersion: ${PROJECT_SOVERSION}")
  if (PROJECT_REVISION)
  message (STATUS "  Revision:  ${PROJECT_REVISION}")
  else ()
  message (STATUS "  Revision:  n/a")
  endif ()

  # resolve dependencies
  include ("${PROJECT_CONFIG_DIR}/Depends.cmake")

  # copy ReadMe file to binary tree (NOT the one specified by PROJECT_README_FILE)
  if (EXISTS "${PROJECT_SOURCE_DIR}/ReadMe")
    if (NOT "${PROJECT_BINARY_DIR}" STREQUAL "${PROJECT_SOURCE_DIR}")
      configure_file ("${PROJECT_SOURCE_DIR}/ReadMe" "${PROJECT_BINARY_DIR}/ReadMe" COPYONLY)
    endif ()
  elseif (EXISTS "${PROJECT_SOURCE_DIR}/ReadMe.txt")
    if (NOT "${PROJECT_BINARY_DIR}" STREQUAL "${PROJECT_SOURCE_DIR}")
      configure_file ("${PROJECT_SOURCE_DIR}/ReadMe.txt" "${PROJECT_BINARY_DIR}/ReadMe.txt" COPYONLY)
    endif ()
  endif ()

  # install ReadMe file (NOT the one specified by PROJECT_README_FILE)
  install (
    FILES       "${PROJECT_SOURCE_DIR}/ReadMe"
    DESTINATION "${CMAKE_INSTALL_PREFIX}"
    OPTIONAL
  )
  install (
    FILES       "${PROJECT_SOURCE_DIR}/ReadMe.txt"
    DESTINATION "${CMAKE_INSTALL_PREFIX}"
    OPTIONAL
  )

  # copy license to binary tree
  if (NOT "${PROJECT_BINARY_DIR}" STREQUAL "${PROJECT_SOURCE_DIR}")
    get_filename_component (TMP "${PROJECT_LICENSE_FILE}" NAME)
    configure_file ("${PROJECT_LICENSE_FILE}" "${PROJECT_BINARY_DIR}/${TMP}" COPYONLY)
    set (TMP)
  endif ()

  # install license
  install (
    FILES       "${PROJECT_LICENSE_FILE}"
    DESTINATION "${CMAKE_INSTALL_PREFIX}"
  )
endmacro ()

# ============================================================================
# Subversion
# ============================================================================

# ****************************************************************************
# \function sbia_svn_get_revision
# \brief    Get current revision of file or directory.
#
# This function makes use of the variable SBIA_CMD_SVN which is set to the
# full path to the svn command in the CMake module SbiaCommands.
#
# \see SbiaCommands
#
# \param [in]  URL Absolute path of directory or file. May also be a URL to the
#                  directory or file in the repository. A leading "file://" is
#                  automatically removed such that the svn command treats it as a
#                  local path.
# \param [out] REV The revision number of URL. If URL is not under revision
#                  control or SBIA_CMD_SVN is invalid, "0" is returned.

function (sbia_svn_get_revision URL REV)
  set (OUT "0")

  if (SBIA_CMD_SVN)
    # remove "file://" from URL
    string (REGEX REPLACE "file://" "" TMP "${URL}")

    # retrieve SVN info
    execute_process (
      COMMAND         "${SBIA_CMD_SVN}" info "${TMP}"
      OUTPUT_VARIABLE OUT
      ERROR_QUIET
      OUTPUT_STRIP_TRAILING_WHITESPACE
    )

    # extract revision
    string (REGEX REPLACE "^(.*\n)?Revision: ([^\n]+).*" "\\2" OUT "${OUT}")

    if (OUT STREQUAL "")
      set (OUT "0")
    endif ()
  endif ()

  # return
  set (${REV} "${OUT}" PARENT_SCOPE)
endfunction ()

# ****************************************************************************
# \function sbia_svn_get_last_changed_revision
# \brief    Get revision number when directory or file was last changed.
#
# This function makes use of the variable SBIA_CMD_SVN which is set to the
# full path to the svn command in the CMake module SbiaCommands.
#
# \see SbiaCommands
#
# \param [in]  URL Absolute path of directory or file. May also be a URL to the
#                  directory or file in the repository. A leading "file://" is
#                  automatically removed such that the svn command treats it as a
#                  local path.
# \param [out] REV Revision number when URL was last modified. If URL is not
#                  under Subversion control or SBIA_CMD_SVN is invalid,
#                  "0" is returned.

function (sbia_svn_get_last_changed_revision URL REV)
  set (OUT "0")

  if (SBIA_CMD_SVN)
    # remove "file://" from URL
    string (REGEX REPLACE "file://" "" TMP "${URL}")

    # retrieve SVN info
    execute_process (
      COMMAND         "${SBIA_CMD_SVN}" info "${TMP}"
      OUTPUT_VARIABLE OUT
      ERROR_QUIET
      OUTPUT_STRIP_TRAILING_WHITESPACE
    )

    # extract last changed revision
    string (REGEX REPLACE "^(.*\n)?Last Changed Rev: ([^\n]+).*" "\\2" OUT "${OUT}")

    if (OUT STREQUAL "")
      set (OUT "0")
    endif ()
  endif ()

  # return
  set (${REV} "${OUT}" PARENT_SCOPE)
endfunction ()

# ****************************************************************************
# \function sbia_svn_status
# \brief    Get status of revision controlled file.
#
# This function makes use of the variable SBIA_CMD_SVN which is set to the
# full path to the svn command in the CMake module SbiaCommands.
#
# \see SbiaCommands
#
# \param [in]  URL    Absolute path of directory or file. May also be a URL to
#                     the directory or file in the repository.
#                     A leading "file://" will be removed such that the svn
#                     command treats it as a local path.
# \param [out] STATUS The status of URL as returned by 'svn status'.
#                     If the local directory or file is unmodified, an
#                     empty string is returned. An empty string is also
#                     returned when SBIA_CMD_SVN is invalid.

function (sbia_svn_status URL STATUS)
  if (SBIA_CMD_SVN)
    # remove "file://" from URL
    string (REGEX REPLACE "file://" "" TMP "${URL}")

    # retrieve SVN status of URL
    execute_process (
      COMMAND         "${SBIA_CMD_SVN}" status "${TMP}"
      OUTPUT_VARIABLE OUT
      ERROR_QUIET
    )

    # return
    set (${STATUS} "${OUT}" PARENT_SCOPE)
  else ()
    set (${STATUS} "" PARENT_SCOPE)
  endif ()
endfunction ()

# ============================================================================
# target properties / dependencies
# ============================================================================

# ****************************************************************************
# \function include_directories
# \brief    Overwrites CMake's include_directories () command.
#
# All arguments are passed on to CMake's include_directories () command.
# Additionally, the list of include directories is stored in the cached CMake
# variable SBIA_INCLUDE_DIRECTORIES. This variable can be used by custom
# commands for the build process, e.g., it is used as argument for the -I
# option of the MATLAB Compiler.
#
# The value of this internal cache variabel is cleared by sbia_project ().
#
# \param ARGN Argument list passed on to CMake's include_directories command.

function (include_directories)
  CMAKE_PARSE_ARGUMENTS (ARGN "" "" "AFTER;BEFORE;SYSTEM" ${ARGN})

  if (ARGN_BEFORE)
    set (
      SBIA_INCLUDE_DIRECTORIES
        "${ARGN_UNPARSED_ARGUMENTS};${SBIA_INCLUDE_DIRECTORIES}"
        CACHE INTERNAL "Include directories." FORCE
    )
  else ()
    set (
      SBIA_INCLUDE_DIRECTORIES
        "${SBIA_INCLUDE_DIRECTORIES};${ARGN_UNPARSED_ARGUMENTS}"
        CACHE INTERNAL "Include directories." FORCE
    )
  endif ()

  if (SBIA_INCLUDE_DIRECTORIES)
    list (REMOVE_DUPLICATES SBIA_INCLUDE_DIRECTORIES)
  endif ()

  _include_directories (${ARGN})
endfunction ()

# ****************************************************************************
# \function target_link_libraries
# \brief    Overwrites CMake's target_link_libraries () command.
#
# The main reason for overwriting this function was, to treat libraries
# such as MEX-files which are supposed to be compiled into a MATLAB
# executable target added by sbia_add_executable () special. In this case,
# these libraries are added to the variable SBIA_DEPENDS defined in the
# file SBIA_DEPENDS_FILE in the temporary build directory. The build of the
# MATLAB executable works fine like this. However, the problem is that the
# custom command which executes the MATLAB Compiler does not depend on these
# additional input files and hence the build command is not re-executed when
# these files were re-build/changed.
#
# Example:
# \code
# sbia_add_library (MyMEXFunc MEX myfunc.c)
# sbia_add_executable (MyMATLABApp main.m)
# target_link_libraries (MyMATLABApp MyMEXFunc OtherMEXFunc.mexa64)
# \endcode
#
# \param [in] TARGET Name of the target.
# \param [in] ARGN   Link libraries.

function (target_link_libraries TARGET)
  get_target_property (SBIA_TYPE ${TARGET} "SBIA_TYPE")

  # --------------------------------------------------------------------------
  # type: MCC_EXECUTABLE
  # --------------------------------------------------------------------------

  if (SBIA_TYPE STREQUAL "MCC_EXECUTABLE")

    # \todo How is it possible to define the dependency to these link libs
    #       AFTER the custom command for the build of the MATLAB executable
    #       has been added ?
    message (
      WARNING
       "When link libraries for a MATLAB executable target such as MEX-files are specified via the"
       " target_link_libraries () command, the MATLAB executable is not rebuild when any link"
       " library changed. Consider specifying the link libraries as arguments to sbia_add_executable () instead.")

    # name of file listing additional MATLAB Compiler build command ARGS
    # such as MEX-files in particular
    get_target_property (DEPENDS_FILE ${TARGET} "SBIA_DEPENDS_FILE")
    get_target_property (TARGET_OUTPUT ${TARGET} "LOCATION")

    # read in dependencies
    set (SBIA_DEPENDS "")
    include ("${DEPENDS_FILE}" OPTIONAL)

    # iterate over link libraries
    foreach (LIB ${ARGN})
      # get filename of link library
      if (TARGET ${LIB})
        get_target_property (LIB_FILE ${LIB} "LOCATION")
        add_dependencies (${TARGET} ${LIB}) # such that LIB is build before TARGET
      else ()
        set (LIB_FILE "${LIB}")
      endif ()

      # add library (e.g., MEX-file) to SBIA_DEPENDS
      list (APPEND SBIA_DEPENDS "${LIB_FILE}")
    endforeach ()

    # write dependencies file
    if (SBIA_DEPENDS)
      list (REMOVE_DUPLICATES SBIA_DEPENDS)
    endif ()

    file (WRITE  "${DEPENDS_FILE}" "# DO NOT edit. This file is automatically generated by the SBIA CMake functions.\n")
    file (APPEND "${DEPENDS_FILE}" "set (SBIA_DEPENDS\n")
    foreach (DEPEND ${SBIA_DEPENDS})
      file (APPEND "${DEPENDS_FILE}" "  \"${DEPEND}\"\n")
    endforeach ()
    file (APPEND "${DEPENDS_FILE}" ")\n")

  # --------------------------------------------------------------------------
  # other
  # --------------------------------------------------------------------------

  else ()

    # simple use CMake target_link_libraries () command
    _target_link_libraries (${TARGET} ${ARGN})

  endif ()

endfunction ()

# ****************************************************************************
# \function sbia_mexext
# \brief    Determine extension of MEX-files for this architecture.
#
# \param [out] EXT The extension of MEX-files (excluding '.'). If the CMake
#                  variable MEX_EXT is set, its value is returned. Otherwise,
#                  this function tries to determine it from the system
#                  information. If the extension could not be determined,
#                  an empty string is returned.

function (sbia_mexext EXT)
  # default return value
  set (MEXEXT "${MEX_EXT}")

  # use MEXEXT if possible
  if (NOT MEXEXT AND SBIA_CMD_MEXEXT)
    execute_process (
      COMMAND         "${SBIA_CMD_MEXEXT}"
      RESULT_VARIABLE RETVAL
      OUTPUT_VARIABLE MEXEXT
      ERROR_QUIET
      OUTPUT_STRIP_TRAILING_WHITESPACE
    )

    if (RETVAL)
      set (MEXEXT "")
    endif ()
  endif ()

  # otherwise, determine extension given CMake variables describing the system
  if (NOT MEXEXT)
    if (${CMAKE_SYSTEM_NAME} STREQUAL "Linux")
      if (${CMAKE_SYSTEM_PROCESSOR} STREQUAL "x86_64")
        set (MEXEXT ".mexa64")
      elseif (${CMAKE_SYSTEM_PROCESSOR} STREQUAL "x86" OR
              ${CMAKE_SYSTEM_PROCESSOR} STREQUAL "i686")
        set (MEXEXT ".mexglx")
      endif ()
    elseif (${CMAKE_SYSTEM_NAME} STREQUAL "Windows")
      if (${CMAKE_SYSTEM_PROCESSOR} STREQUAL "x86_64")
        set (MEXEXT ".mexw64")
      elseif (${CMAKE_SYSTEM_PROCESSOR} STREQUAL "x86" OR
              ${CMAKE_SYSTEM_PROCESSOR} STREQUAL "i686")
        set (MEXEXT ".mexw32")
      endif ()
    elseif (${CMAKE_SYSTEM_NAME} STREQUAL "Darwin")
      set (MEXEXT "mexaci)
    elseif (${CMAKE_SYSTEM_NAME} STREQUAL "SunOS")
      set (MEXEXT "mexs64)
    endif ()
  endif ()

  # return value
  set (${EXT} "${MEXEXT}" PARENT_SCOPE)
endfunction ()

# ****************************************************************************
# \function sbia_create_addpaths_mfile
# \brief    This function writes a MATLAB M-file with addpath () statements.
#
# This function writes the MATLAB M-file addpaths.m into the root directory
# of the build tree which contains an addpath () statement for each
# directory that was added via the CMake command include_directories ().

function (sbia_create_addpaths_mfile)
  set (MFILE "${CMAKE_CURRENT_BINARY_DIR}/Sbia${PROJECT_NAME}Paths.m")

  file (WRITE "${MFILE}" "% DO NOT edit. This file is automatically generated by CMake.\n")
  foreach (P ${SBIA_INCLUDE_DIRECTORIES})
    file (APPEND "${MFILE}" "addpath ('${P}');\n")
  endforeach ()
endfunction ()

# ============================================================================
# targets
# ============================================================================

# ****************************************************************************
# \function verifyTargetName
# \brief    Verifies if a given name is a valid target name.
#
# \param [in] TARGET_NAME Name of the target.

function (verifyTargetName TARGET_NAME)
  if (TARGET_NAME STREQUAL "doc" OR
      TARGET_NAME STREQUAL "changelog")
    message (FATAL_ERROR "Target names 'doc' and 'changelog' are reserved.")
  endif ()
endfunction ()

# ****************************************************************************
# \function sbia_add_mcc_target
# \brief    Add MATLAB Compiler target.
#
# Use sbia_add_executable () and sbia_add_library () instead. This function
# is used by these two when at least one M-file is given as input source file.
#
# \see sbia_add_executable ()
# \see sbia_add_library ()
#
# \param [in] TARGET_NAME Name of the target.
# \param [in] ARGN        Remaining arguments such as in particular the
#                         input source files.
#
#                           TYPE Type of the target.
#                                Either EXECUTABLE (default) or LIBRARY.

function (sbia_add_mcc_target TARGET_NAME)
  verifyTargetName (${TARGET_NAME})

  # required commands available ?
  if (NOT SBIA_CMD_MCC)
    message (FATAL_ERROR "MATLAB Compiler not found. It is required to build target ${TARGET_NAME}."
                         "Set SBIA_CMD_MCC manually and try again.")
  endif ()

  if (NOT SBIA_SCRIPT_EXECUTE_PROCESS)
    message (FATAL_ERROR "CMake script required for execution of process in script mode not found."
                         "It is required to build the target ${TARGET_NAME}. "
                         "Set SBIA_SCRIPT_EXECUTE_PROCESS manually and try again.")
  endif ()

  # parse arguments
  CMAKE_PARSE_ARGUMENTS (ARGN "" "COMPONENT;TYPE" "" ${ARGN})

  # if no component is specified, use default
  if (NOT ARGN_COMPONENT)
    set (ARGN_COMPONENT "${SBIA_DEFAULT_COMPONENT}")
  endif ()
  if (NOT ARGN_COMPONENT)
    set (ARGN_COMPONENT "Unspecified")
  endif ()

  if (NOT ARGN_TYPE)
    set (ARGN_TYPE "EXECUTABLE")
  else ()
    string (TOUPPER "${ARGN_TYPE}" ARGN_TYPE)
  endif ()

  if (NOT ARGN_TYPE STREQUAL "EXECUTABLE" AND NOT ARGN_TYPE STREQUAL "LIBRARY")
    message (FATAL_ERROR "Invalid type for MCC target ${TARGET_NAME}: ${ARGN_TYPE}")
  endif ()

  if (ARGN_TYPE STREQUAL "LIBRARY")
    message (FATAL_ERROR "Build of MATLAB library from M-files not yet supported.")
  endif ()

  # temporary output directory
  set (TEMP_DIR "${CMAKE_CURRENT_BINARY_DIR}/CMakeFiles/${TARGET_NAME}.dir")

  # prefix target output name
  if (SBIA_OUTPUT_NAME_PREFIXED)
    set (TARGET_OUTPUT_NAME "${SBIA_OUTPUT_NAME_PREFIX}${TARGET_NAME}")
  else ()
    set (TARGET_OUTPUT_NAME "${TARGET_NAME}")
  endif ()

  # convert user specified MCC flags to CMake list
  if (MCC_FLAGS)
    string (REPLACE "\\ "    "&nbsp;" USER_FLAGS "${MCC_FLAGS}")
    string (REPLACE " "      ";"      USER_FLAGS "${USER_FLAGS}")
    string (REPLACE "&nbsp;" " "      USER_FLAGS "${USER_FLAGS}")
  else ()
    set (USER_FLAGS "")
  endif ()

  # convert target arguments to the respective target output files (e.g., MEX-file)
  set (SOURCES)
  set (NON_TARGET_SOURCES)

  foreach (ARG ${ARGN_UNPARSED_ARGUMENTS})
    if (TARGET ${ARG})
      get_target_property (SOURCE "${ARG}" "LOCATION")
    else ()
      set (SOURCE "${ARG}")
      list (APPEND NON_TARGET_SOURCES "${SOURCE}")
    endif ()
    list (APPEND SOURCES "${SOURCE}")
  endforeach ()

  # assemble build command
  set (MCC_ARGS ${USER_FLAGS})                       # user specified flags
  foreach (INCLUDE_PATH ${SBIA_INCLUDE_DIRECTORIES}) # add directories added via
    list (FIND MCC_ARGS "${INCLUDE_PATH}" IDX)       # overwritten include_directories ()
    if (INCLUDE_PATH AND IDX EQUAL -1)               # function to search path
      list (APPEND MCC_ARGS "-I" "${INCLUDE_PATH}")
    endif ()
  endforeach ()
  list (FIND MCC_ARGS "${CMAKE_CURRENT_SOURCE_DIR}" IDX)
  if (IDX EQUAL -1)
    # add current source directory to search path,
    # needed for build in MATLAB mode as working directory
    # differs from the current source directory then
    list (APPEND MCC_ARGS "-I" "${CMAKE_CURRENT_SOURCE_DIR}")
  endif ()
  if (ARGN_TYPE STREQUAL "LIBRARY")
    list (APPEND MCC_ARGS "-l")                       # build library
  else ()
    list (APPEND MCC_ARGS "-m")                       # build standalone application
  endif ()
  list (APPEND MCC_ARGS "-d" "${TEMP_DIR}")           # output directory
  list (APPEND MCC_ARGS "-o" "${TARGET_OUTPUT_NAME}") # output name
  list (APPEND MCC_ARGS ${SOURCES})                   # source (M-)file(s)
  list (APPEND MCC_ARGS "ARGS")                       # the ARGS string will be replaced
                                                      # by the script SBIA_SCRIPT_EXECUTE_PROCESS
                                                      # with the content of the CMake variable
                                                      # SBIA_DEPENDS defined in the DEPENDS_FILE.

  # build command for invocation of MATLAB Compiler in standalone mode
  set (BUILD_CMD      "${SBIA_CMD_MCC}" ${MCC_ARGS})
  set (ARGS_SEPARATOR ";")
  set (BUILD_LOG      "${TEMP_DIR}/mccBuild.log")
  set (WORKING_DIR    "${CMAKE_CURRENT_SOURCE_DIR}")
  set (MATLAB_MODE    OFF)

  # build command for invocation of MATLAB Compiler in MATLAB mode
  if (MCC_MATLAB_MODE)
    set (MATLAB_MODE ON)

    if (NOT SBIA_CMD_MATLAB)
      message ("MATLAB not found. It is required to build target ${TARGET_NAME} in MATLAB mode."
               "Set SBIA_CMD_MATLAB manually and try again or set MCC_MATLAB_MODE to OFF.")
      set (MATLAB_MODE OFF)
    endif ()

    if (NOT SBIA_SCRIPT_RUN_MCC)
      message ("MATLAB script for invocation of MATLAB Compiler not found. It is required to build target ${TARGET_NAME} in MATLAB mode."
               "Set SBIA_SCRIPT_RUN_MCC manually and try again or set MCC_MATLAB_MODE to OFF.")
      set (MATLAB_MODE OFF)
    endif ()

    if (MATLAB_MODE)
      get_filename_component (WORKING_DIR "${SBIA_SCRIPT_RUN_MCC}" PATH)
      get_filename_component (MFUNC       "${SBIA_SCRIPT_RUN_MCC}" NAME_WE)

      set (MATLAB_CMD "${MFUNC} -q") # -q option quits MATLAB when build is finished
      foreach (MCC_ARG ${MCC_ARGS})
        set (MATLAB_CMD "${MATLAB_CMD} ${MCC_ARG}")
      endforeach ()

      set (ARGS_SEPARATOR " ") # the additional arguments ARGS need to be separated by ' '
                               # as they are part of the single argument to the -r option

      set (
        BUILD_CMD
          "${SBIA_CMD_MATLAB}" # run MATLAB
          "-nosplash"          # do not display splash screen on start up
          "-nodesktop"         # run in command line mode
          "-nojvm"             # we do not need the Java Virtual Machine
          "-r" "${MATLAB_CMD}" # MATLAB command which invokes MATLAB Compiler
      )
    endif ()
  endif ()

  # relative paths used for comments of commands
  file (RELATIVE_PATH REL_OUTPUT_NAME "${PROJECT_BINARY_DIR}" "${TEMP_DIR}/${TARGET_OUTPUT_NAME}")

  # output files of build command
  if (ARGN_TYPE STREQUAL "EXECUTABLE")
    set (
      BUILD_OUTPUT
        "${EXECUTABLE_OUTPUT_PATH}/${TARGET_OUTPUT_NAME}"
        "${EXECUTABLE_OUTPUT_PATH}/run_${TARGET_OUTPUT_NAME}.sh"
    )

    set (
      POST_BUILD_COMMAND
        COMMAND "${CMAKE_COMMAND}"
                -E copy "${TEMP_DIR}/${TARGET_OUTPUT_NAME}" "${EXECUTABLE_OUTPUT_PATH}/${TARGET_OUTPUT_NAME}"
        COMMAND "${CMAKE_COMMAND}"
                -E copy "${TEMP_DIR}/run_${TARGET_OUTPUT_NAME}.sh" "${EXECUTABLE_OUTPUT_PATH}/run_${TARGET_OUTPUT_NAME}.sh"
    )

    set (BUILD_COMMENT "Building MATLAB executable ${REL_OUTPUT_NAME}...")
  else ()
    set (
      BUILD_OUTPUT
        "${LIBRARY_OUTPUT_PATH}/${TARGET_OUTPUT_NAME}"
    )

    set (
      POST_BUILD_COMMAND
        COMMAND "${CMAKE_COMMAND}"
                -E copy "${TEMP_DIR}/${TARGET_OUTPUT_NAME}" "${LIBRARY_OUTPUT_PATH}/${TARGET_OUTPUT_NAME}"
    )

    set (BUILD_COMMENT "Building MATLAB library ${REL_OUTPUT_NAME}...")
  endif ()

  # create/clear file with additional dependencies
  set (DEPENDS_FILE "${TEMP_DIR}/${SBIA_DEPENDS_FILE}")

  file (WRITE "${DEPENDS_FILE}" "# DO NOT edit. This file is automatically generated by the SBIA CMake functions.\nset (SBIA_DEPENDS)\n")

  # add custom command to build executable using MATLAB Compiler
  add_custom_command (
    OUTPUT ${BUILD_OUTPUT}
    # rebuild when input sources were modified
    DEPENDS "${DEPENDS_FILE}" ${ARGN_UNPARSED_ARGUMENTS}
    # invoke MATLAB Compiler in either MATLAB or standalone mode
    # wrapping command in CMake execute_process () command allows for inspection
    # parsing of command output for error messages and specification of timeout
    COMMAND "${CMAKE_COMMAND}"
            "-DCOMMAND=${BUILD_CMD}"
            "-DARGS=SBIA_DEPENDS"
            "-DARGS_FILE=${DEPENDS_FILE}"
            "-DARGS_SEPARATOR='${ARGS_SEPARATOR}'"
            "-DWORKING_DIRECTORY=${WORKING_DIR}"
            "-DTIMEOUT=${MCC_TIMEOUT}"
            "-DERROR_EXPRESSION=[E|e]rror"
            "-DOUTPUT_FILE=${BUILD_LOG}"
            "-DERROR_FILE=${BUILD_LOG}"
            "-DCMAKE_VERBOSE=OFF"
            "-DLOG_ARGS=ON"
            "-P" "${SBIA_SCRIPT_EXECUTE_PROCESS}"
    # post build command(s)
    ${POST_BUILD_COMMAND}
    # inform user where build log can be found
    COMMAND "${CMAKE_COMMAND}" -E echo "Build log written to ${BUILD_LOG}"
    # comment
    COMMENT "${BUILD_COMMENT}"
    VERBATIM
  )

  # add custom target (which always is considered out-of-date!)
  add_custom_target (
    ${TARGET_NAME} ALL
    DEPENDS ${BUILD_OUTPUT}
    SOURCES ${NON_TARGET_SOURCES}
  )

  # set MATLAB executable specific target properties
  set_target_properties (
    ${TARGET_NAME}
    PROPERTIES
      SBIA_TYPE         "MCC_${ARGN_TYPE}"
      SBIA_DEPENDS_FILE "${DEPENDS_FILE}"
  )

  # cleanup on "make clean"
  set_property (
    DIRECTORY
    APPEND PROPERTY
      ADDITIONAL_MAKE_CLEAN_FILES
        ${BUILD_OUTPUT}
        "${TEMP_DIR}/${TARGET_OUTPUT_NAME}.prj"
        "${TEMP_DIR}/mccExcludedFiles.log"
        "${TEMP_DIR}/mccBuild.log"
        "${TEMP_DIR}/readme.txt"
  )

  if (ARGN_TYPE STREQUAL "EXECUTABLE")
    set_property (
      DIRECTORY
      APPEND PROPERTY
        ADDITIONAL_MAKE_CLEAN_FILES
          "${TEMP_DIR}/${TARGET_OUTPUT_NAME}"
          "${TEMP_DIR}/run_${TARGET_OUTPUT_NAME}.sh"
          "${TEMP_DIR}/${TARGET_OUTPUT_NAME}_main.c"
          "${TEMP_DIR}/${TARGET_OUTPUT_NAME}_mcc_component_data.c"
    )
  else ()
#    set_property (
#      DIRECTORY
#      APPEND PROPERTY
#        ADDITIONAL_MAKE_CLEAN_FILES
#    )
  endif ()

  # install target
  if (ARGN_TYPE STREQUAL "EXECUTABLE")
    install (
      PROGRAMS    ${BUILD_OUTPUT}
      DESTINATION "${INSTALL_BIN_DIR}"
      COMPONENT   "${ARGN_COMPONENT}"
    )
  else ()
  endif ()
endfunction ()

# ****************************************************************************
# \function sbia_add_executable
# \brief    Add executable target. Use as replacement for add_executable ().
#
# This function adds an executable target and optionally sets its OUTPUT_NAME
# property to include the prefix specified by SBIA_OUTPUT_NAME_PREFIX when the
# option SBIA_OUTPUT_NAME_PREFIXED is ON.
#
# An install command for the added executable target is added by this function
# as well. The executable will be installed as part of the component COMPONENT
# in the directory INSTALL_BIN_DIR.
#
# Besides adding usual executable targets build by the set C/CXX language
# compiler, this function inspects the list of source files given and detects
# whether this list contains sources which need to be build using a different
# compiler. In particular, it supports the following languages:
#
# CXX    - The default behavior, adding an executable target build from C/C++
#          source code. The target is added via CMake's add_executable () command.
# MATLAB - Standalone application build from MATLAB sources using the
#          MATLAB Compiler (mcc). This option is used when the list of source
#          files contains one or more *.m files. A custom target is added which
#          depends on custom command(s) that build the executable.
#
#          \attention The *.m file with the entry point / main function of the
#                     executable has to be given before any other *.m file.
#
# \param [in] TARGET_NAME Name of the executable target.
# \param [in] ARGN        This argument list is parsed and the following
#                         arguments are extracted, all other arguments are passed
#                         on to add_executable () or the respective custom commands
#                         used to build the executable.
#
#                           COMPONENT Name of the component.
#                                     Defaults to SBIA_DEFAULT_COMPONENT.
#                           LANGUAGE  Source code language.
#                                     By default determined from the extensions
#                                     of the given source files, where CXX is
#                                     assumed if no other language is detected.

function (sbia_add_executable TARGET_NAME)
  verifyTargetName (${TARGET_NAME})

  message (STATUS "Adding executable '${TARGET_NAME}'...")

  # parse arguments
  CMAKE_PARSE_ARGUMENTS (ARGN "" "COMPONENT;LANGUAGE" "" ${ARGN})

  # if no component is specified, use default
  if (NOT ARGN_COMPONENT)
    set (ARGN_COMPONENT "${SBIA_DEFAULT_COMPONENT}")
  endif ()
  if (NOT ARGN_COMPONENT)
    set (ARGN_COMPONENT "Unspecified")
  endif ()

  # if the language is not explicitly selected, determine it from the
  # extensions of the source files
  if (NOT ARGN_LANGUAGE)
    foreach (ARG ${ARGN})
      if (ARG MATCHES "\\.m$")
        set (ARGN_LANGUAGE "MATLAB")
      endif ()
    endforeach ()
  endif ()

  # by default, consider source files as CXX language
  # (i.e., defaults to behavior of CMake's add_executable ())
  if (NOT ARGN_LANGUAGE)
    set (ARGN_LANGUAGE "CXX")
  endif ()

  # --------------------------------------------------------------------------
  # MATLAB Compiler
  # --------------------------------------------------------------------------

  if (ARGN_LANGUAGE STREQUAL "MATLAB")

    sbia_add_mcc_target (
      ${TARGET_NAME}
      TYPE      "EXECUTABLE"
      COMPONENT "${ARGN_COMPONENT}"
      ${ARGN_UNPARSED_ARGUMENTS}
    )

  # --------------------------------------------------------------------------
  # other (just wrap add_executable () by default)
  # --------------------------------------------------------------------------

  else ()

    # add executable target
    add_executable (${TARGET_NAME} ${ARGN_UNPARSED_ARGUMENTS})

    # prefix target output name
    if (SBIA_OUTPUT_NAME_PREFIXED)
      set_target_properties (
        ${TARGET_NAME}
        PROPERTIES
          OUTPUT_NAME "${SBIA_OUTPUT_NAME_PREFIX}${TARGET_NAME}"
      )
    endif ()

    # target version information
    if (WIN)
      set_target_properties (
        ${TARGET_NAME}
        PROPERTIES
          VERSION   ${PROJECT_VERSION}
          SOVERSION ${PROJECT_SOVERSION}
      )
    endif ()

    # install executable
    install (
      TARGETS     ${TARGET_NAME}
      DESTINATION "${INSTALL_BIN_DIR}"
      COMPONENT   "${ARGN_COMPONENT}"
    )
    
  endif ()

  message (STATUS "Adding executable '${TARGET_NAME}'... - done")
endfunction ()

# ****************************************************************************
# \function sbia_add_library
# \brief    Add library target. To be used as replacement for add_library ().
#
# This function adds a library target and optionally sets its OUTPUT_NAME
# property to include the prefix specified by SBIA_OUTPUT_NAME_PREFIX when the
# option SBIA_OUTPUT_NAME_PREFIXED is ON.
#
# An install command for the added library target is added by this function
# as well. The runtime library will be installed as part of the component
# RUNTIME_COMPONENT in the directory INSTALL_BIN_DIR, the static/import library
# will be installed as part of the component DEVELOPMENT_COMPONENT in the
# directory INSTALL_LIB_DIR, while the corresponding public header files will
# be installed as part of the same component in the directory INSTALL_INCLUDE_DIR.
# By default, all header files are considered public. To declare certain header
# files private and hence exclude them from the installation, add them to the
# variable <TARGET_NAME>_PRIVATE_HEADER before calling this function. Note that
# the header file path must be exacly the same as the one passed to this function
# as part of ARGN!
#
# Example:
#
# \code
# set (MYLIB_PRIVATE_HEADER
#   mylibprivate.h
#   utilities/utility.h
# )
#
# set (MYLIB_SOURCE
#   mylib.c
#   mylibpublic.h
#   ${MYLIB_PRIVATE_HEADER}
# )
#
# sbia_add_library (MyLib1 STATIC ${MYLIB_SOURCE})
# sbia_add_library (MyLib2 STATIC ${MYLIB_SOURCE} COMPONENT dev)
# sbia_add_library (MyLib3 STATIC ${MYLIB_SOURCE} RUNTIME_COMPONENT bin DEVELOPMENT_COMPONENT dev)
# \endcode
#
# \param [in] TARGET_NAME Name of the library target.
# \param [in] ARGN        Arguments passed to add_library () (excluding target name).
#                         This argument list is parsed and the following
#                         arguments are extracted, all other arguments are passed
#                         on to add_library ().
#
#                           COMPONENT                Name of the component.
#                                                    Defaults to SBIA_DEFAULT_COMPONENT.
#                           RUNTIME_COMPONENT        Name of runtime component.
#                                                    Defaults to COMPONENT.
#                           DEVELOPMENT_COMPONENT    Name of development component.
#                                                    Defaults to COMPONENT.
#
#                           STATIC|SHARED|MODULE|MEX Type of the library.
#                           EXTERNAL                 Whether the library target is
#                                                    an external library, i.e., the
#                                                    project version does not apply.

function (sbia_add_library TARGET_NAME)
  verifyTargetName (${TARGET_NAME})

  # parse arguments
  CMAKE_PARSE_ARGUMENTS (ARGN "STATIC;SHARED;MODULE;MEX;EXTERNAL" "COMPONENT;DEVELOPMENT_COMPONENT;RUNTIME_COMPONENT;LANGUAGE" "" ${ARGN})

  if (NOT ARGN_COMPONENT)
    set (ARGN_COMPONENT "${SBIA_DEFAULT_COMPONENT}")
  endif ()
  if (NOT ARGN_COMPONENT)
    set (ARGN_COMPONENT "Unspecified")
  endif ()

  if (NOT ARGN_DEVELOPMENT_COMPONENT)
    set (ARGN_DEVELOPMENT_COMPONENT "${ARGN_COMPONENT}")
  endif ()
  if (NOT ARGN_RUNTIME_COMPONENT)
    set (ARGN_RUNTIME_COMPONENT "${ARGN_COMPONENT}")
  endif ()

  # if the language is not explicitly selected, determine it from the
  # extensions of the source files
  if (NOT ARGN_LANGUAGE)
    foreach (ARG ${ARGN})
      if (ARG MATCHES "\\.m$")
        set (ARGN_LANGUAGE "MATLAB")
      endif ()
    endforeach ()
  endif ()

  # by default, consider source files as CXX language
  # (i.e., defaults to behavior of CMake's add_library ())
  if (NOT ARGN_LANGUAGE)
    set (ARGN_LANGUAGE "CXX")
  endif ()

  # status message including parsing of library type
  if (ARGN_LANGUAGE STREQUAL "MATLAB")
    message (STATUS "Adding MATLAB library '${TARGET_NAME}'...")
    if (NOT ARGN_SHARED OR ARGN_STATIC OR ARGN_MODULE OR ARGN_MEX)
      message (FATAL_ERROR "Invalid type for MATLAB library target '${TARGET_NAME}'. Only SHARED allowed.")
    endif ()
    set (ARGN_TYPE "SHARED")
  else ()
    if (NOT ARGN_SHARED AND NOT ARGN_STATIC AND NOT ARGN_MODULE AND NOT ARGN_MEX)
      if (BUILD_SHARED_LIBS)
        set (ARGN_SHARED 1)
      else ()
        set (ARGN_STATIC 0)
      endif ()
    endif ()

    if (ARGN_STATIC)
      message (STATUS "Adding static library '${TARGET_NAME}'...")
      if (ARGN_TYPE)
        message (FATAL_ERROR "More than one library type specified for target ${TARGET_NAME}.")
      endif ()
      set (ARGN_TYPE "STATIC")
    endif ()
    if (ARGN_SHARED)
      message (STATUS "Adding shared library '${TARGET_NAME}'...")
      if (ARGN_TYPE)
        message (FATAL_ERROR "More than one library type specified for target ${TARGET_NAME}.")
      endif ()
      set (ARGN_TYPE "SHARED")
    endif ()
    if (ARGN_MODULE)
      message (STATUS "Adding shared module '${TARGET_NAME}'...")
      if (ARGN_TYPE)
        message (FATAL_ERROR "More than one library type specified for target ${TARGET_NAME}.")
      endif ()
      set (ARGN_TYPE "MODULE")
    endif ()
    if (ARGN_MEX)
      message (STATUS "Adding MEX-file '${TARGET_NAME}'...")
      if (ARGN_TYPE)
        message (FATAL_ERROR "More than one library type specified for target ${TARGET_NAME}.")
      endif ()
      set (ARGN_TYPE "MEX")
    endif ()
  endif ()

  # --------------------------------------------------------------------------
  # MATLAB Compiler
  # --------------------------------------------------------------------------

  if (ARGN_LANGUAGE STREQUAL "MATLAB")

    sbia_add_mcc_target (
      ${TARGET_NAME}
      TYPE      "LIBRARY"
      COMPONENT ${ARGN_COMPONENT}
      ${ARGN_UNPARSED_ARGUMENTS}
    )

  # --------------------------------------------------------------------------
  # C/C++
  # --------------------------------------------------------------------------

  else ()

    # prefix target output name
    if (NOT EXTERNAL AND SBIA_OUTPUT_NAME_PREFIXED)
      set_target_properties (
        ${TARGET_NAME}
        PROPERTIES
          OUTPUT_NAME "${SBIA_OUTPUT_NAME_PREFIX}${NAME}"
      )
    endif ()

    # public and private headers
    set (${TARGET_NAME}_PUBLIC_HEADER "")

    foreach (SOURCE_FILE ${ARGN})
      if (${SOURCE_FILE} MATCHES "\\.\(h|hpp|hxx|inl|txx\)$")
        list (APPEND ${TARGET_NAME}_PUBLIC_HEADER "${SOURCE_FILE}")
      endif ()
    endforeach ()

    foreach (PRIVATE_HEADER_FILE ${${TARGET_NAME}_PRIVATE_HEADER})
      list (REMOVE_ITEM ${TARGET_NAME}_PUBLIC_HEADER ${PRIVATE_HEADER_FILE})
    endforeach ()

    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    # MEX-file
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    if (ARGN_TYPE STREQUAL "MEX")

      # MATLAB external library found ?
      if (NOT MATLAB_FOUND)
        message (FATAL_ERROR "MATLAB not found (or package not searched). It is required to build target ${TARGET_NAME}."
                             "Set MATLAB_ROOT or MATLAB_INCLUDE_DIR and MATLAB_LIBRARIES manually and try again.")
      endif ()

      # determine extension of MEX-files for this architecture
      if (NOT MEX_EXT)
        sbia_mexext (MEXEXT)
        set (MEX_EXT "${MEXEXT}" CACHE STRING "Extension of MEX-files." FORCE)
        mark_as_advanced (MEX_EXT)
      endif ()

      if (NOT MEX_EXT)
        message (FATAL_ERROR "Failed to determine extension of MEX-files. It is required to build target ${TARGET_NAME}."
                             "Set SBIA_CMD_MEXEXT or MEX_EXT and try again.")
      endif ()

      # add library target
      add_library (${TARGET_NAME} SHARED ${ARGN_UNPARSED_ARGUMENTS})

      target_link_libraries (${TARGET_NAME} ${MATLAB_LIBRARIES})

      # compiler flags and definitions specific to MEX-file build
      set (
        MEX_DEFINITIONS
          "_FILE_OFFSET_BITS=64"
          #"ARGCHECK"     # corresponds to MEX option -argcheck            ==> user may add via add_definitions ()
          #"MX_COMPAT_32" # corresponds to MEX option -compatibleArrayDims ==> user may add via add_definitions ()
          "MATLAB_MEX_FILE"
      )

      set (
        MEX_COMPILE_FLAGS
      )

      if (CMAKE_COMPILER_IS_GNUCC OR CMAKE_COMPILER_ISGNUCXX)
        list (APPEND MEX_DEFINITIONS "_GNU_SOURCE")
        list (APPEND MEX_COMPILE_FLAGS "-pthread -fexceptions -fno-omit-frame-pointer")
      endif ()

      set_target_properties (
        ${TARGET_NAME}
        PROPERTIES
          PREFIX              ""
          SUFFIX              ".${MEX_EXT}"
          COMPILE_FLAGS       "${MEX_COMPILE_FLAGS}"
          COMPILE_DEFINITIONS "${MEX_DEFINITIONS}"
      )

    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    # other libraries
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    else ()

      # add library target
      add_library (${TARGET_NAME} ${ARGN_TYPE} ${ARGN_UNPARSED_ARGUMENTS})

      # headers
      set_target_properties (
        ${TARGET_NAME}
        PROPERTIES
          PUBLIC_HEADER  "${${TARGET_NAME}_PUBLIC_HEADER}"
          PRIVATE_HEADER "${${TARGET_NAME}_PRIVATE_HEADER}"
      )

    endif ()

    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    # common
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    # target version information
    if (WIN AND NOT ARGN_EXTERNAL)
      set_target_properties (
        ${TARGET_NAME}
        PROPERTIES
          VERSION   "${PROJECT_VERSION}"
          SOVERSION "${PROJECT_SOVERSION}"
      )
    endif ()

    # install library
    install (
      TARGETS ${TARGET_NAME}
      RUNTIME
        DESTINATION "${INSTALL_BIN_DIR}"
        COMPONENT   "${ARGN_RUNTIME_COMPONENT}"
      ARCHIVE
        DESTINATION "${INSTALL_LIB_DIR}"
        COMPONENT   "${ARGN_DEVELOPMENT_COMPONENT}"
      LIBRARY
        DESTINATION "${INSTALL_LIB_DIR}"
        COMPONENT   "${ARGN_DEVELOPMENT_COMPONENT}"
      PUBLIC_HEADER
        DESTINATION "${INSTALL_INCLUDE_DIR}"
        COMPONENT   "${ARGN_DEVELOPMENT_COMPONENT}"
    )
  endif ()

  # done
  if (ARGN_LANGUAGE STREQUAL "MATLAB")
    message (STATUS "Adding MATLAB library '${TARGET_NAME}'... - done")
  else ()
    if (ARGN_STATIC)
      message (STATUS "Adding static library '${TARGET_NAME}'... - done")
    elseif (ARGN_SHARED)
      message (STATUS "Adding shared library '${TARGET_NAME}'... - done")
    elseif (ARGN_MODULE)
      message (STATUS "Adding shared module '${TARGET_NAME}'... - done")
    elseif (ARGN_MEX)
      message (STATUS "Adding MEX-file '${TARGET_NAME}'... - done")
    endif ()
  endif ()
endfunction ()

# ****************************************************************************
# \function sbia_add_script
# \brief    Add script ("target").
#
# This function adds a script "target" to the project, where the script is
# simply configured via configure_file () and copied to the directory
# specified by EXECUTABLE_OUTPUT_PATH. If the scripts name ends in ".in",
# the ".in" suffix is removed from the output name. Further, all occurrences
# of ".in." anywhere within the script name, these parts are removed as well.
#
# Example:
#
# \code
# sbia_add_script (MyShellScript.sh.in)
# sbia_add_script (AnotherShellScript.in.sh)
# \endcode
#
# The variable @PROJECT_BIN_DIR@ can be used within the script to get the
# absolute path of the directory where the project's scripts and executables
# are copied or installed, respectively. In case of the binary tree, this
# variable is replaced by EXECUTABLE_OUTPUT_PATH. In case of the install tree,
# it is set to INSTALL_BIN_DIR instead. Therefore, two versions of the scripts
# are configured and copied to the binary tree, one which can be used directly
# within the binary tree and another one which is copied to the install tree
# upon installation. The name of the script which will be installed contains
# the ".install" suffix within the binary tree, which is removed upon installation.
#
# Note that scripts which call other scripts or executables from this project
# should be aware of the variables SBIA_OUTPUT_NAME_PREFIX and
# SBIA_OUTPUT_NAME_PREFIXED. When SBIA_OUTPUT_NAME_PREFIXED is "ON", the
# script needs to prefix the other scripts or executables names by
# SBIA_OUTPUT_NAME_PREFIX. Alternatively, the script can just prepand the
# string passed by the variable PROJECT_OUTPUT_NAME_PREFIX, which is an
# empty string if no prefix is used.
#
# Example MyShellScript.sh.in:
#
# \code
# if [ "@SBIA_OUTPUT_NAME_PREFIXED@" == "ON" ]; then
#   OtherScript='@PROJECT_BIN_DIR@/@SBIA_OUTPUT_NAME_PREFIX@AnotherShellScript.sh'
# else
#   OtherScript='@PROJECT_BIN_DIR@/AnotherShellScript.sh'
# fi
#
# OtherScript
# \endcode
#
# or
#
# \code
# prefix="@PROJECT_OUTPUT_NAME_PREFIX@
# OtherScript=@PROJECT_BIN_DIR@/${prefix}AnotherShellScript.sh
#
# OtherScript
# \endcode
#
# \param [in] SCRIPT_NAME Filename of the script within the current directory.
# \param [in] ARGN        This argument list is parsed and the following
#                         arguments are extracted:
#
#                           COMPONENT Name of the component.
#                                     Defaults to SBIA_DEFAULT_COMPONENT.

function (sbia_add_script SCRIPT_NAME)
  # parse arguments
  CMAKE_PARSE_ARGUMENTS (ARGN "" "COMPONENT" "" ${ARGN})

  if (NOT ARGN_COMPONENT)
    set (ARGN_COMPONENT "${SBIA_DEFAULT_COMPONENT}")
  endif ()
  if (NOT ARGN_COMPONENT)
    set (ARGN_COMPONENT "Unspecified")
  endif ()

  if (ARGN_UNPARSED_ARGUMENTS)
    message ("Unknown arguments given for sbia_add_script ('${SCRIPT_NAME}'): '${ARGN_UNPARSED_ARGUMENTS}'")
  endif ()

  # prefix target output name
  set (PROJECT_OUTPUT_NAME_PREFIX "")
  if (SBIA_OUTPUT_NAME_PREFIXED)
    set (TARGET_NAME "${SBIA_OUTPUT_NAME_PREFIX}${SCRIPT_NAME}")
    set (PROJECT_OUTPUT_NAME_PREFIX "${SBIA_OUTPUT_NAME_PREFIX}")
  else ()
    set (TARGET_NAME "${SCRIPT_NAME}")
  endif ()

  # remove ".in" from script name
  string (REGEX REPLACE "\\.in$"          ""    TARGET_NAME ${TARGET_NAME})
  string (REGEX REPLACE "\\.in\(\\..*\)$" "\\1" TARGET_NAME ${TARGET_NAME})

  # remove extension from script name (if UNIX system and sha-bang directive exists)
  if (UNIX)
    file (STRINGS "${CMAKE_CURRENT_SOURCE_DIR}/${TARGET_NAME}" SHABANG LIMIT_COUNT 2 LIMIT_INPUT 2)
    if (SHABANG STREQUAL "#!")
      get_filename_component (TARGET_NAME "${TARGET_NAME}" NAME_WE)
    endif ()
  endif ()

  message (STATUS "Adding script '${TARGET_NAME}'...")

  # configure/copy script
  set (PROJECT_BIN_DIR "${EXECUTABLE_OUTPUT_PATH}")
  configure_file ("${CMAKE_CURRENT_SOURCE_DIR}/${SCRIPT_NAME}" "${EXECUTABLE_OUTPUT_PATH}/${TARGET_NAME}" @ONLY)

  set (PROJECT_BIN_DIR "${CMAKE_INSTALL_PREFIX}/${INSTALL_BIN_DIR}")
  configure_file ("${CMAKE_CURRENT_SOURCE_DIR}/${SCRIPT_NAME}" "${EXECUTABLE_OUTPUT_PATH}/${TARGET_NAME}.install" @ONLY)

  if (UNIX)
    execute_process (COMMAND chmod +x "${EXECUTABLE_OUTPUT_PATH}/${TARGET_NAME}")
  endif ()

  # cleanup on "make clean"
  set_property (
    DIRECTORY
    APPEND PROPERTY
      ADDITIONAL_MAKE_CLEAN_FILES
        "${EXECUTABLE_OUTPUT_PATH}/${TARGET_NAME}"
        "${EXECUTABLE_OUTPUT_PATH}/${TARGET_NAME}.install"
  )

  # install script
  install (
    PROGRAMS    "${EXECUTABLE_OUTPUT_PATH}/${TARGET_NAME}.install"
    RENAME      "${TARGET_NAME}"
    DESTINATION "${INSTALL_BIN_DIR}"
    COMPONENT   "${ARGN_COMPONENT}"
  )

  message (STATUS "Adding script '${TARGET_NAME}'... - done")
endfunction ()

# ****************************************************************************
# \function sbia_add_scripts_by_extension
# \brief    Adds scripts with specified extension.
#
# This function calls sbia_add_script () for each script within the current
# source directory which has the extension ".<EXT>" or ".<EXT>.in".
#
# \param [in] EXT  Script extension, e.g., "sh" for shell scripts.
# \param [in] ARGN This argument list is parsed and the following
#                  arguments are extracted:
#
#                    COMPONENT Name of the component.
#                              Defaults to SBIA_DEFAULT_COMPONENT.

function (sbia_add_scripts_by_extension EXT)
  if (CMAKE_VERBOSE)
    file (RELATIVE_PATH DIR "${PROJECT_SOURCE_DIR}" "${CMAKE_CURRENT_SOURCE_DIR}")
    message (STATUS "Adding scripts in '${DIR}' with extension '.${EXT}' or '.${EXT}.in'")
    set (DIR)
  endif ()

  # parse arguments
  CMAKE_PARSE_ARGUMENTS (ARGN "" "COMPONENT" "" ${ARGN})

  if (NOT ARGN_COMPONENT)
    set (ARGN_COMPONENT "${SBIA_DEFAULT_COMPONENT}")
  endif ()
  if (NOT ARGN_COMPONENT)
    set (ARGN_COMPONENT "Unspecified")
  endif ()

  if (ARGN_UNPARSED_ARGUMENTS)
    message ("Unknown arguments given for sbia_add_scripts_by_extension ('${EXT}'): '${ARGN_UNPARSED_ARGUMENTS}'")
  endif ()

  # glob script files with given extension
  file (GLOB FILES RELATIVE "${CMAKE_CURRENT_SOURCE_DIR}" "${CMAKE_CURRENT_SOURCE_DIR}/*")

  # add scripts
  foreach (SCRIPT ${FILES})
    if (NOT IS_DIRECTORY "${SCRIPT}")
      if ("${SCRIPT}" MATCHES ".*\\.${EXT}$|.*\\.${EXT}\\.in$")
        sbia_add_script ("${SCRIPT}" COMPONENT "${ARGN_COMPONENT}")
      endif ()
    endif ()
  endforeach ()
endfunction ()

# ****************************************************************************
# \macro sbia_add_scripts_by_extensions
# \brief Adds scripts with specified extensions.
#
# \see sbia_add_scripts_by_extension
#
# \param [in] ARGN This argument list is parsed and the following arguments
#                  are extracted. All other arguments are considered script
#                  extension.
#
#                    COMPONENT Name of the component.

macro (sbia_add_scripts_by_extensions)
  # parse arguments
  CMAKE_PARSE_ARGUMENTS (ARGN "" "COMPONENT" "" ${ARGN})

  if (NOT ARGN_COMPONENT)
    set (ARGN_COMPONENT "${SBIA_DEFAULT_COMPONENT}")
  endif ()
  if (NOT ARGN_COMPONENT)
    set (ARGN_COMPONENT "Unspecified")
  endif ()

  # add scripts by extension
  foreach (EXT ${ARGN_UNPARSED_ARGUMENTS})
    sbia_add_scripts_by_extension ("${EXT}" COMPONENT "${ARGN_COMPONENT}")
  endforeach ()
endmacro ()

# ****************************************************************************
# \macro sbia_add_scripts
# \brief Adds scripts with default extensions.
#
# This macro adds each script within the current source directory which has
# a default extension using sbia_add_script ().
#
# Considered default extensions are "sh" for shell scripts, "py" for Python
# scripts, and "pl" for Perl scripts.
#
# \param [in] ARGN This argument list is parsed and the following arguments
#                  are extracted.
#
#                    COMPONENT Name of the component.

macro (sbia_add_scripts)
  # parse arguments
  CMAKE_PARSE_ARGUMENTS (ARGN "" "COMPONENT" "" ${ARGN})

  if (NOT ARGN_COMPONENT)
    set (ARGN_COMPONENT "${SBIA_DEFAULT_COMPONENT}")
  endif ()
  if (NOT ARGN_COMPONENT)
    set (ARGN_COMPONENT "Unspecified")
  endif ()

  if (ARGN_UNPARSED_ARGUMENTS)
    message ("Unknown arguments given for sbia_add_scripts (): '${ARGN_UNPARSED_ARGUMENTS}'")
  endif ()

  # add scripts with known extensions
  sbia_add_scripts_by_extension (sh) # shell scripts
  sbia_add_scripts_by_extension (py) # python scripts
  sbia_add_scripts_by_extension (pl) # Perl scripts
endmacro ()

# ****************************************************************************
# \function sbia_add_test
# \brief    Add CTest test.
#
# \param [in] TEST_NAME Name of the test.
# \param [in] ARGN      Parameters passed to add_test () (excluding test name).

function (sbia_add_test TEST_NAME)
  message (STATUS "Adding test '${TEST_NAME}'...")

  add_test (${TEST_NAME} ${ARGN})

  message (STATUS "Adding test '${TEST_NAME}'... - done")
endfunction ()

# ****************************************************************************
# \function sbia_add_doc
# \brief    Adds documentation target.
#
# This function is especially used to add a custom target to the "doc" target
# which is used to generate documentation from input files such as in
# particular source code files. Other documentation files such as HTML, Word,
# or PDF documents can be added as well using this function. A component
# as part of which this documentation shall be installed can be specified.
#
# The documentation files are installed in/as INSTALL_DOC_DIR/DOC_NAME as part
# of the component specified by the COMPONENT option.
#
# \note If the option BUILD_DOC is not defined or does not evaluate to true,
#       only documentation generated with the NONE generator is build and
#       installed.
#
# Example:
#
# \code
# sbia_add_doc (UserManual.pdf)
# sbia_add_doc (DeveloperManual.docx COMPONENT dev)
# sbia_add_doc (SourceManual.html    COMPONENT src)
# \endcode
#
# \param [in] TARGET_NAME Name of the documentation target or file.
# \param [in] ARGN        Further options which are given as pairs
#                         "OPTION_NAME <OPTION_VALUE>".
#
# Common options:
#
# COMPONENT Name of the component this documentation belongs to.
# GENERATOR Documentation generator, where the case of the generator name is
#           ignored, i.e., "Doxygen", "DOXYGEN", "doxYgen" are all valid
#           arguments which select the DOXYGEN generator. The arguments for the
#           the different supported generators are documented below.
#           The default generator is "NONE". The NONE generator simply installs
#           the document with the filename TARGET_NAME and has no own options.
#
# Generator: DOXYGEN
# 
# Example:
#
# \code
# sbia_add_doc (
#   API
#   GENERATOR Doxygen
#     DOXYGEN_DOXYFILE        "Doxyfile.in"
#     DOXYGEN_PROJECT_NAME    "${PROJECT_NAME}"
#     DOXYGEN_PROJECT_VERSION "${PROJECT_VERSION}"
#   COMPONENT dev
# )
# \endcode
#
# Options of DOXYGEN generator:
#
# \see http://www.stack.nl/~dimitri/doxygen/config.html
#
# DOXYGEN_DOXYFILE         Name of the template Doxyfile.
# DOXYGEN_PROJECT_NAME     Value for Doxygen's PROJECT_NAME tag which is used
#                          to specify the project name.
#                          Default: "<PROJECT_NAME>".
# DOXYGEN_PROJECT_NUMBER   Value for Doxygen's PROJECT_NUMBER tag which is used
#                          to specify the project version number.
#                          Default: <PROJECT_VERSION>.
# DOXYGEN_INPUT            Value for Doxygen's INPUT tag which is used to
#                          specify input directories/files.
#                          Default: "<PROJECT_SOURCE_DIR>/Code <PROJECT_BINARY_DIR>/Code".
# DOXYGEN_FILTER_PATTERNS  Value for Doxygen's FILTER_PATTERNS tag which
#                          can be used to specify filters on a per file
#                          pattern basis.
# DOXYGEN_EXCLUDE          Value for Doxygen's EXCLUDE tag which can be used
#                          to specify files and/or directories that should 
#                          excluded from the INPUT source files.
#                          Default: ""
# DOXYGEN_OUTPUT_DIRECTORY Value for Doxygen's OUTPUT_DIRECTORY tag which
#                          can be used to specify the output directory.
#                          Default: "<CMAKE_CURRENT_BINARY_DIR>/<TARGET_NAME>".
# DOXYGEN_GENERATE_HTML    Either "YES" or "NO". Default is "YES".
# DOXYGEN_GENERATE_LATEX   Either "YES" or "NO". Default is "NO".
# DOXYGEN_GENERATE_RTF     Either "YES" or "NO". Default is "NO".
# DOXYGEN_GENERATE_MAN     Either "YES" or "NO". Default is "NO".
#
# Generator: SVN2CL
#
# Example:
#
# \code
# sbia_add_doc (
#   ChangeLog
#   GENERATOR svn2cl
#   COMPONENT dev
# )
# \endcode
#

function (sbia_add_doc TARGET_NAME)
  verifyTargetName (${TARGET_NAME})

  message (STATUS "Adding documentation '${TARGET_NAME}'...")

  # --------------------------------------------------------------------------
  # default common options
  # --------------------------------------------------------------------------

  set (GENERATOR "NONE")
  set (COMPONENT "${SBIA_DEFAULT_COMPONENT}")

  # --------------------------------------------------------------------------
  # parse common options
  # --------------------------------------------------------------------------

  set (OPTION_NAME "")

  foreach (ARG ${ARGN})
    # if previous argument was a constant which specifies which argument
    # comes next, set the parameters value to the current argument
    if (OPTION_NAME)
      set (${OPTION_NAME} "${ARG}")
      set (OPTION_NAME "")
    # otherwise, determine whether we encounter another common option
    else ()
      if (
           "${ARG}" STREQUAL "GENERATOR"
        OR "${ARG}" STREQUAL "COMPONENT"
      )
        set (OPTION_NAME "${ARG}")
      endif ()
    endif ()
  endforeach ()

  # generator name is case insensitive
  string (TOUPPER "${GENERATOR}" GENERATOR)

  # --------------------------------------------------------------------------
  # generator: NONE
  # --------------------------------------------------------------------------

  if (GENERATOR STREQUAL "NONE")

    # install documentation file
    install (
      FILES       "${CMAKE_CURRENT_SOURCE_DIR}/${TARGET_NAME}"
      DESTINATION "${INSTALL_DOC_DIR}"
      COMPONENT   "${COMPONENT}"
    )

  # --------------------------------------------------------------------------
  # generator: DOXYGEN
  # --------------------------------------------------------------------------

  elseif (GENERATOR STREQUAL "DOXYGEN")

    # find Doxygen installation
    if (NOT DOXYGEN_EXECUTABLE)
      find_package (Doxygen)
    endif ()

    if (NOT DOXYGEN_EXECUTABLE)
      message ("Could not find Doxygen installation. Skipping build of '${TARGET_NAME}'")
      message (STATUS "Adding documentation '${TARGET_NAME}'... - failed")
      return ()
    endif ()

    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    # default options
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    set (DOXYGEN_DOXYFILE         "")
    set (DOXYGEN_PROJECT_NAME     "${PROJECT_NAME}")
    set (DOXYGEN_PROJECT_NUMBER   "${PROJECT_VERSION}")
    set (DOXYGEN_INPUT            "${PROJECT_SOURCE_DIR}/Code ${PROJECT_BINARY_DIR}/Code")
    set (DOXYGEN_FILTER_PATTERNS  "")
    set (DOXYGEN_EXCLUDE          "")
    set (DOXYGEN_OUTPUT_DIRECTORY "${CMAKE_CURRENT_BINARY_DIR}/${TARGET_NAME}")
    set (DOXYGEN_GENERATE_HTML    "YES")
    set (DOXYGEN_GENERATE_LATEX   "NO")
    set (DOXYGEN_GENERATE_RTF     "NO")
    set (DOXYGEN_GENERATE_MAN     "NO")

    # click & jump in Emacs and Visual Studio
    if (CMAKE_BUILD_TOOL MATCHES "(msdev|devenv)")
      set (DOXYGEN_WARN_FORMAT "\"$file($line) : $text \"")
    else ()
      set (DOXYGEN_WARN_FORMAT "\"$file:$line: $text \"")
    endif ()

    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    # parse options
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    foreach (ARG ${ARGN})
      # if previous argument was a constant which specifies which argument
      # comes next, set the parameters value to the current argument
      if (OPTION_NAME)
        set (${OPTION_NAME} "${ARG}")
        set (OPTION_NAME "")
      # otherwise, determine whether we encountered another DOXYGEN option
      else ()
        if (ARG MATCHES "DOXYGEN_.*")
          set (OPTION_NAME "${ARG}")
        endif ()
      endif ()
    endforeach ()

    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    # check options
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    if (NOT DOXYGEN_DOXYFILE)
      message (FATAL_ERROR "Missing option DOXYGEN_DOXYFILE.")
    endif ()

    if (NOT EXISTS "${DOXYGEN_DOXYFILE}")
      message (FATAL_ERROR "Input Doxyfile '${DOXYGEN_DOXYFILE}' not found")
    endif ()

    if (DOXYGEN_GENERATE_HTML)
      set (DOXYGEN_GENERATE_HTML "YES")
    else ()
      set (DOXYGEN_GENERATE_HTML "NO")
    endif ()

    if (DOXYGEN_GENERATE_LATEX)
      set (DOXYGEN_GENERATE_LATEX "YES")
    else ()
      set (DOXYGEN_GENERATE_LATEX "NO")
    endif ()

    if (DOXYGEN_GENERATE_RTF)
      set (DOXYGEN_GENERATE_RTF "YES")
    else ()
      set (DOXYGEN_GENERATE_RTF "NO")
    endif ()

    if (DOXYGEN_GENERATE_MAN)
      set (DOXYGEN_GENERATE_MAN "YES")
    else ()
      set (DOXYGEN_GENERATE_MAN "NO")
    endif ()
 
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    # Doxygen target
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    # configure Doxyfile
    set (DOXYFILE "${CMAKE_CURRENT_BINARY_DIR}/Sbia${PROJECT_NAME}${TARGET_NAME}.doxy")
    configure_file ("${DOXYGEN_DOXYFILE}" "${DOXYFILE}" @ONLY)

    # add target
    add_custom_target (
      ${TARGET_NAME}
      COMMAND "${DOXYGEN_EXECUTABLE}" "${DOXYFILE}"
    )

    # cleanup on "make clean"
    set_property (
      DIRECTORY
      APPEND PROPERTY
        ADDITIONAL_MAKE_CLEAN_FILES
          "${DOXYGEN_OUTPUT_DIRECTORY}/html"
          "${DOXYGEN_OUTPUT_DIRECTORY}/latex"
          "${DOXYGEN_OUTPUT_DIRECTORY}/rtf"
          "${DOXYGEN_OUTPUT_DIRECTORY}/man"
    )

    # add target as dependency to doc target
    if (NOT TARGET doc)
      if (BUILD_DOC)
        add_custom_target (doc ALL)
      else ()
        add_custom_target (doc)
      endif ()
    endif ()

    add_dependencies (doc ${TARGET_NAME})

    # install documentation
    install (
      DIRECTORY       "${DOXYGEN_OUTPUT_DIRECTORY}/"
      DESTINATION     "${INSTALL_DOC_DIR}/${TARGET_NAME}"
      COMPONENT       "${COMPONENT}"
      FILES_MATCHING
      PATTERN         html/*
      PATTERN         latex/*
      PATTERN         rtf/*
      PATTERN         man/*
      # exclude directories as FILES_MATCHING does not do this
      PATTERN         CMakeFiles EXCLUDE # some CMake files
      PATTERN         .svn       EXCLUDE # for in-source builds
      PATTERN         .git       EXCLUDE # for in-source builds
    )

  # --------------------------------------------------------------------------
  # generator: svn2cl
  # --------------------------------------------------------------------------

  elseif (GENERATOR STREQUAL "SVN2CL")

    if (NOT SBIA_CMD_SVN2CL)
      message ("Could not find svn2cl installation. Skipping build of '${TARGET_NAME}'")
      message (STATUS "Adding documentation '${TARGET_NAME}'... - failed")
      return ()
    endif ()

    sbia_svn_get_revision (${PROJECT_SOURCE_DIR} REV)

    if (NOT REV)
      message ("Project is not under SVN control. Skipping build of '${TARGET_NAME}'")
      return ()
    endif ()

    # svn2cl command arguments
    set (SVN2CL_PATH             "${PROJECT_SOURCE_DIR}")
    set (SVN2CL_OUTPUT           "${PROJECT_BINARY_DIR}/${TARGET_NAME}")
    set (SVN2CL_AUTHORS          "${PROJECT_SOURCE_DIR}/Authors")
    set (SVN2CL_LINELEN          79)
    set (SVN2CL_REPARAGRAPH      0)
    set (SVN2CL_INCLUDE_ACTIONS  1)
    set (SVN2CL_INCLUDE_REV      1)
    set (SVN2CL_BREAK_BEFORE_MSG 1)
    set (SVN2CL_GROUP_BY_DAY     1)

    set (SVN2CL_ARGS "--output=${SVN2CL_OUTPUT}")
    if (SVN2CL_LINELEN)
      list (APPEND SVN2CL_ARGS "--linelen=${SVN2CL_LINELEN}")
    endif ()
    if (SVN2CL_REPARAGRAPH)
      list (APPEND SVN2CL_ARGS "--reparagraph")
    endif ()
    if (SVN2CL_INCLUDE_ACTIONS)
      list (APPEND SVN2CL_ARGS "--include-actions")
    endif ()
    if (SVN2CL_BREAK_BEFORE_MSG)
      list (APPEND SVN2CL_ARGS "--break-before-msg=${SVN2CL_BREAK_BEFORE_MSG}")
    endif ()
    if (SVN2CL_GROUP_BY_DAY)
      list (APPEND SVN2CL_ARGS "--group-by-day")
    endif ()

    if (EXISTS "${SVN2CL_AUTHORS}")
      list (APPEND SVN2CL_ARGS "--authors=${SVN2CL_AUTHORS}")
    elseif (EXISTS "${SVN2CL_AUTHORS}.xml")
      list (APPEND SVN2CL_ARGS "--authors=${SVN2CL_AUTHORS}.xml")
    elseif (EXISTS "${SVN2CL_AUTHORS}.txt")
      list (APPEND SVN2CL_ARGS "--authors=${SVN2CL_AUTHORS}.txt")
    endif ()

    # add target
    add_custom_target (
      ${TARGET_NAME}
      COMMAND           "${SBIA_CMD_SVN2CL}" ${SVN2CL_ARGS} "${SVN2CL_PATH}"
      WORKING_DIRECTORY "${PROJECT_BINARY_DIR}"
      COMMENT           "Generating ${TARGET_NAME} from SVN log..."
    )

    # cleanup on "make clean"
    set_property (
      DIRECTORY
      APPEND PROPERTY
        ADDITIONAL_MAKE_CLEAN_FILES
          "${SVN2CL_OUTPUT}"
    )

    # add target as dependency to changelog target
    # \note svn2cl may require the user to enter the SVN password
    #       hence, exclude this target from ALL!
    if (NOT TARGET changelog)
      add_custom_target (changelog)
    endif ()

    add_dependencies (changelog ${TARGET_NAME})

    # install documentation
    install (
      FILES       "${SVN2CL_OUTPUT}"
      DESTINATION "${CMAKE_INSTALL_PREFIX}"
      COMPONENT   "${COMPONENT}"
      OPTIONAL
    )

  # --------------------------------------------------------------------------
  # generator: unknown
  # --------------------------------------------------------------------------

  else ()
    message (FATAL_ERROR "Unknow documentation generator: ${GENERATOR}.")
  endif ()

  message (STATUS "Adding documentation '${TARGET_NAME}'... - done")
endfunction ()

# ============================================================================
# packaging
# ============================================================================

# ****************************************************************************
# \function sbia_add_component_group
# \brief    Add component group.
#
# \see http://www.cmake.org/pipermail/cmake/2008-August/023336.html
# \see cpack_add_component_group ()
#
# \param [in] GRPNAME Name of the component group.
# \param [in] ARGN    Further arguments passed to cpack_add_component_group ().

function (sbia_add_component_group GRPNAME)
  set (OPTION_NAME)
  set (PARENT_GROUP)

  foreach (ARG ${ARGN})
    if (OPTION_NAME)
      set (${OPTION_NAME} "${ARG}")
      set (OPTION_NAME)
      break ()
    else ()
      if ("${ARG}" STREQUAL "PARENT_GROUP")
        set (OPTION_NAME "PARENT_GROUP")
      endif ()
    endif ()
  endforeach ()

  cpack_add_component_group (${GRPNAME} ${ARGN})

  add_custom_target (install_${GRPNAME})

  if (PARENT_GROUP)
    add_dependencies (install_${PARENT_GROUP} install_${GRPNAME})
  endif ()
endfunction ()

# ****************************************************************************
# \function sbia_add_component
# \brief    Add component.
#
# \see http://www.cmake.org/pipermail/cmake/2008-August/023336.html
# \see cpack_add_component ()
#
# \param [in] COMPNAME Name of the component.
# \param [in] ARGN     Further arguments passed to cpack_add_component ().

function (sbia_add_component COMPNAME)
  set (OPTION_NAME)
  set (GROUP)

  foreach (ARG ${ARGN})
    if (OPTION_NAME)
      set (${OPTION_NAME} "${ARG}")
      set (OPTION_NAME)
      break ()
    else ()
      if ("${ARG}" STREQUAL "GROUP")
        set (OPTION_NAME "GROUP")
      endif ()
    endif ()
  endforeach ()

  cpack_add_component (${COMPNAME} ${ARGN})

  add_custom_target (
    install_${COMPNAME}
    COMMAND
      "${CMAKE_COMMAND}"
        -DCOMPONENT=${COMPNAME}
        -P "${PROJECT_BINARY_DIR}/cmake_install.cmake"
  )

  if (GROUP)
    add_dependencies (install_${GROUP} install_${COMPNAME})
  endif ()
endfunction ()

# ============================================================================
# license
# ============================================================================

# ****************************************************************************
# \function sbia_update_license
# \brief    Updates license information of file.
#
# This function updates the license information of the specified file such
# that it is identical to the content of the file specified by
# PROJECT_LICENSE_FILE. If the file PROJECT_LICENSE_FILE is not found,
# nothing is done by this macro.
#
# \note This function is disabled when PROJECT_REVISION is not a valid
#       revision number, i.e., is equal to zero. Thus, the license information
#       is only kept in sync with PROJECT_LICENSE_FILE when the project is
#       build from a working copy of the revision controlled project.
#
# \note This function is currently not used. The update of the license
#       information of source files should be triggered manually instead
#       by executing a script that recursively traverses the source tree and
#       calls SbiaUpdateLicense.py for each encountered source file whenever
#       the PROJECT_LICENSE_FILE has been modified.
#
# \param [in] FILENAME Name of file relative to current source directory.
# \param [in] ARGN     An additional prefix string can be specified which is
#                      put at the beginning of each new line of the license
#                      notice. For example, "# ".

function (sbia_update_license FILENAME)
  if (ARGC GREATER 1)
    set (OPT "-p ${ARGV2}")
  else ()
    set (OPT "")
  endif ()

  # is project under revision control ?
  if (NOT PROJECT_REVISION)
    return ()
  endif ()

  # check existence of PROJECT_LICENSE_FILE
  if (NOT EXISTS "${PROJECT_LICENSE_FILE}")
    return ()
  endif ()

  # find update license script
  if (NOT SBIA_CMD_PYTHON)
    return ()
  endif ()

  find_file (
    SBIA_UPDATE_LICENSE_SCRIPT
    NAMES SbiaUpdateLicense.py
    PATHS ${CMAKE_MODULE_PATH}
    NO_DEFAULT_PATHS
  )

  if (NOT SBIA_UPDATE_LICENSE_SCRIPT)
    return ()
  endif ()

  # absolute path of project file
  set (CUR "${CMAKE_CURRENT_SOURCE_DIR}/${FILENAME}")

  # get path of file relative to project source directory
  file (RELATIVE_PATH REL "${PROJECT_SOURCE_DIR}" "${CUR}")

  # is file excluded from udpate ?
  list (FIND SBIA_UPDATE_LICENSE_EXCLUDE "${REL}" IDX)

  if (IDX GREATER -1)
    return ()
  endif ()

  # update license information
  execute_process (
    COMMAND "${SBIA_CMD_PYTHON}" "${SBIA_UPDATE_LICENSE_SCRIPT}" ${OPT} -i "${CUR}" -l "${PROJECT_LICENSE_FILE}" -o "${CUR}"
    RESULT_VARIABLE RETVAL
    OUTPUT_QUIET
    ERROR_QUIET
  )

  # notify user
  if (RETVAL EQUAL 0)
    message ("Updated license of file '${REL}'")
  elseif (NOT RETVAL EQUAL 2)
    if (CMAKE_VERBOSE)
      message ("Failed to update license of file '${REL}'")
    endif ()
  endif ()
endfunction ()

# ****************************************************************************
# \macro sbia_update_license_exclude
# \brief Exclude file in current source directory from license update.

macro (sbia_update_license_exclude FILENAME)
  set (CUR "${CMAKE_CURRENT_SOURCE_DIR}/${FILENAME}")
  file (RELATIVE_PATH REL "${PROJECT_SOURCE_DIR}" "${CUR}")
  set (SBIA_UPDATE_LICENSE_EXCLUDE ${SBIA_UPDATE_LICENSE_EXCLUDE} ${REL})
endmacro ()

# ****************************************************************************
# \macro sbia_update_licenses
# \brief Update license of all files in current source directory.

macro (sbia_update_licenses)
  GLOB (FILENAMES RELATIVE "${CMAKE_CURRENT_SOURCE_DIR}" "*")

  foreach (FILENAME ${FILENAMES})
    sbia_update_license (${FILENAME})
  endforeach ()

  set (FILENAMES)
endmacro ()
