function [cmds chk_files] = ...
                run_aladin_ants_registration(target, atlas, labels, ...
                                             out_prefix, out_type, ...
                                             out_warp_dir, out_seg_dir, ...
                                             out_atlas_dir, out_temp_dir,...
                                             regloc, opts);
% RUN_ALADIN_ANTS_REGISTRATION - runs the ALADIN-ANTS registration framework
%                                between a given target and an atlas with labels
%                              - In other words, it runs reg_aladin, followed
%                                by ANTs
%
% [cmds chk_files] = run_aladin_ants_registration(target, atlas, labels, ...
%                                                 out_prefix, out_type, ...
%                                                 out_warp_dir, out_seg_dir, ...
%                                                 out_atlas_dir, ...
%                                                 out_temp_dir, regloc, opts);
%
% Input: target - the target filename
%        atlas - the atlas filename
%        labels - the atlas labels filename
%        out_prefix - the output prefix for the filenames
%        out_type - the output filetype (1 - .nii, 2 - .nii.gz)
%        out_warp_dir - the output directory for the warp files
%        out_seg_dir - the output directory for the segmentation files
%        out_atlas_dir - the output directory for the registred atlas files
%        out_temp_dir - the output directory to store the temporary files
%        regloc - the location of the registration application
%        opts - the registration-specific options
%
% Output: cmds - a cell array of commands to run
%         chk_files - a cell array of files that should be there when done
%

% set some filenames
wrp_pfix = sprintf('%s/%s_', out_warp_dir, out_prefix);
aff_name = sprintf('%s/%s_Affine.txt', out_warp_dir, out_prefix);
aladin_aff_name = sprintf('%s/%s_aladin_Affine.txt', out_warp_dir, out_prefix);
wrp_name = sprintf('%s/%s_0Warp.nii.gz', out_warp_dir, out_prefix);
if out_type == 1
    seg_name = sprintf('%s/%s_seg.nii', out_seg_dir, out_prefix);
    reg_name = sprintf('%s/%s.nii', out_atlas_dir, out_prefix);
    aff_seg_name = sprintf('%s/%s_aff_seg.nii', out_seg_dir, out_prefix);
    aff_reg_name = sprintf('%s/%s_aff.nii', out_atlas_dir, out_prefix);
else
    seg_name = sprintf('%s/%s_seg.nii.gz', out_seg_dir, out_prefix);
    reg_name = sprintf('%s/%s.nii.gz', out_atlas_dir, out_prefix);
    aff_seg_name = sprintf('%s/%s_aff_seg.nii.gz', out_seg_dir, out_prefix);
    aff_reg_name = sprintf('%s/%s_aff.nii.gz', out_atlas_dir, out_prefix);
end

chk_files{1} = seg_name;
chk_files{2} = reg_name;
chk_files{3} = aladin_aff_name;

% ANTS options
if ~isfield(opts, 'simmetric')
    opts.simmetric = 'CC';
end
if ~isfield(opts, 'simopts')
    opts.simopts = '1,3';
end
if ~isfield(opts, 'addopts')
    opts.addopts = ['-d 3 -u 1 -w [0.01,0.99] -c [100x100x50x10,1e-9,15] ', ...
                    '-t SyN[0.1,3,0] -f 6x4x2x1 -s 3x2x1x0'];
end

% reg aladin options
if ~isfield(opts, 'aladin_loc')
    error('opts.aladin_loc must be defined');
end
if ~isfield(opts, 'aladin_addopts')
    opts.aladin_addopts = '-%v 90 -%i 90';
end

% set the locations of the executables
ALADIN_reg = [opts.aladin_loc, '/reg_aladin'];
ALADIN_apply = [opts.aladin_loc, '/reg_resample'];
ANTS_reg = [regloc, '/bin/antsRegistration'];
ANTS_apply = [regloc, '/bin/antsApplyTransforms'];

% initialize the cmds struct
cmds = {};

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% run reg_aladin
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% run the affine registration
cmds{1} = sprintf('%s -ref %s -flo %s -aff %s -res %s %s\n', ...
                  ALADIN_reg, target, atlas, aladin_aff_name, aff_reg_name, ...
                  opts.aladin_addopts);

% apply the affine registration
cmds{2} = sprintf('%s -ref %s -flo %s -aff %s -res %s -NN', ...
                  ALADIN_apply, target, labels, aladin_aff_name, aff_seg_name);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% run ANTS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% run the non-rigid registration
cmds{3} = sprintf('%s -m %s[%s,%s,%s] -o %s %s\n', ...
                  ANTS_reg, opts.simmetric, target, ...
                  aff_reg_name, opts.simopts, wrp_pfix, opts.addopts);

% apply the deformation to the atlas
cmds{4} = sprintf('%s -i %s -o %s -r %s -n BSpline -t %s \n', ...
                  ANTS_apply, aff_reg_name, reg_name, target, wrp_name);

% apply the deformation to the labels
cmds{5} = sprintf('%s -i %s -o %s -r %s -n NearestNeighbor -t %s\n', ...
                  ANTS_apply, aff_seg_name, seg_name, target, wrp_name);

