function [strip_dir] = run_bet_labels(atlases, labels, out_dir)
% RUN_BET_LABELS - runs a brain extraction given a set of labels the easy way
%                  (i.e., it simply removes any voxels with label == 0)
%
% [strip_dir] = run_bet_labels(atlases, labels, out_dir)
%
% Input: atlases - a cell array of atlas filenames
%        labels - a cell array of label filenames
%        out_dir - the output directory
%
% Output: strip_dir - the directory where the stripped atlases are stored

% make sure the output directory exists
if ~exist(out_dir, 'dir')
    error(sprintf('Output directory %s does not exist', out_dir));
end

% set the output directory
strip_dir = sprintf('%s/bet-atlases/', out_dir);

if ~exist(strip_dir, 'dir')
    mkdir(strip_dir)
end

% iterate over all of the atlases
for i = 1:length(atlases)

    [a b c] = fileparts(atlases{i});
    out_file = sprintf('%s/%s%s', strip_dir, b, c);

    if exist(out_file, 'file')
        disp(sprintf('Skipping %s: output files exist', out_file));
        pause(0.01);
        continue;
    end

    % set the output prefix
    [a_dir a_name a_ext] = fileparts(atlases{i});
    [l_dir l_name l_ext] = fileparts(labels{i});

    % set the output type
    out_type = 1;

    % load the atlas
    if strcmp(a_name(end-3:end), '.nii')
        out_type = 2;
        a = load_untouch_nii_gz(atlases{i});
    else
        a = load_untouch_nii(atlases{i});
    end

    % load the labels
    if strcmp(l_name(end-3:end), '.nii')
        b = load_untouch_nii_gz(labels{i});
    else
        b = load_untouch_nii(labels{i});
    end

    % set the output image
    c = a;
    c.img(b.img == 0) = 0;
    c.hdr.dime.glmax = max(c.img(:));
    c.hdr.dime.glmin = min(c.img(:));

    % save the output image
    if out_type == 2
        save_untouch_nii_gz(c, out_file);
    else
        save_untouch_nii(c, out_file);
    end
end

