function [estimate W theta] = COLLATE(varargin)
% COLLATE - performs the COLLATE statistical label fusion algorithm
%
% There are 4 forms:
% 1) [estimate W theta] = COLLATE(obs, epsilon, prior_flag, ...
%                                 init_flag, alphas, cvals);
% 2) [estimate W theta] = COLLATE(obs, epsilon, prior_flag, ...
%                                 init_flag, alpha);
% 3) [estimate W theta] = COLLATE(obs, epsilon, sv_prior, ...
%                                 init_flag, alphas, cvals);
% 4) [estimate W theta] = COLLATE(obs, epsilon, sv_prior, ...
%                                 init_flag, alpha);
%
% Input: obs - observation struct
%        epsilon - the convergence factor (e.g. 0.001)
%        prior_flag - the type of prior
%                   - 0 => global prior
%                   - 1 => voxelwise prior
%        sv_prior - An explicit spatially varying prior (X x Y x Z x L)
%        init_flag - the type of initialization
%                  - 0 => W then theta
%                  - 1 => theta then W
%        alphas - an array indicating the impact of the performance level
%                 parameters on the corresponding consensus level
%        alpha - a single value (indicating a single consensus level) that
%                indicates the relationship between STAPLE and majority vote
%        cvals - Threshold for estimating a voxel to be a consensus voxel
%
% Output: estimate - estimate of the true segmentation
%         W - the probability of the true segmentation for each voxel/label
%         theta - the performance level parameters
%

if nargin ~= 5 && nargin ~= 6
    error('Incorrect COLLATE usage -- too many input arguments');
end

obs = varargin{1};
epsilon = varargin{2};
prior = varargin{3};
init_flag = varargin{4};
alphas = varargin{5};

% create the java observation object
obsjava = create_java_obs(obs);

if nargin == 6
    cvals = varargin{6};

    % run the constructor on the staple class (initialize)
    C = edu.vanderbilt.masi.LabelFusion.COLLATE(obsjava, epsilon, prior, ...
                                                init_flag, alphas, cvals);
elseif nargin == 5
    % run the constructor on the staple class (initialize)
    C = edu.vanderbilt.masi.LabelFusion.COLLATE(obsjava, epsilon, prior, ...
                                                init_flag, alphas);
else
    error('Incorrect COLLATE usage -- too many input arguments');
end

% run the COLLATE algorithm
estimate = uint8(C.run());
W = C.get_W();
theta = C.get_theta();
