function [msd hd ul] = bidir_surface_distance(truth, estimate, res_dims)
% BIDIR_SURFACE_DISTANCE - returns the surface distance metrics between a true
%                          segmentation and an estimate
%
% [msd hd] = bidir_surface_distance(truth, estimate)
%
% Input: truth - the true segmentation
%        estimate - the estimated segmentation
%        res_dims - the resolution of each of the dimensions
%                   should be an array of the form [Rx Ry Rz]
%
% Output: msd - the mean surface distance
%         hd - the hausdorff distance (max distance)
%         ul - the unique labels corresponding to "msd" and "hd"

% set the unique labels from the truth
ul = unique(truth(:));
ul = ul(2:end);

% allocate some space
msd = zeros(size(ul));
hd = zeros(size(ul));

% iterate over all of the unique labels on the "truth"
for l = 1:length(ul)

    % convert the estimate/truth to binary images
    el = estimate == ul(l);
    if nnz(el) == 0
        msd(l) = Inf;
        hd(l) = Inf;
        continue;
    end
    tl = truth == ul(l);

    % crop the truth/estimate to make it a little faster
    cr = determine_cropping_region(el|tl, 1);
    tl = tl(cr(1):cr(2), cr(3):cr(4), cr(5):cr(6));
    el = el(cr(1):cr(2), cr(3):cr(4), cr(5):cr(6));

    % get the perimeter images
    perim1 = bwperim(tl);
    perim2 = bwperim(el);

    % get the number of perimeter voxels from the truth and perimeter
    num1 = nnz(perim1);
    num2 = nnz(perim2);

    % store the distance array
    dist1 = zeros([num1, 1]);
    dist2 = zeros([num2, 1]);

    % if isotropic, use an easy bwdist function
    if res_dims(1)==res_dims(2) && res_dims(1)==res_dims(3)

        % calculate the distance transform on both datasets
        distfunc1 = bwdist(perim1) * prod(res_dims);
        distfunc2 = bwdist(perim2) * prod(res_dims);

        % evaluate at the appropriate locations
        dist1(:) = distfunc2(perim1 > 0);
        dist2(:) = distfunc1(perim2 > 0);

    % else if anisotropic out-of-plane, use the bwdistX function
    elseif res_dims(1)==res_dims(2)

        % calculate the distance transform on both datasets
        distfunc1 = bwdistX(perim1, res_dims);
        distfunc2 = bwdistX(perim2, res_dims);

        % evaluate at the appropriate locations
        dist1(:) = distfunc2(perim1 > 0);
        dist2(:) = distfunc1(perim2 > 0);

    % else, anisotropic in plane, use the slowest version
    else

        % calculate the vertices of the perimeter voxels
        vert1 = zeros(num1, 3);
        vert2 = zeros(num2, 3);
        [vert1(:, 1), vert1(:, 2), vert1(:, 3)] = ind2sub(size(perim1), ...
                                                          find(perim1 > 0));
        [vert2(:, 1), vert2(:, 2), vert2(:, 3)] = ind2sub(size(perim2), ...
                                                          find(perim2 > 0));

        % allocate some temporary matrices to make it faster
        rd1 = repmat(res_dims, [num2 1]);
        om1 = ones(num2, 1);
        rd2 = repmat(res_dims, [num1 1]);
        om2 = ones(num1, 1);

        % calculate the distance from the estimate to the truth for each element
        for i = 1:num1
            dist1(i) = sqrt(min(sum((rd1 .* (vert2 - om1*vert1(i, :))).^2, 2)));
        end

        % calculate the distance from the truth to the estimate for each element
        for i = 1:num2
            dist2(i) = sqrt(min(sum((rd2 .* (vert1 - om2*vert2(i, :))).^2, 2)));
        end

    end

    % get the mean surface distance (msd)
    msd(l) = (mean(dist1) + mean(dist2)) / 2;

    % get the hausdorff distance (hd)
    hd(l) = max([max(dist1), max(dist2)]);

end

